package com.digiwin.athena.executionengine.service.facade.mapping.data;

import com.digiwin.athena.executionengine.constant.CommonConstant;
import com.digiwin.athena.executionengine.model.input.InputParamModule;
import com.digiwin.athena.executionengine.core.holder.ReadContextHolder;
import org.apache.commons.collections.CollectionUtils;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * @description: 实体数据的信息统计对象管理，存储不同的对象信息集合，并提供相关操作，
 * 存储dataCollection，每个Colloction都是一个response统计和分析结果。
 * <p>
 * 在运行时按照 调用方-数据源 的组合创建一个运行信息实例。该实例来自Colloction的深拷贝，
 * 每个运行实例称为pack。
 * @author: fenglei
 * @date: 2020-11-9
 */
public class DataMappingManager {
    /**
     * >>>> 只含有静态数据，对数据解析后，不会再修改。
     * 所有数据集，包括response、param、paras、sysParam
     * ---> 数据应该在第一次使用时被分析，而不是产生后分析。
     * String： response归属的actionid，如果是执行引擎入参则为ExecutionParam/ExecutionSysParam
     */
    private Map<String, HashMap<String, DataNode>> dataCollection;

    /**
     * >>>> 动态数据，运行期的数据版本，会在运行期修改各种pos，运行期拿的也是这个，未来和静态结构会拆开。
     * 不同action在pulling时，参数映射可能包含同一个数据源的映射信息，
     * 不同的action在映射过程中需要有自己独立的映射信息
     * key： requestActionId-reqonseActionId
     * value: 深拷贝自DataCollection
     */
    private Map<String, HashMap<String, DataNode>> dataPackCollection;

    /**
     * action返回值Map
     */
    private Map<String, Object> actionResponseMap = new HashMap<>();

    /**
     * 执行引擎入参数据
     */
    private InputParamModule inputParam;

    /**
     * 一一对应 dataPack
     * key： requestActionId-reqonseActionId
     */
    private Map<String, DataMappingHandler> handlerCollection;

    private ReadContextHolder readContextHolder;

    /**
     * 构造函数，用于执行引擎内部参数映射的构造方法，需要通过actionResponseMap，区别不同action调用场景的数据映射关系
     */
    public DataMappingManager(Map<String, Object> actionResponseMap, InputParamModule inputParam) {
        this.dataCollection = new ConcurrentHashMap<>();
        this.handlerCollection = new ConcurrentHashMap<>();
        this.dataPackCollection = new ConcurrentHashMap<>();
        this.actionResponseMap = actionResponseMap;
        this.inputParam = inputParam;
    }

    /**
     * 构造函数, 提供给外部，用于构建数据的元数据
     * 将传入带分析的数据模拟成执行引擎入参，
     */
    public DataMappingManager(Object data) {
        this.dataCollection = new ConcurrentHashMap<>();
        this.handlerCollection = new ConcurrentHashMap<>();
        this.dataPackCollection = new ConcurrentHashMap<>();
        this.inputParam = new InputParamModule();
        this.inputParam.setActionId(CommonConstant.EXECUTION_PARAM);
        if (data != null) {
            if (data instanceof Map) {
                this.inputParam.setParam((Map) data);
            } else if (data instanceof List) {
                this.inputParam.setParas((List) data);
            } else {
                ///暂时不考虑单字段传入的情形
            }
        }
        this.inputParam.setTenantId("");
        actionResponseMap.put(CommonConstant.EXECUTION_PARAM, data);
    }

    /**
     * 创建数据源
     *
     * @param actionId 数据源在Collection的key，
     *                 一般认为是一个actionid，但是入参可以认为是一个特殊的actionId="ExecutionParam"或者"ExecutionSysParam"
     */
    private HashMap<String, DataNode> addNewCollection(String actionId) {
        HashMap<String, DataNode> data = new HashMap<>();
        dataCollection.put(actionId, data);

        if (!dataCollection.containsKey(actionId)) {
            dataCollection.put(actionId, data);
        }

        return dataCollection.get(actionId);
    }

    /**
     * @param dataPackKey dataPackKey = requestActionId + "-" + reqonseActionId;
     * @param pack        HashMap<String, DataNode> pack 入参映射过程中的数据实例
     */
    private void createPackHandler(String dataPackKey, HashMap<String, DataNode> pack) {
        handlerCollection.put(dataPackKey, new DataMappingHandler(pack, readContextHolder));
    }

    public String getDataPackKey(String requestActionId, String responseActionId) {
        return requestActionId + "-" + responseActionId;
    }

    /**
     * 按照requestActionId 和 reqonseActionId 检查并创建对象，如果数据源未分析则先分析数据源。
     * 如果pack对象已经存在则直接返回
     *
     * @param requestActionId
     * @param responseActionId
     */
    private HashMap<String, DataNode> createPack(String requestActionId, String responseActionId) {
        synchronized (responseActionId) {
            if (!dataCollection.containsKey(responseActionId)) {

                //生成dataCollection
                generateCollection(responseActionId, addNewCollection(responseActionId));
            }
        }
        String dataPackKey = getDataPackKey(requestActionId, responseActionId);
        if (dataPackCollection.containsKey(dataPackKey)) {
            return dataPackCollection.get(dataPackKey);
        }

        //从dataCollection模板中新的pack副本，并放入dataPackCollection中管理
        HashMap<String, DataNode> pack = new HashMap<>();
        pack.putAll(dataCollection.get(responseActionId));
        dataPackCollection.put(dataPackKey, pack);

        return pack;
    }

    /**
     * 按照requestActionId 和 reqonseActionId 取映射信息对象
     *
     * @param requestActionId
     * @param responseActionId
     * @return
     */
    private HashMap<String, DataNode> getPack(String requestActionId, String responseActionId) {
        String dataPackKey = getDataPackKey(requestActionId, responseActionId);

        if (!dataPackCollection.containsKey(dataPackKey)) {
            //创建pack并设置packHandler
            createPackHandler(dataPackKey, createPack(requestActionId, responseActionId));
        }
        return dataPackCollection.get(dataPackKey);
    }

    /**
     * =>>>>>>>>>>>>>>>>>>>>>>> API !
     */
    public DataMappingHandler getDatMappingHandler(String requestActionId, String responseActionId) {
        if (getPack(requestActionId, responseActionId) == null) {
            ///抛错吧，取不到应该自己创建pack的
        }
        return handlerCollection.get(getDataPackKey(requestActionId, responseActionId));
    }

    /**
     * dataCollection
     *
     * @param reqonseActionId
     * @param data
     */
    private void generateCollection(String reqonseActionId, HashMap<String, DataNode> data) {

        Object responseData = null;
        if (CommonConstant.EXECUTION_PARAM.equals(reqonseActionId)) {
            responseData = this.inputParam.getParam() == null ? this.inputParam.getParas() : this.inputParam.getParam();
        } else if (CommonConstant.EXECUTION_SYSPARAM.equals(reqonseActionId)) {
            responseData = this.inputParam.getSysParam();
        } else {
            responseData = this.actionResponseMap.get(reqonseActionId);
        }
        if (responseData == null) {
            return;
        }

        DataMappingHandler handler = new DataMappingHandler(data, readContextHolder);
        handlerCollection.put(reqonseActionId, handler);

        boolean isRootNode = true;
        String nodeName = ".ROOT";
        String parentNodeId = null;
        int totalCnt = 1;
        String lastCollectionNode = null;
        boolean isArrayAsObject = false;
        String pathTemplate = "$";
        if (responseData instanceof List) {
            pathTemplate = pathTemplate + "[*]";
        }
        String paramPath = "$";


        ///重复的逻辑
        boolean isObject = false;
        boolean isArray = false;
        //判断并设置 isArray 和 array中含有Object A[]/A[{},{}...]
        if (responseData instanceof List) {
            isArray = true;
            isObject = checkElementIsObject((List) responseData);
        }
        //判断并设置 isObject A{}
        if (responseData instanceof Map) {
            isObject = true;
        }

        DataNode rootNode = handler.createNode(isRootNode, nodeName, parentNodeId, lastCollectionNode, totalCnt, isObject, isArray, isArrayAsObject, pathTemplate, paramPath);
        String concreteJsonPath = "$";

        ///设置当前路径下的该节点的实际数据量
        int zoneSize = handler.calcZoneSize(rootNode, responseData, reqonseActionId, concreteJsonPath);
        handler.setZone(concreteJsonPath, concreteJsonPath, rootNode, zoneSize);
        handler.appendTotalCnt(rootNode, zoneSize);

        analiyzeData(reqonseActionId, responseData, responseData, rootNode, concreteJsonPath);

        dataCollection.put(reqonseActionId, handler.getDataCollection());
    }

    /**
     * 检查集合里面的元素是否为对象
     *
     * @param responseData
     * @return
     */
    private boolean checkElementIsObject(List responseData) {
        if (CollectionUtils.isEmpty(responseData)) {
            return false;
        }
        for (Object element : responseData) {
            if (element == null) {
                continue;
            }
            if (element instanceof Map) {
                return true;
            }
        }
        return false;
    }

    //分析当前数据的结构

    /**
     * 递归分析数据结构，获得统计数据
     *
     * @param reqonseActionId
     * @param data               上一层解析后，剥离出来的结构
     * @param originData         数据原始的结构，来自paras或者是param或者是response
     * @param parentDataNode
     * @param parentConcretePath
     */
    private void analiyzeData(String reqonseActionId, Object data, Object originData, DataNode parentDataNode, String parentConcretePath) {

        if (!handlerCollection.containsKey(reqonseActionId)) {
            //不应该找不到，这里的调用应该是需要先创建root节点的，创建root一定需要一个handler！
            //报错
        }
        DataMappingHandler nodeHandler = handlerCollection.get(reqonseActionId);

        //结构就是先创建孩子节点，补全自己的信息，字段就是补全自己的信息
        if (data instanceof Map) {
            Map<String, Object> mapData = (Map) data;

            if (mapData.size() <= 0) {
                //set dataNode
                nodeHandler.setEmptyMapNode(parentDataNode);
                return;
            }

            String parentParamPath = nodeHandler.getParamPath(parentDataNode);

            for (Map.Entry<String, Object> subField : mapData.entrySet()) {
                //取当前的参数路径: A.B.C
                String subFieldParamPath = parentParamPath + "." + subField.getKey();

                DataNode dataNode = nodeHandler.getDataNodeByParamPath(subFieldParamPath);


                //如果没有该节点，创建！
                if (dataNode == null) {
                    dataNode = nodeHandler.createNode(subField.getKey(), subField.getValue(), parentDataNode, subFieldParamPath, this.actionResponseMap.get(reqonseActionId));
                }

                /**
                 * 更新DataNode的isObject信息
                 * 1.业务上会存在这种数据：多笔数据 的第一笔数据的某个节点是空集合（不能明确该节点是集合/对象）
                 * 2.datamapping 循环分析数据时，构建dataNode对象时候，如果该节点已存在，则不会重新创建新的节点(只有第一次会创建)
                 * 3.基于1和2 的前提，我们发现在基于第一笔数据创建DataNode的时候，认为该dataNode 的isArray = true ，isObject = false。而执行第二笔数据时
                 * 会依赖前面的创建的DataNode ，此时不会修改DataNode的 isObject属性。因此需要根据每一笔数据的实际情况 来更新DataNode的isObject
                 */
                updateIsObject(subField.getValue(), dataNode);
                //节点已存在，追加信息
                ///创建自己实例化jsonPath
                String concreteJsonPath = parentConcretePath + "." + subField.getKey();
                ///设置当前路径下的该节点的实际数据量
                int zoneSize = nodeHandler.calcZoneSize(dataNode, originData, reqonseActionId, concreteJsonPath);
                nodeHandler.setZone(concreteJsonPath, parentConcretePath, dataNode, zoneSize);
                nodeHandler.appendTotalCnt(dataNode, zoneSize);
                //更新父节点中的childNode
                nodeHandler.addChildNode(parentDataNode, dataNode);

                analiyzeData(reqonseActionId, subField.getValue(), originData, dataNode, concreteJsonPath);
            }

            // 更新父节点中maxSizeChildNodeId  这里有问题，上一层如果是list，那么这里的MaxSizeChild可能不是同一个，
            // 但是只提供了一个存储，会导致记录被覆盖, 暂时用不到，先注释掉
            ///nodeHandler.setMaxSizeChildNodeId(parentDataNode, parentConcretePath);
        } else if (data instanceof List) {

            List<Object> listData = (List) data;
            if (listData.size() <= 0) {
                //set dataNode
                return;
            }

            int listIndex = 0;
            for (Object subData : listData) {
                String concreteJsonPath = parentConcretePath + "[" + listIndex + "]";

                analiyzeData(reqonseActionId, subData, originData, parentDataNode, concreteJsonPath);
                listIndex++;
            }
        } else {
            //最简单的字段或者是简单list中的单值，这里好像没有什么要算的
        }

        return;
    }

    /**
     * 如果数据节点是数组 对象的结构，需要去更新DataNode节点的isObject信息
     *
     * @param jsonDataPart
     * @param dataNode
     */
    private void updateIsObject(Object jsonDataPart, DataNode dataNode) {
        if (dataNode.isArray() && dataNode.isObject()) {
            return;
        }
        //如果是array且是object 需要DataNode的 isObject 需要被更新
        if (jsonDataPart instanceof List) {
            List subCollection = (List) jsonDataPart;
            if (subCollection.size() > 0) {
                if (subCollection.get(0) instanceof Map) {
                    dataNode.setObject(true);
                }
            }
        }
    }

    public void setReadContextHolder(ReadContextHolder readContextHolder) {
        this.readContextHolder = readContextHolder;
    }

    public void release(String sourceKey, String currentKey) {
        String key = sourceKey + "-" + currentKey;
        handlerCollection.remove(key);
        handlerCollection.remove(currentKey);
        dataPackCollection.remove(key);
        dataCollection.remove(currentKey);
    }
}
