package com.digiwin.athena.executionengine.util;


import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.Period;
import java.time.YearMonth;
import java.time.format.DateTimeFormatter;
import java.time.temporal.TemporalAdjusters;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

/**
 * @description: 日期时间工具类
 * @author: zhangww
 * @date: 2020/4/28 15:26
 */
public class DateUtils {

    private static final String YYYY_MM_DD_HH_MM_SS = "yyyy-MM-dd HH:mm:ss";

    private static final DateTimeFormatter DATETIME_FMT_BY_DATE_AND_TIME = DateTimeFormatter.ofPattern(YYYY_MM_DD_HH_MM_SS);
    private static final DateTimeFormatter YMD = DateTimeFormatter.ofPattern("yyyy-MM-dd");

    private static final Logger LOGGER = LoggerFactory.getLogger(DateUtils.class);

    private static final String[] POSSIBLE_PATTERN = {"yyyy-MM-dd", "yyyy-MM", "yyyy-MM-dd HH:mm:ss", "yyyy-MM-dd HH:mm:ss.SSS", "yyyyMMdd", "yyyyMM", "yyyyMMddHHmmss", "yyyyMMddHHmmssSSS", "yyyy/MM/dd", "yyyy/MM", "yyyy/MM/dd HH:mm:ss", "yyyy/MM/dd HH:mm:ss.SSS", "yyyy.MM.dd HH:mm:ss", "yyyy.MM", "yyyy.MM.dd HH:mm:ss.SSS"};


    private DateUtils() {
    }

    /**
     * 时间转换 String类型转为LocalDateTime
     *
     * @param dateTime
     * @return
     */
    public static LocalDateTime parseUseDefaultPattern(String dateTime) {
        if (null == dateTime || dateTime.trim().isEmpty()) {
            return LocalDateTime.MIN;
        }
        return LocalDateTime.parse(dateTime, DATETIME_FMT_BY_DATE_AND_TIME);
    }

    /**
     * 获取当前时间,不包含毫秒
     *
     * @return
     */
    public static String getCurrentDateTime() {
        return LocalDateTime.now().format(DATETIME_FMT_BY_DATE_AND_TIME);
    }

    /**
     * 转换时间日期格式字串为long型
     *
     * @param time   日期
     * @param format 日期格式
     */
    public static long convertTimeToLong(String time, String format) {
        String formatTime = getFormatTime(time, format);
        try {
            SimpleDateFormat sdf = new SimpleDateFormat(format);
            return sdf.parse(formatTime).getTime();
        } catch (Exception e) {
            LOGGER.error("输入日期和给定的格式不匹配,日期:{}，格式:{}", time, format);
            return -1L;
        }
    }

    /**
     * 将数据转换成指定格式
     *
     * @param time
     * @param format
     * @return
     */
    private static String getFormatTime(String time, String format) {
        if (StringUtils.isEmpty(time) || StringUtils.isEmpty(format)) {
            return time;
        }
        if (time.length() > format.length()) {
            return time.substring(0, 10);
        } else if (time.length() < format.length()) {
            return time + " 00:00:00";
        }
        return time;

    }

    /**
     * 日期格式转换
     *
     * @param inTime
     * @return
     * @throws ParseException
     */
    public static String transferFormat(String inTime, String format) {
        SimpleDateFormat s1 = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        SimpleDateFormat s2 = new SimpleDateFormat(format);
        Date tempDate;
        String outTime = null;
        try {
            tempDate = s1.parse(inTime);
            outTime = s2.format(s2.parse(s1.format(tempDate)));
        } catch (ParseException e) {
            LOGGER.error("输入日期和给定的格式不匹配,日期:{}，格式:{}", inTime, format);
        }
        return outTime;
    }

    /**
     * 根据format获取当前时间
     *
     * @param format
     * @return
     */
    public static String getCurrentDateTime(String format) {
        return LocalDateTime.now().format(DateTimeFormatter.ofPattern(format));
    }

    public static int getWeekOfYear(String dateString) throws ParseException {
        SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
        Date date = format.parse(dateString);
        Calendar calendar = Calendar.getInstance();
        calendar.setFirstDayOfWeek(Calendar.MONDAY);
        calendar.setTime(date);
        return calendar.get(Calendar.WEEK_OF_YEAR);
    }

    public static int getWeekOfYear(String dateString, String formatString) throws ParseException {
        SimpleDateFormat format = new SimpleDateFormat(formatString);
        Date date = format.parse(dateString);
        Calendar calendar = Calendar.getInstance();
        calendar.setFirstDayOfWeek(Calendar.MONDAY);
        calendar.setTime(date);
        return calendar.get(Calendar.WEEK_OF_YEAR);
    }

    /**
     * 计算两个日期之间的相对天数，不足一天的按天向下取整。
     *
     * @param formatA
     * @param formatB
     * @param fieldA
     * @param fieldB
     * @return
     * @throws ParseException
     */
    public static BigDecimal getDateDiffInDays(String formatA, String formatB, String fieldA, String fieldB) throws ParseException {
        SimpleDateFormat dateFormatA = new SimpleDateFormat(formatA);
        Date dateA = dateFormatA.parse(fieldA);
        SimpleDateFormat dateFormatB = new SimpleDateFormat(formatB);
        Date dateB = dateFormatB.parse(fieldB);

        BigDecimal millionSeconds = new BigDecimal(dateA.getTime() - dateB.getTime());
        // 86400000 = 1000*60*60*24
        BigDecimal millionSecondsOneDay = new BigDecimal(86400000);
        return millionSeconds.divide(millionSecondsOneDay, 0, RoundingMode.CEILING);
    }

    /**
     * 传入时间,获取到传入时间所在月的第一天,默认时间格式为：yyyy-MM-dd hh:mm:ss
     * 使用 Java8,中的 LocalDateTime 类
     */
    public static String getFirstDayOfMonth(String pattern, String firstDay) {
        if (StringUtils.isBlank(pattern)) {
            pattern = YYYY_MM_DD_HH_MM_SS;
        }
        firstDay = firstDay + " 00:00:00";
        DateTimeFormatter dateFormatter = DateTimeFormatter.ofPattern(pattern);

        LocalDateTime firstDateTime = DateUtils.parseUseDefaultPattern(firstDay);
        return firstDateTime.with(TemporalAdjusters.firstDayOfMonth()).withHour(0).
                withMinute(0).withSecond(0).format(dateFormatter);
    }

    /**
     * 获取传入时间所在月的最后一天,默认时间格式为：yyyy-MM-dd hh:mm:ss
     */
    public static String getLastDayOfMonth(String pattern, String lastDay) {
        if (StringUtils.isBlank(pattern)) {
            pattern = YYYY_MM_DD_HH_MM_SS;
        }
        lastDay = lastDay + " 23:59:59";
        DateTimeFormatter dateFormatter = DateTimeFormatter.ofPattern(pattern);

        LocalDateTime lastDateTime = DateUtils.parseUseDefaultPattern(lastDay);
        return lastDateTime.with(TemporalAdjusters.lastDayOfMonth()).withHour(23)
                .withMinute(59).withSecond(59).format(dateFormatter);
    }

    /**
     * 获取当前时间上个月的第一天,默认时间格式为：yyyy-MM-dd hh:mm:ss
     *
     * @param pattern
     * @return
     */
    public static String getfirstDayOfLastMonth(String pattern) {
        if (StringUtils.isBlank(pattern)) {
            pattern = YYYY_MM_DD_HH_MM_SS;
        }
        // 获取当前时间
        LocalDate currentDate = LocalDate.now();

        // 计算上个月的时间
        LocalDate lastMonthDate = currentDate.minusMonths(1);

        return getFirstDayOfMonth(pattern, lastMonthDate.format(YMD));
    }

    /**
     * 获取当前时间上一年本月的第一天,默认时间格式为：yyyy-MM-dd hh:mm:ss
     *
     * @param pattern
     * @return
     */
    public static String firstDayOfMonthYearAgo(String pattern) {
        if (StringUtils.isBlank(pattern)) {
            pattern = YYYY_MM_DD_HH_MM_SS;
        }
        // 获取当前时间
        LocalDate currentDate = LocalDate.now();

        // 计算上一年的时间
        LocalDate lastMonthDate = currentDate.minusYears(1);

        return getFirstDayOfMonth(pattern, lastMonthDate.format(YMD));
    }

    /**
     * 获取当前时间上个月的最后一天,默认时间格式为：yyyy-MM-dd hh:mm:ss
     *
     * @param pattern
     * @return
     */
    public static String getLastDayOfLastMonth(String pattern) {
        if (StringUtils.isBlank(pattern)) {
            pattern = YYYY_MM_DD_HH_MM_SS;
        }
        // 获取当前时间
        LocalDate currentDate = LocalDate.now();

        // 计算上个月的时间
        LocalDate lastMonthDate = currentDate.minusMonths(1);

        return getLastDayOfMonth(pattern, lastMonthDate.format(YMD));
    }

    /**
     * 获取2个日期间的时间列表
     *
     * @param startTime
     * @param endTime
     * @param pattern
     * @return
     */
    public static Set<String> getTimeListByBetweenTime(String startTime, String endTime, String pattern) {
        LocalDate startDate = LocalDate.parse(startTime);
        LocalDate endDate = LocalDate.parse(endTime);

        Set<String> monthList = new HashSet<>();
        DateTimeFormatter monthFormatter = DateTimeFormatter.ofPattern(pattern);
        while (!startDate.isAfter(endDate)) {
            String monthString = startDate.format(monthFormatter);
            monthList.add(monthString);
            startDate = startDate.plusMonths(1);
        }

        return monthList;

    }

    /**
     * 分析数据的日期格式
     *
     * @param date
     * @return
     */
    public static String analyseFormat(String date) {
        String patternMatch = null;
        for (String pattern : POSSIBLE_PATTERN) {
            SimpleDateFormat sdf = new SimpleDateFormat(pattern);
            sdf.setLenient(false);  // 设置严格模式，不自动修正日期
            try {
                sdf.parse(date);
                patternMatch = pattern;
                break;
            } catch (ParseException e) {
                // Do nothing, continue to the next format check
            }
        }
        return patternMatch;
    }

    public static String convertDateFormat(String inputDate, String inputFormat, String outputFormat) {
        SimpleDateFormat inputDateFormat = new SimpleDateFormat(inputFormat);
        SimpleDateFormat outputDateFormat = new SimpleDateFormat(outputFormat);

        try {
            Date date = inputDateFormat.parse(inputDate);
            return outputDateFormat.format(date);
        } catch (ParseException e) {
            return null;
        }
    }

    public static String firstDayOfThisYear() {
        LocalDateTime firstDay = LocalDateTime.now().withDayOfYear(1).withHour(0).withMinute(0).withSecond(0).withNano(0);
        return firstDay.format(DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss"));
    }

    public static String lastDayOfThisYear() {
        LocalDateTime lastDay = LocalDateTime.now().withDayOfYear(1).withMonth(12).withDayOfMonth(31)
                .withHour(23).withMinute(59).withSecond(59).withNano(999999999);
        return lastDay.format(DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss"));
    }

    public static String firstDayOfThisMonth() {
        LocalDateTime firstDay = LocalDateTime.now().withDayOfMonth(1).withHour(0).withMinute(0).withSecond(0).withNano(0);
        return firstDay.format(DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss"));
    }

    public static String lastDayOfThisMonth() {
        LocalDateTime now = LocalDateTime.now();

        // 提取当前的年月
        YearMonth currentMonth = YearMonth.from(now);

        // 获取当前月份的最后一天
        LocalDate lastDayOfMonth = currentMonth.atEndOfMonth();

        // 将日期转换为 LocalDateTime，并设置时间为 23:59:59
        LocalDateTime lastDayOfMonthAtEndOfDay = lastDayOfMonth.atTime(LocalTime.MAX);
        return lastDayOfMonthAtEndOfDay.format(DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss"));
    }
}
