package com.digiwin.athena.executionengine.core.aop;

import com.digiwin.athena.executionengine.exception.ResponseStatusException;
import com.digiwin.athena.executionengine.util.JsonUtil;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.aspectj.lang.reflect.MethodSignature;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import java.lang.reflect.Method;
import java.util.UUID;

/**
 * 异常捕获重试切面
 */
@Aspect
@Component("engineExceptionRetryAspect")
public class ExceptionRetryAspect {

    private static final Logger LOGGER = LoggerFactory.getLogger(ExceptionRetryAspect.class);

    @Pointcut("@annotation(com.digiwin.athena.executionengine.core.aop.ExceptionRetry)")
    public void retryPointCut() {
    }

    @Around("retryPointCut()")
    public Object around(ProceedingJoinPoint joinPoint) throws Throwable {
        MethodSignature methodSignature = (MethodSignature) joinPoint.getSignature();
        Method method = methodSignature.getMethod();
        ExceptionRetry retry = method.getAnnotation(ExceptionRetry.class);
        String name = method.getName();
        Object[] args = joinPoint.getArgs();
        String uuid = UUID.randomUUID().toString();
        LOGGER.info("执行重试切面{}, 方法名称{}, 方法参数{}", uuid, name, JsonUtil.getJsonString(args));
        int times = retry.times();
        long waitTime = retry.waitTime();
        // check param
        if (times <= 0) {
            times = 1;
        }

        for (; times >= 0; times--) {
            try {
                return joinPoint.proceed();
            } catch (Exception e) {
                //不是ResponseStatusException异常直接向上抛出
                if (e.getClass() != ResponseStatusException.class) {
                    throw e;
                }
                LOGGER.info("异常报文:{}", e.getMessage());
                //判断异常内容是否为502,非502继续向上抛出
                if (e.getMessage().indexOf("502 Bad Gateway") == -1) {
                    throw e;
                }

                // 如果接下来没有重试机会的话，直接报错
                if (times <= 0) {
                    LOGGER.warn("执行重试切面{}失败", uuid);
                    throw e;
                }

                // 休眠 等待下次执行
                if (waitTime > 0) {
                    Thread.sleep(waitTime);
                }

                LOGGER.info("执行重试切面{}, 还有{}次重试机会, 异常类型{}, 异常信息{}, 栈信息{}", uuid, times, e.getClass().getName(), e.getMessage(), e.getStackTrace());
            }
        }
        return false;
    }
}