package com.digiwin.athena.executionengine.service.facade.execution.impl;

import com.alibaba.fastjson.JSONObject;
import com.digiwin.app.service.DWServiceContext;
import com.digiwin.athena.executionengine.component.action.ActionBase;
import com.digiwin.athena.executionengine.component.domain.ActionParam;
import com.digiwin.athena.executionengine.constant.AgileDataErrorCodeConstant;
import com.digiwin.athena.executionengine.constant.CommonConstant;
import com.digiwin.athena.executionengine.constant.FieldNameConstant;
import com.digiwin.athena.executionengine.constant.LogConstant;
import com.digiwin.athena.executionengine.core.container.ExecuteContext;
import com.digiwin.athena.executionengine.dto.LogDataDto;
import com.digiwin.athena.executionengine.dto.LogDto;
import com.digiwin.athena.executionengine.dto.runinfo.DebugDto;
import com.digiwin.athena.executionengine.dto.runinfo.MetricExecutionDto;
import com.digiwin.athena.executionengine.enumtype.ActionTypeEnum;
import com.digiwin.athena.executionengine.enumtype.ErrorCodeEnum;
import com.digiwin.athena.executionengine.exception.BusinessException;
import com.digiwin.athena.executionengine.model.input.InputParamModule;
import com.digiwin.athena.executionengine.model.reporter.ErrorLogReportDto;
import com.digiwin.athena.executionengine.service.client.IThemeMapClient;
import com.digiwin.athena.executionengine.service.facade.execution.IExecutionFacade;
import com.digiwin.athena.executionengine.service.facade.mapping.data.DataMappingHandler;
import com.digiwin.athena.executionengine.service.facade.mapping.data.DataMappingManager;
import com.digiwin.athena.executionengine.service.facade.mapping.data.MetaGenerator;
import com.digiwin.athena.executionengine.service.facade.reporter.IReporter;
import com.digiwin.athena.executionengine.service.facade.router.IDataAnalyzerRouterFacade;
import com.digiwin.athena.executionengine.util.ContextUtils;
import com.digiwin.athena.executionengine.util.ExceptionUtils;
import com.digiwin.athena.executionengine.util.LogUtils;
import com.digiwin.athena.executionengine.util.ThreadPoolUtils;
import com.digiwin.athena.executionengine.vo.MetricResultVo;
import com.google.common.base.Stopwatch;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.ThreadContext;
import org.apache.lucene.util.RamUsageEstimator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;

import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * @description:
 * @author: ZhangJun
 * @create: 2022/5/5
 */
@Service
public class ExecutionFacade implements IExecutionFacade {
    private static final Logger LOGGER = LoggerFactory.getLogger(ExecutionFacade.class);

    //指标action前缀
    public static final String METRIC_ACTION_PREFIX = "METRIC_";

    private IDataAnalyzerRouterFacade dataAnalyzerRouterFacade;

    private IThemeMapClient themeMapClient;

    private IReporter reporter;

    public ExecutionFacade(
        IDataAnalyzerRouterFacade dataAnalysisRouterFacade,
        @Qualifier("engineThemeMapClient") IThemeMapClient themeMapClient,
        @Qualifier("engineErrorLogReporter") IReporter reporter) {
        this.dataAnalyzerRouterFacade = dataAnalysisRouterFacade;
        this.themeMapClient = themeMapClient;
        this.reporter = reporter;
    }

    @Override
    public Object execute(InputParamModule inputParam) {
        LOGGER.info(new LogDto("执行引擎接收请求,开始执行").toString());
        LOGGER.info("执行引擎接收的参数:{} \ntoken: {} ", inputParam, DWServiceContext.getContext().getToken());
        String source = (String) DWServiceContext.getContext().getRequestHeader().get(FieldNameConstant.SOURCE);
        if (StringUtils.isNotEmpty(source)) {
            LOGGER.error(String.format(LogConstant.AGILE_DATA + "pulling入参:%s", source, inputParam.toString()));
        }
        long startTime = System.currentTimeMillis();
        ExecuteContext context = prepare(inputParam);

        ActionDto actionDto = getRootAction(inputParam.getActionId(), context);

        if (null == actionDto.getActionBase()) {
            throw ExceptionUtils.buildBusinessException(ErrorCodeEnum.ACTION_NOT_FOUND, inputParam.getActionId());
        }
        Object obj = doExecuteAction(actionDto.getActionBase(), context, actionDto.getActionParam());
        long timeTaken = (System.currentTimeMillis() - startTime);
        LOGGER.info("执行引擎duringExecute action:{}总耗时 :{}ms", inputParam.getActionId(), timeTaken);
        if (StringUtils.isNotEmpty(source)) {
            LOGGER.error(String.format(LogConstant.AGILE_DATA + "pulling出参:不输出数据,执行耗时:%sms", source, timeTaken));
        }
        LOGGER.info("执行完，上下文大小为:{}M", RamUsageEstimator.sizeOf(context) / 1048576);
        LOGGER.info("执行完，上下文大小actionResponse为:{}M", RamUsageEstimator.sizeOf(context.getActionResponse()) / 1048576);
        /**
         * debug模式，且是指标查询，需要将debug信息返回
         */
        if (context.isDebug() && actionDto.isMetric()) {
            Map<String, Object> map = (Map<String, Object>) obj;
            map.put("runInfo", context.getRunInfo());
            return map;
        }
        return obj;
    }

    @Override
    public MetricResultVo query(InputParamModule inputParam) {
        LOGGER.info(String.format(LogConstant.AGILE_DATA + "接收的参数:%s \nToken:%s", "executionEngine", JSONObject.toJSONString(inputParam), DWServiceContext.getContext().getToken()));

        String source = (String) DWServiceContext.getContext().getRequestHeader().get(FieldNameConstant.SOURCE);
        if (StringUtils.isNotEmpty(source)) {
            LOGGER.error(String.format(LogConstant.AGILE_DATA + "pulling入参:%s", source, inputParam.toString()));
        }
        long startTime = System.currentTimeMillis();
        ExecuteContext context = prepare(inputParam);

        ActionDto actionDto = getRootAction(inputParam.getActionId(), context);

        if (null == actionDto.getActionBase()) {
            throw ExceptionUtils.buildBusinessException(ErrorCodeEnum.ACTION_NOT_FOUND, inputParam.getActionId());
        }
        Object obj = doExecuteAction(actionDto.getActionBase(), context, actionDto.getActionParam());
        long timeTaken = (System.currentTimeMillis() - startTime);
        LOGGER.info("指标取数duringExecute action:{}总耗时 :{}ms", inputParam.getActionId(), timeTaken);
        LogUtils.buildAgileLog(LogConstant.AGILE_CODE_PULL_DATA, LogUtils.SUCCESS, inputParam.getActionId(),"-", "-");
        if (StringUtils.isNotEmpty(source)) {
            LOGGER.error(String.format(LogConstant.AGILE_DATA + "pulling出参:不输出数据,执行耗时:%sms", source, timeTaken));
        }
        MetricExecutionDto metricExecutionDto = context.getRunInfo().getMetricExecutionDto();
        metricExecutionDto.setMetricId(inputParam.getActionId());
        String uuid = UUID.randomUUID().toString();
        metricExecutionDto.setUuid(uuid);
        context.getRunInfo().getActionExecutionMap().put(uuid, context.getRunInfo().getActionExecutionDtos());
        DebugDto debugDto = new DebugDto(context.getRunInfo().getActionExecutionMap(), metricExecutionDto);
        LOGGER.info(String.format(LogConstant.AGILE_DATA + "接收的出参:%s \nToken:%s", "executionEngine", JSONObject.toJSONString(obj), DWServiceContext.getContext().getToken()));
        return new MetricResultVo(obj, debugDto);
    }

    private ActionDto getRootAction(String actionId, ExecuteContext context) {

        if (actionId.startsWith(METRIC_ACTION_PREFIX)) {
            List<ActionParam> collect = context.getActionParamMap().entrySet().stream()
                    .filter(e -> ActionTypeEnum.PULL_DATA.getType().equalsIgnoreCase(e.getValue().getActionType()))
                    .map(e -> e.getValue()).collect(Collectors.toList());
            ActionBase actionBase = ContextUtils.getBean(collect.get(0).getActionName(), ActionBase.class);
            ActionDto actionDto = new ActionDto(actionBase, collect.get(0));
            actionDto.setMetric(true);
            return actionDto;
        } else {
            ActionBase executeAction = Optional.ofNullable(context.getActionParam(actionId)).map(
                    ap -> ContextUtils.getBean(ap.getActionName(), ActionBase.class)).orElse(null);
            return new ActionDto(executeAction, context.getActionParam(actionId));
        }

    }

    private ExecuteContext prepare(InputParamModule inputParam) {

        if (StringUtils.isBlank(inputParam.getTenantId())) {
            throw new BusinessException(ErrorCodeEnum.TENANT_OR_ACTION_EMPTY.getCode(), ErrorCodeEnum.TENANT_OR_ACTION_EMPTY.getMessage());
        }

        //处理后续接口调用需要传递的routerKey
        //规范要求下发请求在head中区分routerKey大小写，确保K8S能够识别和路由，但由于tomcat在接收请求时，会自动小写，解析时需要解析routerkey
        Object routerKey = DWServiceContext.getContext().getRequestHeader().get(FieldNameConstant.CAMEL_CASE_ROUTER_KEY);
        if (routerKey == null || StringUtils.isEmpty(String.valueOf(routerKey))) {
            throw new BusinessException(ErrorCodeEnum.ROUTER_KEY_EMPTY.getCode(), ErrorCodeEnum.ROUTER_KEY_EMPTY.getMessage());
        }

        String token = DWServiceContext.getContext().getToken();

        if (StringUtils.isBlank(inputParam.getActionId())) {
            throw new BusinessException(ErrorCodeEnum.TENANT_OR_ACTION_EMPTY.getCode(), ErrorCodeEnum.TENANT_OR_ACTION_EMPTY.getMessage());
        }

        //解析结构
        ExecuteContext context = dataAnalyzerRouterFacade.initExecuteContext(inputParam, token);
        context.setRouterKey((String) routerKey);
        //分析入参
        dataAnalyzerRouterFacade.analyseInputParam(context);

        Stopwatch stopwatch = Stopwatch.createStarted();
        JSONObject executionRule = themeMapClient.getExecutionRule(inputParam.getActionId(), inputParam.getTenantId(), context.getInputParamKeys(), context.getSysParamKeys(), token);
        stopwatch.stop();
        LOGGER.info("请求ThemeMap 获取 action[{}]信息耗时:{}毫秒", inputParam.getActionId(), stopwatch.elapsed(TimeUnit.MILLISECONDS));

        stopwatch.reset().start();

        if (MapUtils.isEmpty(executionRule)) {
            throw ExceptionUtils.buildBusinessException(ErrorCodeEnum.METADATA_IS_EMPTY, inputParam.getActionId());
        }


        try {
            //分析行动逻辑图谱
            dataAnalyzerRouterFacade.analyseThemeMapData(context, executionRule);
            LogUtils.buildAgileLog(LogConstant.AGILE_CODE_ANALYSE_EXECUTION_RULE, LogUtils.SUCCESS, executionRule.toJSONString(), "", "");
        } catch (Exception e) {
            LogUtils.buildAgileLog(LogConstant.AGILE_CODE_ANALYSE_EXECUTION_RULE, AgileDataErrorCodeConstant.ANALYSE_EXECUTION_RULE_ERROR, executionRule.toJSONString(), "数据流配置解析异常，可能的原因是：MICTRANS节点配置与对应模板配置不一致，失败原因：【" + e.getMessage() + "】：", "请联系系统管理员");
        }

        LOGGER.info("执行引擎解析ThemeMap action[{}]结构耗时:{}毫秒", inputParam.getActionId(), stopwatch.elapsed(TimeUnit.MILLISECONDS));
        return context;

    }

    /**
     * @param executeAction
     * @param context
     * @param actionParam
     * @return
     * @throws Exception
     */
    private Object doExecuteAction(ActionBase executeAction, ExecuteContext context, ActionParam actionParam) {
        boolean executeState = executeAction.execute(context, actionParam);
        if (!executeState) {
            if (context.getErrorLog() != null) {
                doReportErrorLog(context);
                String source = (String) DWServiceContext.getContext().getRequestHeader().get(FieldNameConstant.SOURCE);
                LOGGER.error(String.format(LogConstant.AGILE_DATA + "pulling出参:%s", source, context.getErrorLog().toString()));
                //LogUtils.buildAgileLog(LogConstant.AGILE_CODE_PULL_DATA, ErrorCodeEnum.ACTION_EXECUTE_FAIL.getCode(), "指标id：" + actionParam.getActionId(), "PULLING取数异常，失败原因：【"+context.getErrorLog().getErrorMessage()+"】", "依据不同的错误状况先自行处理，如问题仍未解决，请联系系统管理员");
                throw new BusinessException(context.getErrorLog().getErrorCode(), context.getErrorLog().getErrorMessage());
            }

            LogUtils.buildAgileLog(LogConstant.AGILE_CODE_PULL_DATA, ErrorCodeEnum.ACTION_EXECUTE_FAIL.getCode(), "指标id：" + actionParam.getActionId(), "PULLING取数异常，失败原因：【程序运行时异常】", "依据不同的错误状况先自行处理，如问题仍未解决，请联系系统管理员");

            LOGGER.error(new LogDto("执行引擎执行失败",
                    Arrays.asList(new LogDataDto(actionParam.getActionId(), "actionId", LogConstant.TYPE_CONFIG, LogConstant.LABEL_ACTION_ID))).toString());
            if(Objects.nonNull(context.getExecuteErrorMessage())){
                throw new BusinessException(context.getExecuteErrorMessage().getErrorCode(),context.getExecuteErrorMessage().getErrorMsg());
            }else {
                throw ExceptionUtils.buildBusinessException(ErrorCodeEnum.ACTION_EXECUTE_FAIL, actionParam.getActionId());
            }
        }
        return context.getActionResponse().get(actionParam.getActionId());
    }

    private void doReportErrorLog(ExecuteContext context) {
        CompletableFuture.runAsync(() -> {
            ErrorLogReportDto errorLogReportDto = new ErrorLogReportDto();
            errorLogReportDto.setAppCode(CommonConstant.APP_CODE);
            errorLogReportDto.setAppName(CommonConstant.APP_NAME);
            errorLogReportDto.setMessageId(context.getMessageId());
            errorLogReportDto.setPixBackendId(ThreadContext.get(CommonConstant.PTX_ID));
            errorLogReportDto.setDescription(context.getErrorLog().getErrorDescription());
            errorLogReportDto.setContent(context.getErrorLog().getErrorMessage());
            errorLogReportDto.setExtend(context.getErrorLog().getPossibleCausesAndGuidance());
            errorLogReportDto.setToken(context.getToken());
            errorLogReportDto.setRouterKey(context.getRouterKey());
            reporter.report(errorLogReportDto);
        }, ThreadPoolUtils.getExecutorService());

    }

    @Override
    public Object generateMetaHandler(DataMappingManager dataMappingMgr, String actionId, String tenantId) {
        DataMappingHandler dataMappingHandler = dataMappingMgr.getDatMappingHandler("META", CommonConstant.EXECUTION_PARAM);

        String locale = (String) DWServiceContext.getContext().getRequestHeader().get(FieldNameConstant.LOCALE);
        if (dataMappingHandler == null) {
            LOGGER.error("构建数据的dataMap结构失败");
        }
        MetaGenerator metaGenerator = new MetaGenerator(dataMappingHandler, actionId);

        JSONObject newMeta = metaGenerator.generate();
        if (StringUtils.isBlank(actionId)) {
            return newMeta;
        }

        //根据action取action的response结构
        JSONObject uiMeta = themeMapClient.getUIMetaData(actionId,
                DWServiceContext.getContext().getToken(), tenantId, locale);

        //用meta逐一分析response元数据字段，补充属性
        newMeta = metaGenerator.merge(newMeta, uiMeta);

        return newMeta;
    }

    @Override
    public Object verify(InputParamModule inputParam) {
        ExecuteContext context = prepare(inputParam);
        //校验时，不需要释放资源
        context.setRelease(false);
        ActionBase executeAction = Optional.ofNullable(context.getActionParam(inputParam.getActionId())).map(
                ap -> ContextUtils.getBean(ap.getActionName(), ActionBase.class)).orElse(null);
        executeAction.execute(context, context.getActionParam(inputParam.getActionId()));
        return context.getActionResponse();
    }
}

class ActionDto {
    private ActionBase actionBase;
    private ActionParam actionParam;
    private boolean isMetric = false;

    public boolean isMetric() {
        return isMetric;
    }

    public void setMetric(boolean metric) {
        isMetric = metric;
    }

    public ActionBase getActionBase() {
        return actionBase;
    }

    public void setActionBase(ActionBase actionBase) {
        this.actionBase = actionBase;
    }

    public ActionParam getActionParam() {
        return actionParam;
    }

    public void setActionParam(ActionParam actionParam) {
        this.actionParam = actionParam;
    }

    public ActionDto(ActionBase actionBase, ActionParam actionParam) {
        this.actionBase = actionBase;
        this.actionParam = actionParam;
    }
}