package com.digiwin.athena.executionengine.service.facade.analyzer.impl;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.digiwin.athena.executionengine.component.domain.ActionParam;
import com.digiwin.athena.executionengine.constant.CommonConstant;
import com.digiwin.athena.executionengine.constant.MetaDataConstant;
import com.digiwin.athena.executionengine.constant.TransConstant;
import com.digiwin.athena.executionengine.enumtype.ActionTypeEnum;
import com.digiwin.athena.executionengine.enumtype.ValueTypeEnum;
import com.digiwin.athena.executionengine.model.ParamElement;
import com.digiwin.athena.executionengine.service.facade.analyzer.AbstractAnalyzerBase;
import com.digiwin.athena.executionengine.service.facade.template.ITransSchemaCreator;
import com.digiwin.athena.executionengine.util.AnalysisUtils;
import com.digiwin.athena.executionengine.util.ReplaceUtils;
import com.google.common.collect.Lists;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.*;

/**
 * @description: Metadata 数据结构分析
 * @author: renwm
 * @date: 2020/6/15 13:31
 */
@Service("engineMetadataAnalyzer")
public class MetadataAnalyzer extends AbstractAnalyzerBase {

    private static final Logger LOGGER = LoggerFactory.getLogger(MetadataAnalyzer.class);

    @Autowired
    private ITransSchemaCreator transSchemaCreator;

    /**
     * 分析处理
     *
     * @param executionRule
     */
    @Override
    protected void analysisHandler(JSONObject executionRule) {
        if (MapUtils.isEmpty(executionRule)) {
            return;
        }
        //获取结构中的dataMetaDatas结构数据
        JSONArray metadatas = executionRule.getJSONArray(MetaDataConstant.DATA_ACTION_METADATAS);
        JSONObject standardParameterMapping = executionRule.getJSONObject(MetaDataConstant.STANDARD_PARAMETER_MAPPING);

        if (CollectionUtils.isEmpty(metadatas)) {
            LOGGER.warn("没有需要分析的metadatas");
        }
        //分析metadatas中的参数
        analysisActionMetaParam(metadatas, standardParameterMapping);
    }

    /**
     * 分析元数据参数
     *
     * @param metadatas
     * @param standardParameterMapping
     */
    private void analysisActionMetaParam(JSONArray metadatas, JSONObject standardParameterMapping) {

        //循环metadatas中的结构对象，解析action和对应的request 和response对象
        for (int i = 0; i < metadatas.size(); i++) {
            JSONObject metaDataObj = metadatas.getJSONObject(i);

            String actionType = metaDataObj.getString(MetaDataConstant.ACTION_TYPE);
            ActionTypeEnum typeEnum = ActionTypeEnum.getEnumByType(actionType);
            String actionId = metaDataObj.getString(MetaDataConstant.ACTION_ACTION_ID);

            ActionParam actionParam = new ActionParam(actionId, typeEnum.getClassName(), actionType, metaDataObj);
            actionParam.setStandardParameterMapping(standardParameterMapping);
            if (metaDataObj.containsKey(MetaDataConstant.ACTION_TYPE)
                    && "MICRO_TRANS".equals(metaDataObj.get(MetaDataConstant.ACTION_TYPE).toString())) {
                JSONArray microTransList = metaDataObj.getJSONArray(MetaDataConstant.ACTION_MICRO_TRANS);
                JSONObject microTransSchema = createTransSchema(microTransList);
                actionParam.setMicroTrans(microTransSchema);
            }
            if (metaDataObj.containsKey(MetaDataConstant.ACTION_API_META)) {
                JSONArray apiMeta = metaDataObj.getJSONArray(MetaDataConstant.ACTION_API_META);
                List<ParamElement> paramElements = analyseApiMeta(actionId, apiMeta);
                actionParam.setApiMeta(paramElements);
            }


            //解析querySchema
            JSONObject querySchema = metaDataObj.getJSONObject(MetaDataConstant.METADATAS_ACTION_QUERY_SCHEMA);
            if (querySchema != null) {
                actionParam.setQuerySchema(querySchema);
            }

            JSONObject aggregation = metaDataObj.getJSONObject(MetaDataConstant.METADATAS_ACTION_AGGREGATION);
            if (aggregation != null) {
                actionParam.setAggregation(aggregation);
            }

            getLocalContext().addActionParam(actionParam);
            //解析action对象中的request和response
            analysisReqMetaDatas(metaDataObj, actionParam);
            analysisRespMetaDatas(metaDataObj);
        }
    }

    private List<ParamElement> analyseApiMeta(String actionId, JSONArray apiMeta) {
        List<ParamElement> paramElements = new ArrayList<>();
        for (int i = 0; i < apiMeta.size(); i++) {
            JSONObject param = apiMeta.getJSONObject(i);
            analysisParameter(param, actionId, null, null);
            //action.参数名称作为 参数的路径，后面用来查找到对应的参数对象ParamElement
            String path = AnalysisUtils.concatByDot(actionId, "$", param.getString(MetaDataConstant.METADATAS_PARAM_DATA_NAME));
            //添加入参父节点到action中，后面拼接入参的时候 从父节点开始拼接（父->子的关系 拼接所有入参的结构和节点）
            ParamElement element = getLocalContext().getParamElement(path);
            paramElements.add(element);
        }
        return paramElements;
    }

    /**
     * 分析元数据入参
     *
     * @param metadata
     * @param actionParam
     */
    private void analysisReqMetaDatas(JSONObject metadata, ActionParam actionParam) {
        //获取actionId
        String actionId = metadata.getString(MetaDataConstant.ACTION_ACTION_ID);
        //获取元数据中定义的 request结构
        JSONObject requestObj = metadata.getJSONObject(MetaDataConstant.METADATAS_ACTION_REQUEST);
        //获取request结构中的parameters结构
        JSONArray parameters = requestObj.getJSONArray(MetaDataConstant.METADATAS_ACTION_REQUEST_PARAMETERS);
        if (CollectionUtils.isEmpty(parameters)) {
            return;
        }

        //循环request中的参数对象描述，并解析，将解析结果添加到上下文中
        for (int i = 0; i < parameters.size(); i++) {
            JSONObject param = parameters.getJSONObject(i);
            analysisParameter(param, actionId, null, null);
            //action.参数名称作为 参数的路径，后面用来查找到对应的参数对象ParamElement
            String path = AnalysisUtils.concatByDot(actionId, "$", param.getString(MetaDataConstant.METADATAS_PARAM_DATA_NAME));
            //添加入参父节点到action中，后面拼接入参的时候 从父节点开始拼接（父->子的关系 拼接所有入参的结构和节点）
            ParamElement element = getLocalContext().getParamElement(path);
            if (null != element) {
                actionParam.getRequestParams().add(element);
            }
        }
    }

    /**
     * 分析元数据中的resp参数
     *
     * @param metadata
     */
    private void analysisRespMetaDatas(JSONObject metadata) {
        String actionId = metadata.getString(MetaDataConstant.ACTION_ACTION_ID);
        JSONObject respObject = metadata.getJSONObject(MetaDataConstant.METADATAS_ACTION_RESPONSE);

        if (MapUtils.isEmpty(respObject)) {
            return;
        }

        JSONObject parameter = respObject.getJSONObject(MetaDataConstant.METADATAS_ACTION_RESPONSE_DATA);
        if (MapUtils.isEmpty(parameter)) {
            return;
        }
        analysisParameter(parameter, actionId, null, null);
    }

    /**
     * 分析元数据中的 parameter
     *
     * @param param
     * @param actionId
     * @param parentElement
     * @param children
     */
    private void analysisParameter(JSONObject param, String actionId, ParamElement parentElement, List<ParamElement> children) {

        //1.添加当前节点 为paramElement，初始化对象属性
        ParamElement paramElement = new ParamElement();
        paramElement.setActionId(actionId);
        paramElement.setParentElement(parentElement);
        paramElement.setParamName(param.getString(MetaDataConstant.METADATAS_PARAM_DATA_NAME));
        paramElement.setValueType(ValueTypeEnum.getEnumByType(param.getString(MetaDataConstant.METADATAS_PARAM_DATA_TYPE)));
        paramElement.setArray(Optional.ofNullable(param.getBoolean(MetaDataConstant.METADATAS_PARAM_IS_ARRAY)).orElse(false));
        paramElement.setIsDataKey(Optional.ofNullable(param.getBoolean(MetaDataConstant.METADATAS_PARAM_IS_DATAKEY)).orElse(false));
        paramElement.setIsRequired(Optional.ofNullable(param.getBoolean(MetaDataConstant.METADATAS_PARAM_REQUIRED)).orElse(false));

        //判断子节点集合是否存在，如果存在将当前节点添加到子节点集合中
        if (null != children) {
            children.add(paramElement);
        }

        //拼接节点的路径（如果当前节点没有父节点，路径=actionId.当前节点参数的名称，
        // 如果有父节点 路径=父节点路径(父节点路径中包含了actionId).当前节点参数的名称）
        String elementPath = Optional.ofNullable(parentElement)
                .filter(parent -> StringUtils.isNotBlank(parent.getParamPath()))
                .map(parent -> AnalysisUtils.concatByDot(parent.getParamPath(), paramElement.getParamName()))
                .orElse(AnalysisUtils.concatByDot(actionId, "$", paramElement.getParamName()));
        paramElement.setParamPath(elementPath);

        //初始化参数的jsonPath，有父节点 直接是父节点.当前节点，如果当前节点是数组  直接是 父节点.当前节点[*]
        String parentJsonPath = CommonConstant.JSON_PATH_PREFIX;
        if ((null != parentElement)) {
            parentJsonPath = parentElement.getElementJsonPath();
            if (parentElement.getArray()) {
                parentJsonPath = StringUtils.join(parentJsonPath, CommonConstant.JSON_PATH_ARRAY_SUFFIX);
            }
        }
        paramElement.setElementJsonPath(parentJsonPath);
        if (StringUtils.isNotBlank(paramElement.getParamName())) {
            paramElement.setElementJsonPath(AnalysisUtils.concatByDot(parentJsonPath, paramElement.getParamName()));
        }
        //初始化好的参数ParamElement对象添加到上下文中
        getLocalContext().addParamElement(paramElement.getParamPath(), paramElement);

        JSONArray fields = param.getJSONArray(MetaDataConstant.METADATAS_PARAM_FIELD);
        //2.判断当前节点是不是Object 且 有没有子的 field
        if (!ValueTypeEnum.OBJECT.equals(paramElement.getValueType()) || CollectionUtils.isEmpty(fields)) {
            return;
        }

        //3.添加子节点
        List<ParamElement> childList = Lists.newArrayList();
        paramElement.setChildElements(childList);

        for (int i = 0; i < fields.size(); i++) {
            JSONObject fieldObj = fields.getJSONObject(i);

            analysisParameter(fieldObj, actionId, paramElement, childList);
        }
    }


    private JSONObject createTransSchema(JSONArray microTrans) {
        JSONObject micTrans = microTrans.getJSONObject(0);
        buildDistinctParam(micTrans);
        String template = transSchemaCreator.createTransSchemaByTemplate(micTrans);
        try {
            return JSONObject.parseObject(ReplaceUtils.replace(micTrans, template));
        } catch (Exception e) {
            LOGGER.error("micTrans定义转换失败,micTrans:{},template:{}", micTrans, template, e);
            throw e;
        }
    }

    private void buildDistinctParam(JSONObject micTrans) {
        String technique = micTrans.getString("technique");
        if (TransConstant.DISTINCT_FUNCTION.indexOf(technique) == -1) {
            return;
        }
        JSONArray groupFields = micTrans.getJSONArray("groupFields");

        JSONArray statistics = micTrans.getJSONArray("statistics");

        statistics.forEach(stat -> groupFields.add(((JSONObject) stat).getString("newField")));

        micTrans.put("distinctFields", groupFields);

    }
}
