package com.digiwin.athena.executionengine.service.facade.export;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.digiwin.app.service.DWServiceContext;
import com.digiwin.athena.executionengine.constant.MetaDataConstant;
import com.digiwin.athena.executionengine.constant.MetaDefinitionConstant;
import com.digiwin.athena.executionengine.constant.ModuleProperty;
import com.digiwin.athena.executionengine.dto.ExportDataDto;
import com.digiwin.athena.executionengine.exception.BusinessException;
import com.digiwin.athena.executionengine.model.BasePage;
import com.digiwin.athena.executionengine.model.DataDescription;
import com.digiwin.athena.executionengine.model.Metric;
import com.digiwin.athena.executionengine.model.input.ExportInput;
import com.digiwin.athena.executionengine.service.client.IThemeMapClient;
import com.digiwin.athena.executionengine.service.facade.schema.ISchemaBuilder;
import com.digiwin.athena.executionengine.util.JsonUtil;
import com.digiwin.athena.smartdata.sdk.DatasourceProxyInitialize;
import com.google.common.collect.Lists;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * @description:
 * @author: ZhangJun
 * @create: 2024/9/23
 */
@Service("engineExportFacade")
public class ExportFacade implements IExportFacade {
    private static final Logger LOGGER = LoggerFactory.getLogger(ExportFacade.class);

    @Autowired
    private IThemeMapClient themeMapClient;

    @Autowired
    private ISchemaBuilder schemaBuilder;

    @Override
    public List<ExportDataDto> exportData(ExportInput exportInput) {

        String token = DWServiceContext.getContext().getToken();
        String locale = (String) DWServiceContext.getContext().getRequestHeader().get(MetaDataConstant.ACTION_LOCALE);
        List<Metric> metrics = exportInput.getMetrics();
        if (CollectionUtils.isEmpty(metrics)) {
            return Lists.newArrayList();
        }

        Set<String> metricIds = metrics.parallelStream().map(Metric::getMetricId).collect(Collectors.toSet());
        if (metricIds.size() > ModuleProperty.METRICS_NUM) {
            LOGGER.warn("导出指标个数不能大于5个");
            return Lists.newArrayList();
        }
        JSONObject jsonObject = themeMapClient.queryMetricQuerySchema(metricIds);

        if (!checkQuery(metrics, jsonObject, exportInput, token, locale)) {
            LOGGER.warn("存在明细超过5000条数据，无法导出");
            return Lists.newArrayList();
        }

        Map<String, List<Object>> dataMap = new HashMap<>();
        List<ExportDataDto> exportDataDos = Lists.newArrayList();
        metrics.forEach(item -> {
            JSONArray metricQuerySchema = jsonObject.getJSONArray(item.getMetricId());
            JSONObject metricCondition = item.getMetricCondition();
            metricQuerySchema.parallelStream().forEach(metricDefine -> {
                JSONObject metricDefineObj = (JSONObject) metricDefine;
                String actionId = metricDefineObj.getString("actionId");
                String actionName = ((JSONObject) metricDefine).getString("actionName");
                String schema = metricDefineObj.getString("querySchema");
                String productLine = metricDefineObj.getString("productLine");
                JSONArray responseFields = new JSONArray();
                if (metricDefineObj.getJSONObject(MetaDataConstant.METADATAS_ACTION_RESPONSE) != null) {
                    responseFields = metricDefineObj.
                            getJSONObject(MetaDataConstant.METADATAS_ACTION_RESPONSE).
                            getJSONArray(MetaDataConstant.METADATAS_PARAM_FIELD);
                }
                Map<String, Object> retMap = doQuery(schema, metricCondition, exportInput, token, locale, productLine);

                dataMap.computeIfAbsent(actionId, k -> Lists.newArrayList());
                List<Object> resultList = dataMap.get(actionId);
                resultList.addAll((Collection<?>) retMap.get("list"));
                //这里需要替换一下metadata的名称，将其改为BMD节点要输出的名称
                List<Object> metadata = (List<Object>) retMap.get("metadata");
                metadata = buildMetadata(responseFields, metadata, locale);

                boolean exists = exportDataDos.stream().anyMatch(dto -> dto.getActionId().equals(actionId));
                if (!exists) {
                    exportDataDos.add(new ExportDataDto(actionId, actionName, metadata));
                }
            });
        });

        exportDataDos.forEach(dto -> dto.setData(dataMap.get(dto.getActionId())));
        return exportDataDos;
    }

    /**
     * 替换metadata的名称，将其改为BMD节点要输出的名称
     *
     * @param responseFields
     * @param metadata
     * @param locale
     */
    private List<Object> buildMetadata(JSONArray responseFields, List<Object> metadata, String locale) {
        List<Object> tempMetadata = Lists.newArrayList();
        if (CollectionUtils.isNotEmpty(metadata)) {
            for (Object fieldObject : responseFields) {
                Map<String, Object> field = (Map<String, Object>) fieldObject;
                for (Object o : metadata) {
                    Map<String, Object> data = (Map<String, Object>) o;
                    if (data.get(MetaDataConstant.BMD_METADATAS_COLUMN_NAME).equals(field.get(MetaDataConstant.METADATAS_PARAM_DATA_NAME))) {
                        tempMetadata.add(o);
                        break;
                    }
                }
            }
        }
        return tempMetadata;
    }

    private boolean checkQuery(List<Metric> metrics, JSONObject jsonObject, ExportInput exportInput, String token, String locale) {
        final Integer[] totalCnt = {0};
        metrics = Collections.synchronizedList(metrics);
        metrics.forEach(metric -> {
            JSONArray metricQuerySchema = jsonObject.getJSONArray(metric.getMetricId());
            metricQuerySchema.parallelStream().forEach(obj -> {
                JSONObject metricDefineObj = (JSONObject) obj;
                String schema = metricDefineObj.getString("querySchema");
                String productLine = metricDefineObj.getString("productLine");
                int count = doCount(schema, metric.getMetricCondition(), exportInput, token, locale, productLine);
                totalCnt[0] += count;
            });
        });
//        for (Metric metric : metrics) {
//            JSONArray metricQuerySchema = jsonObject.getJSONArray(metric.getMetricId());
//            JSONObject metricCondition = metric.getMetricCondition();
//            for (Object obj : metricQuerySchema) {
//                JSONObject metricDefineObj = (JSONObject) obj;
//                String schema = metricDefineObj.getString("querySchema");
//                int count = doCount(schema, metricCondition, exportInput);
//                totalCnt[0] += count;
//                if (totalCnt[0] > ModuleProperty.METRICS_DATA_SIZE) {
//                    return false;
//                }
//            }
//        }
        if (totalCnt[0] > ModuleProperty.METRICS_DATA_SIZE) {
            return false;
        }
        return true;
    }

    private int doCount(String schema, JSONObject metricCondition,
                        ExportInput exportInput, String token, String locale, String productLine) {
        DataDescription dataDescription = new DataDescription();
        dataDescription.getDynamicSchema().putAll(metricCondition);
        JSONObject permissions = exportInput.getPermissions();
        List roles = null;
        JSONArray rows = null;
        if (MapUtils.isNotEmpty(permissions)) {
            roles = permissions.getJSONArray("roles");
            rows = permissions.getJSONArray("rows");
        }
        if (CollectionUtils.isNotEmpty(rows)) {
            for (int i = 0; i < rows.size(); i++) {
                JSONObject rowJson = rows.getJSONObject(i);
                if (productLine.equals(rowJson.getString("productLine"))) {
                    dataDescription.setPermissions(rowJson.getJSONObject("row"));
                    break;
                }
            }
        }
        Map<String, Object> querySchema = schemaBuilder.buildExportQuerySchema(schema, new HashMap<>(), dataDescription);
        querySchema.put("productLine", productLine);
        String datasource = exportInput.getDatasource();
        String userId = exportInput.getUserId();

        Map<String, Object> bmd = DatasourceProxyInitialize.countData(datasource, querySchema, exportInput.getTenantId(),
                locale, token, userId, exportInput.getTenantId(), exportInput.getEocMaps(), roles, "");

        Map<String, Object> detail = (Map<String, Object>) bmd.get("data");
        if (detail == null || detail.isEmpty() || !detail.containsKey("total")) {
            return 0;
        }

        return (Integer) detail.get("total");
    }


    private Map<String, Object> doQuery(String schema, JSONObject metricCondition, ExportInput exportInput, String token, String locale, String productLine) {
        Map<String, Object> result = new HashMap<>();
        DataDescription dataDescription = new DataDescription();
        dataDescription.getDynamicSchema().putAll(metricCondition);
        JSONObject permissions = exportInput.getPermissions();
        List roles = null;
        JSONArray rows = null;
        if (MapUtils.isNotEmpty(permissions)) {
            roles = permissions.getJSONArray("roles");
            rows = permissions.getJSONArray("rows");
        }
        if (CollectionUtils.isNotEmpty(rows)) {
            for (int i = 0; i < rows.size(); i++) {
                JSONObject rowJson = rows.getJSONObject(i);
                if (productLine.equals(rowJson.getString("productLine"))) {
                    dataDescription.setPermissions(rowJson.getJSONObject("row"));
                    break;
                }
            }
        }

        Map<String, Object> querySchema = schemaBuilder.buildExportQuerySchema(schema, new HashMap<>(), dataDescription);
        querySchema.put("productLine", productLine);
        try {
            BasePage basePage = createBasePage(querySchema);
            querySchema.put("page", basePage);
            List<Object> list = new ArrayList<>();
            String datasource = exportInput.getDatasource();
            String userId = exportInput.getUserId();
            long startTime = System.currentTimeMillis();

            List<Object> meta = null;
            while (true) {
                Map<String, Object> bmd = DatasourceProxyInitialize.getData(datasource, querySchema, exportInput.getTenantId(),
                        locale, token, userId, exportInput.getTenantId(), exportInput.getEocMaps(), roles, "");
                LOGGER.error("bmd的sql为:{}", JsonUtil.getJsonString(((Map) (bmd.get("data"))).get("querySql")));
                if (bmd == null || bmd.isEmpty()) {
                    break;
                }
                Map<String, Object> detail = (Map<String, Object>) bmd.get("data");
                if (detail == null || detail.isEmpty() || !detail.containsKey("total") || !detail.containsKey("data")) {
                    break;
                }

                Integer total = "dcp".equals(datasource) ? Integer.parseInt((String) detail.get("total")) : (Integer) detail.get("total");
                if (total == 0) {
                    //createDefaultData(list, detail);
                    break;
                }
                if (total >= BasePage.MAX_DATA_SIZE) {
                    throw new BusinessException("单个查询方案数据量超出限制");
                }

                meta = (List) detail.get("meta");

                list.addAll((List) detail.get("data"));

                if (list.size() >= total) {
                    break;
                }

                basePage.setPageNo(basePage.getPageNo() + 1);
                querySchema.put("page", basePage);
            }


            long timeTaken = (System.currentTimeMillis() - startTime);
            LOGGER.info(String.format("查询耗时:%sms，查询笔数:%s,查询方案:%s", timeTaken, list.size(), JSONObject.toJSONString(querySchema)));
            result.put("list", list);
            result.put("metadata", meta);
            return result;
        } catch (Exception e) {
            LOGGER.error("调用BMD获取数据异常,查询方案:{}", JSONObject.toJSONString(querySchema), e);
            throw e;
        }

    }


    private BasePage createBasePage(Map<String, Object> querySchema) {
        BasePage pageObj = Optional.ofNullable(querySchema).map(map -> map.get("page")).map(page -> JsonUtil.getObject(JsonUtil.getJsonString(page), BasePage.class)).orElse(BasePage.createDefaultPage());
        /*if (pageObj.getPageSize() < BasePage.DEFAULT_PAGE_SIZE) {
            pageObj.setPageSize(BasePage.DEFAULT_PAGE_SIZE);
        }*/
        return pageObj;
    }
}

