package com.digiwin.athena.executionengine.service.facade.mapping.param;

import com.digiwin.athena.executionengine.constant.CommonConstant;
import com.digiwin.athena.executionengine.core.container.ExecuteContext;
import com.digiwin.athena.executionengine.enumtype.ErrorCodeEnum;
import com.digiwin.athena.executionengine.enumtype.ValueTypeEnum;
import com.digiwin.athena.executionengine.exception.BusinessException;
import com.digiwin.athena.executionengine.model.ParamElement;
import com.digiwin.athena.executionengine.model.ServiceResult;
import com.digiwin.athena.executionengine.component.param.ParamBase;
import com.digiwin.athena.executionengine.util.ExceptionUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * @description: action入参处理
 * 由于逻辑比较复杂，定义如下的描述，明确代码的所表述的逻辑
 * - 向上构建中构建方法以   construct开头
 * - 向下构建中构建方法以   build开头
 * - 构建中对属性赋值方法以 assemble开头
 * - 主体功能实现的入口以   mapping开头
 * - 映射JSON结构分为：
 * === 集合属性：collection
 * === 复杂属性：complex
 * === 简单属性：simple property
 * @author: zhangww
 * @date: 2020/11/4 14:34
 */
public class ParamMappingUtils {

    ExecuteContext context;
    String actionId;
    List<ParamBase> actionParams;
    ConstructionUtils constructionUtils;

    private static final Logger LOGGER = LoggerFactory.getLogger(ParamMappingUtils.class);

    public ParamMappingUtils(ExecuteContext context, String actionId) {
        this.context = context;
        this.actionId = actionId;
        this.actionParams = context.getActionParam(actionId).getParams();
        this.constructionUtils = new ConstructionUtils(this.context, this.actionParams);
    }

    /**
     * 根据映射关系计算根节点的数据量大小
     *
     * @param paramBaseList
     * @return
     */
    private int calculateRootSize(Map<String, ParamBase> paramBaseList) {
        //如果是单来源，paramBaseList的个数是1，无需判断根size
        if (paramBaseList.size() == CommonConstant.SIZE_ONE) {
            return CommonConstant.SIZE_ONE;
        }
        //多来源，只判断集合类型的size,如果个数不同set集合个数大于1
        Set<Integer> rootSizeSet = new HashSet<>();
        int loopNum = 0;
        for (Map.Entry<String, ParamBase> paramBaseEntry : paramBaseList.entrySet()) {
            ParamBase paramBase = paramBaseEntry.getValue();
            //非集合的不判断
            if (!paramBase.isCollectionRoot(context.getDataMappingManager(), actionId)) {
                continue;
            }
            int rootSize = paramBase.getRootSize(context.getDataMappingManager(), actionId);
            rootSizeSet.add(rootSize);
            //除第一个来源外，其他来源均需要检测下rootSizeSet的个数 是否大于1，如果大于1 说明多来源rootSize 不相同，需要抛出异常
            if (loopNum > 0 && rootSizeSet.size() > CommonConstant.SIZE_ONE) {
                String srcActionId = paramBase.getSrcActionId(context);
                LOGGER.error("构建退出！参数多来源长度不一致: actionId:{}, valueParam:{}", srcActionId, paramBase.getValue());
                break;
            }
            loopNum++;
        }
        return rootSizeSet.size() > CommonConstant.SIZE_ONE ? CommonConstant.SIZE_NO_EQUAL : rootSizeSet.size();
    }

    /**
     * 根据requestMetadatas数据构建action的入参
     *
     * @param apiParams
     * @param requestParameters
     * @throws Exception
     */
    public ServiceResult mappingReqParam(List<Object> apiParams, List<ParamElement> requestParameters) {

        ///处理p1，p2，p3
        for (ParamElement paramElement : requestParameters) {
            //获取p1，p2，p3对应的映射关系
            //key：数据源的paramPath，
            //value: 参数映射关系对象
            Map<String, ParamBase> paramBaseMap = constructionUtils.getSrcParamBase(paramElement);

            ///校验映射关系
            if (MapUtils.isEmpty(paramBaseMap)) {
                //判断该字段是否为必传字段，
                // 如果是必传字段则必须要有对应的映射关系，
                // 如果不是必传字段则不构建该字段
                //这里由于在解析后已经做了必传字段的映射校验，所以不再需要判断该字段是否为必传字段，如果为空则直接退出本次构建
                LOGGER.info("循环requestParameters时，检查出ParamElement对应的映射关系不存在,即将continue,paramName:{}", paramElement.getParamName());
                continue;
            }

            //校验映射关系是否在源数据中存在，如果是必填 不存在，直接抛出异常，如果非必填 ，继续下个p构建
            if (!mappingRelationCheck(paramBaseMap, paramElement)) {
                continue;
            }

            ///校验根节点数量
            /**
             * FIXME, 如果P1的映射数据源action1和action2的根都是集合，第一个size=1，第二个size>1，这里应该校验失败，但是实际是通过
             * 这的校验是如果根都是集合，那么根size要相同，如果不同的映射源一个是集合一个是对象，是允许的。
             * 如果其中一个映射源数据为空，这里也不会卡，由后面映射的时候决定是否合理
             */
            int rootSize = calculateRootSize(paramBaseMap);
            if (rootSize < 0) {
                //取size校验错误，终止构建
                LOGGER.error("计算多来源的rootSize数据量不相同，检查数据");
                return new ServiceResult(false, "计算多来源的rootSize数据量不相同，检查数据");
            }

            //获取多源数据所有集合节点的字段名，可能含有$
            //key：paramBase.value  //数据源的paramPath，
            //value: 数据源的paramPath中出现的所有集合属性的paramPath
            ///
            /// A
            /// A.B.C
            /// A.B.C.D.E     -> P1
            ///
            /// O
            /// O.P           -> P1
            Map<String, List<String>> pathInfoGroup = constructionUtils.getPathInfo(paramBaseMap, actionId);

            //判断是否需要向上构建
            /**
             * FIXME，这里的处理可能存在问题，需要研究讨论，现在看起来，如果当前P1映射来自$.A.B.C的数据，如果C是一个集合，P1是一个简单集合，这里会认为不需要向上构建。
             * 如果说P1还要对应$和A的都是个集合，而P1就是要映射简单集合（但是我们之前说，我们现在不支持简单集合的映射），那么这里就有问题。
             * 这时构建的结果将是P1是否继续映射未知，如果映射也只会拿到$[0].A[0].B.C的数据。
             */
            boolean isNeedConstructUpper = isNeedConstructUpper(paramElement, pathInfoGroup);

            /**
             * 这里有个问题需要开发时关注：（未来需要考虑如何组织这块代码，让逻辑更能顺畅）
             *   paramBase在数据层面可能是找不到这个节点的，当所有数据都缺失这个字段时。
             *   虽然一开始有校验必填字段的数据侧是否存在数据字段，但是非必填字段是允许缺失情形的。
             *   dataMapping中提供的一些方法isCollectionParam(...)和isObjectCollectionParam(...)本身不会校验数据侧是否有字段的情形，
             * 所以在使用这两个方法前必须做校验。而这都在isNeedConstructUpper(...)中完成，但即使字段在数据中没有找到，也会返回true。
             *   但如果映射在数据侧缺失节点（或者其上一级节点），会有个必然结果：pathInfoGroup中是没有向上构建的层次的
             *   最终是否向上构建，还是通过pathInfoGroup判定。所以目前看处理结果不会有问题、
             *
             *  >>>>> 但是这段逻辑其实没有很好的整理，看起来很乱，需要针对数据侧拿不到数据情形做一些做法的调整。 <<<<<
             */

            ///数据源的上层结构中不存在集合，则直接从自身开始构建入参
            if (!isNeedConstructUpper) {
                apiParams.add(assemble(paramElement));
            } else {
                ///p1，p2，p3是集合属性
                /// 获取所有的集合层的jsonPath实例
                /// key: A.B.C
                /// value: A.B.C的实例jsonPath
                List<Object> paramList = new ArrayList<>();

                //构建上层的结构
                //pos: pathInfo中的具体某个层
                boolean constructResult = constructUpperLevel(paramList, paramElement, paramBaseMap,
                        pathInfoGroup, CommonConstant.POS_INIT);
                if (!constructResult) {
                    return new ServiceResult(false, "不同源数据，在同层次的数量不一致");
                }

                //因为目前结构是从根节点开始构建的，而解析组装action入参是按照P1,P2,P3处理的，所以这里要取出一层数据
                //paramList的size大于1说明没有上层没有集合层次，则直接取值
                apiParams.add(paramList.size() > CommonConstant.SIZE_ONE ? paramList : paramList.get(CommonConstant.POS_INIT));
            }
        }
        return new ServiceResult(true);
    }

    /**
     * 映射关系检查
     *
     * @param paramBaseMap
     * @param paramElement
     * @return
     */
    private boolean mappingRelationCheck(Map<String, ParamBase> paramBaseMap, ParamElement paramElement) {
        boolean pathExists = false;
        for (Map.Entry<String, ParamBase> entry : paramBaseMap.entrySet()) {
            ParamBase paramBase = entry.getValue();
            pathExists = paramBase.pathIsExistsInData(context.getDataMappingManager(), actionId, paramBase.getValue());
            if (!pathExists && paramElement.getIsRequired()) {
                throw ExceptionUtils.buildVerificationException(ErrorCodeEnum.MAPPING_RELATION_NOT_FOUND_IN_SOURCE_DATA, paramElement.getParamPath());
            }
        }
        return pathExists;
    }

    /**
     * 判断是否需要向上构建
     *
     * @param paramElement  p1,p2
     * @param pathInfoGroup 包含的集合层次
     * @return
     */
    private boolean isNeedConstructUpper(ParamElement paramElement, Map<String, List<String>> pathInfoGroup) {
        Map<String, ParamBase> paramBaseList = constructionUtils.getSrcParamBase(paramElement);
        for (Map.Entry<String, ParamBase> paramBaseEntry : paramBaseList.entrySet()) {
            ParamBase paramBase = paramBaseEntry.getValue();
            //在mappingRelationCheck中已经对该paramElement所有的数据源做了数据节点不存在 是否合理判断。
            //此处如果不存在，虽然业务上是合理的，但是这应该跳过，防止isCollectionParam和isObjectCollectionParam方法处理错误
            /////另外这里是判断数据节点之间的关系，是不会用到下标的，只要数据中出现过这个节点，就一定会判断出该节点的关系
            if (!paramBase.pathIsExistsInData(context.getDataMappingManager(), actionId, paramBase.getValue())) {
                continue;
            }
            /**
             * 特殊的需要向上构建的场景
             * 1. P1是复杂映射数据来自复杂集合
             */
            if (isComplexProperties(paramElement)
                    && paramBase.isObjectCollectionParam(context.getDataMappingManager(), actionId, paramBase.getValue())) {
                List<String> pathInfo = pathInfoGroup.get(constructionUtils.getMultipleSourceKey(paramBase));
                pathInfo.add(paramBase.getValue());
            }
        }

        /**
         * 是否需要向上构建,pathInfoGroup中存在集合，就需要向上构建
         */
        return isDataHaveCollection(pathInfoGroup);
    }

    /**
     * 数据源的上层结构中是否存在集合
     *
     * @param pathInfoGroup 对应数据源包含的所有集合层次路径
     * @return
     */
    private boolean isDataHaveCollection(Map<String, List<String>> pathInfoGroup) {
        for (Map.Entry<String, List<String>> pathInfoEntry : pathInfoGroup.entrySet()) {
            //判断自己的上层是否包含集合
            if (pathInfoEntry.getValue().size() > CommonConstant.SIZE_ZERO) {
                return true;
            }
        }
        return false;
    }

    /**
     * 构建整个api入参的数据结构，这是一个list，是否要foreach，根据该list的大小决定。
     *
     * @param apiParamList list里存的是P1,P2,P3的数据，Object即是P1，P2...的一个实例
     *                     例如：
     *                     apiParamList[0] = {1}   // P1
     *                     apiParamList[1] = {1}   // P2
     *                     apiParamList[2] = [[{1},{2},{3}]]   // P3
     *                     apiParamList[3] = [{1},{2},{3}]   // P4
     * @return
     */
    public List<Map<String, Object>> mergeApiData(List<Object> apiParamList) {
        /** 层次和实例矩阵 matrix
         *  假定apiParamList如下
         *  p1 = [ {p1:11}, {p1:12} ]
         *  p2 = [ [[{p2:21}],[{p2:22},{p2:23}]], [[{p2:24}]] ]
         *  p3 = [ [[{p3:31},{p3:32}], [{p3:33}]] ]
         *  p4 = [ [[{p4:41},{p4:42}], [{p4:43}]] ]
         *  p5 = {p5:"机制变量"}
         *
         *  matrix
         *  层号:     第一层pos 第二层pos 第三层pos   存储
         *  Pn     P1   0                         // 11
         *         P1   1                         // 12
         *         P2   0        0        0       // 21
         *         P2   0        1        0       // 22
         *         P2   0        1        1       // 23
         *         P2   1        0        0       // 24
         *         P3   0        0                // 31
         *         P3   0        1                // 32
         *         P3   1        0                // 33
         *         P4   0        0                // 41
         *         P4   0        1                // 42
         *         P4   1        0                // 43
         *         P5                             // 机制变量
         *
         *  matrixIndex
         *  key(matrix中的pos索引) : value(List<String> : P1,P2,P3...)
         *  0: P1
         *  1: P1
         *  00: P3,P4
         *  01: P3,P4
         *  ...
         *  000: P2
         *  010: P2
         *  "": P5
         *
         *  1、构建matrix和matrixIndex
         *  2、组装P1，P2，P3的时候，先找出哪个层次最深，从层次最深的开始组装，找到P2
         *  3、组装的做法：循环组装matrix的p2数据(原先的做法是按照apiDataCnt循环)：
         *               从P2开始组装，先全匹配pos:000，在matrix发现只有P2，装入结果
         *               再匹配P2的前两层pos:00，找到P3和P4，装入结果
         *               再匹配P2的前两层pos:0，找到P1，装入结果
         *               最后找是否存在没有层次的，找到P5，装入结果
         *
         *               再继续组装matrix中P2的pos索引：010
         *               ....
         */

        //循环构建每个API入参对象
        return setApiData(apiParamList);
    }

    /**
     * 构建每个API的入参Json结构
     *
     * @param apiParamList
     * @return
     */
    public List<Map<String, Object>> setApiData(List<Object> apiParamList) {
        //获取apiParamList下集合属性中下标最大的一个size
        int paramSize = getParamSize(apiParamList);
        List<Map<String, Object>> apiData = new ArrayList<>();
        if (paramSize == 0) {
            //paramSize等于0，存在2种情况，一种是空集合，一种是所有的param都是对象。如果是空集合应该过滤掉
            Map<String, Object> apiParam = new HashMap<>();
            for (Object apiParamData : apiParamList) {
                if (apiParamData instanceof List) {
                    continue;
                }
                apiParam.putAll((Map<String, Object>) apiParamData);
            }
            //详见故事756 任务单3375
            if (!MapUtils.isEmpty(apiParam)) {
                apiData.add(apiParam);
            }
            return apiData;
        }

        for (int pos = 0; pos < paramSize; pos++) {
            List<Object> deepApiParamList = new ArrayList<>();
            //Pn结构中的所有param取相同的下标实例，递归组装
            for (Object apiParam : apiParamList) {
                if (!(apiParam instanceof List)) {
                    deepApiParamList.add(apiParam);
                    continue;
                }
                ///向上构建中存在非必填字段，层次结构将找不到P，也就是说可能是一个空的list,找不到P就不组装
                if (CollectionUtils.isEmpty((List) apiParam)) {
                    continue;
                }
                //说明已经向上构建的层次结构已经找完，内部就是P，这时只会有一个map，重复取这个值
                if (((List) apiParam).get(0) instanceof Map) {
                    deepApiParamList.add(((List) apiParam).get(0));
                    continue;
                }
                //再往下递归中发现同层的集合size不相同，直接停止合并api,抛出异常.pos超出了当期的apiParam的size
                if (pos > ((List) apiParam).size() - 1) {
                    throw new BusinessException(ErrorCodeEnum.SAME_LEVER_SIZE_DIFF.getCode(), ErrorCodeEnum.SAME_LEVER_SIZE_DIFF.getMessage());
                }
                deepApiParamList.add(((List) apiParam).get(pos));
            }
            apiData.addAll(setApiData(deepApiParamList));
        }
        return apiData;
    }

    /**
     * 获取单个param的size
     *
     * @param apiParamList
     * @return
     */
    private int getParamSize(List<Object> apiParamList) {
        int paramSize = 0;
        for (Object apiParam : apiParamList) {
            if (!(apiParam instanceof List)) {
                continue;
            }
            if (paramSize == 0 || (paramSize != 0 && paramSize < ((List) apiParam).size())) {
                paramSize = ((List) apiParam).size();
            }
        }
        return paramSize;
    }

    /**
     * 根据dataSizeInPathInfo计算不同数据源的同一层数据数量是否一致
     *
     * @param dataSizeInPathInfo key：paramBase.value
     *                           value：一个数据源的数据量
     * @return
     */
    private boolean isSameDataSizeInDifferentSource(Map<String, Integer> dataSizeInPathInfo) {
        int lastSize = 0;
        boolean isFirstRun = true;
        boolean isSameSize = true;
        for (Map.Entry<String, Integer> dataSizeInPathInfoEntry : dataSizeInPathInfo.entrySet()) {
            int size = dataSizeInPathInfoEntry.getValue().intValue();
            if (isFirstRun) {
                isFirstRun = false;
                lastSize = size;
            }

            if (lastSize != size) {
                isSameSize = false;
                break;
            }
            lastSize = size;
        }
        return isSameSize;
    }

    /**
     * 根据pathInfo递归创建上层结构，并在最后一层，向下构建具体入参
     *
     * @param parentList       最终构建好的入参结构
     * @param requestParameter p1,p2,p3
     * @param paramBaseList    参数映射关系，包含多数据源
     * @param pathInfoGroup    多数据源对应的所有集合层次路径
     * @param pos              当前正在构建的集合层次，pos表示第几层
     *                         p1,p2,p3...节点对应的多数据实际路径，但是这里并不是最终的jsonPath，
     *                         因为没有完成该路径的拼接，所以暂时不能往 concreteJsonPath 里存，
     *                         只要是入参，一定是未完成的，完成的path会set进concreteJsonPath的Map
     *                         key：paramBase.value
     *                         value: 未完成的jsonPath
     * @return
     */
    private boolean constructUpperLevel(List<Object> parentList, ParamElement requestParameter,
                                        Map<String, ParamBase> paramBaseList,
                                        Map<String, List<String>> pathInfoGroup,
                                        int pos) {
        Map<String, Integer> dataSizeInPathInfo = new HashMap<>();
        int pathLevel = CommonConstant.SIZE_ZERO;
        //计算上层结构中含有的集合的层级
        for (Map.Entry<String, ParamBase> paramEntry : paramBaseList.entrySet()) {
            List<String> pathList = pathInfoGroup.get(paramEntry.getKey());
            if (pathList.size() > pathLevel) {
                pathLevel = pathList.size();
            }
            ParamBase param = paramEntry.getValue();

            ///pathInfo : 0:A, 1:B, 2:C SIZE=3
            /// A
            /// A.B.C
            /// A.B.C.D.E
            //如果pathList遍历完，则完成构建，退出递归
            if (pos != CommonConstant.SIZE_ZERO && pos >= pathList.size()) {
                return false;
            }

            ///jsonPath = $[0]
            ///concreteCollectionPath = $[0].A 或者是 $[0].A.B
            /// A.B.C  ->  $[0]
            int size = param.getPathDataSize(context.getDataMappingManager(), actionId, pathList.get(pos));
            dataSizeInPathInfo.put(param.getValue(), size);
        }

        if (!isSameDataSizeInDifferentSource(dataSizeInPathInfo)) {
            LOGGER.error("不同源数据，在同层次的数量不一致！");
            return false;
        }

        /**
         * 因为上面isSameDataSizeInDifferentSource方法已经判断过多来源size是否相同，此处看似没必要重复判断
         * 此处逻辑先保留，后续再观察
         */
        int sourceSize = checkAndGetMultiSrcSizeOnSameLevel(pathInfoGroup, paramBaseList, pos);
        if (sourceSize < 0) {
            return false;
        }

        /**
         * 保存当前jsonpath最后一个节点不带索引的map
         * 如：$
         *     $[0].A
         *     $[0].A[0].B
         *     $[0].A[0].B[0].C
         */
        //按照数据源的结构（pathInfo）开始构建上层层次
        for (int i = 0; i < sourceSize; i++) {
            //在request的参数结构中构建一个新层次
            List<Object> list = new ArrayList();
            parentList.add(list);

            for (Map.Entry<String, ParamBase> paramEntry : paramBaseList.entrySet()) {
                List<String> pathList = pathInfoGroup.get(paramEntry.getKey());
                //这里处理的是数据映射，向上构建结构中的最后一个数据集合
                constructionUtils.setJsonPath(paramEntry.getValue(), actionId, pathList.get(pos), i);
            }

            //判断是否有层级结构或是集合的最后一层结构，是则递归构建参数并赋值，这时requestParameter可以得到数据的实例路径
            if (pos == pathLevel - 1 || pathLevel == 0) {
                if (!buildLowerLevel(list, requestParameter)) {
                    return false;
                }
            } else {
                //递归下一层次结构
                if (!constructUpperLevel(list, requestParameter, paramBaseList, pathInfoGroup, pos + CommonConstant.INCREASE_ONE)) {
                    return false;
                }
            }
        }

        return true;
    }

    /**
     * 获取来源数据的数据大小
     *
     * @param pathInfoMap 所有的集合路径
     * @param paramBase   来源映射
     * @return
     */
    private int getDataSize(Map<String, List<String>> pathInfoMap, ParamBase paramBase, int pos) {
        List<String> pathList = pathInfoMap.get(constructionUtils.getMultipleSourceKey(paramBase));
        return paramBase.getPathDataSize(context.getDataMappingManager(), actionId, pathList.get(pos));
    }

    /**
     * 创建一个简单的属性字段
     *
     * @param param
     * @param requestParameter
     */
    private boolean assembleSimpleProperty(HashMap<String, Object> param, ParamElement requestParameter) {
        Map<String, ParamBase> paramBaseMap = constructionUtils.getSrcParamBase(requestParameter);
        if (MapUtils.isEmpty(paramBaseMap)) {
            return false;
        }

        String[] keys = paramBaseMap.keySet().toArray(new String[0]);
        if (keys.length == 0) {
            return false;
        }
        ParamBase paramBase = paramBaseMap.get(keys[0]);
        if (paramBase == null) {
            if (requestParameter.getIsRequired()) {
                throw ExceptionUtils.buildVerificationException(ErrorCodeEnum.REQUIRE_FIELD_NONE_MAPPING_RELATION, requestParameter.getParamPath());
            }
        } else {
            Object elementValue = getValueByParamBase(paramBase);

            if (requestParameter.getIsRequired()) {
                //如果是必传字段则一定要有数据，否则报错退出
                if (elementValue == null) {
                    throw ExceptionUtils.buildVerificationException(ErrorCodeEnum.REQUIRE_FIELD_NONE_DATA, buildExceptionInfo(requestParameter.getActionId(), paramBase));
                }
                param.put(requestParameter.getParamName(), elementValue);
            } else {
                //如果是非必传字段，值为空不传
                /*if (null != elementValue) {
                }*/
                param.put(requestParameter.getParamName(), elementValue);
            }
        }

        return true;
    }

    /**
     * 必传参数没有取到值时的 异常信息
     *
     * @param actionId
     * @param paramBase
     * @return
     */
    private String buildExceptionInfo(String actionId, ParamBase paramBase) {
        StringBuilder sBuilder = new StringBuilder();
        sBuilder.append("actionId:");
        sBuilder.append(actionId);
        sBuilder.append(";name:");
        sBuilder.append(paramBase.getName());
        sBuilder.append(";value:");
        sBuilder.append(paramBase.getValue());
        return sBuilder.toString();
    }

    /**
     * 组装参数 根据当前requestParameter的属性对最后一层的数据进行赋值
     *
     * @param lastLevelList    最终返回的组装好的数据集
     * @param requestParameter request Metadata
     */
    private boolean buildLowerLevel(List<Object> lastLevelList, ParamElement requestParameter) {
        boolean result;
        HashMap<String, Object> param = new HashMap<>();

        if (isSingleProperties(requestParameter) || isSingleCollectionProperties(requestParameter)) {
            if (!assembleSimpleProperty(param, requestParameter)) {
                return false;
            }

            lastLevelList.add(param);
        } else if (isComplexProperties(requestParameter)) {
            HashMap<String, Object> paramItem = new HashMap<>();
            param.put(requestParameter.getParamName(), paramItem);
            result = assembleComplex(paramItem, requestParameter.getChildElements(), requestParameter);
            if (!result) {
                return false;
            }

            lastLevelList.add(param);
        } else if (isObjectCollectionProperties(requestParameter)) {
            List<HashMap<String, Object>> paraList = new ArrayList<>();
            param.put(requestParameter.getParamName(), paraList);
            //当前是一个复杂&集合的属性，这里是构建集合的每个复杂属性结构

            Map<String, ParamBase> paramBaseList = constructionUtils.getSrcParamBase(requestParameter);

            int compareSize = 0;
            boolean isCompareSizeInCollectionPath = false;
            ///paramBaseList是childElement的多数据源映射关系
            for (Map.Entry<String, ParamBase> paramBaseEntry : paramBaseList.entrySet()) {
                ParamBase paramBase = paramBaseEntry.getValue();

                int size = paramBase.getDataSize(context.getDataMappingManager(), actionId,
                        paramBase.getValue(), constructionUtils.getJsonPath(paramBase, actionId));

                //这里不可能有size为0的情况，因为在处理P1,P2时已经把为0的情况处理掉了
                if (compareSize <= 0) {
                    compareSize = size;
                    //看当前数据是否来自集合路径
                    if (paramBase.countCollectionInPath(context.getDataMappingManager(), actionId, paramBase.getValue()) > 0) {
                        isCompareSizeInCollectionPath = true;
                    }
                } else {
                    if (compareSize != size) {
                        return false;
                    } else {
                        //比较下来是数量相等，但是数量为1，需要确认这1个数据是来自含有集合的路径还是复杂属性的路径
                        //如果当前路径含有集合，但是多数据源的另一个来源是一个非集合路径，这里不认为是合法的。
                        if (size == 1) {
                            int collectionLevelInPath = paramBase.countCollectionInPath(context.getDataMappingManager(), actionId, paramBase.getValue());
                            boolean isPathSame = (collectionLevelInPath > 0 && !isCompareSizeInCollectionPath)
                                    || (collectionLevelInPath == 0 && isCompareSizeInCollectionPath);
                            if (isPathSame) {
                                LOGGER.error("路径结构不同，虽然数据量都是1，但是依然不认为是合法结构：paramBase.name:{}", paramBase.getName());
                                return false;
                            }
                        }
                    }
                }
            }

            //根据集合的大小，循环构建集合内的对象（复杂属性）
            for (int i = 0; i < compareSize; i++) {
                setDataSrcJsonPath(paramBaseList, i);

                HashMap<String, Object> paramItem = new HashMap<>();
                //构建集合中的复杂属性
                boolean isSuccess = assembleComplex(paramItem, requestParameter.getChildElements(), requestParameter);
                if (!isSuccess) {
                    return false;
                }
                if (MapUtils.isNotEmpty(paramItem)) {
                    paraList.add(paramItem);
                }
            }

            handleIsRequire(lastLevelList, requestParameter, param, paraList);
        }
        return true;
    }

    /**
     * 设置并拼接当前层次数据的jsonPath路径
     *
     * @param paramBaseList
     * @param i
     */
    private void setDataSrcJsonPath(Map<String, ParamBase> paramBaseList, int i) {
        for (Map.Entry<String, ParamBase> paramBaseEntry : paramBaseList.entrySet()) {
            //新增当前路径，留作后续更深层次递归的读取
            constructionUtils.setJsonPath(paramBaseEntry.getValue(), actionId, paramBaseEntry.getValue().getValue(), i);
        }
    }

    /**
     * 获取相同层次的数据源的数量大小
     * 同层级数据量相同则返回具体大小值
     * 不同则返回-1
     *
     * @param pathInfoGroup 包含的所有上层集合路径
     * @param paramBaseList 数据来源映射
     * @return 大于0则同层级数据量相同，-1则同层级数据量不相同或来源于不同类型的数据源
     */
    private int checkAndGetMultiSrcSizeOnSameLevel(Map<String, List<String>> pathInfoGroup,
                                                   Map<String, ParamBase> paramBaseList,
                                                   int pos) {
        ///paramBaseList是childElement的多数据源映射关系
        int compareSize = 0;
        boolean isCompareSizeInCollectionPath = false;
        for (Map.Entry<String, ParamBase> paramBaseEntry : paramBaseList.entrySet()) {
            ParamBase paramBase = paramBaseEntry.getValue();

            //上层没有集合或者首次向上构建并且根节点是集合则最外层的循环次数取值于根节点的数量
            int size = getDataSize(pathInfoGroup, paramBase, pos);

            //这里不可能有size为0的情况，因为在处理P1,P2时已经把为0的情况处理掉了
            if (compareSize <= CommonConstant.SIZE_ZERO) {
                compareSize = size;
                //看当前数据是否来自集合路径
                if (paramBase.countCollectionInPath(context.getDataMappingManager(), actionId, paramBase.getValue())
                        > CommonConstant.SIZE_ZERO) {
                    isCompareSizeInCollectionPath = true;
                }
            } else {
                if (compareSize != size) {
                    compareSize = -1;
                    return compareSize;
                } else {
                    //比较下来是数量相等，但是数量为1，需要确认这1个数据是来自含有集合的路径还是复杂属性的路径
                    //如果当前路径含有集合，但是多数据源的另一个来源是一个非集合路径，这里不认为是合法的。
                    if (size == CommonConstant.SIZE_ONE) {
                        int collectionCountInPath = paramBase.countCollectionInPath(context.getDataMappingManager(), actionId, paramBase.getValue());
                        boolean isPathSame = (collectionCountInPath > CommonConstant.SIZE_ZERO && !isCompareSizeInCollectionPath)
                                || (collectionCountInPath == CommonConstant.SIZE_ZERO && isCompareSizeInCollectionPath);
                        if (isPathSame) {
                            LOGGER.error("路径结构不同，虽然数据量都是1，但是依然不认为是合法结构：paramBase.name:{}", paramBase.getName());
                            compareSize = -1;
                            return compareSize;
                        }
                    }
                }
            }
        }

        return compareSize;
    }

    /**
     * 对数据是否是必填字段进行处理
     *
     * @param lastLevelList
     * @param requestParameter
     * @param param
     * @param paraList
     */
    private void handleIsRequire(List<Object> lastLevelList, ParamElement requestParameter, HashMap<String, Object> param, List<HashMap<String, Object>> paraList) {
        if (requestParameter.getIsRequired()) {
            //如果是必传字段则一定要有数据，否则报错退出
            if (MapUtils.isEmpty(param)) {
                throw ExceptionUtils.buildVerificationException(ErrorCodeEnum.REQUIRE_FIELD_NONE_DATA, requestParameter.getParamPath());
            }
            if (param.get(requestParameter.getParamName()) instanceof List) {
                if (CollectionUtils.isEmpty((Collection) param.get(requestParameter.getParamName()))) {
                    throw ExceptionUtils.buildVerificationException(ErrorCodeEnum.REQUIRE_FIELD_NONE_DATA, requestParameter.getParamPath());
                }
            }
            lastLevelList.add(param);
        } else {
            //如果是非必传字段，值为空不传
            if (paraList.size() > 0) {
                lastLevelList.add(param);
            }
        }
    }

    /**
     * 组装参数
     *
     * @param requestParameter p1,p2,p3
     */
    private Object assemble(ParamElement requestParameter) {
        List<Object> lastLevelList = new ArrayList<>();
        buildLowerLevel(lastLevelList, requestParameter);
        if (lastLevelList.size() > CommonConstant.SIZE_ZERO) {
            return lastLevelList.get(CommonConstant.POS_INIT);
        }
        return lastLevelList;
    }

    /**
     * 向下构建复杂属性
     *
     * @param param            最终返回的组装好的对象
     * @param childElements    复杂属性下的所有字段
     * @param requestParameter request Metadata
     * @return
     */
    private boolean assembleComplex(HashMap<String, Object> param, List<ParamElement> childElements,
                                    ParamElement requestParameter) {
        boolean isSuccess = true;

        Map<String, ParamBase> parentParamBaseList = constructionUtils.getSrcParamBase(requestParameter);
        for (ParamElement childElement : childElements) {
            Map<String, ParamBase> paramBaseMap = constructionUtils.getSrcParamBase(childElement);
            if (MapUtils.isEmpty(paramBaseMap)) {
                LOGGER.debug("向下构建时，检查出childElement对应的映射关系不存在,即将continue,paramName:{}", childElement.getParamName());
                continue;
            }
            //校验映射关系是否在源数据中存在，如果是必填 不存在，直接抛出异常，如果非必填 ，继续下个p构建
            if (!mappingRelationCheck(paramBaseMap, childElement)) {
                continue;
            }
            if (isSingleProperties(childElement) || isSingleCollectionProperties(childElement)) {
                //调用简单属性的取值并赋值给request结构
                isSuccess = assembleSimpleProperty(param, childElement);
                if (!isSuccess) {
                    break;
                }
            } else if (isComplexProperties(childElement)) {
                HashMap<String, Object> paramItem = new HashMap<>();
                param.put(childElement.getParamName(), paramItem);
                isSuccess = assembleComplex(paramItem, childElement.getChildElements(), childElement);
                if (!isSuccess) {
                    break;
                }
            } else if (isObjectCollectionProperties(childElement)) {
                boolean result = assembleCollection(paramBaseMap, parentParamBaseList, childElement, param);
                //如果失败直接return
                if (!result) {
                    return false;
                }
            }
        }
        return isSuccess;
    }

    /**
     * 向下构建集合属性
     *
     * @param paramBaseList       数据源映射
     * @param parentParamBaseList 父数据源映射
     * @param childElement        集合下的字段
     * @param param               最终组装的对象
     * @return
     */
    private boolean assembleCollection(Map<String, ParamBase> paramBaseList,
                                       Map<String, ParamBase> parentParamBaseList,
                                       ParamElement childElement,
                                       HashMap<String, Object> param) {
        ///计算当前节点需要创建多少（compareSize）个实例，并校验多数据源结构合法性
        int compareSize = 0;
        boolean isCompareSizeInCollectionPath = false;
        ///paramBaseList是childElement的多数据源映射关系
        for (Map.Entry<String, ParamBase> paramBaseEntry : paramBaseList.entrySet()) {
            ParamBase paramBase = paramBaseEntry.getValue();

            ParamBase parentParamBase = constructionUtils.getRealParentParamBase(paramBase, parentParamBaseList);

            int size = paramBase.getPathDataSize(context.getDataMappingManager(), actionId,
                    paramBase.getValue(), constructionUtils.getJsonPath(parentParamBase, actionId));

            //这里不可能有size为0的情况，因为在处理P1,P2时已经把为0的情况处理掉了
            if (compareSize <= 0) {
                compareSize = size;
                //看当前数据是否来自集合路径
                if (paramBase.countCollectionInPath(context.getDataMappingManager(), actionId, paramBase.getValue()) > 0) {
                    isCompareSizeInCollectionPath = true;
                }
            } else {
                if (compareSize != size) {
                    return false;
                } else {
                    //比较下来是数量相等，但是数量为1，需要确认这1个数据是来自含有集合的路径还是复杂属性的路径
                    //如果当前路径含有集合，但是多数据源的另一个来源是一个非集合路径，这里不认为是合法的。
                    if (size == 1) {
                        int collectionLevelInPath = paramBase.countCollectionInPath(context.getDataMappingManager(), actionId, paramBase.getValue());
                        boolean isPathSame = (collectionLevelInPath > 0 && !isCompareSizeInCollectionPath)
                                || (collectionLevelInPath == 0 && isCompareSizeInCollectionPath);
                        if (isPathSame) {
                            LOGGER.error("路径结构不同，虽然数据量都是1，但是依然不认为是合法结构：paramBase.name:{}", paramBase.getName());
                            return false;
                        }
                    }
                }
            }
        }

        List<HashMap<String, Object>> paraList = new ArrayList<>();
        //当前是一个复杂&集合的属性，这里是构建集合的每个复杂属性结构
        for (int i = 0; i < compareSize; i++) {
            HashMap<String, Object> paramItem = new HashMap<>();
            //创建多数据源的实例jsonPath
            setEveryParamJsonPath(paramBaseList, i);
            boolean isSuccess = assembleComplex(paramItem, childElement.getChildElements(), childElement);
            if (!isSuccess) {
                return false;
            }
            paraList.add(paramItem);
        }

        param.put(childElement.getParamName(), paraList);
        return true;
    }

    /**
     * 使用父层次数据源的实际jsonPath拼接当前字段对应数据源的jsonPath，该方法适用一个集合属性
     * 这里考虑多数据源的情形
     * -- 当前字段含有多个数据源，和父层次的数据源，进行匹配，然后根据父层次数据源找到其实例jsonPath
     * -- 对该jsonpPath进行拼接加工，再交给多数据源管理进行处理。
     * <p>
     * 取多数据源父层的json实例路径, 这里做的事情，打个比方：
     * . 同母异父的很多孩子（paramBaseList），
     * . 在他们的爸爸中找出各自的亲爸爸（parentParamBaseList）
     * . 找到亲爸爸后找出这个孩子的路径
     *
     * @param paramBaseList
     */
    private void setEveryParamJsonPath(Map<String, ParamBase> paramBaseList, int pos) {
        //创建多数据源的实例jsonPath
        for (Map.Entry<String, ParamBase> paramBaseEntry : paramBaseList.entrySet()) {
            //新增当前路径，留作后续更深层次递归的读取
            constructionUtils.setJsonPath(paramBaseEntry.getValue(), actionId, paramBaseEntry.getValue().getValue(), pos);
        }
    }

    /**
     * 使用paramBase取值
     *
     * @param paramBase
     * @return
     */
    private Object getValueByParamBase(ParamBase paramBase) {
        String concreteCollectionPath = constructionUtils.getJsonPath(paramBase, actionId);
        return paramBase.getData(context, concreteCollectionPath);
    }

    /**
     * 判断是否是简单属性
     *
     * @param element
     * @return
     */
    private boolean isSingleProperties(ParamElement element) {
        return ValueTypeEnum.OBJECT != element.getValueType() && !element.getArray();
    }

    /**
     * 判断是否是简单集合属性
     *
     * @param element
     * @return
     */
    private boolean isSingleCollectionProperties(ParamElement element) {
        return ValueTypeEnum.OBJECT != element.getValueType() && element.getArray();
    }

    /**
     * 判断是否是对象集合属性
     *
     * @param element
     * @return
     */
    private boolean isObjectCollectionProperties(ParamElement element) {
        return ValueTypeEnum.OBJECT == element.getValueType() && element.getArray();
    }

    /**
     * 判断是否是复杂属性
     *
     * @param element
     * @return
     */
    private boolean isComplexProperties(ParamElement element) {
        return ValueTypeEnum.OBJECT == element.getValueType() && !element.getArray();
    }
}