package com.digiwin.athena.executionengine.trans.components;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.digiwin.athena.executionengine.constant.CommonConstant;
import com.digiwin.athena.executionengine.enumtype.ValueTypeEnum;
import com.digiwin.athena.executionengine.model.trans.DealResult;
import com.digiwin.athena.executionengine.model.trans.StepElement;
import com.digiwin.athena.executionengine.trans.Step;
import com.digiwin.athena.executionengine.trans.TransAbstractStep;
import com.digiwin.athena.executionengine.util.CloneUtil;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.jayway.jsonpath.Configuration;
import com.jayway.jsonpath.JsonPath;
import com.jayway.jsonpath.Option;
import com.jayway.jsonpath.ReadContext;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import java.util.Map;

/**
 * 目前只支持对json根节点上数据抓取
 *
 * @description: 从执行引擎入参结构中读取数据作为trans流处理入参的组件
 * @author: fenglei
 * @date:
 */
@Component("input")
public class InputStep extends TransAbstractStep {

    private static final Logger LOGGER = LoggerFactory.getLogger(InputStep.class);

    @Override
    public boolean defineCheck(StepElement stepElement) {
        String nullDefaultConstruct = stepElement.getNullDefaultConstruct();
        String jsonPath = stepElement.getPath();
        if (StringUtils.isEmpty(jsonPath)) {
            LOGGER.error("input,rule定义path节点为空，请检查");
            return false;
        }
        if (StringUtils.isNotEmpty(nullDefaultConstruct)) {
            return nullDefaultConstruct.equalsIgnoreCase(ValueTypeEnum.ARRAY.getType()) || nullDefaultConstruct.equalsIgnoreCase(ValueTypeEnum.OBJECT.getType());
        }
        return true;
    }

    /**
     * 数据处理的前置处理
     *
     * @return
     * @throws Exception
     */
    @Override
    public DealResult doDealData(Step step) {
        DealResult dealResult = new DealResult();
        try {
            Object originData = getOriginData();
            if (!(originData instanceof Map)) {
                LOGGER.error("入参读取组件需要输入的是一个Map结构！但是当前是：{}", originData.getClass().getSimpleName());
                dealResult.setDataNullFail();
                return dealResult;
            }
            StepElement stepElement = step.getStepElement();
            String jsonPath = stepElement.getPath();

            Object originDataPart = getJsonPathVal(originData, jsonPath);
            if (originDataPart == null) {
                dealResult.setSuccess(getDefaultData(stepElement));
                return dealResult;
            }
            //数据提取后，深拷贝成一个新的数据源
            Object newData = CloneUtil.clone(originDataPart);

            dealResult.setSuccess(newData);
            return dealResult;
        } catch (Exception e) {
            LOGGER.error("入参读取组件出现异常");
            throw e;
        }
    }

    /**
     * 获取jsonpath对应的值 option 使用REQUIRE_PROPERTIES方式，抑制抛出异常策略，如果是节点不存在，返回的结果可能为null 或者empty数组
     *
     * @param obj
     * @param path
     * @return
     */
    private Object getJsonPathVal(Object obj, String path) {
        if (null == obj || StringUtils.isBlank(path)) {
            return null;
        }

        Configuration conf = Configuration.defaultConfiguration();
        try {
            Configuration readConf = conf.addOptions(Option.REQUIRE_PROPERTIES);
            ReadContext ctx = JsonPath.using(readConf).parse(JSON.toJSONString(obj, SerializerFeature.WriteMapNullValue));
            return ctx.read(path);
        } catch (Exception e) {
            LOGGER.error("解析json:{}中的路径{}异常:{}", obj, path, e);
        }
        return null;
    }

    /**
     * 如果基于jsonpath取到的input为null,则需要返回一个默认的结构
     *
     * @param stepElement
     * @return
     */
    private Object getDefaultData(StepElement stepElement) {
        String nullDefaultConstruct = stepElement.getNullDefaultConstruct();
        if (nullDefaultConstruct != null) {
            if (nullDefaultConstruct.equalsIgnoreCase(ValueTypeEnum.ARRAY.getType())) {
                return Lists.newArrayList();
            } else if (nullDefaultConstruct.equalsIgnoreCase(ValueTypeEnum.OBJECT.getType())) {
                return Maps.newHashMap();
            }
        }

        String jsonPath = stepElement.getPath();
        if (jsonPath.indexOf(CommonConstant.JSON_PATH_ARRAY_SUFFIX) != -1) {
            return Lists.newArrayList();
        } else {
            return Maps.newHashMap();
        }
    }

}
