package com.digiwin.athena.executionengine.component.param;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.digiwin.athena.executionengine.constant.CommonConstant;
import com.digiwin.athena.executionengine.constant.MetaDataConstant;
import com.digiwin.athena.executionengine.core.container.ExecuteContext;
import com.digiwin.athena.executionengine.model.ParamElement;
import com.digiwin.athena.executionengine.service.facade.mapping.data.DataMappingManager;
import com.jayway.jsonpath.JsonPath;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;

import java.util.List;

/**
 * @description: action中的 param对象
 * @author: renwm
 * @date: 2020/6/12 16:36
 */
public abstract class ParamBase {

    String name;
    String value;
    String valJsonPath;
    /**
     * 会带上actionId.
     */
    String valParamPath;
    JSONArray except;

    public void initInstance(JSONObject paramObj) {
        this.name = paramObj.getString(MetaDataConstant.ACTIONS_PARAMS_NAME);
        this.value = paramObj.getString(MetaDataConstant.ACTIONS_PARAMS_VALUE);
        this.except = paramObj.getJSONArray(MetaDataConstant.ACTIONS_PARAMS_EXCEPT);
        initProperty(paramObj);
    }

    /**
     * 在不同类型的params结构中，有些类型含有属性字段，这些字段是该类型特有的。这里会对这些特有属性字段做处理
     *
     * @param paramObj
     */
    protected abstract void initProperty(JSONObject paramObj);

    /**
     * 检查，需要时执行pulling
     *
     * @param context
     * @return
     * @throws Exception
     */
    protected abstract boolean eval(ExecuteContext context) ;

    /**
     * 取值并且去除不需要的传参
     *
     * @param context
     * @return
     */
    public boolean parEvalAndExclude(ExecuteContext context)  {
        return eval(context);
    }

    /**
     * 获取参数数据来源的actionid
     * 如果是入参返回""
     *
     * @param context
     * @return
     * @throws Exception
     */
    public abstract String getSrcActionId(ExecuteContext context);

    /**
     * 根据jsonpath从上下文读取数据
     *
     * @param context
     * @param jsonPath
     * @return
     * @throws Exception
     */
    public abstract Object getData(ExecuteContext context, String jsonPath);

    /**
     * 根据path从上下文读取数据
     *
     * @param context
     * @param dataMappingManager
     * @param requestActionId    映射中输入的actionId
     * @param paramPath          映射中数据来源的paramPath
     * @return
     * @throws Exception
     */
    public abstract Object getData(ExecuteContext context, DataMappingManager dataMappingManager, String requestActionId, String paramPath) throws Exception;

    /**
     * 根据实际jsonpath取该路径下的数据量
     *
     * @param dataMappingManager
     * @param requestActionId
     * @param paramPath
     * @param jsonPath
     * @return
     */
    public abstract int getDataSize(DataMappingManager dataMappingManager, String requestActionId, String paramPath, String jsonPath);

    /**
     * 取当前数据源的root层的大小
     *
     * @param dataMappingManager
     * @param requestActionId
     * @return
     */
    public abstract int getRootSize(DataMappingManager dataMappingManager, String requestActionId);

    /**
     * 取当前数据源对应到的所有集合层次路径
     *
     * @param dataMappingManager
     * @param requestActionId
     * @param paramPath
     * @return
     */
    public abstract List<String> getCollectionPath(DataMappingManager dataMappingManager, String requestActionId, String paramPath);

    /**
     * 返回该path末端节点是否是json的集合结构
     *
     * @param dataMappingManager
     * @param requestActionId
     * @return
     */
    public abstract boolean isCollectionRoot(DataMappingManager dataMappingManager, String requestActionId);

    /**
     * 返回该path末端节点是否是json的集合结构
     *
     * @param dataMappingManager
     * @param requestActionId
     * @param param
     * @return
     */
    public abstract boolean isCollectionParam(DataMappingManager dataMappingManager, String requestActionId, String param);

    /**
     * 返回该path末端节点是否是json的对象集合结构
     *
     * @param dataMappingManager
     * @param requestActionId
     * @param param
     * @return
     */
    public abstract boolean isObjectCollectionParam(DataMappingManager dataMappingManager, String requestActionId, String param);

    /**
     * 获取当前路径集合结构的数量
     *
     * @param dataMappingManager
     * @param requestActionId
     * @param param
     * @return
     */
    public abstract int countCollectionInPath(DataMappingManager dataMappingManager, String requestActionId, String param);

    /**
     * 根据paramPath和限定范围路径，取该范围路径下的数据量
     *
     * @param dataMappingManager
     * @param requestActionId
     * @param paramPath
     * @param fixedPath
     * @return
     */
    public abstract int getPathDataSize(DataMappingManager dataMappingManager, String requestActionId, String paramPath, String fixedPath);

    /**
     * 根据paramPath，获取上一级节点的实际路径作为限定范围路径，取该范围路径下的数据量
     *
     * @param dataMappingManager
     * @param requestActionId
     * @param paramPath
     * @return
     */
    public abstract int getPathDataSize(DataMappingManager dataMappingManager, String requestActionId, String paramPath);

    /**
     * 判断映射关系在数据上是否存在
     *
     * @param dataMappingManager
     * @param requestActionId
     * @param paramPath
     * @return
     */
    public abstract boolean pathIsExistsInData(DataMappingManager dataMappingManager, String requestActionId, String paramPath);

    /**
     * 去除不需要的传参
     *
     * @param val
     * @param context
     * @return
     */
    private Object exclude(Object val, ExecuteContext context) {
        if (null == val || CollectionUtils.isEmpty(except)) {
            return val;
        }
        boolean isArray = false;
        String jsonPathPrefix = CommonConstant.JSON_PATH_PREFIX;
        if (val instanceof List) {
            isArray = true;
            jsonPathPrefix = CommonConstant.JSON_ARRAY_PATH_PREFIX + CommonConstant.SEPARATOR_DOT;
        }
        String valJson = JSON.toJSONString(val);
        String elementPathPrefix = StringUtils.replace(valParamPath, value, "");
        for (int i = 0; i < except.size(); i++) {
            ParamElement paramElement = context.getParamElement(elementPathPrefix + except.getString(i));
            if (null != paramElement) {
                String path = StringUtils.replace(paramElement.getElementJsonPath(), valJsonPath, jsonPathPrefix);
                valJson = JsonPath.parse(valJson).delete(path).jsonString();
            }
        }

        if (isArray) {
            return JSONArray.parse(valJson);
        }
        return JSONObject.parse(valJson);
    }

    public String getName() {
        return name;
    }

    public String getValue() {
        return value;
    }

    /**
     * 通过参数映射关系设置数据的实际下标
     *
     * @param paramPath
     * @param pos
     */
    public abstract void setParentPathPos(DataMappingManager dataMappingManager, String requestActionId, String paramPath, int pos);

    /**
     * 通过参数映射关系获取数据的实际下标
     *
     * @param paramPath
     */
    public abstract void setPathPos(DataMappingManager dataMappingManager, String requestActionId, String paramPath, int pos);

    /**
     * 通过参数映射关系获取数据的真实的JsonPath
     *
     * @param paramPath
     */
    public abstract String getDataJsonPath(DataMappingManager dataMappingManager, String requestActionId, String paramPath);
}
