package com.digiwin.athena.executionengine.service.facade.analyzer.impl;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.digiwin.athena.executionengine.constant.CommonConstant;
import com.digiwin.athena.executionengine.constant.LogConstant;
import com.digiwin.athena.executionengine.core.container.SolutionStepContext;
import com.digiwin.athena.executionengine.model.Metric;
import com.digiwin.athena.executionengine.model.input.SolutinStep.*;
import com.digiwin.athena.executionengine.service.facade.analyzer.IDataValueExtractor;
import com.digiwin.athena.executionengine.service.facade.analyzer.ISolutionStepMetricAnalyzerFacade;
import com.digiwin.athena.executionengine.model.solutionStep.SolutionStepMetric;
import com.digiwin.athena.executionengine.service.facade.data.IScraper;
import com.digiwin.athena.executionengine.service.facade.execution.IExecutionFacade;
import com.digiwin.athena.executionengine.service.facade.execution.ISolutionStepExecutionFacade;
import com.digiwin.athena.executionengine.service.facade.execution.impl.SolutionStepExecutionFacade;
import com.digiwin.athena.executionengine.util.ContextUtils;
import com.digiwin.athena.executionengine.util.LogUtils;
import com.digiwin.athena.executionengine.vo.MetricResultVo;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * @description:
 * @author: Lee
 * @create: 2024/09/19
 */
@Service
public class SolutionStepMetricAnalyzer implements ISolutionStepMetricAnalyzerFacade {

    private static final Logger LOGGER = LoggerFactory.getLogger(SolutionStepMetricAnalyzer.class);

    @Autowired
    private IDataValueExtractor dataValueExtractor;
    private IExecutionFacade executionFacade;

    @Autowired
    public void setExecutionFacade(IExecutionFacade executionFacade) {
        this.executionFacade = executionFacade;
    }

    private ISolutionStepExecutionFacade solutionStepExecutionFacade;

    @Autowired
    public void setSolutionStepExecutionFacade(ISolutionStepExecutionFacade solutionStepExecutionFacade) {
        this.solutionStepExecutionFacade = solutionStepExecutionFacade;
    }

    /**
     * 指标分析
     *
     * @param solutionStep      solutionStep
     * @param context           context
     * @param solutionStepIndex solutionStepIndex
     */
    @Override
    public void analyseMetric(SolutionStep solutionStep, SolutionStepContext context, Integer solutionStepIndex) {
        findMetrics(solutionStep, solutionStepIndex, context);
    }

    /**
     * 查找solutionStep中的所有Metric
     *
     * @param solutionStep      solutionStep
     * @param solutionStepIndex 步骤序列
     * @param context           context
     */
    private void findMetrics(SolutionStep solutionStep, int solutionStepIndex, SolutionStepContext context) {
        LOGGER.info("开始查找solutionStep中的Metric:{},stepIndex:{}", solutionStep.getAlias(), solutionStepIndex);
        //1.先判断datasetObject
        DatasetObject datasetObject = solutionStep.getDatasetObject();
        datasetObjectMetricAnalyzer(solutionStep, datasetObject, solutionStepIndex, context);
        //2.判断select中的datasetObject
        List<Select> selectList = solutionStep.getSelect();
        if (selectList != null) {
            LOGGER.info("开始查找selectList中的Metric:{},stepIndex:{}", selectList.size(), solutionStepIndex);
            for (Select select : selectList) {
                datasetObjectMetricAnalyzer(null, select.getDatasetObject(), solutionStepIndex, context);
            }
        }
        //3.判断join中的datasetObject
        List<Join> joinList = solutionStep.getJoin();
        if (joinList != null) {
            LOGGER.info("开始查找joinList中的Metric:{},stepIndex:{}", joinList.size(), solutionStepIndex);
            for (Join join : joinList) {
                datasetObjectMetricAnalyzer(null, join.getDatasetObject(), solutionStepIndex, context);
            }
        }
        //4.判断filter中的datasetObject
        Filter filter = solutionStep.getFilter();
        if (filter != null && filter.getChildren() != null) {
            LOGGER.info("开始查找filter中的Metric:filter,stepIndex:{}", solutionStepIndex);
            List<FilterHavingChildren> childrenList = filter.getChildren();
            filterHavingMetricAnalyzer(childrenList, solutionStepIndex, context);
        }
        //5.判断having中的datasetObject
        Having having = solutionStep.getHaving();
        if (having != null && having.getChildren() != null) {
            LOGGER.info("开始查找having中的Metric:having,stepIndex:{}", solutionStepIndex);
            filterHavingMetricAnalyzer(having.getChildren(), solutionStepIndex, context);
        }
        //6.判断union中的datasetObject
        List<Union> unionList = solutionStep.getUnion();
        if (unionList != null) {
            LOGGER.info("开始查找unionList中的Metric:{},stepIndex:{}", unionList.size(), solutionStepIndex);
            for (Union union : unionList) {
                SolutionStep unionSolutionStep = union.getSubQuery();
                findMetrics(unionSolutionStep, solutionStepIndex, context);
            }
        }
    }

    /**
     * 分析datasetObject中的Metric
     *
     * @param baseSolutionStep  datasetObject所在的solutionStep
     * @param datasetObject     datasetObject
     * @param solutionStepIndex 大步骤序列
     * @param context           context
     */
    private void datasetObjectMetricAnalyzer(SolutionStep baseSolutionStep, DatasetObject datasetObject, int solutionStepIndex, SolutionStepContext context) {
        if (datasetObject != null) {
            LOGGER.info("datasetObjectMetricAnalyzer开始...");
            String datasetType = datasetObject.getDatasetType();
            if (CommonConstant.METRIC.equalsIgnoreCase(datasetType)
                    || CommonConstant.METRIC_DATASET.equalsIgnoreCase(datasetType) || CommonConstant.DATASET.equals(datasetType)) {
                //指标
                LOGGER.info("datasetObjectMetricAnalyzer分析datasetObject为Metric...");
                String alias = datasetObject.getAlias();
                List<SolutionStepMetric> metrics = SolutionStepExecutionFacade.getSolutionStepMetrics();
                if (metrics.stream().noneMatch(item -> item.getAlias().equals(alias))) {
                    LOGGER.info("datasetObjectMetricAnalyzer分析datasetObject为Metric，且指标没有重复出现，开始指标取数...");
                    SolutionStepMetric metric = new SolutionStepMetric();
                    metric.setMetric(datasetObject.getMetric());
                    metric.setAlias(alias);
                    metric.setCount(metric.getCount() == null || metric.getCount() == 0 ? 1 : metric.getCount() + 1);
                    Filter filter = baseSolutionStep == null ? null : createrMetricFilter(baseSolutionStep.getFilter(), alias, solutionStepIndex, context);
                    metric.setFilter(filter);
                    Having having = baseSolutionStep == null ? null : createMetricHaving(baseSolutionStep.getHaving(), alias, solutionStepIndex, context);
                    metric.setHaving(having);
                    List<GroupBy> groupByList = baseSolutionStep == null ? null : createMetricGroupBy(baseSolutionStep.getGroupby(), baseSolutionStep.getJoin(), baseSolutionStep.getSelect(), alias);
                    metric.setGroupBy(groupByList);
                    //目前看，所有的计算中字段的source都应该是当前SolutionStep，不可能跨步骤计算
                    metric.setComputeList(baseSolutionStep == null ? null : baseSolutionStep.getComputeList());
                    metric.setSort(baseSolutionStep == null ? null : baseSolutionStep.getSort());
                    metric.setSelect(baseSolutionStep.getSelect());
                    metric.setModelId(context.getModelIdMapping().get(metric.getMetric()));
                    metric.setModelCode(context.getModelCodeMapping().get(metric.getMetric()));
                    if (context.isNeedData()) {
                        fetchData(metric, context, datasetType);
                    } else {
                        buildMetricExportConditions(metric);
                    }
                    metrics.add(metric);
                }
            } else if (CommonConstant.SUB_QUERY.equalsIgnoreCase(datasetType)) {
                //如果是子查询,递归进行判断
                LOGGER.info("datasetObjectMetricAnalyzer分析datasetObject为subquery,继续往下查找...");
                findMetrics(datasetObject.getSubQuery(), solutionStepIndex, context);
            }
        }
    }

    private void buildMetricExportConditions(SolutionStepMetric metric) {
        JSONObject dynamicSchema = new JSONObject();
        dynamicSchema.put("groupBy", CollectionUtils.isEmpty(metric.getGroupBy()) ? null : JSONArray.parseArray(JSONArray.toJSONString(metric.getGroupBy())));
        dynamicSchema.put("filter", metric.getFilter() == null || CollectionUtils.isEmpty(metric.getFilter().getChildren()) ? null : JSONObject.parseObject(JSONObject.toJSONString(metric.getFilter())));
        dynamicSchema.put("having", metric.getHaving() == null || CollectionUtils.isEmpty(metric.getHaving().getChildren()) ? null : JSONObject.parseObject(JSONObject.toJSONString(metric.getHaving())));
        dynamicSchema.put("computeList", metric.getComputeList() == null || metric.getComputeList().isEmpty() ? null : metric.getComputeList());
        dynamicSchema.put("sort", CollectionUtils.isEmpty(metric.getSort()) ? null : JSONArray.parseArray(JSONObject.toJSONString(metric.getSort())));
        List<Metric> metricList = SolutionStepExecutionFacade.getExportMetrics();
        if (metricList.stream().noneMatch(m -> m.getAlias().equals(metric.getAlias()))) {
            Metric mObj = new Metric(metric.getMetric(), dynamicSchema, metric.getAlias());
            metricList.add(mObj);
        }
        SolutionStepExecutionFacade.setExportMetrics(metricList);
    }

    /**
     * 分析filter or having的children中的指标
     *
     * @param childrenList list
     * @param context      context
     */
    private void filterHavingMetricAnalyzer(List<FilterHavingChildren> childrenList, int solutionStepIndex, SolutionStepContext context) {
        for (FilterHavingChildren child : childrenList) {
            //4.1 判断其right中的
            List<FilterHavingChildrenRight> childrenRightList = child.getRight();
            if (childrenRightList != null) {
                for (FilterHavingChildrenRight childRight : childrenRightList) {
                    datasetObjectMetricAnalyzer(null, childRight.getDatasetObject(), solutionStepIndex, context);
                }
            }
            //4.2 判断其子children
            List<FilterHavingChildren> childrenInChildrenList = child.getChildren();
            if (childrenInChildrenList != null) {
                filterHavingMetricAnalyzer(childrenInChildrenList, solutionStepIndex, context);
            }
        }
    }

    /**
     * 创建取指标filter,取出可以直接用的，剔除指标、子查询
     *
     * @param filter
     * @param source
     * @return filter
     */
    private Filter createrMetricFilter(Filter filter, String source, int solutionStepIndex, SolutionStepContext context) {
        Filter newFilter = new Filter();
        if (filter != null) {
            newFilter.setLogic(filter.getLogic());
            List<FilterHavingChildren> childrenList = filter.getChildren();
            List<FilterHavingChildren> newChildrenList = createFilterOrHavingChildren(childrenList, source, solutionStepIndex, context);
            newFilter.setChildren(newChildrenList);
        }
        return newFilter;
    }

    /**
     * 创建取指标having,取出可以直接用的，剔除指标、子查询
     *
     * @param having
     * @param source
     * @return
     */
    private Having createMetricHaving(Having having, String source, int solutionStepIndex, SolutionStepContext context) {
        Having newHaving = new Having();
        if (having == null) {
            return newHaving;
        }
        having.setLogic(having.getLogic());
        List<FilterHavingChildren> childrenList = having.getChildren();
        List<FilterHavingChildren> newChildrenList = createFilterOrHavingChildren(childrenList, source, solutionStepIndex, context);
        newHaving.setChildren(newChildrenList);
        return newHaving;
    }

    /**
     * 创建filter中children节点
     *
     * @param childrenList
     * @param source
     * @return
     */
    private List<FilterHavingChildren> createFilterOrHavingChildren(List<FilterHavingChildren> childrenList, String source, int solutionStepIndex, SolutionStepContext context) {
        List<FilterHavingChildren> newChildrenList = new ArrayList<>();
        if (childrenList != null) {
            for (FilterHavingChildren child : childrenList) {
                FilterHavingChildren newChild = new FilterHavingChildren();
                newChild.setLogic(child.getLogic());
                newChild.setOperator(child.getOperator());
                BaseObject newChildLeft = new BaseObject();
                if (child.getLeft() != null && child.getLeft().getSource().equals(source)) {
                    //来源一致用于取指标
                    BeanUtils.copyProperties(child.getLeft(), newChildLeft);
                    newChild.setLeft(newChildLeft);
                } else if (child.getLeft() == null) {
                    newChild.setLeft(null);
                }
                List<FilterHavingChildrenRight> childrenRigthList = child.getRight();
                LOGGER.info("createFilterOrHavingChildren.right:{}", JSONObject.toJSONString(childrenRigthList));
                if (!CollectionUtils.isEmpty(childrenRigthList)) {
                    List<FilterHavingChildrenRight> newChildrenRigthList = new ArrayList<>();
                    for (FilterHavingChildrenRight childRight : childrenRigthList) {
                        if (childRight.getDataObject() != null) {
                            FilterHavingChildrenRight newChildRight = new FilterHavingChildrenRight();
                            if (childRight.getDataObject().getSource().equals(source)) {
                                BaseObject newBaseObject = new BaseObject();
                                BeanUtils.copyProperties(childRight.getDataObject(), newBaseObject);
                                newChildRight.setDataObject(newBaseObject);
                            }//剔除了指标或者子查询的情况
                            newChildrenRigthList.add(newChildRight);
                        } else if (childRight.getDatasetObject() != null) {
                            //如果是子查询
                            Object rightData = null;
                            DatasetObject datasetObject = childRight.getDatasetObject();
                            if ("metric".equals(datasetObject.getDatasetType())) {
                                //直接去指标取数，且不存在指标的过滤条件
                                datasetObjectMetricAnalyzer(null, datasetObject, solutionStepIndex, context);
                                //对取到的指标处理
                                rightData = SolutionStepExecutionFacade.getMetricData(datasetObject.getAlias());
                            } else if ("step".equals(datasetObject.getDatasetType())) {
                                if (SolutionStepExecutionFacade.getTransDataMap().containsKey(datasetObject.getStep())) {
                                    rightData = SolutionStepExecutionFacade.getTransDataMap().get(datasetObject.getStep());
                                    rightData = dataValueExtractor.extract(datasetObject.getContent(), child.getOperator(), rightData);
                                }
                            } else if ("subQuery".equals(datasetObject.getDatasetType())) {
                                //递归执行数据集的操作
                                solutionStepExecutionFacade.doSolutionStep(datasetObject.getSubQuery(), context, solutionStepIndex);
                                //去取数据 通过-alias
                                if (SolutionStepExecutionFacade.getTransDataMap().containsKey(datasetObject.getAlias())) {
                                    rightData = SolutionStepExecutionFacade.getTransDataMap().get(datasetObject.getAlias());
                                }
                            }
                            //处理结果数据
                            if (rightData == null) {
                                LOGGER.info(String.format(LogConstant.AGILE_DATA + "右child为datasetObject未取到数据:%s", "SolutionStepMetricAnalyzer.createFilterOrHavingChildren", JSONObject.toJSONString(childRight.getDatasetObject())));
                            } else {
                                //处理取到的数据----作为常量dataObject处理
                                BaseObject newBaseObject = new BaseObject();
                                newBaseObject.setContentType("const");
                                newBaseObject.setDataType("string");
                                newBaseObject.setSource(source);
                                newBaseObject.setContent((String) rightData);
                                FilterHavingChildrenRight newChildRight = new FilterHavingChildrenRight();
                                newChildRight.setDataObject(newBaseObject);
                                newChildrenRigthList.add(newChildRight);
                            }
                        }
                    }
                    newChild.setRight(newChildrenRigthList);
                } else {
                    newChild.setRight(null);
                }

                //处理left.children.children
                List<FilterHavingChildren> childChildrenList = child.getChildren();
                if (childChildrenList != null) {
                    newChild.setChildren(createFilterOrHavingChildren(childChildrenList, source, solutionStepIndex, context));
                }
                newChildrenList.add(newChild);
            }
        }
        LOGGER.info("createFilterOrHavingChildren:{}", JSONObject.toJSONString(newChildrenList));
        return newChildrenList;
    }


    /**
     * 创建指标的groupby数据
     *
     * @param groupByList
     * @param joinList
     * @param selectList
     * @param source
     * @return
     */
    private List<GroupBy> createMetricGroupBy(List<GroupBy> groupByList, List<Join> joinList, List<Select> selectList, String source) {
        List<GroupBy> newGroupByList = new ArrayList<>();
        if (groupByList == null) {
            groupByList = new ArrayList<>();
        }
        List<GroupBy> onGroupByList = new ArrayList<>();
        if (!CollectionUtils.isEmpty(joinList)) {
            //如果存在关联表，找到关联的字段----关联字段一定会在groupby中
            for (Join join : joinList) {
                List<On> on = join.getOn();
                for (On onItem : on) {
                    //暂时不考虑其他类型，默认field TODO：增加其他类型支持
                    if (source.equals(onItem.getLeft().getSource()) && "field".equals(onItem.getLeft().getContentType())) {
                        GroupBy groupBy = new GroupBy();
                        BeanUtils.copyProperties(onItem.getLeft(), groupBy);
                        onGroupByList.add(groupBy);
                    }
                    if (source.equals(onItem.getRight().getSource()) && "field".equals(onItem.getRight().getContentType())) {
                        GroupBy groupBy = new GroupBy();
                        BeanUtils.copyProperties(onItem.getRight(), groupBy);
                        onGroupByList.add(groupBy);
                    }
                }
            }
        }
        newGroupByList.addAll(onGroupByList);
        //处理groupBy中的字段
        for (GroupBy groupBy : groupByList) {
            if (groupBy.getSource().equals(source)) {
                if (newGroupByList.stream().noneMatch(
                        item -> item.getContent().equals(groupBy.getContent())
                                && item.getSource().equals(groupBy.getSource()))) {
                    GroupBy newGroupBy = new GroupBy();
                    BeanUtils.copyProperties(groupBy, newGroupBy);
                    newGroupByList.add(newGroupBy);
                }
            }
        }
        //将select中非number类型且是field类型的的字段也加入到group by中----如果group by中不存在的话
        for (Select select : selectList) {
            DataObject dataObject = select.getDataObject();
            if (dataObject != null && !dataObject.getDataType().equals("number") && dataObject.getContentType().equals("field")) {
                String content = dataObject.getContent();
                if (newGroupByList.stream().noneMatch(groupBy -> groupBy.getContent().equals(content))) {
                    GroupBy newGroupBy = new GroupBy();
                    BeanUtils.copyProperties(dataObject, newGroupBy);
                    newGroupByList.add(newGroupBy);
                }
            }
        }
        return newGroupByList;
    }

    /**
     * 获取指标数据
     *
     * @param metric      指标信息
     * @param context     context
     * @param dataSetType 数据集类型  1.0 metric / metric_table  2.0 table
     */
    public void fetchData(SolutionStepMetric metric, SolutionStepContext context, String dataSetType) {
        MetricResultVo result = ContextUtils.getBean(dataSetType, IScraper.class).fetchData(metric, context);
        JSONObject resultJson = JSONObject.parseObject(JSONObject.toJSONString(result.getData(), SerializerFeature.WriteMapNullValue));
        if (context.getParam().containsKey("debug") && "true".equals(String.valueOf(context.getParam().get("debug")))) {
            SolutionStepExecutionFacade.setDebugDtoListLocal(result.getDebugDto());
        }

        if (resultJson.get("data") != null) {
            JSONArray jsonArray = resultJson.getJSONArray("data");
            metric.setData(jsonArray);
            LogUtils.buildAgileLog(LogConstant.AGILE_CODE_GET_METRIC_DATA, LogUtils.SUCCESS, "", "指标取数成功，具体数据不展示", "");
            LOGGER.info(String.format(LogConstant.AGILE_DATA + "取指标数据出参:指标取数共取出数据%s条", "SolutionStepMetricAnalyzer.getMetricData", jsonArray.size()));
        } else {
            LOGGER.info(String.format(LogConstant.AGILE_DATA + "取指标数据出参:%s", "SolutionStepMetricAnalyzer.getMetricData", "未取到数据"));
            LogUtils.buildAgileLog(LogConstant.AGILE_CODE_GET_METRIC_DATA, LogUtils.WARING, "", JSONObject.toJSONString(resultJson), "检查下入参与pulldata日志，检查是否取数过程中存在问题");
        }
    }
}