package com.digiwin.athena.executionengine.component.action;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.TypeReference;
import com.digiwin.athena.executionengine.component.domain.ActionParam;
import com.digiwin.athena.executionengine.component.param.CaseParam;
import com.digiwin.athena.executionengine.component.param.MultiPathParam;
import com.digiwin.athena.executionengine.component.param.PullingParam;
import com.digiwin.athena.executionengine.constant.CommonConstant;
import com.digiwin.athena.executionengine.constant.MetaDataConstant;
import com.digiwin.athena.executionengine.constant.MetaDefinitionConstant;
import com.digiwin.athena.executionengine.constant.ResultStatusConstant;
import com.digiwin.athena.executionengine.core.container.ExecuteContext;
import com.digiwin.athena.executionengine.model.DataDescription;
import com.digiwin.athena.executionengine.service.facade.execution.asyc.AsyncTask;
import com.digiwin.athena.executionengine.service.facade.execution.asyc.IAsyncExecutor;
import com.digiwin.athena.executionengine.service.facade.mapping.param.ParamMappingUtils;
import com.digiwin.athena.executionengine.model.ServiceResult;
import com.digiwin.athena.executionengine.component.param.ParamBase;
import com.digiwin.athena.executionengine.service.facade.schema.dynamic.IDynamicCalculateBuilder;
import com.digiwin.athena.executionengine.service.facade.transform.IAggregator;
import com.digiwin.athena.executionengine.service.facade.transform.IFlattener;
import com.digiwin.athena.executionengine.service.facade.transform.MetaNode;
import com.digiwin.athena.executionengine.service.srp.ITransSrp;
import com.digiwin.athena.executionengine.util.ContextUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;

import java.util.*;
import java.util.stream.Collectors;

/**
 * @description:
 * @author: renwm
 * @date: 2020/6/12 16:06
 */
public abstract class ActionBase implements IActionExecutionProcessor {

    /**
     * 子类实现具体执行操作
     *
     * @param context
     * @param reqMap
     * @param actionParam
     * @return
     * @throws Exception
     */
    protected abstract Object actionExecute(ExecuteContext context, Map<String, Object> reqMap, ActionParam actionParam);


    /**
     * 执行action具体操作
     *
     * @param context
     * @param actionParam
     * @return
     */
    public boolean execute(ExecuteContext context, ActionParam actionParam) {

        prepare(context, actionParam);
        //主线程等待
        try {
            synchronized (context) {
                context.wait();
            }
        } catch (InterruptedException e) {
            throw new RuntimeException(e);
        }
        return actionParam.getActionState() == ResultStatusConstant.EXECUTE_SUCCESS.getCode();
    }


    public void prepare(ExecuteContext context, ActionParam actionParam) {

        if (actionParam.getActionState() == ResultStatusConstant.EXECUTE_SUCCESS.getCode()) {
            return;
        }

        //预处理参数
        doPrepare(context, actionParam);

    }

    @Override
    public List<Map<String, Object>> postProcessBeforeExecution(List<Map<String, Object>> request, ActionParam actionParam) {
        if (CollectionUtils.isEmpty(actionParam.getApiMeta()) || CollectionUtils.isEmpty(request)) {
            return request;
        }

        Map<String, Object> stringObjectMap = request.get(0);

        List<Map<String, Object>> data = (List<Map<String, Object>>) stringObjectMap.get("data");


        //api类型需要将展平的数据聚合
        IAggregator iAggregator = ContextUtils.getBean(IAggregator.class);
        Map<String, MetaNode> nodeMap = iAggregator.analyzeAPIMetaData(actionParam.getApiMeta());
        MetaNode node = nodeMap.values().stream().max(Comparator.comparingInt(MetaNode::getDepth)).get();
        int maxDepth = node.getDepth();
        if (maxDepth == 1) {
            return data;
        }

        JSONObject result = new JSONObject();
        iAggregator.processAggregation(result, data, 1, maxDepth, nodeMap);
        return (List<Map<String, Object>>) result.get("result");
    }

    @Override
    public Object postProcessAfterExecution(Object response, ActionParam actionParam) {
        if (CollectionUtils.isEmpty(actionParam.getApiMeta())) {
            return response;
        }

        Map<String, Object> inputMap = new HashMap<>();
        //先这样处理
        inputMap.put(CommonConstant.EMPTY_KEY, response);
        List<Map<String, Object>> flattenData = ContextUtils.getBean(IFlattener.class).flatten(CommonConstant.EMPTY_KEY, inputMap);
        Map<String, Object> result = new HashMap<>();
        result.put(CommonConstant.DATA_KEY, flattenData);
        return result;
    }

    @Override
    public Object dynamicCompute(ExecuteContext context, Object response, ActionParam actionParam) {
        JSONObject metaObj = actionParam.getMetaObj();
        String actionMode = metaObj.getString(MetaDataConstant.ACTION_MODE);
        if (!MetaDefinitionConstant.DYNAMIC.equals(actionMode)) {
            return response;
        }
        if (response == null || !(response instanceof Map)) {
            return response;
        }

        Map<String, Object> result = (Map<String, Object>) response;
        if (!result.containsKey("data")) {
            return response;
        }

        List<Map<String, Object>> dataList = (List<Map<String, Object>>) result.get("data");

        //动态模式下，需要对数据进行动态计算
        DataDescription dataDescription = context.getInputParam().getDataDescription();
        JSONObject dynamicSchema = dataDescription.getDynamicSchema();
        JSONArray recast = new JSONArray();
        if (MapUtils.isNotEmpty(dynamicSchema)) {
            dynamicSchema.forEach((k, v) -> {
                String beanName = CommonConstant.CALCULATE_BUILDER_MAP.get(k);
                if (StringUtils.isNotEmpty(beanName)) {
                    JSONObject buildSchema = ContextUtils.getBean(beanName, IDynamicCalculateBuilder.class).build(v, actionParam);
                    if (MapUtils.isNotEmpty(buildSchema)) {
                        recast.add(buildSchema);
                    }
                }
            });
        }

        List<Map<String, Object>> listOfMaps = JSON.parseObject(JSON.toJSONString(recast), new TypeReference<List<Map<String, Object>>>() {
        });
        ServiceResult serviceResult = ContextUtils.getBean(ITransSrp.class).runTrans(context.getTenantId(), listOfMaps, dataList);
        if (serviceResult.getSuccess() && serviceResult.getData() != null && dynamicSchema.get("groupBy") != null) {
            List<Map<String, Object>> data = (List<Map<String, Object>>) serviceResult.getData();
            //取设计器的维度
            JSONObject aggregation = actionParam.getAggregation();
            JSONArray dimensions = aggregation.getJSONArray("dimensions");
            List<String> dimensionNames = new ArrayList<>();
            for (int i = 0; i < dimensions.size(); i++) {
                dimensionNames.add(dimensions.getJSONObject(i).getString("data_name"));
            }
            //语义给到了groupBy同时取到数据
            JSONArray groupArray = dynamicSchema.getJSONArray("groupBy");
            //存储语义的维度字段
            List<String> groupList = new ArrayList<>();
            for (int i = 0; i < groupArray.size(); i++) {
                String groupField = groupArray.getJSONObject(i).getString("content");
                groupList.add(groupField);
            }
            //遍历数据判断是否需要对没有问到的维度设置default
            for (Map<String, Object> datum : data) {
                for (Map.Entry<String, Object> entry : datum.entrySet()) {
                    //字段不在语义的group维度中，且不是度量(用设计器给的维度字段判断，这个字段肯定在设计器维度中)
                    if (!groupList.contains(entry.getKey()) && dimensionNames.contains(entry.getKey())) {
                        //不在语义给出的维度字段，设置为default
                        datum.put(entry.getKey(), "default");
                    }
                }
            }
        }
        result.put(CommonConstant.DATA_KEY, serviceResult.getData());
        return result;
    }

    public boolean exec(ExecuteContext context, ActionParam actionParam) {
        String actionId = actionParam.getActionId();
        boolean isForeach = actionParam.isForeach();
        ParamMappingUtils paramMappingUtils = new ParamMappingUtils(context, actionId);
        //最终会发起api调用的数据集合
        List<Object> apiData = new ArrayList<>();
        //3执行action
        ServiceResult serviceResult = paramMappingUtils.mappingReqParam(apiData, actionParam.getRequestParams());

        if (!serviceResult.getSuccess()) {
            return false;
        }

        //合并apiData，生成多个完整的含有key结构的api的入参
        List<Map<String, Object>> apiParams = paramMappingUtils.mergeApiData(apiData);

        //设置循环执行结果的状态
        boolean foreachRes = true;

        if (apiParams.size() > 1 && !isForeach) {
            //throw ExceptionUtils.buildVerificationException(ErrorCodeEnum.IS_FOREACH_MALFORMATION, actionId);
            return false;
        }

        apiParams = postProcessBeforeExecution(apiParams, actionParam);

        //循环执行action
        for (Map<String, Object> apiParam : apiParams) {
            //先将状态设置false，actionExecute方法中会根据执行结果重置executeStatus
            context.setCurrentActionId(actionId);
            //此处是为了拿到代理对象，使得异常重试切面生效
            Object resp = ContextUtils.getBean(actionParam.getActionName(), ActionBase.class).actionExecute(context, apiParam, actionParam);
            context.addActionResponse(actionId, resp, isForeach, actionParam.getActionType());

            if (foreachRes && !context.isExecuteStatus()) {
                foreachRes = false;
            }
        }

        ///如果走到这里说明
        // 1、执行规则是合法的，有metaData.request必然要有至少一个字段的映射关系
        // -> FIXME,这个合法性校验应该在之前就做了，如果不合法，就不应该发起action的调用。
        // 2、如果有metaData.request字段，并且有映射关系，但是数据源没有数据：
        //    2.1、metaData.request允许为空，这里应该允许下发一次。
        //    2.2、metaData.request有不允许为空的字段，这些字段都没有值，这种情况应该要屏蔽，不应该下发调用！
        ////-> FIXME,2.2的场景应该在合并apiDataList的时候做一定的处理，校验api数据合法性，不合法这里不下发。
        // 3、如果没有metaData.request字段，并没有映射关系。这时候是合理的，应该下发一次。
        //====> 结论:走到这里如果数据合法，那么应该至少要调用一次。
        if (apiParams.size() == 0) {
            context.setCurrentActionId(actionId);
            Object resp = ContextUtils.getBean(actionParam.getActionName(), ActionBase.class).actionExecute(context, new HashMap<>(), actionParam);
            context.addActionResponse(actionId, resp, isForeach, actionParam.getActionType());

            if (foreachRes && !context.isExecuteStatus()) {
                foreachRes = false;
            }
        }

        Object res = postProcessAfterExecution(context.getActionResponse(actionId), actionParam);

        Object dynamicComputeRes = dynamicCompute(context, res, actionParam);


        if (dynamicComputeRes != null) {
            context.getActionResponse().put(actionId, dynamicComputeRes);
        }

        //当前action执行完成 清空当前上下文中action 依赖的action数据 包含readContext对象，dataMappingHandler对象
        if (context.isRelease()) {
            releaseMemory(context, actionParam);
        }
        return foreachRes;
    }

    private void releaseMemory(ExecuteContext context, ActionParam actionParam) {
        List<ParamBase> params = actionParam.getParams();
        params.forEach(param -> {
            if (param instanceof PullingParam) {
                PullingParam pullingParam = (PullingParam) param;
                context.getActionResponse().remove(pullingParam.getSource());
                context.getParamGetDataHandler().removeReadContext(pullingParam.getSource());
                context.getDataMappingManager().release(actionParam.getActionId(), pullingParam.getSource());
            }

        });
    }

    /**
     * 参数预处理,获取到的参数缓存到 paramValMap
     *
     * @return
     */
    private void doPrepare(ExecuteContext context, ActionParam actionParam) {
        //同步处理
        //specialParamHandle(context, actionParam);


        Set<String> sourceSet = new HashSet<>();
        List<PullingParam> finalParams = actionParam.getParams().stream().filter(paraObj -> paraObj instanceof PullingParam).map(pullingParam -> (PullingParam) pullingParam).filter(pullingParam -> sourceSet.add(pullingParam.getSource())).collect(Collectors.toList());

        //统计上下文中准备好的依赖
        long count = finalParams.stream().filter(pullingParam -> context.getActionResponse().containsKey(pullingParam.getSource())).count();

        long failedCnt = sourceSet.stream().filter(k -> context.getActionParam(k).getActionState() == ResultStatusConstant.EXECUTED_FAILD.getCode()).count();
        if (finalParams.size() == count) {
            ContextUtils.getBean(IAsyncExecutor.class).addTask(new AsyncTask(context, actionParam));
            return;
        } else if (failedCnt > 0) {
            ContextUtils.getBean(IAsyncExecutor.class).addTask(new AsyncTask(context, actionParam, false));
            return;
        } else {
            //放入等待队列
            context.getActionQueue().addNode(actionParam.getActionId(), sourceSet);
        }

        for (ParamBase paraObj : finalParams) {
            //获取参数的value值
            paraObj.parEvalAndExclude(context);
        }
    }

    private boolean specialParamHandle(ExecuteContext context, ActionParam actionParam) {
        List<ParamBase> specialParams = actionParam.getParams().stream().filter(paraObj -> (paraObj instanceof CaseParam) || (paraObj instanceof MultiPathParam)).collect(Collectors.toList());

        if (CollectionUtils.isEmpty(specialParams)) {
            return true;
        }
        for (ParamBase paraObj : specialParams) {
            //获取参数的value值
            boolean parVal = paraObj.parEvalAndExclude(context);
            if (!parVal) {
                return false;
            }
        }

        return true;
    }

}
