package com.digiwin.athena.executionengine.component.action;

import com.alibaba.fastjson.JSONObject;
import com.digiwin.app.service.utils.DWServiceChainUtils;
import com.digiwin.athena.esp.sdk.exception.InvocationException;
import com.digiwin.athena.esp.sdk.model.RequestModel;
import com.digiwin.athena.esp.sdk.model.ResponseModel;
import com.digiwin.athena.executionengine.component.domain.ActionParam;
import com.digiwin.athena.executionengine.constant.CommonConstant;
import com.digiwin.athena.executionengine.constant.FieldNameConstant;
import com.digiwin.athena.executionengine.constant.LogConstant;
import com.digiwin.athena.executionengine.constant.MetaDataConstant;
import com.digiwin.athena.executionengine.constant.ModuleProperty;
import com.digiwin.athena.executionengine.core.container.ExecuteContext;
import com.digiwin.athena.executionengine.dto.LogDataDto;
import com.digiwin.athena.executionengine.dto.LogDto;
import com.digiwin.athena.executionengine.enumtype.ErrorCodeEnum;
import com.digiwin.athena.executionengine.exception.ResponseStatusException;
import com.digiwin.athena.executionengine.util.JsonResolverUtils;
import com.digiwin.athena.executionengine.util.PlatformServiceUtils;
import com.digiwin.athena.executionengine.util.ServiceResponseUtils;
import com.google.common.collect.Maps;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.LoggerFactory;
import org.slf4j.Logger;
import org.springframework.stereotype.Service;

import java.util.*;

/**
 * @description:
 * @author: renwm
 * @date: 2020/6/15 9:29
 */
@Service("EspAction")
public class EspAction extends ActionBase {
    private static final Logger LOGGER = LoggerFactory.getLogger(EspAction.class);
    private static final String STD_DATA = "std_data";
    private static final String STD_DATA_PARAMETER = "parameter";
    private static final String STD_DATA_PARAMETER_CALL_ID = "call_id";
    private static final String STD_DATA_PARAMETER_SITE_NO = "site_no";
    private static final String STD_DATA_PARAMETER_ENTERPRISE_NO = "enterprise_no";


    @Override
    public Object actionExecute(ExecuteContext context, Map<String, Object> reqMap, ActionParam actionParam) {
        ResponseModel responseModel = null;
        try {
            Map<String, Object> bodyMap = new HashMap<>();
            Map<String, Object> dataMap = new HashMap<>();
            Map<String, Object> parameter = new HashMap<>();
            bodyMap.put(STD_DATA, dataMap);
            dataMap.put(STD_DATA_PARAMETER, parameter);
            parameter.put(STD_DATA_PARAMETER_ENTERPRISE_NO, "");
            parameter.put(STD_DATA_PARAMETER_SITE_NO, "");
            parameter.put(STD_DATA_PARAMETER_CALL_ID, UUID.randomUUID().toString());
            parameter.putAll(reqMap);

            RequestModel requestModel = getRequestModel(context, bodyMap, actionParam);
            responseModel = PlatformServiceUtils.callEsp(requestModel);
            LOGGER.info("执行引擎调用ESP执行{}:{}返回的结果为：{}", actionParam.getActionId(), parameter, responseModel.getBodyJsonString());
            logTrace(responseModel, true);
            JSONObject dataObj = ServiceResponseUtils.getEspRespData(responseModel.getBodyJsonString());
            boolean status = ServiceResponseUtils.getEspRespStatus(dataObj);
            context.setExecuteStatus(status);
            return ServiceResponseUtils.getEspRespParameter(dataObj);
        } catch (InvocationException e) {
            logTrace(responseModel, false);
            LOGGER.error("执行引擎调用esp执行{}异常:{}", actionParam.getActionId(), e);
            ResponseStatusException responseStatusException = new ResponseStatusException(e.getErrorCode()
                    , e.getErrorMessage());
            responseStatusException.getInstructors().put(CommonConstant.CHAIN_INFO,
                    e.getChainInfo());
            throw responseStatusException;
        } catch (Exception e) {
            logTrace(responseModel, false);
            LOGGER.error("执行引擎调用esp执行{}异常:{}", actionParam.getActionId(), e);
            context.setExecuteStatus(false);
            throw new ResponseStatusException(ErrorCodeEnum.CALL_ESP_EXCEPTION.getCode()
                    , ErrorCodeEnum.CALL_ESP_EXCEPTION.getMessage(), e);
        }
    }

    /**
     * 组装请求esp的RequestModel对象
     *
     * @param context
     * @param bodyMap
     * @param actionParam
     * @return
     */
    private RequestModel getRequestModel(ExecuteContext context, Map<String, Object> bodyMap, ActionParam actionParam) {
        String serverName = actionParam.getMetaObj().getString(MetaDataConstant.ACTION_SERVICE_NAME);
        String productName = actionParam.getActionJson().getString(MetaDataConstant.ACTION_VARIABLE_PRODUCT_NAME);

        RequestModel requestModel = new RequestModel();
        requestModel.setHostProd("ExecutionEngine");
        requestModel.setHostVer("1.0");
        requestModel.setHostId(ModuleProperty.ESP_AP_ID);
        requestModel.setHostAcct(ModuleProperty.ESP_HOSTACCT);
        requestModel.setServiceName(serverName);
        requestModel.setBodyJsonString(JsonResolverUtils.toJsonString(bodyMap));
        requestModel.setTenantId(context.getTenantId());
        requestModel.setServiceProd(productName);
        requestModel.setLanguage(context.getLocale());
        Map<String, String> header = new HashMap<>();
        DWServiceChainUtils.beforeInvokeOutterAPI(header::put);
        header.put(FieldNameConstant.USER_TOKEN, context.getToken());
        header.put(FieldNameConstant.CAMEL_CASE_ROUTER_KEY, context.getRouterKey());
        header.put(FieldNameConstant.APP_TOKEN, ModuleProperty.MULTIPLEAPPENV?ModuleProperty.MERGED_IAM_APP_TOKEN:ModuleProperty.APP_TOKEN_VALUE);
        requestModel.setHeaderMap(header);
        requestModel.setEocMap(context.getEocMap());
        if (context.getLocale() != null) {
            requestModel.setLanguage(context.getLocale());
        }
        // 拿到 appCode（application），透传给 esp，esp 给到业务中台用作运营统计分析
        Map<String, String> datakeymap = Maps.newHashMap();
        // 有些应用在解析 digi-datakey 的时候，如果里面的值有 null 会报错
        if (StringUtils.isNotBlank(context.getApplication())) {
            datakeymap.put("appCode", context.getApplication());
        }
        if (StringUtils.isNotBlank(context.getTaskType())) {
            datakeymap.put("taskType", context.getTaskType());
        }
        if (StringUtils.isNotBlank(context.getTaskCode())) {
            datakeymap.put("taskCode", context.getTaskCode());
        }
        if (MapUtils.isNotEmpty(datakeymap)) {
            requestModel.setDatakeyMap(datakeymap);
        }
        return requestModel;
    }


    /**
     * 埋点 调用esp
     *
     * @param responseModel
     * @param success
     */
    private void logTrace(ResponseModel responseModel, boolean success) {
        if (responseModel == null) {
            return;
        }
        String message = success ? "调用esp成功" : "调用esp异常";
        LogDataDto logDataDto = new LogDataDto(responseModel.getReqid(), "esp请求id", LogConstant.TYPE_LINK, LogConstant.LABEL_ESP);
        LogDto logDto = new LogDto(message, Arrays.asList(logDataDto));
        if (success) {
            LOGGER.info(logDto.toString());
        } else {
            LOGGER.error(logDto.toString());
        }
    }
}
