package com.digiwin.athena.executionengine.component.action;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.digiwin.athena.executionengine.component.domain.ActionParam;
import com.digiwin.athena.executionengine.constant.AgileDataErrorCodeConstant;
import com.digiwin.athena.executionengine.constant.LogConstant;
import com.digiwin.athena.executionengine.constant.MetaDataConstant;
import com.digiwin.athena.executionengine.core.aop.ActionMock;
import com.digiwin.athena.executionengine.core.aop.Debug;
import com.digiwin.athena.executionengine.core.container.ExecuteContext;
import com.digiwin.athena.executionengine.dto.ErrorLog;
import com.digiwin.athena.executionengine.enumtype.ErrorCodeEnum;
import com.digiwin.athena.executionengine.exception.BusinessException;
import com.digiwin.athena.executionengine.util.DateUtils;
import com.digiwin.athena.executionengine.util.LogUtils;
import com.digiwin.athena.executionengine.util.ReplaceUtils;
import jdk.nashorn.api.scripting.ScriptObjectMirror;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import javax.script.ScriptContext;
import javax.script.ScriptEngine;
import javax.script.ScriptEngineManager;
import javax.script.SimpleScriptContext;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

/**
 * @description:
 * @author: ZhangJun
 * @create: 2024/2/29
 */
@Service("FormulaCAction")
public class FormulaCAction extends ActionBase {
    private static final Logger LOGGER = LoggerFactory.getLogger(FormulaAction.class);
    private static final String SCRIPT_ENGINE_SHORT_NAME = "nashorn";
    private static final String SCRIPT_METHOD_TEMPLATE = "function transform(){%s} transform();";

    @ActionMock
    @Debug
    @Override
    public Object actionExecute(ExecuteContext context, Map<String, Object> exprParams, ActionParam actionParam) {
        JSONObject actionJson = actionParam.getActionJson();
        String expression = actionJson.getString(MetaDataConstant.ACTION_EXPRESSION);
        String target = actionJson.getString(MetaDataConstant.ACTION_TARGET);
        if (StringUtils.isEmpty(expression)) {
            LOGGER.warn("执行引擎执行{} FormulaC {} 没有对应的参数表达式", actionParam.getActionId(), expression);
            context.setExecuteStatus(false);
            return null;
        }
        try {
            //运行表达式脚本并存储计算结果
            Map<String, Object> returnMap = new HashMap<>();
            //如果入参没有，就不执行表达式，避免错误
            if (actionParam.getRequestParams().size() == exprParams.size()) {
                //运行表达式脚本并存储计算结果
                Object result = doExpression(expression, exprParams);
                returnMap.put(target, convert(result));
            }
            context.setExecuteStatus(true);
            LogUtils.buildAgileLog(LogConstant.AGILE_CODE_EXECUTE_FORMULAC_ACTION, LogUtils.SUCCESS, "actionId:" + actionParam.getActionId() + ",action入参:" + JSONObject.toJSONString(exprParams) + "，js脚本:" + expression, JSONObject.toJSONString(returnMap), "");
            return returnMap;
        } catch (Exception e) {
            LOGGER.error("执行引擎Formula[{}]执行表达式异常:{}", actionParam.getActionId(), e);
            context.setExecuteStatus(false);
            ErrorLog errorLog = new ErrorLog();
            errorLog.setErrorCode(AgileDataErrorCodeConstant.AGILE_FORMULA_ERROR_CODE);
            errorLog.setErrorLocation("计算节点执行异常，节点id:" + actionParam.getActionId());
            errorLog.setErrorDescription("javascript脚本执行异常");
            errorLog.setErrorTimestamp(DateUtils.getCurrentDateTime());
            errorLog.setErrorMessage(String.format("入参:%s, javascript脚本: %s", JSONObject.toJSONString(exprParams), expression));
            errorLog.setPossibleCausesAndGuidance("请检查脚本接收的入参变量是否正确，以及脚本本身是否正确");
            context.setErrorLog(errorLog);

            LogUtils.buildAgileLog(LogConstant.AGILE_CODE_EXECUTE_FORMULAC_ACTION, errorLog.getErrorCode(), "actionId:" + actionParam.getActionId() + ",action入参:" + JSONObject.toJSONString(exprParams) + "，js脚本:" + expression, "JAVASCRIPT脚本执行报错，失败原因:【"+e.getMessage()+"】", "\"1.打开常用的浏览器，按下F12键打开开发者工具，切换到console面板。\n" +
                    "2.将需要调试的脚本复制粘贴到console中。\n" +
                    "3.观察console中的报错信息，根据提示修改脚本中的错误。\n" +
                    "4.多次测试确保脚本在console中能正确运行。\n" +
                    "5.回到相关的应用或系统的发布界面，修改对应的预算节点，重新发布。\"");

            throw new BusinessException(ErrorCodeEnum.FORMULA_EXECUTE_EXCEPTION.getCode(),
                    ErrorCodeEnum.FORMULA_EXECUTE_EXCEPTION.getMessage(), e);
        }
    }

    private Object convert(Object original) {
        if (original == null) {
            return null;
        } else if (original instanceof String || original instanceof Integer || original instanceof Long || original instanceof Boolean || original instanceof Double) {
            return original;
        } else if (original instanceof ScriptObjectMirror) {
            ScriptObjectMirror jsOriginal = (ScriptObjectMirror) original;
            if (jsOriginal.isArray()) {
                List<Object> listResult = new ArrayList<>();
                Integer length = (Integer) jsOriginal.get("length");
                for (int i = 0; i < length; i++) {
                    listResult.add(convert(jsOriginal.get("" + i)));
                }
                return listResult;
            } else if (jsOriginal.isFunction()) {
                // can't convert it...
                return null;
            }
            Map<String, Object> mapResult = new LinkedHashMap<>();
            for (Map.Entry<String, Object> entry : jsOriginal.entrySet()) {
                mapResult.put(entry.getKey(), convert(entry.getValue()));
            }
            return mapResult;
        }
        return original;
    }


    /**
     * Script 表达式计算
     *
     * @param expression
     * @param exprParams
     * @return
     * @throws Exception
     */
    private Object doExpression(String expression, Map<String, Object> exprParams) throws Exception {
        String execScript = String.format(SCRIPT_METHOD_TEMPLATE, expression);
        ScriptEngineManager factory = new ScriptEngineManager();
        ScriptEngine engine = factory.getEngineByName(SCRIPT_ENGINE_SHORT_NAME);
        ScriptContext ctx = new SimpleScriptContext();
        ctx.setBindings(engine.createBindings(), ScriptContext.ENGINE_SCOPE);

        execScript = ReplaceUtils.replace(convert2ValueString(exprParams), execScript);
        //返回表达式计算结果
        return engine.eval(execScript, ctx);
    }


    private Map<String, String> convert2ValueString(Map<String, Object> variables) {
        Map<String, String> newMap = new HashMap<>();
        variables.forEach((k, v) -> {
            if (v instanceof Map) {
                Map tTmpValue = (Map) v;
                newMap.put(k, (new JSONObject(tTmpValue)).toString());
            } else if (v instanceof List) {
                List tTmpValue = (List) v;
                newMap.put(k, (new JSONArray(tTmpValue)).toString());
            } else {
                newMap.put(k, v + "");
            }
        });
        return newMap;
    }

}
