package com.digiwin.athena.executionengine.trans.components;

import com.digiwin.athena.executionengine.model.trans.DealResult;
import com.digiwin.athena.executionengine.model.trans.StepElement;
import com.digiwin.athena.executionengine.trans.Step;
import com.digiwin.athena.executionengine.trans.TransAbstractStep;
import com.google.common.collect.Lists;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.LoggerFactory;
import org.slf4j.Logger;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;

/**
 * @description: 数据去重组件
 * @author: xijj
 * @date: 2020/08/11 15:09
 */
@Component("distinct")
public class DistinctStep extends TransAbstractStep {

    private static final Logger LOGGER = LoggerFactory.getLogger(DistinctStep.class);

    @Override
    public boolean defineCheck(StepElement stepElement) {
        return true;
    }

    @Override
    public DealResult doDealData(Step step) {
        DealResult dealResult = new DealResult();
        try {
            Object inputData = getCurrentData(step.getPrevStepNameList());
            if (!(inputData instanceof List)) {
                LOGGER.warn("数据去重组件只能接受表状数据，非展平数据需要先执行展平节点！Trans执行 {} 失败，数据结构错误！",
                        step.getName());
                dealResult.setDataNullFail();
                return dealResult;
            }
            List<Map<String, Object>> dataList = (List<Map<String, Object>>) (inputData);

            // 数据源为空直接返回
            if (CollectionUtils.isEmpty(dataList)) {
                dealResult.setSuccess(Lists.newArrayList());
                return dealResult;
            }
            // 获取去重的字段
            List<String> fieldList = step.getStepElement().getFields();
            if (CollectionUtils.isEmpty(fieldList)) {
                LOGGER.warn("对应的distinct元素中没有field字段");
                dealResult.setSuccess(dataList);
                return dealResult;
            }
            dealResult.setSuccess(distinctByFieldMap(dataList, fieldList,step.getStepElement().isReturnAll()));
            return dealResult;
        } catch (Exception e) {
            throw e;
        }
    }


    /**
     * 根据数据中指定的字段进行去重
     *
     * @param dataList
     * @param fieldList
     * @return
     */
    private List<Map<String, Object>> distinctByFieldMap(List<Map<String, Object>> dataList, List<String> fieldList,boolean returnAll) {
        List<Map<String, Object>> collect = dataList.stream().filter(distinctByKey(map -> createFunction(map, fieldList))).collect(Collectors.toList());
        if(returnAll){
            return collect;
        }
        //只取fieldList的值
        List<Map<String, Object>> resultList = new ArrayList<>();
        collect.forEach(value -> {
            Map<String, Object> rowMap = new LinkedHashMap<>();
            fieldList.forEach(field -> rowMap.put(field, value.get(field)));
            resultList.add(rowMap);
        });
        return resultList;
    }

    private Map<String, Object> createFunction(Map<String, Object> map, List<String> fieldList) {
        Map<String, Object> fieldMap = new HashMap<>();
        fieldList.forEach(field -> fieldMap.put(field, String.valueOf(map.get(field))));
        return fieldMap;
    }

    public static <T> Predicate<T> distinctByKey(Function<? super T, Object> keyExtractor) {
        Map<Object, Boolean> map = new ConcurrentHashMap<>();
        return t -> map.putIfAbsent(keyExtractor.apply(t), Boolean.TRUE) == null;
    }
}
