package com.digiwin.athena.executionengine.service.facade.execution.impl;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.TypeReference;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.digiwin.app.service.DWServiceContext;
import com.digiwin.athena.executionengine.constant.LogConstant;
import com.digiwin.athena.executionengine.core.container.SolutionStepContext;
import com.digiwin.athena.executionengine.dto.ExportDataDto;
import com.digiwin.athena.executionengine.dto.runinfo.ActionExecutionDto;
import com.digiwin.athena.executionengine.dto.runinfo.DebugDto;
import com.digiwin.athena.executionengine.model.Metric;
import com.digiwin.athena.executionengine.model.input.ExportInput;
import com.digiwin.athena.executionengine.model.input.SolutinStep.DataObject;
import com.digiwin.athena.executionengine.model.input.SolutinStep.Select;
import com.digiwin.athena.executionengine.model.input.SolutinStep.SolutionStep;
import com.digiwin.athena.executionengine.model.input.SolutionStepInput;
import com.digiwin.athena.executionengine.model.solutionStep.SolutionStepMetric;
import com.digiwin.athena.executionengine.service.facade.analyzer.ISolutionStepMetricAnalyzerFacade;
import com.digiwin.athena.executionengine.service.facade.analyzer.impl.SolutionStepManager;
import com.digiwin.athena.executionengine.service.facade.execution.ISolutionStepExecutionFacade;
import com.digiwin.athena.executionengine.service.facade.export.IExportFacade;
import com.digiwin.athena.executionengine.util.LogUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.*;
import java.util.stream.Collectors;

/**
 * @description: 新架构实现
 * @author: Lee
 * @create: 2024/9/18
 */
@Service
public class SolutionStepExecutionFacade implements ISolutionStepExecutionFacade {

    private static final Logger LOGGER = LoggerFactory.getLogger(SolutionStepExecutionFacade.class);

    /**
     * 管理Metric数据
     */
    private static final ThreadLocal<List<SolutionStepMetric>> metricLocal = new ThreadLocal<>();

    /**
     * 管理step数据
     */
    private static final ThreadLocal<Map<String, Object>> transDataLocal = new ThreadLocal<>();

    /**
     * 管理导数的指标数据
     */
    private static final ThreadLocal<List<Metric>> exportMetricLocal = new ThreadLocal<>();


    /**
     * 调试模式数据
     */
    private static final ThreadLocal<List<DebugDto>> debugDtoListLocal = new ThreadLocal<>();

    private ISolutionStepMetricAnalyzerFacade solutionStepMetricAnalyzerFacade;

    @Autowired
    private IExportFacade exportFacade;

    @Autowired
    public void setSolutionStepMetricAnalyzerFacade(ISolutionStepMetricAnalyzerFacade solutionStepMetricAnalyzerFacade) {
        this.solutionStepMetricAnalyzerFacade = solutionStepMetricAnalyzerFacade;
    }

    private SolutionStepManager solutionStepManager;

    @Autowired
    public void setSolutionStepManager(SolutionStepManager solutionStepManager) {
        this.solutionStepManager = solutionStepManager;
    }

    @Override
    public Object execute(SolutionStepInput solutionStepInput) {
        try {
            LOGGER.info(String.format(LogConstant.AGILE_DATA + "指标接收的参数:%s \nToken:%s", "SolutionStepExecutionFacade.execute", JSONObject.toJSONString(solutionStepInput), DWServiceContext.getContext().getToken()));
            long startTime = System.currentTimeMillis();
            if (!checkSolutionStepParams(solutionStepInput)) {
                return null;
            }
            SolutionStepContext context = createSolutionStepContext(solutionStepInput);
            //需要取数
            context.setNeedData(true);
            List<SolutionStep> solutionStepList = JSONArray.parseArray(JSONObject.toJSONString(solutionStepInput.getSolutionStep()), SolutionStep.class);
            //1.解析取数
            int solutionStepIndex = 0;
            String lastStepKey = "";
            for (SolutionStep solutionStep : solutionStepList) {
                doSolutionStep(solutionStep, context, solutionStepIndex);
                if (transDataLocal.get() != null) {
                    transDataLocal.get().put(solutionStep.getAlias(), washResultData(transDataLocal.get().get(solutionStep.getAlias()), solutionStep.getSelect(), solutionStep.getComputeList()));
                }
                solutionStepIndex = solutionStepIndex + 1;
                if (solutionStepIndex == solutionStepList.size()) {
                    lastStepKey = solutionStep.getAlias();
                }
            }
            long timeTaken = (System.currentTimeMillis() - startTime);
            LOGGER.info("执行引擎SolutionStepExecute总耗时 :{}ms", timeTaken);
            //处理最终数据，保留语义给的度量+查询方案里的度量
            Map<String, Object> dataMap = transDataLocal.get();
            LogUtils.buildAgileLog(LogConstant.AGILE_CODE_ANALYSE_SOLUTION_STEP_DEFINE, LogUtils.SUCCESS, String.format("指标接收的参数:%s  token:%s", JSONObject.toJSONString(solutionStepInput), DWServiceContext.getContext().getToken()), String.format("指标接收的出参:%s ", JSONObject.toJSONString(dataMap)), "如果对返回结果有异议，可以查看后续详细的执行步骤日志");
            JSONObject response = new JSONObject();
            if (dataMap == null || !dataMap.containsKey(lastStepKey)) {
                LOGGER.info(String.format(LogConstant.AGILE_DATA + "指标接收的出参:*未查询到数据* \nToken:%s", "SolutionStepExecutionFacade.execute", DWServiceContext.getContext().getToken()));
                response.put("data", new JSONArray());
                return response;
            }

            List<Object> dictionary = context.getDatasetDictionaryMapping().values().stream().flatMap(List::stream)
                    .collect(Collectors.toList());
            response.put("data", doTranslate((JSONArray) dataMap.get(lastStepKey), dictionary));
            response.put("debug", createDebugInfo(context));
            LOGGER.info(String.format(LogConstant.AGILE_DATA + "指标接收的出参:%s \nToken:%s", "SolutionStepExecutionFacade.execute", response, DWServiceContext.getContext().getToken()));
            return response;
        } catch (Exception exception) {
            LOGGER.error(String.format(LogConstant.AGILE_DATA + "指标接收的出参:%s \nToken:%s", "SolutionStepExecutionFacade.execute", exception, DWServiceContext.getContext().getToken()));
            throw exception;
        } finally {
            transDataLocal.remove();
            debugDtoListLocal.remove();
            metricLocal.remove();
        }
    }

    /**
     * 字典翻译
     *
     * @param dataCenterData
     * @param dictionary
     * @return
     */
    private JSONArray doTranslate(JSONArray dataCenterData, List<Object> dictionary) {
        if (CollectionUtils.isEmpty(dictionary) || CollectionUtils.isEmpty(dataCenterData)) {
            return dataCenterData;
        }
        JSONArray dic = JSONArray.parseArray(JSONObject.toJSONString(dictionary));
        Map<String, JSONObject> allDicMap = new HashMap<>();
        dic.stream().forEach(obj -> {
            JSONArray enums = ((JSONObject) obj).getJSONArray("enums");
            String name = ((JSONObject) obj).getString("name");
            JSONObject dicMap = new JSONObject();
            enums.forEach(item -> {
                JSONObject enumItem = (JSONObject) item;
                dicMap.put(enumItem.getString("value"), enumItem.getString("description"));
            });
            allDicMap.put(name, dicMap);
        });
        dataCenterData.forEach(item -> {
            JSONObject row = (JSONObject) item;
            row.forEach((key, value) -> {
                if (allDicMap.containsKey(key)) {
                    row.put(key, allDicMap.get(key).get(value.toString()));
                }
            });
        });
        return dataCenterData;
    }

    /**
     * 进行指标分析，以及步骤处理
     *
     * @param solutionStep      步骤
     * @param context           权限及基础信息
     * @param solutionStepIndex 步骤序号
     */
    public void doSolutionStep(SolutionStep solutionStep, SolutionStepContext context, int solutionStepIndex) {
        solutionStepMetricAnalyzerFacade.analyseMetric(solutionStep, context, solutionStepIndex);
        if (context.isNeedData()) {
            solutionStepManager.executeSolutionStep(solutionStep, solutionStepIndex, context);
        }
    }


    @Override
    public List<ExportDataDto> export(SolutionStepInput solutionStepInput) {
        try {
            LOGGER.info(String.format(LogConstant.AGILE_DATA + "接收的参数:%s \nToken:%s", "SolutionStepExecutionFacade.export", JSONObject.toJSONString(solutionStepInput), DWServiceContext.getContext().getToken()));
            if (solutionStepInput.getSolutionStep() == null || solutionStepInput.getSolutionStep().isEmpty()) {
                LOGGER.error(String.format(LogConstant.AGILE_DATA + "接收的step参数:%s", "SolutionStepExecutionFacade.export", JSONObject.toJSONString(solutionStepInput.getSolutionStep())));
            }
            SolutionStepContext context = createSolutionStepContext(solutionStepInput);
            List<SolutionStep> solutionStepList = JSONArray.parseArray(JSONObject.toJSONString(solutionStepInput.getSolutionStep()), SolutionStep.class);
            int solutionStepIndex = 0;
            for (SolutionStep solutionStep : solutionStepList) {
                doSolutionStep(solutionStep, context, solutionStepIndex);
            }
            List<Metric> metricList = exportMetricLocal.get();
            ExportInput exportInput = new ExportInput();
            exportInput.setDatasource(!solutionStepInput.getSysParam().isEmpty() && solutionStepInput.getSysParam().containsKey("dataSource") ? String.valueOf(solutionStepInput.getSysParam().get("dataSource")) : null);
            exportInput.setMetrics(metricList);
            exportInput.setEocMaps(convertEocMaps(solutionStepInput.getEocMaps()));
            exportInput.setPermissions(JSONObject.parseObject(JSONObject.toJSONString(context.getPermissions())));
            exportInput.setUserId(!solutionStepInput.getSysParam().isEmpty() && solutionStepInput.getSysParam().containsKey("userId") ? String.valueOf(solutionStepInput.getSysParam().get("userId")) : null);
            exportInput.setTenantId(solutionStepInput.getTenantId());
            return exportFacade.exportData(exportInput);
        } catch (Exception ex) {
            throw ex;
        } finally {
            exportMetricLocal.remove();
            metricLocal.remove();
        }
    }

    /**
     * 创建SolutionStepContext
     *
     * @param solutionStepInput
     * @return
     */
    private SolutionStepContext createSolutionStepContext(SolutionStepInput solutionStepInput) {
        SolutionStepContext context = new SolutionStepContext();
        context.setTenantId(solutionStepInput.getTenantId());
        context.setEocMaps(convertEocMaps(solutionStepInput.getEocMaps()));
        context.setEocMap(solutionStepInput.getEocMaps() == null || solutionStepInput.getEocMaps().isEmpty() ? new HashMap<>() : solutionStepInput.getEocMaps().get(0));
        context.setParam(solutionStepInput.getSysParam());
        context.setToken(DWServiceContext.getContext().getToken());
        //多方案
        if (solutionStepInput.getExtension() != null && !solutionStepInput.getExtension().isEmpty()) {
            if (solutionStepInput.getExtension().containsKey("permissions")) {
                JSONObject permissions = solutionStepInput.getExtension().getJSONObject("permissions");
                context.setPermissions(permissions.toJavaObject(new TypeReference<Map<String, Object>>() {
                }));
                if (permissions.containsKey("roles")) {
                    context.setRoles(permissions.getJSONArray("roles").toJavaObject(new TypeReference<List<String>>() {
                    }));
                } else {
                    context.setRoles(new ArrayList<>());
                }
            }
            if (solutionStepInput.getExtension().containsKey("productLineInfo")) {
                context.setProductLineInfo(solutionStepInput.getExtension().getJSONArray("productLineInfo"));
            }
            if (!solutionStepInput.getExtension().isEmpty() && solutionStepInput.getExtension().containsKey("datasets")) {
                JSONArray datasets = solutionStepInput.getExtension().getJSONArray("datasets");
                List<JSONObject> javaObject = datasets.toJavaObject(new TypeReference<List<JSONObject>>() {
                });
                initDatasetInfo(javaObject, context);
            }
        }
        return context;
    }

    /**
     * 初始化数据集映射信息
     *
     * @param datasets
     * @param context
     */
    private void initDatasetInfo(List<JSONObject> datasets, SolutionStepContext context) {
        datasets.forEach(datasetInfo -> {
            String datasetId = datasetInfo.getString("datasetId");
            String modelId = datasetInfo.getString("modelId");
            String modelCode = datasetInfo.getString("modelCode");
            JSONObject mappingFields = datasetInfo.getJSONObject("mappingFields");
            context.getModelIdMapping().put(datasetId, modelId);
            context.getModelCodeMapping().put(datasetId, modelCode);
            context.getDatasetFieldsMapping().put(datasetId, mappingFields);
            context.getDatasetDictionaryMapping().put(datasetId, datasetInfo.getJSONArray("dictionary"));
        });
    }

    /**
     * 站换eocMap
     *
     * @param eocMaps
     * @return
     */
    private List<Map<String, Object>> convertEocMaps(List<Map<String, Object>> eocMaps) {
        List<Map<String, Object>> eocMapList = new ArrayList<>();
        if (eocMaps.isEmpty()) {
            return eocMapList;
        }
        for (Map<String, Object> eocMap : eocMaps) {
            Map<String, Object> objectMap = new HashMap<>();
            if (eocMap.isEmpty()) {
                continue;
            }
            for (String key : eocMap.keySet()) {
                objectMap.put(key, eocMap.get(key));
            }
            eocMapList.add(objectMap);
        }
        return eocMapList;
    }

    /**
     * 根据别名获取指标数据
     *
     * @param alias
     * @return
     */
    public static Object getMetricData(String alias) {
        List<SolutionStepMetric> metrics = SolutionStepExecutionFacade.getSolutionStepMetrics();
        if (CollectionUtils.isNotEmpty(metrics)) {
            Optional<SolutionStepMetric> metricOptional = metrics.stream().filter(solutionStepMetric -> alias.equals(solutionStepMetric.getAlias())).findFirst();
            if (metricOptional.isPresent()) {
                SolutionStepMetric metric = metricOptional.get();
                if (metric.getData() != null) {
                    if (metric.getCount() - 1 == 0) {
                        metrics.remove(metric);
                    }
                    return metric.getData();
                }
            }
        }
        return null;
    }

    /**
     * 清洗结果数据，将列改成alias别名
     *
     * @param data
     * @param selectList
     */
    private Object washResultData(Object data, List<Select> selectList, JSONObject computeList) {
        LOGGER.info("washResultData开始...");
        if (data == null) {
            LOGGER.info("washResultData的数据为null直接返回...");
            return data;
        }
        JSONArray jsonArray = JSONArray.parseArray(JSONArray.toJSONString(data, SerializerFeature.WriteMapNullValue));
        if (jsonArray.isEmpty()) {
            LOGGER.info("washResultData的数据为空集合直接返回...");
            return data;
        }
        LOGGER.info("washResultData的数据条目数量:{}", jsonArray.size());
        JSONArray newData = new JSONArray();
        for (int i = 0; i < jsonArray.size(); i++) {
            JSONObject jsonObject = jsonArray.getJSONObject(i);
            JSONObject newJSONObject = JSONObject.parseObject(JSONObject.toJSONString(jsonObject));
            Set<String> keySet = jsonObject.keySet();
            for (String key : keySet) {
                selectList.forEach(select -> {
                    //目前只处理select为非子查询的情况 TODO:后期考虑子查询的支持
                    DataObject selectObject = select.getDataObject();
                    if (selectObject != null) {
                        if (!"calculate".equals(selectObject.getContentType())) {
                            if (key.equals(selectObject.getContent()) && !key.equals(selectObject.getAlias())) {
                                String alias = selectObject.getAlias();
                                //新增一列
                                newJSONObject.put(alias, newJSONObject.get(key));
                                //移除旧列
                                newJSONObject.remove(key);
                            }
                        } else {
                            //找计算的值
                            JSONArray computeArray = computeList.getJSONArray(selectObject.getContent());
                            JSONObject lastComputeObject = computeArray.getJSONObject(computeArray.size() - 1);
                            String newField = lastComputeObject.getString("newField");
                            if (key.equals(newField) && !key.equals(selectObject.getAlias())) {
                                String alias = selectObject.getAlias();
                                //新增一列
                                newJSONObject.put(alias, newJSONObject.get(key));
                                //移除旧列
                                newJSONObject.remove(key);
                            }
                        }
                    }
                });
            }
            newData.add(newJSONObject);
        }

        return newData;
    }

    /**
     * 创建debug返回数据
     *
     * @param context context
     * @return json
     */
    public JSONObject createDebugInfo(SolutionStepContext context) {
        JSONObject jsonObject = new JSONObject();
        if (context.getParam().containsKey("debug") && "true".equals(String.valueOf(context.getParam().get("debug")))) {
            List<DebugDto> debugDtoList = getDebugDtoListLocal();
            JSONArray metricRunningInfoArray = new JSONArray();
            JSONObject metricRunningDetail = new JSONObject();
            for (DebugDto debugDto : debugDtoList) {
                JSONObject metricRunningInfo = JSONObject.parseObject(JSONObject.toJSONString(debugDto.getMetricExecutionDto()));
                metricRunningInfoArray.add(metricRunningInfo);
                //组装metricRunningDetail
                if (debugDto.getActionExecutionDtosMap() != null) {
                    for (Map.Entry<String, List<ActionExecutionDto>> entry : debugDto.getActionExecutionDtosMap().entrySet()) {
                        List<ActionExecutionDto> actionExecutionDtoList = entry.getValue();
                        Collections.reverse(actionExecutionDtoList);
                        metricRunningDetail.put(entry.getKey(), JSONObject.parseArray(JSONObject.toJSONString(actionExecutionDtoList)));
                    }
                }
            }
            jsonObject.put("metricRunningInfo", metricRunningInfoArray);
            jsonObject.put("metricRunningDetail", metricRunningDetail);
        }
        return jsonObject;
    }

    /**
     * 校验入参的是否正确
     *
     * @param solutionStepInput 入参
     * @return true-正确 false-反之
     */
    public boolean checkSolutionStepParams(SolutionStepInput solutionStepInput) {
        if (solutionStepInput.getSysParam() == null || solutionStepInput.getSysParam().isEmpty()) {
            LogUtils.buildAgileLog(LogConstant.AGILE_CODE_ANALYSE_SOLUTION_STEP_DEFINE, LogUtils.SUCCESS, String.format("指标接收的参数:%s  token:%s", JSONObject.toJSONString(solutionStepInput), DWServiceContext.getContext().getToken()), "指标接收的出参:{null}", "调用入参缺少sysParam内容，确认边缘数据中台当前是否有发版操作");
            return false;
        }
        if (solutionStepInput.getSolutionStep() == null || solutionStepInput.getSolutionStep().isEmpty()) {
            LogUtils.buildAgileLog(LogConstant.AGILE_CODE_ANALYSE_SOLUTION_STEP_DEFINE, LogUtils.SUCCESS, String.format("指标接收的参数:%s  token:%s", JSONObject.toJSONString(solutionStepInput), DWServiceContext.getContext().getToken()), "指标接收的出参:{null}", "调用入参信息异常，请联系语义相关人员排查语义分析否存在问题");
            return false;
        }
        if (StringUtils.isEmpty(solutionStepInput.getTenantId())) {
            LogUtils.buildAgileLog(LogConstant.AGILE_CODE_ANALYSE_SOLUTION_STEP_DEFINE, LogUtils.SUCCESS, String.format("指标接收的参数:%s  token:%s", JSONObject.toJSONString(solutionStepInput), DWServiceContext.getContext().getToken()), "指标接收的出参:{null}", "调用入参缺少TenantId，请联系敏捷数据引擎(ade)相关人员检查他们调用是否存在问题");
            return false;
        }
        for (int i = 0; i < solutionStepInput.getSolutionStep().size(); i++) {
            JSONObject jsonObject = solutionStepInput.getSolutionStep().getJSONObject(i);
            JSONArray selectArray = jsonObject.getJSONArray("select");
            for (int j = 0; j < selectArray.size(); j++) {
                JSONObject selectObject = selectArray.getJSONObject(j);
                JSONObject dataObject = selectObject.getJSONObject("dataObject");
                if (dataObject == null || !dataObject.containsKey("dataType") || !dataObject.containsKey("alias") || !dataObject.containsKey("contentType") || !dataObject.containsKey("content")) {
                    LogUtils.buildAgileLog(LogConstant.AGILE_CODE_ANALYSE_SOLUTION_STEP_DEFINE, LogUtils.SUCCESS, String.format("指标接收的参数:%s  token:%s", JSONObject.toJSONString(solutionStepInput), DWServiceContext.getContext().getToken()), "指标接收的出参:{null}", "调用入参的select内容异常，请联系语义相关人员排查语义分析否存在问题");
                    return false;
                }
            }
        }

        return true;
    }

    //region 管理的指标数据与step的数据的获取存储方法
    public static List<SolutionStepMetric> getSolutionStepMetrics() {
        if (metricLocal.get() == null) {
            metricLocal.set(new ArrayList<>());
        }
        return metricLocal.get();
    }

    public static void setSolutionStepMetrics(List<SolutionStepMetric> solutionStepMetrics) {
        metricLocal.set(solutionStepMetrics);
    }

    public static void releaseSolutionStepMetrics() {
        metricLocal.remove();
    }

    public static Map<String, Object> getTransDataMap() {
        return transDataLocal.get();
    }

    public static void setTransDataMap(Map<String, Object> transDataMap) {
        transDataLocal.set(transDataMap);
    }

    public static void releaseTransDataMap() {
        transDataLocal.remove();
    }

    public static List<Metric> getExportMetrics() {
        if (exportMetricLocal.get() == null) {
            exportMetricLocal.set(new ArrayList<>());
        }
        return exportMetricLocal.get();
    }

    public static void setExportMetrics(List<Metric> exportMetrics) {
        exportMetricLocal.set(exportMetrics);
    }

    public static void releaseExportMetrics() {
        exportMetricLocal.remove();
    }

    public static List<DebugDto> getDebugDtoListLocal() {
        if (debugDtoListLocal.get() == null) {
            debugDtoListLocal.set(new ArrayList<>());
        }
        return debugDtoListLocal.get();
    }

    public static void setDebugDtoListLocal(DebugDto debugDto) {
        if (debugDtoListLocal.get() == null) {
            debugDtoListLocal.set(new ArrayList<>());
        }
        debugDtoListLocal.get().add(debugDto);
    }
    //endregion
}