package com.digiwin.athena.executionengine.service.facade.mapping.data;

import java.util.*;

/**
 * @description: 存储字段数据信息，包含字段名、字段的上下关系
 *               数据量统计信息、层次分区信息、数据形态认知信息
 * @author: fenglei
 * @date: 2020-11-9
 */
public class DataNode {
    /**
     * 本节点id-uuid
     */
    private String nodeId;

    /**
     * 本节点的字段名
     * 当字段名为".ROOT"时，为系统级顶层节点，此节点为虚拟节点，只有一个。
     */
    private String nodeName;

    /**
     * 父节点id，为".ROOT"时表示当前节点已是最上层节点。当前节点不是“.ROOT”，但是为null时，表示该节点没有初始化
     */
    private String parentNodeId;

    /**
     * 该节点的数据实例的总数量
     */
    private int totalCnt;

    /**
     * 当前节点外层中最后出现集合的节点的nodeId
     */
    private String lastCollectionNode;

    /**
     * 相对属性
     * 上一层存在循环的节点（lastCollectionNode），可能也存在多个不同的实例，不同的父节点划分出不同的数据域
     * zone描述的是父节点下标对应的数据实例数量
     * key: JsonPathInst  一个jsonPath实例：父层的jsonPath实例
     * value: 当前父节点中的数据实例量
     *       $[].A[].B[].C[]的结构，描述C的zone，Map的实际例子：
     *       key: $[0].A[0].B[0], value: 5   //$[0].A[0].B[0].C.length = 5
     *       key: $[0].A[0].B[1], value: 8   //$[0].A[0].B[1].C.length = 8
     *       key: $[0].A[0].B[2], value: 2   //$[0].A[0].B[2].C.length = 2
     *       key: $[0].A[1].B[0], value: 6   //$[0].A[1].B[0].C.length = 6
     * 如果该字段为null，则上层没有出现过集合节点
     */
    private Map<String, Integer> zone;

    /**
     * 当前读取的数据在某个zone中的下标
     * 分区索引
     */
    private int zonePos;

    /**
     * 当前所处的数据实例下标在该节点数据实例中的下标
     * 全局索引
     */
    private int globalPos;

    /**
     * 当前节点是否是Map结构
     */
    private boolean isObject;

    /**
     * 子节点信息
     * String: 子节点id
     */
    private List<String> children;

    /**
     * 当前节点是否是集合
     */
    private boolean isArray;

    /**
     * 是否将当前的array看成一个对象
     * 看成对象以后，取值时会直接将array当对象进行赋值，而不是用array的成员赋值
     */
    private boolean isArrayAsObject;

    /**
     * 当前节点的jsonPath，但是这里的path会记录[*]，实例化时需要吧[*]替换掉
     */
    private String pathTemplate;

    /**
     * 当前节点的参数路径，与pathTemplate不同的是，这里不含[*]结构性描述
     */
    private String paramPath;

    /**
     * 子节点中size最大的节点的nodeId
     */
    private String maxSizeChildNodeId;

    /**
     * 当前节点在json中的路径，list中每个元素都是nodeId，最上面的根层节点，最后一个是当前节点
     * 第一个节点是".ROOT"，".ROOT"节点没有该字段，为null
     */
    private List<String> jsonRoute;

    public DataNode () {
        this.nodeId = UUID.randomUUID().toString();
        this.nodeName = "";
        this.parentNodeId = null;
        this.totalCnt = 0;
        this.lastCollectionNode = "";
        this.zone = new HashMap<>();
        this.zonePos= -1;
        this.globalPos = -1;
        this.children = null;
        this.isObject = false;
        this.isArray = false;
        this.isArrayAsObject = false;
        this.pathTemplate = "";
        this.paramPath = "";
        this.maxSizeChildNodeId = null;
        this.jsonRoute = null;
    }

    public DataNode (String nodeName, String parentNodeId, String lastCollectionNode, int totalCnt, boolean isObject,
                     boolean isArray, boolean isArrayAsObject, String pathTemplate, String paramPath, List<String> lastRoute) {
        this.nodeId = UUID.randomUUID().toString();
        this.zone = new HashMap<>();
        this.children = new ArrayList<>();
        this.nodeName = nodeName;
        this.parentNodeId = parentNodeId;
        this.totalCnt = 0;
        this.lastCollectionNode = lastCollectionNode;
        this.zonePos= 0;

        this.globalPos = -1;
        if (isArray) {
            this.globalPos = 0;
        }

        this.isObject = isObject;
        this.isArray = isArray;
        this.isArrayAsObject = isArrayAsObject;
        this.pathTemplate = pathTemplate;
        this.paramPath = paramPath;
        this.maxSizeChildNodeId = null;
        this.jsonRoute = new ArrayList<>();
        addAllJsonRoute(lastRoute);
    }

    /**
     * 设定当前的节点是一个空的map节点
     */
    public void setEmptyMap() {
        this.totalCnt = 0;
        this.zonePos= 0;

        this.globalPos = -1;
        if (this.isArray) {
            this.globalPos = 0;
        }
        
        this.maxSizeChildNodeId = null;
    }

    protected void addChildren(String childId) {
        this.children.add(childId);
    }

    protected void addAllJsonRoute(List<String> lastRoute) {
        if (lastRoute!=null && !lastRoute.isEmpty()) {
            this.jsonRoute.addAll(lastRoute);
        }
    }


    protected String getNodeId() {
        return nodeId;
    }

    protected void setNodeId(String nodeId) {
        this.nodeId = nodeId;
    }

    protected String getNodeName() {
        return nodeName;
    }

    protected void setNodeName(String nodeName) {
        this.nodeName = nodeName;
    }

    protected String getParentNodeId() {
        return parentNodeId;
    }

    protected void setParentNodeId(String parentNodeId) {
        this.parentNodeId = parentNodeId;
    }

    protected int getTotalCnt() {
        return totalCnt;
    }

    protected void setTotalCnt(int totalCnt) {
        this.totalCnt = totalCnt;
    }

    protected String getLastCollectionNode() {
        return lastCollectionNode;
    }

    protected void setLastCollectionNode(String lastCollectionNode) {
        this.lastCollectionNode = lastCollectionNode;
    }

    protected Map<String, Integer> getZone() {
        return zone;
    }

    protected void setZone(Map<String, Integer> zone) {
        this.zone = zone;
    }

    protected void addZone(String key, Integer value) {
        this.zone.put(key, value);
    }

    protected int getZonePos() {
        return zonePos;
    }

    protected void setZonePos(int zonePos) {
        this.zonePos = zonePos;
    }

    protected int getGlobalPos() {
        return globalPos;
    }

    protected void setGlobalPos(int globalPos) {
        this.globalPos = globalPos;
    }

    protected List<String> getChildren() {
        return children;
    }

    protected void setChildren(List<String> children) {
        this.children = children;
    }

    protected boolean isObject() {
        return isObject;
    }

    protected void setObject(boolean isObject) {
        this.isObject = isObject;
    }

    protected boolean isArray() {
        return isArray;
    }

    protected void setArray(boolean isArray) {
        this.isArray = isArray;
    }

    protected boolean isArrayAsObject() {
        return isArrayAsObject;
    }

    protected void setArrayAsObject(boolean isArrayAsObject) {
        this.isArrayAsObject = isArrayAsObject;
    }

    protected String getPathTemplate() {
        return pathTemplate;
    }

    protected void setPathTemplate(String pathTemplate) {
        this.pathTemplate = pathTemplate;
    }

    protected String getParamPath() {
        return paramPath;
    }

    protected void setParamPath(String paramPath) {
        this.paramPath = paramPath;
    }

    protected String getMaxSizeChildNodeId() {
        return maxSizeChildNodeId;
    }

    protected void setMaxSizeChildNodeId(String maxSizeChildNodeId) {
        this.maxSizeChildNodeId = maxSizeChildNodeId;
    }

    protected List<String> getJsonRoute() {
        return jsonRoute;
    }

    protected void setJsonRoute(List<String> jsonRoute) {
        this.jsonRoute = jsonRoute;
    }

    protected void addTotalCnt(int count){
        this.totalCnt += count;
    }
}
