package com.digiwin.athena.executionengine.trans;

import com.digiwin.athena.executionengine.exception.VerificationException;
import com.digiwin.athena.executionengine.trans.util.TransAnalysisUtils;
import com.digiwin.athena.executionengine.constant.TransConstant;
import com.digiwin.athena.executionengine.model.trans.OrderElement;
import com.digiwin.athena.executionengine.model.trans.StepElement;
import com.digiwin.athena.executionengine.util.ContextUtils;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.LoggerFactory;
import org.slf4j.Logger;
import org.springframework.stereotype.Component;

import java.util.*;

/**
 * @description: Trans数据流处理
 * @author: fenglei
 * @date: 2020-12-3
 */
@Component
public class TransEngine {
    private static final Logger LOGGER = LoggerFactory.getLogger(TransEngine.class);

    private static ThreadLocal<TransDataManger> LOCAL_CONTEXT = new ThreadLocal<>();

    public Object runEngine(TransDataManger transDataManger) {
        try {
            LOCAL_CONTEXT.set(transDataManger);
            StepLine stepLine = runTrans();
            //最后一个step处理过的数据作为最终数据返回出去
            return transDataManger.getCurrentData(stepLine.getEndStep().getName());
        } finally {
            LOCAL_CONTEXT.remove();
        }
    }

    protected StepLine runTrans() {
        StepLine stepLine = buildStepLine();
        if (stepLine == null) {
            throw new VerificationException("trans rule is error");
        }
        //对多个开始点一次执行trans计算流
        int runResult = TransConstant.TRANS_FAIL;
        for (Step startStep : stepLine.getStartStepList()) {
            //任何一个节点都可能会有多个to，匹配from，找出order
            List<OrderElement> stepOrders = (List) stepLine.getOrderFromIndex().get(startStep.getName());

            //trans规则中只有一个step的时候，是没有order的
            if (stepOrders == null || stepOrders.size() <= 0) {
                runResult = run(startStep, null, stepLine);
            } else {
                for (OrderElement order : stepOrders) {
                    runResult = run(startStep, order, stepLine);
                    if (runResult == TransConstant.TRANS_FAIL) {
                        break;
                    }
                }
            }

            if (runResult == TransConstant.TRANS_FAIL) {
                break;
            }
        }
        return stepLine;
    }

    /**
     * 构建stepLine
     *
     * @return
     */
    private StepLine buildStepLine() {
        TransDataManger transDataManger = getLocalContext();
        if (transDataManger.getOriginData() == null) {
            LOGGER.error("传入的originData为null");
            return null;
        }

        StepLine stepLine = new StepLine();
        // 解析trans.step
        stepLine.setStepElementList(TransAnalysisUtils.parseStep(transDataManger.getRuleJsonObj()));
        if (CollectionUtils.isEmpty(stepLine.getStepElementList())) {
            LOGGER.error("trans无需要执行的计算节点，请检查trans定义是否完整！");
            return null;
        }

        // 解析trans.order
        stepLine.setOrderElementList(TransAnalysisUtils.parseOrder(transDataManger.getRuleJsonObj()));
        if (stepLine.getStepElementList().size() > 1 && CollectionUtils.isEmpty(stepLine.getOrderElementList())) {
            LOGGER.error("缺少order部分的规则，有些节点将无法被执行！");
            return null;
        }

        //设置from to
        setIndexOfOrder(stepLine);

        // 3.创建trans节点对象
        for (StepElement stepElement : stepLine.getStepElementList()) {
            List<String> prevStepNameList = getFromNameInOrderToIndex(stepElement, stepLine);
            Step step = new Step(prevStepNameList, stepElement);
            ITransStep component = ContextUtils.getBean(step.getTechnique());
            if (!component.defineCheck(stepElement)) {
                LOGGER.error("规则定义校验失败，step name:{},technique:{}",
                        stepElement.getName(), stepElement.getTechnique());
                return null;
            }
            stepLine.getStepMap().put(stepElement.getName(), step);
        }

        //设置开始和结束step
        setStartAndEndStep(stepLine);
        return stepLine;
    }

    /**
     * 设置开始和结束节点
     *
     * @param stepLine
     */
    private void setStartAndEndStep(StepLine stepLine) {
        List<Step> steps = new ArrayList<>();

        //没有order的情况是允许的，这时候规则只会有一个可以执行的step
        if (CollectionUtils.isEmpty(stepLine.getOrderElementList())) {
            String stepName = stepLine.getStepElementList().get(0).getName();
            steps.add(stepLine.getStepMap().get(stepName));
            stepLine.setStartStepList(steps);
            stepLine.setEndStep(stepLine.getStepMap().get(stepName));
            return;
        }

        for (StepElement step : stepLine.getStepElementList()) {
            //如果节点名称不在到达节点列表中，那么它一定是开始节点
            if (!stepLine.getOrderToIndex().containsKey(step.getName())) {
                steps.add(stepLine.getStepMap().get(step.getName()));
            }

            if (stepLine.getEndStep() == null && !stepLine.getOrderFromIndex().containsKey(step.getName())) {
                stepLine.setEndStep(stepLine.getStepMap().get(step.getName()));
            }
        }
        stepLine.setStartStepList(steps);
    }

    /**
     * 处理一个明确的order。并判断是否需要继续处理
     * 传入当前节点的name，并且传入当前节点完成后的order
     * 并管理当前节点的后续order集，完成所有order集以后才会回归到上一步骤。
     *
     * @param order
     * @param stepLine
     * @return
     */
    private int run(Step step, OrderElement order, StepLine stepLine) {
        String stepName = step.getName();
        try {
            ITransStep component = ContextUtils.getBean(step.getTechnique());
            int result = component.dealData(step);
            if (result == TransConstant.TRANS_FAIL) {
                String orderStr = (order == null) ? "EndStep"
                        : (order.getFrom() + "->" + order.getTo());
                LOGGER.error("TransEngine执行step：{} 失败，order：{}", stepName, orderStr);
                return result;
            }

            if (result == TransConstant.TRANS_MISS_DATA) {
                String orderStr = (order == null) ? "EndStep"
                        : (order.getFrom() + "->" + order.getTo());
                LOGGER.warn("TransEngine执行step：{} 缺少数据，order：{}", stepName, orderStr);
                return result;
            }

            //如果为空，显示的理解为当前节点后面没有order操作了，说明是最后一个节点。
            if (order == null) {
                return result;
            } else {
                //处理完后判断是否有后续的order，如果没有说明后续
                List<OrderElement> stepOrders = (List) stepLine.getOrderFromIndex().get(order.getTo());
                //trans规则中只有一个step的时候，是没有order的
                if (stepOrders == null || stepOrders.size() <= 0) {
                    result = run(stepLine.getStepMap().get(order.getTo()), null, stepLine);
                } else {
                    for (OrderElement nextOrder : stepOrders) {
                        //这里并不会回归，需要完成所有分支的order后才能回归
                        result = run(stepLine.getStepMap().get(order.getTo()), nextOrder, stepLine);
                        if (result == TransConstant.TRANS_FAIL) {
                            break;
                        }
                    }
                }

                return result;
            }
        } catch (Exception e) {
            LOGGER.error("TransEngine执行[" + stepName + "]step时异常", e);
            return TransConstant.TRANS_FAIL;
        }
    }

    /**
     * 这是一个有非常大局限性的取order元素的方法
     * 只会返回目标节点是非合并节点的order
     *
     * @param stepElement
     * @param stepLine
     * @return
     */
    private List<String> getFromNameInOrderToIndex(StepElement stepElement, StepLine stepLine) {
        List<String> stepNameList = new ArrayList<>();
        if (CollectionUtils.isEmpty(stepLine.getOrderElementList())) {
            return stepNameList;
        }

        List<OrderElement> orderElementList = (List<OrderElement>) stepLine.getOrderToIndex().get(stepElement.getName());

        if (orderElementList == null) {
            return stepNameList;
        }

        ///上一步只有一个源，那么返回这个stepElement，否则返回空
        for (OrderElement orderElement : orderElementList) {
            stepNameList.add(orderElement.getFrom());
        }

        return stepNameList;
    }

    /**
     * orderFromIndex.key: order.from
     * toFromIndex.key: order.to
     *
     * @param stepLine
     * @return
     */
    private void setIndexOfOrder(StepLine stepLine) {
        if (CollectionUtils.isEmpty(stepLine.getOrderElementList())) {
            return;
        }
        for (OrderElement order : stepLine.getOrderElementList()) {
            stepLine.getOrderFromIndex().put(order.getFrom(), order);
            stepLine.getOrderToIndex().put(order.getTo(), order);
        }
    }

    protected static TransDataManger getLocalContext() {
        return LOCAL_CONTEXT.get();
    }

}
