package com.digiwin.athena.executionengine.trans.components;

import com.digiwin.athena.executionengine.constant.CommonConstant;
import com.digiwin.athena.executionengine.constant.FieldNameConstant;
import com.digiwin.athena.executionengine.constant.TransConstant;
import com.digiwin.athena.executionengine.model.trans.DealResult;
import com.digiwin.athena.executionengine.model.trans.Statistic;
import com.digiwin.athena.executionengine.model.trans.StepElement;
import com.digiwin.athena.executionengine.trans.Step;
import com.digiwin.athena.executionengine.trans.TransAbstractStep;
import com.digiwin.athena.executionengine.util.RegexUtil;
import com.google.common.collect.Lists;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.math3.analysis.UnivariateFunction;
import org.apache.commons.math3.analysis.solvers.BrentSolver;
import org.apache.commons.math3.stat.descriptive.DescriptiveStatistics;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;
import org.springframework.util.ObjectUtils;

import java.math.BigDecimal;
import java.util.*;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.BiConsumer;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import static com.digiwin.athena.executionengine.trans.components.CalculateStep.*;

/**
 * @description: 处理分组组件
 * @author: xijj
 * @date: 2020/7/21 18:04
 */
@Component("group")
public class GroupStep extends TransAbstractStep {

    private static final Logger LOGGER = LoggerFactory.getLogger(GroupStep.class);

    private static Map<String, BiConsumer<Statistic, Map<Map, List<Map<String, Object>>>>> FUNCTION_MAP = new HashMap<>();

    static {
        initFunction();
    }

    @Override
    public boolean defineCheck(StepElement stepElement) {
        return true;
    }

    /**
     * @return 分组组件返回值结构是：List<Map<String, Object>>
     * @throws Exception
     */
    @Override
    public DealResult doDealData(Step step) {
        DealResult dealResult = new DealResult();
        try {
            Object inputData = getCurrentData(step.getPrevStepNameList());
            if (!(inputData instanceof List)) {
                LOGGER.warn("分组组件只能接受表状数据，非展平数据需要先执行展平节点！Trans执行 {} 失败，数据结构错误！",
                        step.getName());
                dealResult.setDataNullFail();
                return dealResult;
            }
            List<Map<String, Object>> dataList = (List<Map<String, Object>>) (inputData);

            // 数据源为空直接返回
            if (CollectionUtils.isEmpty(dataList)) {
                dealResult.setSuccess(Lists.newArrayList());
                return dealResult;
            }
            StepElement stepElement = step.getStepElement();
            // 没有分组条件返回原数据
            List<String> fieldList;
            if (CollectionUtils.isNotEmpty(stepElement.getFields())) {
                fieldList = stepElement.getFields();
            } else {
                fieldList = new ArrayList<>();
            }
            // 按照给定的field分组
            Map<Map, List<Map<String, Object>>> resultMap = dataList.stream().collect(Collectors.groupingBy(map -> {
                Map<String, Object> fieldMap = new HashMap<>();
                for (String field : fieldList) {
                    fieldMap.put(field, String.valueOf(map.get(field)));
                }
                return fieldMap;
            }));

            List<Statistic> statisticList = stepElement.getStatistics();
            //分析排序字段
            if (CollectionUtils.isNotEmpty(statisticList)) {
                List<Statistic> sortStatistics = statisticList.stream().filter(statistic -> statistic.getStatisticalMethod().equals("sort")).collect(Collectors.toList());
                List<Statistic> otherStatistics = statisticList.stream().filter(statistic -> !statistic.getStatisticalMethod().equals("sort")).collect(Collectors.toList());

                //先处理排序
                Map<Map, List<Map<String, Object>>> sortMap = doGroupSort(fieldList, sortStatistics, resultMap);

                // 处理非排序的字段
                for (Statistic statistic : otherStatistics) {
                    doCalculate(statistic, sortMap);
                }

                // 拆分数据返回处理完的数据
                List<Map<String, Object>> resultList = new ArrayList<>();
                for (Map.Entry<Map, List<Map<String, Object>>> entry : sortMap.entrySet()) {
                    resultList.addAll(entry.getValue());
                }
                dealResult.setSuccess(resultList);
                return dealResult;
            }

            dealResult.setSuccess(new ArrayList<>());
            return dealResult;
        } catch (Exception e) {
            LOGGER.error("执行分组组件出现异常");
            throw e;
        }
    }

    private Map<Map, List<Map<String, Object>>> doGroupSort(List<String> fieldList, List<Statistic> sortStatistics, Map<Map, List<Map<String, Object>>> resultMap) {
        Set<String> fieldListSet = new HashSet<>(fieldList);
        List<Statistic> groupSortStatistics = sortStatistics.stream().filter(statistic -> fieldListSet.contains(statistic.getStatisticalFields().get(0)))
                .collect(Collectors.toList());

        Map<Map, List<Map<String, Object>>> sortedMap = new LinkedHashMap<>();
        if (CollectionUtils.isNotEmpty(groupSortStatistics)) {
            List<Map.Entry<Map, List<Map<String, Object>>>> entries = new ArrayList<>(resultMap.entrySet());
            entries.sort(createGroupComparator(groupSortStatistics));
            for (Map.Entry<Map, List<Map<String, Object>>> entry : entries) {
                sortedMap.put(entry.getKey(), entry.getValue());
            }
        }
        if (sortedMap.isEmpty()) {
            sortedMap.putAll(resultMap);
        }
        List<Statistic> realSortStatistics = sortStatistics.stream().filter(statistic -> !fieldListSet.contains(statistic.getStatisticalFields().get(0)))
                .collect(Collectors.toList());

        String newLineField = null;
        for (Statistic sortStatistic : realSortStatistics) {
            if (StringUtils.isEmpty(sortStatistic.getMode())) {
                continue;
            }

            if (TransConstant.GROUP_MODE_NORMAL.equals(sortStatistic.getMode())) {
                continue;
            }

            if (TransConstant.GROUP_MODE_CUSTOM.equals(sortStatistic.getMode())) {
                newLineField = sortStatistic.getSortName();
                break;
            }

            if (TransConstant.GROUP_MODE_LOGICAL.equals(sortStatistic.getMode())) {
                newLineField = FieldNameConstant.TRANS_SORT_LINENUMBER;
                break;
            }
        }
        boolean isAddLineNumber = newLineField != null;
        if (CollectionUtils.isNotEmpty(realSortStatistics)) {
            String finalNewLineField = newLineField;
            sortedMap.forEach((k, v) -> {
                v.sort(createComparator(realSortStatistics));
                if (isAddLineNumber) {
                    int i = 1;
                    for (Map<String, Object> stringObjectMap : v) {
                        stringObjectMap.put(finalNewLineField, i);
                        i++;
                    }
                }
            });
        }

        return sortedMap;

    }

    /**
     * 动态分组key排序
     */
    private Comparator<Map.Entry<Map, List<Map<String, Object>>>> createGroupComparator(List<Statistic> sortStatistics) {
        Comparator<Map.Entry<Map, List<Map<String, Object>>>> comparator = Comparator.comparing(map -> 0);;
        for (Statistic statistic : sortStatistics) {
            String field = statistic.getStatisticalFields().get(0);
            String direction = statistic.getType();

            // 根据字段名构建 Comparator
            Comparator<Map.Entry<Map, List<Map<String, Object>>>> fieldComparator = Comparator.comparing(
                    map -> map.getKey().get(field), // 获取字段值
                    (value1, value2) -> {
                        if (value1 == null || value2 == null) {
                            // 处理空值情况
                            return (value1 == null ? -1 : 1);
                        }
                        // 使用辅助方法进行动态比较
                        return compareValues(value1, value2);
                    }
            );

            // 如果是降序，反转 Comparator
            if ("desc".equalsIgnoreCase(direction)) {
                fieldComparator = fieldComparator.reversed();
            }

            // 链接排序规则
            comparator = comparator.thenComparing(fieldComparator);
        }
        return comparator;
    }


    /**
     * 动态组内排序
     *
     * @param sortStatistics
     * @return
     */
    private Comparator<Map<String, Object>> createComparator(List<Statistic> sortStatistics) {
        Comparator<Map<String, Object>> comparator = Comparator.comparing(map -> 0);

        for (Statistic statistic : sortStatistics) {
            String field = statistic.getStatisticalFields().get(0);
            String direction = statistic.getType();

            // 根据字段名构建 Comparator
            Comparator<Map<String, Object>> fieldComparator = Comparator.comparing(
                    map -> map.get(field), // 获取字段值
                    (value1, value2) -> {
                        if (value1 == null || value2 == null) {
                            // 处理空值情况
                            return (value1 == null ? -1 : 1);
                        }
                        // 使用辅助方法进行动态比较
                        return compareValues(value1, value2);
                    }
            );

            // 如果是降序，反转 Comparator
            if ("desc".equalsIgnoreCase(direction)) {
                fieldComparator = fieldComparator.reversed();
            }

            // 链接排序规则
            comparator = comparator.thenComparing(fieldComparator);
        }
        return comparator;
    }

    /**
     * 数值比较逻辑
     *
     * @param value1
     * @param value2
     * @return
     */

    private int compareValues(Object value1, Object value2) {
        if (value1 instanceof Number && value2 instanceof Number) {
            // 比较数值类型，统一转为 BigDecimal
            return BigDecimal.valueOf(((Number) value1).doubleValue())
                    .compareTo(BigDecimal.valueOf(((Number) value2).doubleValue()));
        }

        if (value1 instanceof String && value2 instanceof String) {
            String str1 = (String) value1;
            String str2 = (String) value2;
            // 如果字符串是数值，按数值比较，否则按字符串自然顺序
            if (RegexUtil.isNumber(str1) && RegexUtil.isNumber(str2)) {
                return new BigDecimal(str1).compareTo(new BigDecimal(str2));
            }
            return str1.compareTo(str2);
        }

        if (value1 instanceof Boolean && value2 instanceof Boolean) {
            // 布尔值比较（true 大于 false）
            return ((Boolean) value1).compareTo((Boolean) value2);
        }

        // 默认按字符串表示比较
        return value1.toString().compareTo(value2.toString());
    }

    public void doCalculate(Statistic statistic, Map<Map, List<Map<String, Object>>> resultMap) {
        BiConsumer function = FUNCTION_MAP.get(statistic.getStatisticalMethod());
        if (function == null) {
            throw new RuntimeException("groupStep method not found,method:" + statistic.getStatisticalMethod());
        }
        function.accept(statistic, resultMap);
    }

    /**
     * 增加排序列行号
     *
     * @param entry
     */
    private static void addSortLineNumber(Map.Entry<Map, List<Map<String, Object>>> entry) {
        int i = 1;
        for (Map dataMap : entry.getValue()) {
            dataMap.put(FieldNameConstant.TRANS_SORT_LINENUMBER, i);
            i++;
        }
    }

    /**
     * 增加排序列行号
     *
     * @param entry
     * @param sortColumnName 排序列名称
     */
    private static void addSortLineNumber(Map.Entry<Map, List<Map<String, Object>>> entry, String sortColumnName) {
        int i = 1;
        for (Map dataMap : entry.getValue()) {
            dataMap.put(sortColumnName, i);
            i++;
        }
    }

    /**
     * 按照所给字段排序
     *
     * @param sortFields 排序字段
     * @param entry      需要排序数据
     * @param type       升降序
     */
    private static void sort(List<String> sortFields, Map.Entry<Map, List<Map<String, Object>>> entry, String type) {
        Collections.sort(entry.getValue(), (o1, o2) -> {
            int i = 0;
            return compareData(sortFields, o1, o2, i++, type);
        });
    }

    /**
     * 进行数据比较
     *
     * @param sortFields 排序字段
     * @param o1         对象
     * @param o2         比较对象
     * @param i          比较的排序字段索引
     * @param type       升降序
     * @return
     */
    private static int compareData(List<String> sortFields, Map<String, Object> o1, Map<String, Object> o2, int i, String type) {
        if (i > sortFields.size() - 1) {
            return CommonConstant.COMPARE_EQUAL;
        }
        String sortField = sortFields.get(i);
        //如果o1为要排序的字段为空，如果是ASC则向后排，代表o1>o2，故取1；DESC也是向后排，代表o1<o2，故取-1
        if (o1.get(sortField) == null || ObjectUtils.isEmpty(o1.get(sortField))) {
            if (CommonConstant.ORDER_ASC.equals(type)) {
                return -1;
            } else {
                return 1;
            }
        }
        //如果o2为要排序的字段为空，如果是ASC则向后排，代表o1<o2，故取-1；DESC也是向后排，代表o1>o2，故取1
        if (o2.get(sortField) == null || ObjectUtils.isEmpty(o2.get(sortField))) {
            if (CommonConstant.ORDER_ASC.equals(type)) {
                return 1;
            } else {
                return -1;
            }
        }

        if (o1.get(sortField) instanceof String) {
            String sortValueA = String.valueOf(o1.get(sortField));
            String sortValueB = String.valueOf(o2.get(sortField));
            if (RegexUtil.isNumber(sortValueA) && RegexUtil.isNumber(sortValueB)) {
                BigDecimal bigDecimaA = new BigDecimal(sortValueA);
                BigDecimal bigDecimaB = new BigDecimal(sortValueB);
                int result = CommonConstant.ORDER_ASC.equals(type) ? bigDecimaA.compareTo(bigDecimaB) : bigDecimaB.compareTo(bigDecimaA);
                if (result == CommonConstant.COMPARE_EQUAL) {
                    return compareData(sortFields, o1, o2, i + CommonConstant.INCREASE_ONE, type);
                } else {
                    return result;
                }
            }

            int result = CommonConstant.ORDER_ASC.equals(type) ? sortValueA.compareTo(sortValueB) : sortValueB.compareTo(sortValueA);
            if (result == CommonConstant.COMPARE_EQUAL) {
                return compareData(sortFields, o1, o2, i + CommonConstant.INCREASE_ONE, type);
            } else {
                return result;
            }
        } else {
            BigDecimal sortValueA = new BigDecimal(String.valueOf(o1.get(sortField)));
            BigDecimal sortValueB = new BigDecimal(String.valueOf(o2.get(sortField)));
            int result = CommonConstant.ORDER_ASC.equals(type) ? sortValueA.compareTo(sortValueB) : sortValueB.compareTo(sortValueA);
            if (result == CommonConstant.COMPARE_EQUAL) {
                return compareData(sortFields, o1, o2, i + CommonConstant.INCREASE_ONE, type);
            } else {
                return result;
            }
        }
    }

    /**
     * 获取最小日期
     *
     * @param statistic
     * @param min
     * @param dataMap
     * @return
     */
    private static String getMinDate(Statistic statistic, String min, Map dataMap) {
        String date = String.valueOf(dataMap.get(statistic.getStatisticalFields().get(0)));
        if (StringUtils.isEmpty(min)) {
            min = date;
        } else {
            min = date.compareTo(min) < 0 ? date : min;
        }
        return min;
    }

    /**
     * 获取最大日期
     *
     * @param statistic
     * @param max
     * @param dataMap
     * @return
     */
    private static String getMaxDate(Statistic statistic, String max, Map dataMap) {
        String date = String.valueOf(dataMap.get(statistic.getStatisticalFields().get(0)));
        if (StringUtils.isEmpty(max)) {
            max = date;
        } else {
            max = date.compareTo(max) > 0 ? date : max;
        }
        return max;
    }

    /**
     * 获取最小值
     *
     * @param statistic
     * @param min
     * @param dataMap
     * @return
     */
    private static BigDecimal getMinNumber(Statistic statistic, BigDecimal min, Map dataMap) {
        Object data = dataMap.get(statistic.getStatisticalFields().get(0));
        //double类型下可能会造成精度缺失，故使用BigDecimal.valueOf.但是需注意传入参数小数点后的位数超过double允许的16位精度还是可能会出现问题的
        BigDecimal bigDecimal;

        if (data == null) {
            bigDecimal = new BigDecimal(0);
        } else if (data instanceof Double) {
            bigDecimal = BigDecimal.valueOf((Double) data);
        } else {
            bigDecimal = new BigDecimal(String.valueOf(data));
        }
        //如果严格限制精度的比较，那么则可考虑使用equals方法。1.00和1.0使用equals返回的是false
        if (min != null) {
            min = bigDecimal.compareTo(min) < 0 ? bigDecimal : min;
        } else {
            min = bigDecimal;
        }
        return min;
    }


    /**
     * 获取最大值
     *
     * @param statistic
     * @param max
     * @param dataMap
     * @return
     */
    private static BigDecimal getMaxNumber(Statistic statistic, BigDecimal max, Map dataMap) {
        Object data = dataMap.get(statistic.getStatisticalFields().get(0));
        //double类型下可能会造成精度缺失，故使用BigDecimal.valueOf.但是需注意传入参数小数点后的位数超过double允许的16位精度还是可能会出现问题的
        BigDecimal bigDecimal;
        if (data == null) {
            bigDecimal = new BigDecimal(0);
        } else if (data instanceof Double) {
            bigDecimal = BigDecimal.valueOf((Double) data);
        } else {
            bigDecimal = new BigDecimal(String.valueOf(data));
        }
        //如果严格限制精度的比较，那么则可考虑使用equals方法。1.00和1.0使用equals返回的是false
        if (max != null) {
            max = bigDecimal.compareTo(max) > 0 ? bigDecimal : max;
        } else {
            max = bigDecimal;
        }
        return max;
    }

    /**
     * 初始化function
     * *************
     * 《******如果新增支持的方法，请同步更新到TransConstant中的GROUP_FUNCTION常量重的方法名称******》
     * 《在TransConstant中的GROUP_FUNCTION中对应的是全部大写》
     * ************
     */
    private static void initFunction() {
        //分组求和
        FUNCTION_MAP.put("sum", (statistic, resultMap) -> {
            for (Map.Entry<Map, List<Map<String, Object>>> entry : resultMap.entrySet()) {
                entry.getValue().forEach(dataMap -> dataMap.put(statistic.getNewField(), getSumValue(statistic, entry)));
            }
        });

        //分组求和
        FUNCTION_MAP.put("accSum", (statistic, resultMap) -> {
            for (Map.Entry<Map, List<Map<String, Object>>> entry : resultMap.entrySet()) {
                BigDecimal cumulativeSum = new BigDecimal(0);
                for (Map<String, Object> dataMap : entry.getValue()) {
                    cumulativeSum = cumulativeSum.add(getBaseValue(statistic.getStatisticalFields(), dataMap));
                    dataMap.put(statistic.getNewField(), cumulativeSum);
                }
            }
        });

        //分组排序
        FUNCTION_MAP.put("sort", (statistic, resultMap) -> {
            //排序字段
            List<String> sortFields = statistic.getStatisticalFields();
            //当为normal时，不需要产生新字段
            if (TransConstant.GROUP_MODE_NORMAL.equals(statistic.getMode())) {
                for (Map.Entry<Map, List<Map<String, Object>>> entry : resultMap.entrySet()) {
                    sort(sortFields, entry, statistic.getType());
                }
            } else {
                for (Map.Entry<Map, List<Map<String, Object>>> entry : resultMap.entrySet()) {
                    sort(sortFields, entry, statistic.getType());
                    if (TransConstant.GROUP_MODE_CUSTOM.equals(statistic.getMode())) {
                        addSortLineNumber(entry, statistic.getSortName());
                    } else {
                        addSortLineNumber(entry);
                    }
                }
            }
        });

        //分组排序取数
        FUNCTION_MAP.put("sortLimit", (statistic, resultMap) -> {
            int startPods = dealPods(statistic.getStartPos());
            int endPods = dealPods(statistic.getEndPos());
            //排序字段
            List<String> sortFields = statistic.getStatisticalFields();
            for (Map.Entry<Map, List<Map<String, Object>>> entry : resultMap.entrySet()) {
                sort(sortFields, entry, statistic.getType());
                if (CollectionUtils.isEmpty(entry.getValue())) {
                    continue;
                }
                int dataSize = entry.getValue().size();
                if (startPods > dataSize) {
                    resultMap.put(entry.getKey(), Lists.newArrayList());
                    continue;
                }
                endPods = Math.min(endPods, entry.getValue().size());
                resultMap.put(entry.getKey(), entry.getValue().subList(startPods, endPods));
            }
        });

        //分组统计组内记录里总量
        FUNCTION_MAP.put("count", (statistic, resultMap) -> {
            for (Map.Entry<Map, List<Map<String, Object>>> entry : resultMap.entrySet()) {
                int count = entry.getValue().size();
                // 将count计数添加到map的数据列中
                entry.getValue().forEach(dataMap -> dataMap.put(statistic.getNewField(), count));
            }
        });

        //分组求取指定字段的最小值
        FUNCTION_MAP.put("min", (statistic, resultMap) -> {
            for (Map.Entry<Map, List<Map<String, Object>>> entry : resultMap.entrySet()) {
                Object finalMin;
                if (TransConstant.STATISTIC_TYPE_DATE.equals(statistic.getType())) {
                    String min = "";
                    for (Map dataMap : entry.getValue()) {
                        min = getMinDate(statistic, min, dataMap);
                    }
                    finalMin = min;
                } else if (TransConstant.STATISTIC_TYPE_NUMBER.equals(statistic.getType())) {
                    BigDecimal min = null;
                    for (Map dataMap : entry.getValue()) {
                        min = getMinNumber(statistic, min, dataMap);
                    }
                    finalMin = min;
                } else {
                    finalMin = "";
                }
                entry.getValue().forEach(dataMap -> dataMap.put(statistic.getNewField(), finalMin));

            }
        });

        //分组求取指定字段的最大值
        FUNCTION_MAP.put("max", (statistic, resultMap) -> {
            for (Map.Entry<Map, List<Map<String, Object>>> entry : resultMap.entrySet()) {
                Object finalMax;
                if (TransConstant.STATISTIC_TYPE_DATE.equals(statistic.getType())) {
                    String max = "";
                    for (Map dataMap : entry.getValue()) {
                        max = getMaxDate(statistic, max, dataMap);
                    }
                    finalMax = max;
                } else if (TransConstant.STATISTIC_TYPE_NUMBER.equals(statistic.getType())) {
                    BigDecimal max = null;
                    for (Map dataMap : entry.getValue()) {
                        max = getMaxNumber(statistic, max, dataMap);
                    }
                    finalMax = max;
                } else if (TransConstant.STATISTIC_TYPE_STRING.equals(statistic.getType())) {
                    String max = "";
                    for (Map dataMap : entry.getValue()) {
                        max = getMaxString(statistic, max, dataMap);
                    }
                    finalMax = max;
                } else {
                    finalMax = "";
                }
                entry.getValue().forEach(dataMap -> dataMap.put(statistic.getNewField(), finalMax));

            }
        });

        //分组求取平均值
        FUNCTION_MAP.put("average", (statistic, resultMap) -> {


            for (Map.Entry<Map, List<Map<String, Object>>> entry : resultMap.entrySet()) {
                AtomicInteger counter = new AtomicInteger();
                // sum的值
                BigDecimal sumValue = entry.getValue().stream().map(map -> {
                    boolean allFieldsNonEmpty = statistic.getStatisticalFields().stream().allMatch(item ->
                            (map.get(item) instanceof Number) ||
                                    (map.get(item) instanceof String && StringUtils.isNotEmpty((String) map.get(item)))
                    );

                    if (allFieldsNonEmpty) {
                        counter.incrementAndGet();
                    }

                    return statistic.getStatisticalFields().stream().map(item -> {
                        if (map.get(item) instanceof Number) {
                            return new BigDecimal(((Number) map.get(item)).toString());
                        }
                        if (map.get(item) instanceof String && StringUtils.isNotEmpty((String) map.get(item))) {
                            return new BigDecimal((String) map.get(item));
                        }
                        return BigDecimal.ZERO;
                    }).reduce(BigDecimal.ZERO, BigDecimal::add);
                }).reduce(BigDecimal.ZERO, BigDecimal::add);

                // count的值
                int count = counter.get();
                BigDecimal aveValue;

                // 平均值 = sum / count
                if (!sumValue.equals(BigDecimal.ZERO) && count != 0) {
                    aveValue = sumValue.divide(new BigDecimal(count), getPrecision(statistic.getPrecision()), getRoundingMode(statistic.getPrecisionType()));
                } else {
                    aveValue = BigDecimal.ZERO;
                }

                entry.getValue().forEach(dataMap -> dataMap.put(statistic.getNewField(), aveValue));
            }

        });

        //分组求指定值的范围
        FUNCTION_MAP.put("limit", (statistic, resultMap) -> {
            //取值顺序--倒叙
            Integer startPods = dealPods(statistic.getStartPos());
            Integer endPods = dealPods(statistic.getEndPos());
            for (Map.Entry<Map, List<Map<String, Object>>> entry : resultMap.entrySet()) {
                List<Map<String, Object>> newValues = new ArrayList<>(endPods - startPods + 1);
                List<Map<String, Object>> values = entry.getValue();
                if (startPods >= 0 || startPods <= endPods || startPods <= values.size()) {
                    if (TransConstant.GROUP_DIRECTION_TAIL.equals(statistic.getDirection())) {
                        int newEndPods = values.size() - startPods - 1;
                        int newStartPods;
                        if (endPods <= values.size()) {
                            newStartPods = values.size() - endPods - 1;
                        } else {
                            newStartPods = values.size() - 1;
                        }
                        limitValue(newStartPods, newEndPods, newValues, values);
                    } else {
                        if (endPods >= values.size()) {
                            endPods = values.size() - 1;
                        }
                        limitValue(startPods, endPods, newValues, values);
                    }
                }
                resultMap.put(entry.getKey(), newValues);
            }
        });

        //分组将指定字段的值逗号拼接起来
        FUNCTION_MAP.put("collection", (statistic, resultMap) -> {
            for (Map.Entry<Map, List<Map<String, Object>>> entry : resultMap.entrySet()) {
                HashSet set = new HashSet();
                for (Map dataMap : entry.getValue()) {
                    set.add(dataMap.get(statistic.getStatisticalFields().get(0)));
                }

                String newFieldValue = StringUtils.join(set, ",");
                entry.getValue().forEach(dataMap -> dataMap.put(statistic.getNewField(), newFieldValue));
            }
        });

        //占比
        FUNCTION_MAP.put("ratio", (statistic, resultMap) -> {
            for (Map.Entry<Map, List<Map<String, Object>>> entry : resultMap.entrySet()) {
                BigDecimal sumValue = getSumValue(statistic, entry);
                // 处理sum为0的情况
                if (sumValue.compareTo(BigDecimal.ZERO) == 0) {
                    if ("divideDefault".equals(statistic.getExceptionHandling())) {
                        entry.getValue().forEach(dataMap -> dataMap.put(statistic.getNewField(), BigDecimal.ZERO));
                    }
                    continue;
                }
                entry.getValue().forEach(map -> {
                    for (String item : statistic.getStatisticalFields()) {
                        Object valueObj = map.get(item);
                        if (valueObj == null) {
                            LOGGER.warn("Group ratio error, value is null for field: {}", item);
                            map.put(statistic.getNewField(), BigDecimal.ZERO);
                            continue;
                        }
                        double value;
                        // 进行类型判断与转换
                        if (valueObj instanceof Number) {
                            value = ((Number) valueObj).doubleValue();
                        } else if (valueObj instanceof String && StringUtils.isNotEmpty((String) valueObj)) {
                            value = Double.parseDouble((String) valueObj);
                        } else {
                            LOGGER.warn("Group ratio error, value is not a valid number or field, key: {}, value: {}", item, valueObj);
                            continue;
                        }
                        BigDecimal valueA = BigDecimal.valueOf(value);
                        BigDecimal divide = valueA.divide(sumValue, getPrecision(statistic.getPrecision()), getRoundingMode(statistic.getPrecisionType()));
                        map.put(statistic.getNewField(), new BigDecimal(divide.stripTrailingZeros().toPlainString()));
                    }
                });
            }
        });

        //方差
        FUNCTION_MAP.put("variance", (statistic, resultMap) -> {
            for (Map.Entry<Map, List<Map<String, Object>>> entry : resultMap.entrySet()) {
                DescriptiveStatistics stats = new DescriptiveStatistics();
                handleStatsList(statistic, entry, stats);
                if (stats.getN() > 0) {
                    BigDecimal resultValue = getResultValue(statistic.getPrecision(),
                            BigDecimal.valueOf(stats.getVariance()), getRoundingMode(statistic.getPrecisionType()));
                    entry.getValue().forEach(dataMap -> dataMap.put(statistic.getNewField(), resultValue));
                } else {
                    entry.getValue().forEach(dataMap -> dataMap.put(statistic.getNewField(), BigDecimal.ZERO));
                }

            }
        });

        //样本标准差
        FUNCTION_MAP.put("stddev_samp", (statistic, resultMap) -> {
            for (Map.Entry<Map, List<Map<String, Object>>> entry : resultMap.entrySet()) {
                DescriptiveStatistics stats = new DescriptiveStatistics();
                handleStatsList(statistic, entry, stats);
                if (stats.getN() > 0) {
                    BigDecimal resultValue = getResultValue(statistic.getPrecision(),
                            BigDecimal.valueOf(stats.getStandardDeviation()), getRoundingMode(statistic.getPrecisionType()));
                    entry.getValue().forEach(dataMap -> dataMap.put(statistic.getNewField(), resultValue));
                } else {
                    entry.getValue().forEach(dataMap -> dataMap.put(statistic.getNewField(), BigDecimal.ZERO));
                }
            }
        });

        //总体标准差
        FUNCTION_MAP.put("stddev", (statistic, resultMap) -> {
            for (Map.Entry<Map, List<Map<String, Object>>> entry : resultMap.entrySet()) {
                DescriptiveStatistics stats = new DescriptiveStatistics();
                handleStatsList(statistic, entry, stats);
                if (stats.getN() > 0) {
                    double populationStdDev = stats.getStandardDeviation() * Math.sqrt((double) (stats.getN() - 1) / stats.getN());
                    BigDecimal resultValue = getResultValue(statistic.getPrecision(),
                            BigDecimal.valueOf(populationStdDev), getRoundingMode(statistic.getPrecisionType()));
                    entry.getValue().forEach(dataMap -> dataMap.put(statistic.getNewField(), resultValue));
                } else {
                    entry.getValue().forEach(dataMap -> dataMap.put(statistic.getNewField(), BigDecimal.ZERO));
                }
            }
        });

        //irr
        FUNCTION_MAP.put("irr", (statistic, resultMap) -> {
            for (Map.Entry<Map, List<Map<String, Object>>> entry : resultMap.entrySet()) {
                List<Double> cashFlows = handleDoubleValue(statistic, entry);
                BigDecimal resultValue;
                if (!cashFlows.isEmpty()) {
                    UnivariateFunction npvFunction = r -> {
                        double npv = 0.0;
                        for (int i = 0; i < cashFlows.size(); i++) {
                            npv += cashFlows.get(i) / Math.pow(1 + r, i);
                        }
                        return npv;
                    };
                    //最大迭代次数
                    int maximumIterations = StringUtils.isBlank(statistic.getMaximumIterations()) ?
                            Integer.parseInt(CommonConstant.STRING_ONE_HUNDRED) :
                            Integer.parseInt(statistic.getMaximumIterations());
                    //搜索区间上界
                    int startPos = StringUtils.isBlank(statistic.getStartPos()) ?
                            Integer.parseInt(CommonConstant.STRING_NEGATIVE_ONE) :
                            Integer.parseInt(statistic.getStartPos());
                    //搜索区间下界
                    int endPos = StringUtils.isBlank(statistic.getEndPos()) ?
                            Integer.parseInt(CommonConstant.STRING_ONE) :
                            Integer.parseInt(statistic.getEndPos());

                    BrentSolver solver = new BrentSolver();
                    double irr = solver.solve(maximumIterations, npvFunction, startPos, endPos);
                    resultValue = getResultValue(statistic.getPrecision(),
                            BigDecimal.valueOf(irr), getRoundingMode(statistic.getPrecisionType()));
                } else {
                    resultValue = BigDecimal.ZERO;
                }
                entry.getValue().forEach(dataMap -> dataMap.put(statistic.getNewField(), resultValue));
            }
        });

        //npv
        FUNCTION_MAP.put("npv", (statistic, resultMap) -> {
            for (Map.Entry<Map, List<Map<String, Object>>> entry : resultMap.entrySet()) {
                List<Double> cashFlows = handleDoubleValue(statistic, entry);
                BigDecimal resultValue;
                if (!cashFlows.isEmpty()) {
                    double rate = Double.parseDouble(statistic.getRate());
                    // 计算净现值
                    BigDecimal value = BigDecimal.valueOf(
                            IntStream.range(0, cashFlows.size()).mapToDouble(i -> cashFlows.get(i) / Math.pow(1 + rate, i)).sum()
                    );
                    resultValue = getResultValue(statistic.getPrecision(),
                            value, getRoundingMode(statistic.getPrecisionType()));
                } else {
                    resultValue = BigDecimal.ZERO;
                }
                entry.getValue().forEach(dataMap -> dataMap.put(statistic.getNewField(), resultValue));
            }
        });


    }

    private static BigDecimal getBaseValue(List<String> statisticalFields, Map<String, Object> dataMap) {
        return statisticalFields.stream().map(item -> {
            Object value = dataMap.get(item);
            if (value instanceof Number) {
                return new BigDecimal(value.toString());
            } else if (value instanceof String && StringUtils.isNotEmpty((String) value)) {
                return new BigDecimal((String) value);
            }
            return BigDecimal.ZERO;
        }).reduce(BigDecimal.ZERO, BigDecimal::add);
    }

    /**
     * 计算sum值
     *
     * @param statistic
     * @param entry
     * @return
     */
    private static BigDecimal getSumValue(Statistic statistic, Map.Entry<Map, List<Map<String, Object>>> entry) {
        try {
            return entry.getValue().stream().map(map ->
                    statistic.getStatisticalFields().stream().map(item -> {
                        Object value = map.get(item);
                        if (value instanceof Number) {
                            return new BigDecimal(value.toString());
                        } else if (value instanceof String && StringUtils.isNotEmpty((String) value)) {
                            return new BigDecimal((String) value);
                        }
                        return BigDecimal.ZERO;
                    }).reduce(BigDecimal.ZERO, BigDecimal::add)
            ).reduce(BigDecimal.ZERO, BigDecimal::add);
        } catch (NumberFormatException e) {
            LOGGER.error("数值转换错误: {}", e.getMessage());
            return BigDecimal.ZERO;
        } catch (Exception e) {
            LOGGER.error("计算总和时发生异常: {}", e.getMessage());
            return BigDecimal.ZERO;
        }
    }


    private static void handleStatsList(Statistic statistic, Map.Entry<Map, List<Map<String, Object>>> entry, DescriptiveStatistics stats) {
        entry.getValue().forEach(map -> {
            for (String item : statistic.getStatisticalFields()) {
                Object valueObj = map.get(item);
                if (valueObj == null) {
                    LOGGER.warn("Group error, value is null for field: {}", item);
                    continue;
                }
                double value;
                if (valueObj instanceof Number) {
                    value = ((Number) valueObj).doubleValue();
                } else if (valueObj instanceof String && (String.valueOf(valueObj)).matches("-?\\d+")) {
                    value = Double.parseDouble(String.valueOf(valueObj));
                } else {
                    LOGGER.warn("Group error, value is not a valid number or field,key: {}, value: {}", item, valueObj);
                    continue;
                }
                stats.addValue(value);
            }
        });
    }

    private static String getMaxString(Statistic statistic, String max, Map dataMap) {
        String date = String.valueOf(dataMap.get(statistic.getStatisticalFields().get(0)));
        if (StringUtils.isEmpty(max)) {
            max = date;
        } else {
            max = date.compareTo(max) > 0 ? date : max;
        }
        return max;
    }

    private static void limitValue(int startPods, int endPods,
                                   List<Map<String, Object>> newValues, List<Map<String, Object>> values) {
        for (int i = startPods; i <= endPods; i++) {
            newValues.add(values.get(i));
        }
    }

    private static Integer dealPods(String pods) {
        if (StringUtils.isBlank(pods)) {
            return 0;
        } else {
            return Integer.valueOf(pods);
        }
    }

    private static List<Double> handleDoubleValue(Statistic statistic, Map.Entry<Map, List<Map<String, Object>>> entry) {
        List<Double> values = new ArrayList<>();
        entry.getValue().forEach(map -> {
            for (String item : statistic.getStatisticalFields()) {
                Object valueObj = map.get(item);
                if (valueObj == null) {
                    LOGGER.warn("Group error, value is null for field: {}", item);
                    continue;
                }
                double value;
                if (valueObj instanceof Number) {
                    value = ((Number) valueObj).doubleValue();
                } else if (valueObj instanceof String && (String.valueOf(valueObj)).matches("-?\\d+")
                ) {
                    value = Double.parseDouble(String.valueOf(valueObj));
                } else {
                    LOGGER.warn("Group error, value is not a valid number or field,key: {}, value: {}", item, valueObj);
                    continue;
                }
                values.add(value);
            }
        });
        return values;
    }

}