package com.digiwin.athena.executionengine.component.action;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.digiwin.athena.executionengine.component.domain.ActionParam;
import com.digiwin.athena.executionengine.constant.AgileDataErrorCodeConstant;
import com.digiwin.athena.executionengine.constant.CommonConstant;
import com.digiwin.athena.executionengine.constant.FieldNameConstant;
import com.digiwin.athena.executionengine.constant.LogConstant;
import com.digiwin.athena.executionengine.core.aop.ActionMock;
import com.digiwin.athena.executionengine.core.aop.Debug;
import com.digiwin.athena.executionengine.core.container.ExecuteContext;
import com.digiwin.athena.executionengine.dto.ErrorLog;
import com.digiwin.athena.executionengine.enumtype.ErrorCodeEnum;
import com.digiwin.athena.executionengine.exception.BusinessException;
import com.digiwin.athena.executionengine.model.BasePage;
import com.digiwin.athena.executionengine.model.DataDescription;
import com.digiwin.athena.executionengine.util.DateUtils;
import com.digiwin.athena.executionengine.util.HttpClientUtils;
import com.digiwin.athena.executionengine.util.JsonResolverUtils;
import com.digiwin.athena.executionengine.util.LogUtils;
import com.digiwin.athena.executionengine.util.ReplaceUtils;
import com.digiwin.athena.smartdata.sdk.config.DatasourceUrl;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

/**
 * @description:
 * @author: ZhangJun
 * @create: 2024/5/9
 */
@Service("SqlAction")
public class SqlAction extends ActionBase {
    private static final Logger LOGGER = LoggerFactory.getLogger(SqlAction.class);
    //超管
    private static final String SUPER_USER = "superadmin";
    private static final String PLACE_HOLDER = "$(param)";
    private static final String SQLSERVER = "sqlServer";
    private static final String ORACLE = "oracle";
    private static final String MYSQL = "mySql";
    private static final String MYSQL_PAGE = " LIMIT $(offset),$(pageSize)";
    private static final String SQLSERVER_PAGE = " OFFSET $(offset) ROWS FETCH NEXT $(pageSize) ROWS ONLY";
    private static final String ORACLE_PAGE = "SELECT t3.* FROM(SELECT t2.*,ROWNUM rn FROM(%s) t2 WHERE ROWNUM <= $(endNo)) t3 WHERE t3.rn >= $(startNo)";

    @ActionMock
    @Debug
    @Override
    protected Object actionExecute(ExecuteContext context, Map<String, Object> reqMap, ActionParam actionParam) {
        String logSql = "";
        try {
            //多产品线处理
            String productLine;
            DataDescription dataDescription = context.getInputParam().getDataDescription();
            if (dataDescription.getProductLineRows() != null && !dataDescription.getProductLineRows().isEmpty()) {
                if (actionParam.getMetaObj().containsKey("productLine")) {
                    productLine = actionParam.getMetaObj().getString("productLine");
                } else {
                    productLine = "ERP";
                }
                for (int i = 0; i < dataDescription.getProductLineRows().size(); i++) {
                    JSONObject rowJson = dataDescription.getProductLineRows().getJSONObject(i);
                    if (productLine.equals(rowJson.getString("productLine"))) {
                        dataDescription.setPermissions(rowJson.getJSONObject("row"));
                        break;
                    }
                }
            } else {
                //兼容老
                productLine = "ERP";
            }
            JSONObject querySchema = actionParam.getQuerySchema();
            boolean isSuperUser = isSuperUser(context.getInputParam().getRoles());
            Map<String, String> paramMap = generateParam(querySchema, reqMap, actionParam.getActionId());

            if (!isSuperUser) {
                String whereSql = generatePermissionParam(context.getInputParam().getDataDescription());
                if (StringUtils.isNotEmpty(whereSql)) {
                    paramMap.put(CommonConstant.PARAM, whereSql);
                }
            }

            String selectSql = querySchema.getString("selectSql");
            selectSql = processSql(isSuperUser, context.getInputParam().getDataDescription(), selectSql);

            String totalSql = querySchema.getString("totalSql");
            totalSql = processSql(isSuperUser, context.getInputParam().getDataDescription(), totalSql);

            selectSql = ReplaceUtils.replace(paramMap, selectSql);
            totalSql = ReplaceUtils.replace(paramMap, totalSql);

            //用作日志打印
            logSql = selectSql;
            JSONArray list = new JSONArray();
            int pageNo = 0;
            long startTime = System.currentTimeMillis();
            String dbType = getDbType(context.getInputParam().getDataDescription(), productLine);
            boolean isFirst = true;
            while (true) {
                //分页设置
                //String newSelectSql = appendPagination(selectSql, BasePage.DEFAULT_PAGE_SIZE * pageNo);
                String newSelectSql = appendPagination(selectSql, dbType, pageNo);
                if (isFirst && context.isDebug()) {
                    isFirst = false;
                    context.getQuerySqlMap().put(actionParam.getActionId(), Arrays.asList(newSelectSql, totalSql));
                }
                JSONObject data = doQuery(newSelectSql, totalSql, context, productLine);
                if (data == null || data.isEmpty()) {
                    break;
                }
                JSONObject dataNode = data.getJSONObject("data");
                if (dataNode == null || dataNode.isEmpty() || !dataNode.containsKey("detail")) {
                    break;
                }
                JSONObject detail = dataNode.getJSONObject("detail");
                if (detail == null || detail.isEmpty() || !detail.containsKey("total") || !detail.containsKey("list")) {
                    break;
                }

                Integer total = Integer.parseInt(detail.getString("total"));
                if (total == 0) {
                    break;
                }

                if (total >= BasePage.MAX_DATA_SIZE) {
                    throw new BusinessException(ErrorCodeEnum.ACTION_PULLING_DATA_OVERFLOW);
                }

                list.addAll(detail.getJSONArray("list"));

                if (list.size() >= total) {
                    break;
                }

                pageNo++;
            }

            long timeTaken = (System.currentTimeMillis() - startTime);
            LOGGER.info(String.format("actionId：%s，查询耗时:%sms，查询笔数:%s,查询sql:%s", actionParam.getActionId(), timeTaken, list.size(), selectSql));
            LogUtils.buildAgileLog(LogConstant.AGILE_CODE_EXECUTE_SQL_ACTION, LogUtils.SUCCESS, "actionId:" + actionParam.getActionId() + ",action入参:" + JSONObject.toJSONString(reqMap) + "，查询sql:" + selectSql, "数据笔数：" + list.size(), "");

            context.setExecuteStatus(true);
            return list;
        } catch (Exception e) {
            context.setExecuteStatus(false);

            ErrorLog errorLog = new ErrorLog();
            if (e instanceof BusinessException) {
                BusinessException be = (BusinessException) e;
                errorLog.setErrorCode(be.getErrorCode());
                errorLog.setErrorMessage(be.getMessage());
            } else {
                errorLog.setErrorCode(AgileDataErrorCodeConstant.EXECUTE_SQL_ACTION_ERROR);
                errorLog.setErrorMessage(e.getMessage());
            }
            errorLog.setErrorLocation("sql执行节点，actionId:" + actionParam.getActionId());
            errorLog.setErrorDescription(e.getMessage());
            errorLog.setErrorTimestamp(DateUtils.getCurrentDateTime());

            errorLog.setPossibleCausesAndGuidance("请检查sql语法是否正确,节点id:" + actionParam.getActionId());
            if (context.getErrorLog() == null) {
                context.setErrorLog(errorLog);
            }
            LOGGER.error(String.format(LogConstant.AGILE_DATA + "sqlAction获取数据异常:%s", "executionengine", e.getMessage()));
            LogUtils.buildAgileLog(LogConstant.AGILE_CODE_EXECUTE_SQL_ACTION, AgileDataErrorCodeConstant.EXECUTE_SQL_ACTION_ERROR, "actionId:" + actionParam.getActionId() + ",action入参:" + JSONObject.toJSONString(reqMap) + "，查询sql:" + logSql, "SQL执行失败，失败原因：【" + e.getMessage() + "】", "\"1.打开数据库管理工具（如MySQLWorkbench、SQLServerManagementStudio等）把报错的SQL语句粘贴到查询窗口。\n" +
                    "2.运行SQL语句，根据数据库返回的错误信息修改语法错误，并检查SQL中表名、字段名、关键字和函数使用是否正确。\n" +
                    "3.若仍报错，将SQL语句和错误信息提供给系统管理员。\"");
            throw e;
        }
    }

    private String appendPagination(String selectSql, String dbType, int pageNo) {
        switch (dbType) {
            case ORACLE:
                return genOraclePageSql(selectSql, pageNo + 1);
            case MYSQL:
                return genMysqlPageSql(selectSql, BasePage.DEFAULT_PAGE_SIZE * pageNo);
            default:
                return genSqlServerPageSql(selectSql, BasePage.DEFAULT_PAGE_SIZE * pageNo);
        }
    }

    private String processSql(boolean isSuperUser, DataDescription dataDescription, String sql) {

        if (StringUtils.isEmpty(sql)) {
            return sql;
        }
        if (sql.indexOf(PLACE_HOLDER) == -1) {
            return sql;
        }

        if (isSuperUser || CollectionUtils.isEmpty(dataDescription.getPermissions())) {
            return sql.replaceAll("(\\s+and\\s+\\$\\(param\\))", "");
        }
        return sql;
    }

    //判断是不是超级管理员
    private boolean isSuperUser(List<String> roles) {
        if (CollectionUtils.isEmpty(roles)) {
            return false;
        }
        if (roles.contains(SUPER_USER)) {
            return true;
        }
        return false;
    }

    private String generatePermissionParam(DataDescription dataDescription) {
        JSONObject permissions = dataDescription.getPermissions();
        if (CollectionUtils.isEmpty(permissions)) {
            return null;
        }
        return processFilter(permissions);
    }

    private String processFilter(JSONObject filter) {
        StringBuilder whereBuilder = new StringBuilder();
        String logic = filter.getString("filterType");
        if (CommonConstant.LOGITYPE_AND.equalsIgnoreCase(logic)
                || CommonConstant.LOGITYPE_OR.equalsIgnoreCase(logic)) {
            whereBuilder.append("(");

            JSONArray children = filter.getJSONArray("filterValue");
            for (int i = 0; i < children.size(); i++) {
                JSONObject child = children.getJSONObject(i);
                whereBuilder.append(processFilter(child));
                if (i < children.size() - 1) {
                    whereBuilder.append(" ").append(logic).append(" ");
                }
            }
            whereBuilder.append(")");
        } else {
            String filterField = filter.getString("filterField");
            String operator = getSqlOpt(filter.getString("filterType"));
            String filterValue = filter.getString("filterValue");

            switch (operator) {
                case "like":
                    filterValue = "%" + filterValue + "%";
                    break;
                case "rl":
                    operator = "like";
                    filterValue = filterValue + "%";
                    break;
                case "nrl":
                    operator = "not like";
                    filterValue = filterValue + "%";
                    break;
                case "ll":
                    operator = "like";
                    filterValue = "%" + filterValue;
                    break;
                case "nll":
                    operator = "not like";
                    filterValue = "%" + filterValue;
                    break;
                case "in":
                case "not in":
                    filterValue = "(" + filterValue + ")";
                    break;
                default:
                    break;
            }

            if ("null".equalsIgnoreCase(operator)) {
                whereBuilder.append(filterField).append(" is null");
            } else if ("not null".equalsIgnoreCase(operator)) {
                whereBuilder.append(filterField).append(" is not null");
            } else {
                whereBuilder.append(filterField).append(" ").append(operator).append(" '").append(filterValue).append("'");
            }
        }
        return whereBuilder.toString();
    }


    private String getSqlOpt(String filterType) {
        switch (filterType) {
            case "eq":
                return "=";
            case "ne":
                return "!=";
            case "like":
                return "like";
            case "gt":
                return ">";
            case "gte":
                return ">=";
            case "lt":
                return "<";
            case "lte":
                return "<=";
            case "in":
                return "in";
            case "nin":
                return "not in";
            // Add more cases for different filter types if needed
            default:
                return filterType;
        }
    }

    private JSONObject doQuery(String selectSql, String totalSql, ExecuteContext context, String productLine) {
        Map<String, String> headerMap = new HashMap<>();
        headerMap.put("token", context.getToken());
        headerMap.put("Content-Type", "application/json;charset=UTF-8");
        headerMap.put(FieldNameConstant.CAMEL_CASE_ROUTER_KEY, context.getRouterKey());
        headerMap.put("Accept", "application/json;charset=UTF-8");
        Map<String, Object> param = new HashMap<>();
        param.put("totalSQL", totalSql);
        param.put("selectSQL", selectSql);
        param.put("productLine", productLine);

        String jsonParam = JsonResolverUtils.toJsonString(param);
        String httpRespContent = HttpClientUtils.doPost(DatasourceUrl.dcpUrl + "/api/ddl/execute/and/return", headerMap, jsonParam);
        return JSONObject.parseObject(httpRespContent);
    }

    private String getDbType(DataDescription dataDescription, String productLine) {
        JSONArray productLineInfo = dataDescription.getProductLineInfo();
        if (CollectionUtils.isEmpty(productLineInfo)) {
            return SQLSERVER;
        }
        JSONObject dbObject = null;
        if (productLine != null) {
            for (int i = 0; i < productLineInfo.size(); i++) {
                JSONObject productLineObj = productLineInfo.getJSONObject(i);
                if (productLine.equals(productLineObj.getString("productLine"))) {
                    dbObject = productLineObj;
                    break;
                }
            }
            if (dbObject == null) {
                return SQLSERVER;
            }
        } else {
            //兼容老场景
            dbObject = productLineInfo.getJSONObject(0);
        }
        switch (dbObject.getString("dbType")) {
            case ORACLE:
                return ORACLE;
            case MYSQL:
                return MYSQL;
            default:
                return SQLSERVER;
        }
    }

    /**
     * sqlServer 分页
     *
     * @param selectSql
     * @param offset
     * @return
     */
    private String genSqlServerPageSql(String selectSql, int offset) {
        Map<String, Object> replaceParam = new HashMap<>(2);
        replaceParam.put("offset", offset);
        replaceParam.put("pageSize", BasePage.DEFAULT_PAGE_SIZE);
        String sql = selectSql + SQLSERVER_PAGE;
        return ReplaceUtils.replace(replaceParam, sql);
    }

    /**
     * mysql 分页
     *
     * @param selectSql
     * @param offset
     * @return
     */
    private String genMysqlPageSql(String selectSql, int offset) {
        Map<String, Object> replaceParam = new HashMap<>(2);
        replaceParam.put("offset", offset);
        replaceParam.put("pageSize", BasePage.DEFAULT_PAGE_SIZE);
        String sql = selectSql + MYSQL_PAGE;
        return ReplaceUtils.replace(replaceParam, sql);
    }

    /**
     * oracle 分页
     *
     * @param selectSql
     * @param pageNo
     * @return
     */
    private String genOraclePageSql(String selectSql, int pageNo) {
        Map<String, Object> replaceParam = new HashMap<>(2);
        replaceParam.put("startNo", (pageNo - 1) * BasePage.DEFAULT_PAGE_SIZE + 1);
        replaceParam.put("endNo", pageNo * BasePage.DEFAULT_PAGE_SIZE);
        String sql = String.format(ORACLE_PAGE, selectSql);
        return ReplaceUtils.replace(replaceParam, sql);
    }


    private Map<String, String> generateParam(JSONObject querySchema, Map<String, Object> reqData, String actionId) {
        JSONArray params = querySchema.getJSONArray("params");
        Map<String, String> paramMap = new HashMap<>();
        params.forEach(param -> {
            JSONObject paramJsonObject = (JSONObject) param;
            if (paramJsonObject.containsKey("source")) {
                String inputParamKey = paramJsonObject.getString("source");
                String dataName = paramJsonObject.getString("data_name");
                String interval = paramJsonObject.getString("interval");
                String inputValue = (String) reqData.get(inputParamKey);

                if (StringUtils.isNotEmpty(inputValue) && inputValue.contains("#")) {
                    int splitPos = inputValue.indexOf('#');
                    if ("between-left".equals(interval)) {
                        inputValue = inputValue.substring(0, splitPos);
                    } else if ("between-right".equals(interval)) {
                        inputValue = inputValue.substring(splitPos + 1);
                    } else {
                        LOGGER.info("BMDAction:{}替换查询参数source:{}时遇到未识别的interval:{}", actionId, inputParamKey, interval);
                    }
                }
                if (StringUtils.isEmpty(inputValue)) {
                    inputValue = getDefaultSetting(paramJsonObject);
                }
                paramMap.put(dataName, inputValue);
            } else {
                String dataName = paramJsonObject.getString("data_name");
                String inputValue = (String) reqData.get(dataName);
                if (StringUtils.isEmpty(inputValue)) {
                    inputValue = getDefaultSetting(paramJsonObject);
                }
                paramMap.put(dataName, inputValue);
            }
        });
        return paramMap;
    }

    private String getDefaultSetting(JSONObject paramJsonObject) {
        String val = null;
        if (!paramJsonObject.containsKey("paramDefault")) {
            LOGGER.warn("图谱中的条件参数并未获得且未设定default默认值");
            return null;
        }

        JSONObject paramDefault = paramJsonObject.getJSONObject("paramDefault");

        String method = paramDefault.getString("method");
        switch (method) {
            case CommonConstant.FIRSTDAYOFMONTHYEARAGO:
                val = DateUtils.firstDayOfMonthYearAgo("");
                break;
            case CommonConstant.FIRSTDATEOFLASTMONTH:
                val = DateUtils.getfirstDayOfLastMonth("");
                break;
            case CommonConstant.LASTDATEOFLASTMONTH:
                val = DateUtils.getLastDayOfLastMonth("");
                break;
            case CommonConstant.FIRSTDAYOFTHISYEAR:
                val = DateUtils.firstDayOfThisYear();
                break;
            case CommonConstant.LASTDAYOFTHISYEAR:
                val = DateUtils.lastDayOfThisYear();
                break;
            case CommonConstant.FIRSTDATEOFMONTH:
                val = DateUtils.firstDayOfThisMonth();
                break;
            case CommonConstant.LASTDATEOFMONTH:
                val = DateUtils.lastDayOfThisMonth();
                break;
            case CommonConstant.CUSTOM:
                val = paramDefault.getString("value");
                break;
            default:
                LOGGER.warn("未识别的 method 类型: {}", method);
                break;
        }
        return val;
    }
}
