package com.digiwin.athena.executionengine.service.facade.schema;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.digiwin.athena.executionengine.bo.DataCenterParamBo;
import com.digiwin.athena.executionengine.constant.CommonConstant;
import com.digiwin.athena.executionengine.constant.FieldNameConstant;
import com.digiwin.athena.executionengine.dto.schema.SchemaTransDto;
import com.digiwin.athena.executionengine.model.DataDescription;
import com.digiwin.athena.executionengine.model.DataItem;
import com.digiwin.athena.executionengine.model.action.DataCenterFilterNode;
import com.digiwin.athena.executionengine.model.action.FilterNode;
import com.digiwin.athena.executionengine.util.ContextUtils;
import com.digiwin.athena.executionengine.util.DateUtils;
import com.digiwin.athena.executionengine.util.JsonUtil;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.stream.Collectors;

/**
 * @description:
 * @author: ZhangJun
 * @create: 2024/9/13
 */
@Service
public class SchemaBuilder implements ISchemaBuilder {
    private static final Logger LOGGER = LoggerFactory.getLogger(SchemaBuilder.class);

    @Override
    public Map<String, Object> buildQuerySchema(JSONObject querySchema, Map<String, Object> reqData, String actionId, DataDescription dataDescription) {
        JSONObject currentQuerySchema = JsonUtil.getObject(JsonUtil.getJsonString(querySchema));
        buildDynamicFilters(currentQuerySchema, dataDescription);
        buildPermissionFilters(currentQuerySchema, dataDescription);
        doParamReplace(reqData, actionId, currentQuerySchema);
        return currentQuerySchema;
    }

    /**
     * 查询方案变量 替换
     *
     * @param reqData
     * @param actionId
     * @param currentQuerySchema
     */
    private void doParamReplace(Map<String, Object> reqData, String actionId, JSONObject currentQuerySchema) {
        JSONArray params = currentQuerySchema.getJSONArray("params");
        for (int i = 0; i < params.size(); i++) {
            JSONObject paramJsonObject = params.getJSONObject(i);
            String inputParamKey = "";
            String inputValue = "";
            /*有source需要按照区间处理，没有source直接替换参数值*/
            if (paramJsonObject.containsKey("source")) {
                inputParamKey = paramJsonObject.getString("source");
                String interval = paramJsonObject.getString("interval");
                inputValue = (String) reqData.get(inputParamKey);

                //可能入参确实为空，但要走下去，可能会有默认值
                if (StringUtils.isNotEmpty(inputValue) && StringUtils.isNotEmpty(interval)) {
                    switch (interval) {
                        case "between-left": {
                            int splitPos = inputValue.indexOf('#');
                            inputValue = inputValue.substring(0, splitPos);
                            break;
                        }
                        case "between-right": {
                            int splitPos = inputValue.indexOf('#') + 1;
                            inputValue = inputValue.substring(splitPos);
                            break;
                        }
                        default:
                            LOGGER.info("BMDAction:{}替换查询参数souce:{}时遇到未识别的interval:{}", actionId, inputParamKey, interval);
                    }
                }
                String value = getValue(paramJsonObject, inputValue);
                paramJsonObject.put("value", value);
                paramJsonObject.fluentRemove("source");
                paramJsonObject.fluentRemove("interval");
            } else {
                inputParamKey = paramJsonObject.getString("value");
                inputValue = (String) reqData.get(inputParamKey);
                paramJsonObject.put("value", inputValue);
            }
        }
    }

    @Override
    public Map<String, Object> buildQuerySchemaByNewStrategy(JSONObject querySchema, Map<String, Object> reqData, DataDescription dataDescription) {
        JSONObject currentQuerySchema = JsonUtil.getObject(JsonUtil.getJsonString(querySchema));
        buildPermissionFilters(currentQuerySchema, dataDescription);

        JSONObject dynamicSchema = dataDescription.getDynamicSchema();
        JSONObject computeObj = dynamicSchema.getJSONObject(FieldNameConstant.DYNAMIC_SCHEMA_COMPUTELIST);

        dynamicSchema.forEach((k, v) -> {
            String beanName = CommonConstant.CONVERTOR_MAP.get(k);
            if (StringUtils.isNotEmpty(beanName)) {
                ContextUtils.getBean(beanName, ISchemaConvertor.class).convert(new SchemaTransDto(currentQuerySchema, v, computeObj, reqData));
            }

        });

        //查询方案变量替换
        doParamReplace(reqData, "", currentQuerySchema);

        return currentQuerySchema;
    }

    @Override
    public Map<String, Object> buildExportQuerySchema(String querySchema, Map<String, Object> reqData, DataDescription dataDescription) {
        JSONObject currentQuerySchema = JsonUtil.getObject(querySchema);
        buildPermissionFilters(currentQuerySchema, dataDescription);

        JSONObject dynamicSchema = dataDescription.getDynamicSchema();
        JSONObject computeObj = dynamicSchema.getJSONObject(FieldNameConstant.DYNAMIC_SCHEMA_COMPUTELIST);
        dynamicSchema.remove(FieldNameConstant.DYNAMIC_SCHEMA_COMPUTELIST);

        dynamicSchema.forEach((k, v) -> {
            //只处理filter
            if ("filter".equalsIgnoreCase(k)) {
                String beanName = CommonConstant.CONVERTOR_MAP.get(k);
                ContextUtils.getBean(beanName, ISchemaConvertor.class).convert(new SchemaTransDto(currentQuerySchema, v, computeObj, reqData));
            }
        });

        //查询方案变量替换
        doParamReplace(reqData, "", currentQuerySchema);
        return currentQuerySchema;
    }

    @Override
    public DataCenterParamBo buildDataCenterSchema(DataDescription dataDescription) {
        JSONObject querySchema = createBaseQuerySchema();

        JSONObject dynamicSchema = dataDescription.getDynamicSchema();
        JSONObject mappingFields = (JSONObject) dynamicSchema.get("fieldMapping");
        buildDatasetPermission(querySchema, dataDescription.getDatasetPermissions(), mappingFields);

        JSONObject computeObj = dynamicSchema.getJSONObject(FieldNameConstant.DYNAMIC_SCHEMA_COMPUTELIST);

        dynamicSchema.forEach((k, v) -> {
            String beanName = CommonConstant.DATA_CENTER_CONVERTOR_MAP.get(k);
            if (StringUtils.isNotEmpty(beanName)) {
                ContextUtils.getBean(beanName, ISchemaConvertor.class).convert(new SchemaTransDto(querySchema, v, computeObj, mappingFields));
            }
        });
        JSONArray orderColumns = buildOrderColumns(dynamicSchema.getJSONArray("sort"), computeObj, mappingFields);
        return new DataCenterParamBo(querySchema, orderColumns);
    }

    private JSONObject createBaseQuerySchema() {
        JSONObject querySchema = new JSONObject();
        querySchema.put("dataType", "1");
        querySchema.put("distinct", "false");
        querySchema.put("dimensions", new JSONArray());
        querySchema.put("measures", new JSONArray());
        querySchema.put("filter", new JSONObject());
        querySchema.put("havingFilters", new JSONArray());
        querySchema.put("computedFields", new JSONArray());
        return querySchema;
    }

    private JSONArray buildOrderColumns(JSONArray sort, JSONObject computeObj, JSONObject mappingFields) {
        if (CollectionUtils.isEmpty(sort)) {
            return new JSONArray();
        }
        JSONArray newSort = new JSONArray();
        sort.forEach(item -> {
            JSONObject jsonObject = (JSONObject) item;
            JSONObject dataObject = jsonObject.getJSONObject("dataObject");
            String field = dataObject.getString(FieldNameConstant.DYNAMIC_SCHEMA_CONTENT);
            String contentType = dataObject.getString(FieldNameConstant.DYNAMIC_SCHEMA_CONTENT_TYPE);
            if ("calculate".equals(contentType)) {
                JSONArray function = computeObj.getJSONArray(field);
                JSONArray params = function.getJSONObject(0).getJSONArray("params");
                JSONObject fieldParam = params.getJSONObject(0);
                field = fieldParam.getString(FieldNameConstant.DYNAMIC_SCHEMA_CONTENT);
            }
            String order = jsonObject.getString("order");
            JSONObject sortSchemaObj = new JSONObject();
            sortSchemaObj.put("field", mappingFields.get(field));
            sortSchemaObj.put("order", order);
            newSort.add(sortSchemaObj);
        });
        return newSort;
    }

    /**
     * 构建数据集权限过滤条件
     *
     * @param querySchema
     * @param permissions
     * @param mappingFields
     */
    private void buildDatasetPermission(JSONObject querySchema, JSONArray permissions, JSONObject mappingFields) {
        if (CollectionUtils.isEmpty(permissions)) {
            return;
        }
        JSONArray children = new JSONArray();
        permissions.forEach(permission -> {
            DataCenterFilterNode filterNode = processDatasetFilter((JSONObject) permission, mappingFields);
            if (filterNode != null) {
                children.add(JSONObject.toJSON(filterNode));
            }
        });
        JSONObject filter = new JSONObject();
        filter.put("logical", "and");
        filter.put("children", children);
        querySchema.put(FieldNameConstant.DYNAMIC_SCHEMA_FILTER, filter);
    }

    private DataCenterFilterNode processDatasetFilter(JSONObject filter, JSONObject mappingFields) {
        if (MapUtils.isEmpty(filter)) {
            return null;
        }
        DataCenterFilterNode filterNode = new DataCenterFilterNode();
        String logic = filter.getString("filterType");
        if (CommonConstant.LOGITYPE_AND.equalsIgnoreCase(logic) || CommonConstant.LOGITYPE_OR.equalsIgnoreCase(logic)) {
            filterNode.setLogical(logic);
            JSONArray children = filter.getJSONArray("filterValue");
            for (int i = 0; i < children.size(); i++) {
                JSONObject child = children.getJSONObject(i);
                DataCenterFilterNode childNode = processDatasetFilter(child, mappingFields);
                if (childNode != null) {
                    filterNode.getChildren().add(childNode);
                }
            }
        } else {
            filterNode.setLeftValue(mappingFields.getString(filter.getString("filterField")));
            filterNode.setOperator(CommonConstant.DATASET_PERMISSION_FILTER_TYPE_COMPARISON_MAP.get(filter.getString("filterType")));
            filterNode.setRightValue(filter.getString("filterValue"));
            filterNode.setRightType("const");
        }
        return filterNode;
    }

    /**
     * 获取模板查询方案
     *
     * @return
     */


    private void buildDynamicFilters(JSONObject querySchema, DataDescription dataDescription) {
        if (dataDescription == null) {
            return;
        }
        if (CollectionUtils.isEmpty(querySchema.getJSONArray("measures"))) {
            //明细查询,不做动态维度
            return;
        }
        List<DataItem> items = dataDescription.getDataItems().stream().filter(dataItem -> dataItem.getType().equalsIgnoreCase("string")).collect(Collectors.toList());

        JSONObject showFieldsMapping = querySchema.getJSONObject("showFieldsMapping");
        // k v 反转一下
        Map<String, String> newMap = showFieldsMapping.entrySet().stream().collect(Collectors.toMap(entry -> entry.getValue().toString(), entry -> entry.getKey()));

        if (!CollectionUtils.isEmpty(items)) {
            for (DataItem item : items) {
                if (!newMap.containsKey(item.getName())) {
                    continue;
                }
                String uuid = UUID.randomUUID().toString();
                JSONObject paramJsonObject = new JSONObject();
                paramJsonObject.put("field", item.getName());
                paramJsonObject.put("name", item.getName());
                paramJsonObject.put("dataType", 0);
                paramJsonObject.put("releaseFlag", 0);
                paramJsonObject.put("key", uuid);
                paramJsonObject.put("value", item.getName());
                paramJsonObject.put("required", "0");
                querySchema.getJSONArray("params").add(paramJsonObject);

                JSONObject filterJsonObject = new JSONObject();
                filterJsonObject.put("leftValue", newMap.get(item.getName()));
                filterJsonObject.put("operator", "eq");
                filterJsonObject.put("rightValue", item.getName());
                filterJsonObject.put("rightType", "param");
                filterJsonObject.put("key", uuid);
                filterJsonObject.put("logical", "and");
                filterJsonObject.put("children", new JSONArray());
                querySchema.getJSONObject("filter").getJSONArray("children").add(filterJsonObject);
            }
        }


        if (!CollectionUtils.isEmpty(dataDescription.getDimensions())) {
            boolean match = dataDescription.getDimensions().stream().anyMatch(item -> newMap.containsKey(item));
            if (match) {
                Map<String, String> map = dataDescription.getDimensions().stream().filter(item -> newMap.get(item) != null).collect(Collectors.toMap(item -> newMap.get(item), item -> item));
                querySchema.getJSONArray("dimensions").stream().filter(o -> !"N".equals(((JSONObject) o).getString("isEnableAggregate"))).filter(o -> ((JSONObject) o).getInteger("dataType") != 7 && ((JSONObject) o).getInteger("dataType") != 1).filter(o -> !map.containsKey(((JSONObject) o).getString("field"))).forEach(o -> ((JSONObject) o).put("defaultValue", "auto"));
            } else {
                //都没匹配上
                querySchema.getJSONArray("dimensions").stream().filter(o -> !"N".equals(((JSONObject) o).getString("isEnableAggregate"))).filter(o -> ((JSONObject) o).getInteger("dataType") != 7 && ((JSONObject) o).getInteger("dataType") != 1).forEach(o -> ((JSONObject) o).put("defaultValue", "auto"));
            }
        } else {
            querySchema.getJSONArray("dimensions").stream().filter(o -> !"N".equals(((JSONObject) o).getString("isEnableAggregate"))).filter(o -> ((JSONObject) o).getInteger("dataType") != 7 && ((JSONObject) o).getInteger("dataType") != 1).forEach(o -> ((JSONObject) o).put("defaultValue", "auto"));
        }
    }

    private void buildPermissionFilters(JSONObject querySchema, DataDescription dataDescription) {
        if (dataDescription == null) {
            return;
        }
        JSONObject showFieldsMapping = querySchema.getJSONObject("showFieldsMapping");
        // k v 反转一下
        Map<String, String> mapping = showFieldsMapping.entrySet().stream().collect(Collectors.toMap(entry -> entry.getValue().toString(), entry -> entry.getKey()));

        JSONObject permissions = dataDescription.getPermissions();
        //1.0指标
        if (!CollectionUtils.isEmpty(permissions)) {
            FilterNode filterNode = processFilter(permissions, mapping);
            if (filterNode != null) {
                JSONObject filter = new JSONObject();
                JSONArray children = new JSONArray();
                filter.put("logical", "and");
                filter.put("children", children);
                children.add(querySchema.getJSONObject("filter"));
                children.add(JSONObject.toJSON(filterNode));
                querySchema.put(FieldNameConstant.DYNAMIC_SCHEMA_FILTER, filter);
            }
        } else if (!CollectionUtils.isEmpty(dataDescription.getDatasetPermissions())) {
            //1.0数据集
            JSONArray children = new JSONArray();
            dataDescription.getDatasetPermissions().forEach(permission -> {
                FilterNode filterNode = processFilter((JSONObject) permission, mapping);
                if (filterNode != null) {
                    children.add(JSONObject.toJSON(filterNode));
                }
            });
            JSONObject filter = new JSONObject();
            filter.put("logical", "and");
            filter.put("children", children);
            querySchema.put(FieldNameConstant.DYNAMIC_SCHEMA_FILTER, filter);
        }
    }

    private FilterNode processFilter(JSONObject filter, Map<String, String> mapping) {
        FilterNode filterNode = new FilterNode();
        String logic = filter.getString("filterType");
        if (CommonConstant.LOGITYPE_AND.equalsIgnoreCase(logic) || CommonConstant.LOGITYPE_OR.equalsIgnoreCase(logic)) {
            filterNode.setLogical(logic);
            JSONArray children = filter.getJSONArray("filterValue");
            for (int i = 0; i < children.size(); i++) {
                JSONObject child = children.getJSONObject(i);
                FilterNode childNode = processFilter(child, mapping);
                if (childNode != null) {
                    filterNode.getChildren().add(childNode);
                }
            }
        } else {
            if (!mapping.containsKey(filter.getString("filterField"))) {
                return null;
            }
            filterNode.setLeftValue(mapping.get(filter.getString("filterField")));
            filterNode.setOperator(CommonConstant.PERMISSION_FILTER_TYPE_COMPARISON_MAP.get(filter.getString("filterType")));
            filterNode.setRightValue(filter.getString("filterValue"));
            filterNode.setRightType("const");
        }
        return filterNode;
    }


    private String getValue(JSONObject paramJsonObject, String val) {
        if (val != null) {
            return val;
        }

        if (!paramJsonObject.containsKey("paramDefault")) {
            LOGGER.warn("图谱中的条件参数并未获得且未设定default默认值");
            return null;
        }

        JSONObject paramDefault = paramJsonObject.getJSONObject("paramDefault");

        String method = paramDefault.getString("method");
        switch (method) {
            case CommonConstant.FIRSTDAYOFMONTHYEARAGO:
                val = DateUtils.firstDayOfMonthYearAgo("");
                break;
            case CommonConstant.FIRSTDATEOFLASTMONTH:
                val = DateUtils.getfirstDayOfLastMonth("");
                break;
            case CommonConstant.LASTDATEOFLASTMONTH:
                val = DateUtils.getLastDayOfLastMonth("");
                break;
            case CommonConstant.FIRSTDAYOFTHISYEAR:
                val = DateUtils.firstDayOfThisYear();
                break;
            case CommonConstant.LASTDAYOFTHISYEAR:
                val = DateUtils.lastDayOfThisYear();
                break;
            case CommonConstant.FIRSTDATEOFMONTH:
                val = DateUtils.firstDayOfThisMonth();
                break;
            case CommonConstant.LASTDATEOFMONTH:
                val = DateUtils.lastDayOfThisMonth();
                break;
            case CommonConstant.CUSTOM:
                val = paramDefault.getString("value");
                break;
            default:
                LOGGER.warn("未识别的 method 类型: {}", method);
                break;
        }
        return val;
    }
}
