package com.digiwin.athena.executionengine.trans.components;

import com.digiwin.athena.executionengine.constant.LogConstant;
import com.digiwin.athena.executionengine.constant.TransConstant;
import com.digiwin.athena.executionengine.model.trans.DealResult;
import com.digiwin.athena.executionengine.model.trans.JoinSelectElement;
import com.digiwin.athena.executionengine.model.trans.StepElement;
import com.digiwin.athena.executionengine.trans.Step;
import com.digiwin.athena.executionengine.trans.TransAbstractStep;
import com.digiwin.athena.executionengine.util.LogUtils;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import java.util.*;
import java.util.stream.Collectors;

/**
 * @description: Join计算节点，未来实现left、right、inner、full四种类型的join
 * @author: fenglei
 * @date:
 */
@Component("join")
public class JoinStep extends TransAbstractStep {

    private static final Logger LOGGER = LoggerFactory.getLogger(InputStep.class);

    @Override
    public boolean defineCheck(StepElement stepElement) {
        return true;
    }

    /**
     * 数据处理的前置处理
     *
     * @return
     * @throws Exception
     */
    @Override
    public DealResult doDealData(Step step) {
        DealResult dealResult = new DealResult();
        StepElement stepElement = step.getStepElement();
        try {
            String dataLtName = stepElement.getDataLt();
            Object dataLt = getCurrentData(dataLtName);
            String dataRtName = stepElement.getDataRt();
            Object dataRt = getCurrentData(dataRtName);
            if (dataLt == null && TransConstant.COMPONENT_TYPE_LEFT_JOIN.equals(stepElement.getType())) {
                dealResult.setSuccess(new ArrayList<>());
                return dealResult;
            }

            if (dataRt == null && TransConstant.COMPONENT_TYPE_LEFT_JOIN.equals(stepElement.getType())) {
                dealResult.setSuccess(dataLt);
                return dealResult;
            }

            if (dataRt == null && TransConstant.COMPONENT_TYPE_RIGHT_JOIN.equals(stepElement.getType())) {
                dealResult.setSuccess(new ArrayList<>());
                return dealResult;
            }

            if (dataLt == null && TransConstant.COMPONENT_TYPE_RIGHT_JOIN.equals(stepElement.getType())) {
                dealResult.setSuccess(dataRt);
                return dealResult;
            }

            if (dataLt == null && TransConstant.COMPONENT_TYPE_FULL_JOIN.equals(stepElement.getType())) {
                dealResult.setSuccess(dataRt);
                return dealResult;
            }

            if (dataRt == null && TransConstant.COMPONENT_TYPE_FULL_JOIN.equals(stepElement.getType())) {
                dealResult.setSuccess(dataLt);
                return dealResult;
            }

            if (dataLt == null && TransConstant.COMPONENT_TYPE_INNER_JOIN.equals(stepElement.getType())) {
                dealResult.setSuccess(dataRt);
                return dealResult;
            }

            if (dataRt == null && TransConstant.COMPONENT_TYPE_INNER_JOIN.equals(stepElement.getType())) {
                dealResult.setSuccess(dataLt);
                return dealResult;
            }

            if (dataLt == null && TransConstant.COMPONENT_TYPE_CROSS_JOIN.equals(stepElement.getType())) {
                dealResult.setSuccess(dataRt);
                return dealResult;
            }

            if (dataRt == null && TransConstant.COMPONENT_TYPE_CROSS_JOIN.equals(stepElement.getType())) {
                dealResult.setSuccess(dataLt);
                return dealResult;
            }

            List<Map<String, Object>> leftList = (List<Map<String, Object>>) dataLt;
            List<Map<String, Object>> rightList = (List<Map<String, Object>>) dataRt;
            List<String> leftFields = stepElement.getLeftFields();
            List<String> rightFields = stepElement.getRightFields();
            List<JoinSelectElement> selectLtFieldList = stepElement.getSelectLtFields();
            List<JoinSelectElement> selectRtFieldList = stepElement.getSelectRtFields();
            //full,inner,left获取左Left查询的字段，如果是right join则将select_lt_fields放入selectRtFieldList
            if (TransConstant.COMPONENT_TYPE_FULL_JOIN.equals(stepElement.getType()) ||
                    TransConstant.COMPONENT_TYPE_INNER_JOIN.equals(stepElement.getType()) ||
                    TransConstant.COMPONENT_TYPE_LEFT_JOIN.equals(stepElement.getType()) ||
                    TransConstant.COMPONENT_TYPE_CROSS_JOIN.equals(stepElement.getType())) {
                selectLtFieldList = analyseSelectKeys(dataLtName, selectLtFieldList);
            } else {
                //right join
                List<JoinSelectElement> selectFields = new ArrayList();
                selectRtFieldList = analyseSelectKeys(dataRtName, selectLtFieldList);

                ///左连接，替换掉右表选取字段，因为左连接用右表追加
                if (TransConstant.COMPONENT_TYPE_RIGHT_JOIN.equals(stepElement.getType())) {
                    for (JoinSelectElement jse : selectLtFieldList) {
                        if (!leftFields.contains(jse.getNewField())) {
                            //存放查出左表字段的别名
                            selectFields.add(jse);
                        }
                    }
                    selectRtFieldList = selectFields;
                }
            }
            //full,right，left,inner获取右Right查询字段,目前看下面的第一层胖段逻辑一致，只是《full》有特殊处理，？？？？逻辑有疑问？？？？
            if (TransConstant.COMPONENT_TYPE_FULL_JOIN.equals(stepElement.getType()) ||
                    TransConstant.COMPONENT_TYPE_RIGHT_JOIN.equals(stepElement.getType())
            ) {
                selectRtFieldList = analyseSelectKeys(dataRtName, selectRtFieldList);
            } else {
                //left inner join
                ////可能出现左连接右表没有对应记录或者右连接没有对应左表记录，导致后一个表在merge时若newField字段相同时出现覆盖了""
                selectRtFieldList = analyseSelectKeys(dataRtName, selectRtFieldList);
            }

            //dataLt不为空，则合并后一定会返回数据
            List<Map<String, Object>> data = new ArrayList<>();
            if (TransConstant.COMPONENT_TYPE_LEFT_JOIN.equals(stepElement.getType())) {
                data = joinData(leftList, rightList, selectLtFieldList,
                        selectRtFieldList, leftFields, rightFields);
            } else if (TransConstant.COMPONENT_TYPE_RIGHT_JOIN.equals(stepElement.getType())) {
                data = joinData(rightList, leftList, selectRtFieldList, selectLtFieldList,
                        rightFields, leftFields);
            } else if (TransConstant.COMPONENT_TYPE_FULL_JOIN.equals(stepElement.getType())) {
                List<Map<String, Object>> leftDatas = joinData(leftList, rightList, selectLtFieldList,
                        selectRtFieldList, leftFields, rightFields);
                List<Map<String, Object>> rightDatas = joinData(rightList, leftList, selectRtFieldList, selectLtFieldList,
                        rightFields, leftFields);
                leftDatas.addAll(rightDatas);
                List<Map<String, Object>> mergedDeduplicatedList = new ArrayList<>();
                Set<Map<String, Object>> set = new HashSet<>();
                for (Map<String, Object> map : leftDatas) {
                    Map<String, Object> hashMap = new HashMap<>(map);
                    if (set.add(hashMap)) {
                        mergedDeduplicatedList.add(hashMap);
                    }
                }
                data = mergedDeduplicatedList;
            } else if (TransConstant.COMPONENT_TYPE_INNER_JOIN.equals(stepElement.getType())) {
                data = innerJoinData(leftList, rightList, selectLtFieldList,
                        selectRtFieldList, leftFields, rightFields);
            } else if (TransConstant.COMPONENT_TYPE_CROSS_JOIN.equals(stepElement.getType())) {
                //cross
                data = crossJoinData(leftList, rightList, selectLtFieldList,
                        selectRtFieldList);
            }
            LogUtils.buildAgileLog(LogConstant.AGILE_CODE_JOIN_DATA, LogUtils.SUCCESS,
                    String.format(
                            "leftData条数:%s ,joinData条数:%s ,join方式:%s",
                            CollectionUtils.isEmpty(leftList) ? "null or empty" : leftList.size(),
                            CollectionUtils.isEmpty(rightList) ? "null or empty" : leftList.size(),
                            stepElement.getType()),
                    String.format("join后数据条数:%s ", CollectionUtils.isEmpty(data) ? "null or empty" : data.size()),
                    "1.检查关联操作查看使用的关联语句（如JOIN语句）及其条件。\n" +
                            "2.对于聚合查询，确认维度字段已包含在关联条件中。\n" +
                            "3.在数据库中使用TRIM()函数检查关联字段是否含空格。\n" +
                            "4.调整关联语句，重新运行查询，若仍不能解决，联系平台技术支持人员。");
            dealResult.setSuccess(data);
            return dealResult;
        } catch (Exception e) {
            LOGGER.error("join组件出现异常");
            throw e;
        }
    }

    /**
     * 分析需要取出来的字段:字段 别名 类型
     *
     * @param stepName        步骤名称
     * @param selectFieldList 步骤里面需要取出的字段
     * @return JoinSelectElement list
     */
    private List<JoinSelectElement> analyseSelectKeys(String stepName, List<JoinSelectElement> selectFieldList) {
        List<JoinSelectElement> selectList = new ArrayList<>();
        Object record = null;
        Object stepData = getTransDataManager().getStepDataCollection(stepName);
        if (stepData instanceof List
                && CollectionUtils.isNotEmpty((List) stepData)) {
            record = ((ArrayList) getTransDataManager().getStepDataCollection(stepName)).get(0);
        } else if (stepData instanceof Map) {
            record = stepData;
        }
        if (record == null) {
            return Collections.EMPTY_LIST;
        }

        Map<String, Object> map = (Map<String, Object>) record;
        if (CollectionUtils.isEmpty(selectFieldList)) {
            //为空去所有的字段
            map.forEach((k, v) -> {
                if (v instanceof Number) {
                    selectList.add(new JoinSelectElement(k, k, "number"));
                } else if (v instanceof Boolean) {
                    selectList.add(new JoinSelectElement(k, k, "boolean"));
                } else {
                    selectList.add(new JoinSelectElement(k, k, "string"));
                }
            });
            return selectList;
        }


        for (JoinSelectElement jse : selectFieldList) {
            if (map.get(jse.getField()) instanceof Number) {
                selectList.add(new JoinSelectElement(jse.getField(), jse.getField(), "number"));
            } else if (map.get(jse.getField()) instanceof Boolean) {
                selectList.add(new JoinSelectElement(jse.getField(), jse.getField(), "boolean"));
            } else {
                selectList.add(new JoinSelectElement(jse.getField(), jse.getField(), "string"));
            }
        }
        return selectList;


    }

    /**
     * 根据规则对数据进行join操作
     *
     * @param leftList          left数据
     * @param rightList         right数据
     * @param selectLtFieldList 需要join的列
     * @param selectRtFieldList 需要join的列
     * @param leftFields        left数据中参与join的条件字段   ///允许为空，为空则返回left所有字段
     * @param rightFields       right数据中参与join的条件字段  ///允许为空，为空则返回left所有字段
     * @return
     */

    private List<Map<String, Object>> joinData(List<Map<String, Object>> leftList, List<Map<String, Object>> rightList,
                                               List<JoinSelectElement> selectLtFieldList,
                                               List<JoinSelectElement> selectRtFieldList,
                                               List<String> leftFields, List<String> rightFields) {
        List<Map<String, Object>> list = new ArrayList<>();
        Map<String, List<Map<String, Object>>> map2Index = rightList.stream().collect(Collectors.groupingBy(map -> buildIndexKey(map, rightFields)));
        for (Map<String, Object> left : leftList) {
            String key = buildIndexKey(left, leftFields);
            List<Map<String, Object>> rightData = map2Index.get(key);
            if (CollectionUtils.isEmpty(rightData)) {
                //左表未匹配到右表，将左表放入数据集中,右表补空
                mergeData(selectLtFieldList, selectRtFieldList, list, left, null);
                continue;
            }

            //匹配到了，需要将匹配到的数据都放入join集中
            for (Map<String, Object> right : rightData) {
                mergeData(selectLtFieldList, selectRtFieldList, list, left, right);
            }

        }
        return list;
    }

    private String buildIndexKey(Map<String, Object> data, List<String> fields) {
        StringBuilder indexKey = new StringBuilder();
        for (String field : fields) {
            indexKey.append(data.get(field));
            indexKey.append("&&");
        }
        return indexKey.toString();
    }

    /**
     * 检查是否有匹配的数据
     *
     * @param leftFields  left数据中参与join的条件字段
     * @param rightFields right数据中参与join的条件字段
     * @param left        left数据
     * @param right       right数据
     * @return
     */
    private boolean checkIsMatch(List<String> leftFields, List<String> rightFields, Map<String, Object> left, Map<String, Object> right) {
        boolean isMatch = true;
        for (int i = 0; i < leftFields.size(); i++) {
            if (!Objects.equals(left.get(leftFields.get(i)), right.get(rightFields.get(i)))) {
                isMatch = false;
                break;
            }
        }
        return isMatch;
    }

    /**
     * 合并数据
     *
     * @param selectLtFieldList 需要join的列
     * @param selectRtFieldList 需要join的列
     * @param list              结果集
     * @param left              left数据
     * @param right             right数据
     */
    private void mergeData(List<JoinSelectElement> selectLtFieldList, List<JoinSelectElement> selectRtFieldList, List<Map<String, Object>> list, Map<String, Object> left, Map<String, Object> right) {
        Map<String, Object> map = new HashMap<>();

        for (JoinSelectElement element : selectLtFieldList) {
            map.put(element.getNewField(), left.get(element.getField()));
        }

        for (JoinSelectElement element : selectRtFieldList) {
            if (right != null) {
                map.put(element.getNewField(), right.get(element.getField()));
            } else {
                if (!map.containsKey(element.getNewField())) {
                    map.put(element.getNewField(), getJoinDefaultValue(element.getType()));
                }
            }
        }

        list.add(map);
    }

    private Object getJoinDefaultValue(String type) {
        switch (type) {
            case "number":
                return 0;
            case "boolean":
                return false;
            default:
                return "";
        }
    }

    /**
     * 根据规则对数据进行join操作
     *
     * @param leftList          left数据
     * @param rightList         right数据
     * @param selectLtFieldList 需要join的列
     * @param selectRtFieldList 需要join的列
     * @param leftFields        left数据中参与join的条件字段   ///允许为空，为空则返回left所有字段
     * @param rightFields       right数据中参与join的条件字段  ///允许为空，为空则返回left所有字段
     * @return
     */

    private List<Map<String, Object>> innerJoinData(List<Map<String, Object>> leftList, List<Map<String, Object>> rightList,
                                                    List<JoinSelectElement> selectLtFieldList,
                                                    List<JoinSelectElement> selectRtFieldList,
                                                    List<String> leftFields, List<String> rightFields) {
        List<Map<String, Object>> list = new ArrayList<>();
        Map<String, List<Map<String, Object>>> map2Index = rightList.stream().collect(Collectors.groupingBy(map -> buildIndexKey(map, rightFields)));
        for (Map<String, Object> left : leftList) {
            String key = buildIndexKey(left, leftFields);
            List<Map<String, Object>> rightData = map2Index.get(key);
            if (!CollectionUtils.isEmpty(rightData)) {
                //匹配到了，需要将匹配到的数据都放入join集中
                for (Map<String, Object> right : rightData) {
                    mergeData(selectLtFieldList, selectRtFieldList, list, left, right);
                }
            }
        }
        return list;
    }

    /**
     * 根据规则对数据进行cross join操作
     *
     * @param leftList          left数据
     * @param rightList         right数据
     * @param selectLtFieldList 需要join的列
     * @param selectRtFieldList 需要join的列
     * @return
     */
    private List<Map<String, Object>> crossJoinData(List<Map<String, Object>> leftList, List<Map<String, Object>> rightList,
                                                    List<JoinSelectElement> selectLtFieldList,
                                                    List<JoinSelectElement> selectRtFieldList) {
        if (CollectionUtils.isEmpty(leftList)) {
            return rightList;
        }

        if (CollectionUtils.isEmpty(rightList)) {
            return leftList;
        }

        List<Map<String, Object>> list = new ArrayList<>();
        for (Map<String, Object> left : leftList) {
            for (Map<String, Object> right : rightList) {
                mergeData(selectLtFieldList, selectRtFieldList, list, left, right);
            }
        }
        return list;
    }

}