package com.digiwin.athena.executionengine.trans.components;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.digiwin.athena.executionengine.constant.LogConstant;
import com.digiwin.athena.executionengine.constant.TransConstant;
import com.digiwin.athena.executionengine.model.trans.DealResult;
import com.digiwin.athena.executionengine.model.trans.FilterItem;
import com.digiwin.athena.executionengine.model.trans.StepElement;
import com.digiwin.athena.executionengine.service.client.impl.ThemeMapClient;
import com.digiwin.athena.executionengine.trans.Step;
import com.digiwin.athena.executionengine.trans.TransAbstractStep;
import com.digiwin.athena.executionengine.trans.operator.OperatorFactory;
import com.digiwin.athena.executionengine.util.DateUtils;
import com.digiwin.athena.executionengine.util.LogUtils;
import com.google.common.collect.Lists;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.*;


/**
 * @description: 处理过滤组件
 * @author: xijj
 * @date: 2020/7/21 18:04
 */
@Component("filter")
public class FilterStep extends TransAbstractStep {

    private static final Logger LOGGER = LoggerFactory.getLogger(FilterStep.class);

    @Resource
    private ThemeMapClient themeMapClient;

    @Override
    public boolean defineCheck(StepElement stepElement) {
        return true;
    }

    /**
     * @return 过滤组件返回值结构是：List<Map<String, Object>>
     * @throws Exception
     */
    @Override
    public DealResult doDealData(Step step) {
        DealResult dealResult = new DealResult();
        try {
            Object inputData = getCurrentData(step.getPrevStepNameList());
            if (!(inputData instanceof List)) {
                LOGGER.warn("过滤组件只能接受表状数据，非展平数据需要先执行展平节点！Trans执行 {} 失败，数据结构错误！",
                        step.getName());
                dealResult.setDataNullFail();
                return dealResult;
            }
            List<Map<String, Object>> dataList = (List<Map<String, Object>>) (inputData);

            // 数据源为空直接返回
            if (CollectionUtils.isEmpty(dataList)) {
                dealResult.setSuccess(Lists.newArrayList());
                return dealResult;
            }


            dealResult.setSuccess(doFilter(dataList, step.getStepElement()));
            return dealResult;
        } catch (Exception e) {
            LOGGER.error("执行filter组件过滤异常");
            throw e;
        }
    }

    /**
     * 重构后的filterStep
     *
     * @param dataList
     * @param stepElement
     * @return
     */
    private List<Map<String, Object>> doFilter(List<Map<String, Object>> dataList, StepElement stepElement) {
        List<FilterItem> itemList = stepElement.getRules();
        if (CollectionUtils.isEmpty(itemList)) {
            return dataList;
        }
        List<Map<String, Object>> matchData = Lists.newArrayList();
        List<Map<String, Object>> noMatchData = Lists.newArrayList();
        FilterItem outerFilterItem = itemList.get(0);
        dataList.forEach(data -> {
            boolean isMatch = dataFilter(data, itemList, outerFilterItem.getLogiType(), stepElement.getNullJudge());
            if (isMatch) {
                matchData.add(data);
            } else {
                noMatchData.add(data);
            }
        });
        LogUtils.buildAgileLog(LogConstant.AGILE_CODE_TRANS_FILTER_DATA, LogUtils.SUCCESS,
                String.format("过滤前数据条数:%s  过滤条件:%s", CollectionUtils.isEmpty(dataList) ? "null or empty" : dataList.size(), JSONObject.toJSON(itemList)),
                String.format("过滤后数据条数:%s  不符合记录条数:%s", CollectionUtils.isEmpty(matchData) ? "null or empty" : matchData.size(), CollectionUtils.isEmpty(noMatchData) ? "null or empty" : noMatchData.size())
                , "1.打开指标配置界面，找到相关指标的计算节点设置。\n" +
                        "2.检查过滤条件，确认其逻辑运算符、比较值等是否正确。\n" +
                        "3.调整过滤设置，保存并重新发版。\n" +
                        "4.若问题未解决，联系平台技术支持人员并提供指标信息。");
        return stepElement.getIsEject() ? noMatchData : matchData;
    }

    /**
     * 数据过滤
     *
     * @param data
     * @param itemList
     * @param logiType
     * @return
     */
    private boolean dataFilter(Map<String, Object> data, List<FilterItem> itemList, String logiType, String nullJudge) {
        boolean isMatch = true;
        for (FilterItem filterItem : itemList) {
            if (!TransConstant.LOGITYPE_SINGLE.equalsIgnoreCase(filterItem.getLogiType())) {
                isMatch = dataFilter(data, filterItem.getFilterItems(), filterItem.getLogiType(), nullJudge);
            } else {
                Object leftValue = getLeftValue(filterItem, data);
                Map.Entry<String, Object> rightValueEntry = getRightValue(filterItem, data);
                if (rightValueEntry.getKey().equals("0")) {
                    //未启用
                    isMatch = true;
                } else {
                    //其他情况
                    if (leftValue == null) {
                        isMatch = TransConstant.RETAIN.equalsIgnoreCase(nullJudge) ? true : false;
                    } else {
                        try {
                            isMatch = OperatorFactory.getOperatorFunction(filterItem.getOp()).apply(leftValue, rightValueEntry.getValue());
                        } catch (Exception e) {
                            LOGGER.error("dataFilter执行失败", e);
                            isMatch = false;
                            break;
                        }
                    }
                }

            }
            if (TransConstant.LOGITYPE_AND.equalsIgnoreCase(logiType) && !isMatch) {
                break;
            }
            if (TransConstant.LOGITYPE_OR.equalsIgnoreCase(logiType) && isMatch) {
                break;
            }
        }
        return isMatch;
    }

    /**
     * 获取比较符左边的值
     *
     * @param filterItem
     * @param data
     * @return
     */
    private Object getLeftValue(FilterItem filterItem, Map<String, Object> data) {
        Object leftValue = data.get(filterItem.getLeft());
        if (leftValue == null) {
            return null;
        }
        if (TransConstant.VALUE_TYPE_DATE.equalsIgnoreCase(filterItem.getLeftValueType())
                || TransConstant.VALUE_TYPE_DATE_TIME.equalsIgnoreCase(filterItem.getLeftValueType())) {
            //如果时日期格式，需要将日期格式转换成long
            return DateUtils.convertTimeToLong(String.valueOf(leftValue), filterItem.getLeftDateFormat());
        }
        return leftValue;
    }

    /**
     * 获取比较符号右边的值
     * 《1107版本》新增业务变量启用不启用开关 0-未启用 1-启用
     * 不启用的时候需要忽略这个条件查所有，及isMatch是true
     * 启用了一定有值，按照值过滤
     *
     * @param filterItem
     * @param data
     * @return 返回值的key就是启用状态，未启用业务变量 key=0 其他key=0
     */
    private Map.Entry<String, Object> getRightValue(FilterItem filterItem, Map<String, Object> data) {
        Object rightValue = null;
        String rightValueStatus = "1";
        if (TransConstant.SOURCE_TYPE_CONSTANT.equalsIgnoreCase(filterItem.getType())) {
            rightValue = filterItem.getRight();
        } else if (TransConstant.SOURCE_TYPE_COLUMN.equalsIgnoreCase(filterItem.getType())) {
            rightValue = data.get(filterItem.getRight());
        } else if (TransConstant.SOURCE_TYPE_MECHANISM.equalsIgnoreCase(filterItem.getType())) {
            rightValue = getTransDataManager().getMechanismData(filterItem.getRight());
        } else if (TransConstant.SOURCE_TYPE_COMMON_VAR.equalsIgnoreCase(filterItem.getType())) {
            //新增commonvar类型逻辑
            Map<String, Object> commonVarMap = getTransDataManager().getCommonVarCollection();
            if (commonVarMap == null || !commonVarMap.containsKey(filterItem.getRight())) {
                List<String> varList = Lists.newArrayList();
                varList.add(filterItem.getRight());
                JSONObject respData = themeMapClient.querySmartDataVariable(getTransDataManager().getTenantId(), varList, getTransDataManager().getToken());
                commonVarMap = commonVarMap == null
                        ? new HashMap<>() : commonVarMap;
                commonVarMap.put(filterItem.getRight(),
                        Optional.ofNullable(respData).map(item -> item.get(filterItem.getRight())).orElse(null));
                getTransDataManager().setCommonVarCollection(commonVarMap);
            }
            if (commonVarMap.get(filterItem.getRight()) != null) {
                JSONObject commonVar = (JSONObject) commonVarMap.get(filterItem.getRight());
                String status = commonVar.getString("status");
                if ("1".equals(status)) {
                    rightValue = ((JSONObject) commonVarMap.get(filterItem.getRight())).get("value");
                } else {
                    //业务变量停用了--目前只有设置未启用则为未启用，否则都为启用，或者是启用未设置
                    rightValueStatus = "0";
                }
            } else {
                rightValueStatus = "0";
            }
        } else if (TransConstant.SOURCE_TYPE_SYSTEM_VAR.equalsIgnoreCase(filterItem.getType())) {
            //系统业务变量
            switch (filterItem.getRight()) {
                case "s_runTime":
                    rightValue = DateUtils.getCurrentDateTime("yyyy-MM-dd HH:mm:ss");
                    filterItem.setRightDateFormat("yyyy-MM-dd HH:mm:ss");
                    break;
                case "s_runDate":
                    rightValue = DateUtils.getCurrentDateTime("yyyy-MM-dd");
                    filterItem.setRightDateFormat("yyyy-MM-dd");
                    break;
                case "s_userName":
                    rightValue = getTransDataManager().getLoginUserName();
                    break;
                case "s_userId":
                    rightValue = getTransDataManager().getUserId();
                    break;
            }
        } else {
            LOGGER.warn("dataFilter, 在逻辑判断的右侧，未预期的数据类型");
        }
        if (TransConstant.VALUE_TYPE_DATE.equalsIgnoreCase(filterItem.getRightValueType())
                || TransConstant.VALUE_TYPE_DATE_TIME.equalsIgnoreCase(filterItem.getLeftValueType())) {
            //如果时日期格式，需要将日期格式转换成long
            rightValue = DateUtils.convertTimeToLong(String.valueOf(rightValue), filterItem.getRightDateFormat());
        }
        if (TransConstant.VALUE_TYPE_COLLECTION.equalsIgnoreCase(filterItem.getRightValueType())) {
            //如果是collection类型，需要将字符串转换成jsonArray类型
            rightValue = JSONArray.parseArray(String.valueOf(rightValue));
        }
        return new AbstractMap.SimpleImmutableEntry<>(rightValueStatus, rightValue);
    }


}
