package com.digiwin.athena.executionengine.component.action;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.digiwin.athena.executionengine.component.domain.ActionParam;
import com.digiwin.athena.executionengine.constant.AgileDataErrorCodeConstant;
import com.digiwin.athena.executionengine.constant.FieldNameConstant;
import com.digiwin.athena.executionengine.constant.LogConstant;
import com.digiwin.athena.executionengine.core.aop.ActionMock;
import com.digiwin.athena.executionengine.core.aop.Debug;
import com.digiwin.athena.executionengine.core.container.ExecuteContext;
import com.digiwin.athena.executionengine.dto.ErrorLog;
import com.digiwin.athena.executionengine.enumtype.ErrorCodeEnum;
import com.digiwin.athena.executionengine.exception.BusinessException;
import com.digiwin.athena.executionengine.model.BasePage;
import com.digiwin.athena.executionengine.model.DataDescription;
import com.digiwin.athena.executionengine.service.facade.schema.ISchemaBuilder;
import com.digiwin.athena.executionengine.util.ContextUtils;
import com.digiwin.athena.executionengine.util.DateUtils;
import com.digiwin.athena.executionengine.util.JsonUtil;
import com.digiwin.athena.executionengine.util.LogUtils;
import com.digiwin.athena.smartdata.sdk.DatasourceProxyInitialize;
import com.digiwin.athena.smartdata.sdk.exception.CustomException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * @description: 使用模型方式通过路由访问业务中台/边缘数据中台
 * @author: ZhangJun
 * @create: 2024/3/5
 */
@Service("BMDGeneralAction")
public class BMDGeneralAction extends ActionBase {

    /**
     * 循环调用超过20次报警
     */
    private final static int MAX_CYCLES = 20;
    private static final Logger LOGGER = LoggerFactory.getLogger(BMDGeneralAction.class);

    @ActionMock
    @Debug
    @Override
    protected Object actionExecute(ExecuteContext context, Map<String, Object> reqMap, ActionParam actionParam) {
        Map<String, Object> querySchema = null;
        try {
            //多产品线处理
            String productLine;
            if (actionParam.getMetaObj().containsKey("productLine")) {
                productLine = actionParam.getMetaObj().getString("productLine");
            } else {
                productLine = "ERP";
            }
            DataDescription dataDescription = context.getInputParam().getDataDescription();
            if (!dataDescription.isDataset() && dataDescription.getProductLineRows() != null && !dataDescription.getProductLineRows().isEmpty()) {
                for (int i = 0; i < dataDescription.getProductLineRows().size(); i++) {
                    JSONObject rowJson = dataDescription.getProductLineRows().getJSONObject(i);
                    if (productLine.equals(rowJson.getString("productLine"))) {
                        dataDescription.setPermissions(rowJson.getJSONObject("row"));
                        break;
                    }
                }
            } else if (dataDescription.isDataset()) {
                JSONArray datasetPermissions = JSONArray.parseArray(JSONObject.toJSONString(dataDescription.getProductLineRows()));
                dataDescription.setDatasetPermissions(datasetPermissions);
            }


            if (!CollectionUtils.isEmpty(context.getInputParam().getParam())) {
                reqMap.putAll(context.getInputParam().getParam());
            }
            if (!CollectionUtils.isEmpty(context.getInputParam().getDataDescription().getDynamicSchema())) {
                querySchema = ContextUtils.getBean(ISchemaBuilder.class).buildQuerySchemaByNewStrategy(actionParam.getQuerySchema(),
                        reqMap, context.getInputParam().getDataDescription());
            } else {
                querySchema = ContextUtils.getBean(ISchemaBuilder.class).buildQuerySchema(actionParam.getQuerySchema(), reqMap,
                        actionParam.getActionId(), context.getInputParam().getDataDescription());
            }
            BasePage basePage = createBasePage(querySchema);
            querySchema.put("page", basePage);
            querySchema.put("productLine", productLine);
            List<Object> list = new ArrayList<>();
            String datasource = (String) context.getInputParam().getSysParam().get("dataSource");
            String userId = (String) context.getInputParam().getSysParam().get("userId");
            long startTime = System.currentTimeMillis();
            List<String> querySql = null;
            int cycles = 1;
            while (true) {
                Map<String, Object> bmd = DatasourceProxyInitialize.getData(datasource, querySchema, context.getTenantId(), context.getLocale(), context.getToken(), userId, context.getRouterKey(), context.getInputParam().getEocMaps(), context.getInputParam().getRoles(), String.valueOf(context.getMessageId()));
                if (bmd == null || bmd.isEmpty()) {
                    break;
                }
                Map<String, Object> detail = (Map<String, Object>) bmd.get("data");
                if (detail == null || detail.isEmpty() || !detail.containsKey("total") || !detail.containsKey("data")) {
                    break;
                }

                Integer total = "dcp".equals(datasource) ? Integer.parseInt((String) detail.get("total")) : (Integer) detail.get("total");
                if (total == 0) {
                    //createDefaultData(list, detail);
                    break;
                }
                querySql = (List<String>) detail.get("querySql");

                if (total >= BasePage.MAX_DATA_SIZE) {
                    context.setExecuteStatus(false);
                    context.setErrorLog(buildErrorLog(actionParam));
                    throw new BusinessException(ErrorCodeEnum.ACTION_PULLING_DATA_OVERFLOW);
                }

                list.addAll((List) detail.get("data"));

                if (list.size() >= total) {
                    break;
                }

                basePage.setPageNo(basePage.getPageNo() + 1);
                querySchema.put("page", basePage);
                cycles++;
            }
            if (context.isDebug() && !CollectionUtils.isEmpty(querySql)) {
                context.getQuerySqlMap().put(actionParam.getActionId(), querySql);
            }
            Integer pageSize = Optional.ofNullable(querySchema).map(map -> map.get("page"))
                    .map(page -> JsonUtil.getObject(JsonUtil.getJsonString(page), BasePage.class))
                    .orElse(BasePage.createDefaultPage())
                    .getPageSize();

            if (pageSize < BasePage.DEFAULT_PAGE_SIZE && cycles > MAX_CYCLES) {
                LogUtils.buildAgileLog(LogConstant.AGILE_CODE_EXECUTE_BMD_ACTION, AgileDataErrorCodeConstant.EXECUTE_BMD_TOO_MANY_QUERIES, "actionId:" + actionParam.getActionId() + ",action入参:" + JSONObject.toJSONString(reqMap) + "，查询方案:" + JSONObject.toJSONString(querySchema), "BMD调用边缘次数超过20次", "建议将pageSize设置为大于等于" + BasePage.DEFAULT_PAGE_SIZE + "的值");
            }

            long timeTaken = (System.currentTimeMillis() - startTime);
            LOGGER.info(String.format("actionId：%s，查询耗时:%sms，查询笔数:%s,查询方案:%s", actionParam.getActionId(), timeTaken, list.size(), JSONObject.toJSONString(querySchema)));
            LogUtils.buildAgileLog(LogConstant.AGILE_CODE_EXECUTE_BMD_ACTION, LogUtils.SUCCESS, "actionId:" + actionParam.getActionId() + ",action入参:" + JSONObject.toJSONString(reqMap) + "，查询方案:" + JSONObject.toJSONString(querySchema), "数据笔数：" + list.size(), "");
            context.setExecuteStatus(true);
            return list;
        } catch (Exception e) {
            if (context.getErrorLog() == null && e instanceof CustomException) {
                ErrorLog errorLog = getErrorLog(actionParam, (CustomException) e);
                context.setErrorLog(errorLog);
            }
            String message = e.getMessage();
            if (e instanceof NullPointerException) {
                message = "NullPointerException";
            } else if (e instanceof CustomException) {
                message = ((CustomException) e).getErrorMessage();
            }
            LogUtils.buildAgileLog(LogConstant.AGILE_CODE_EXECUTE_BMD_ACTION, AgileDataErrorCodeConstant.EXECUTE_BMD_ACTION_ERROR, "actionId:" + actionParam.getActionId() + ",action入参:" + JSONObject.toJSONString(reqMap) + "，查询方案:" + JSONObject.toJSONString(querySchema), "代码报错，失败原因:【" + message + "】", "请联系系统管理员");

            context.setExecuteStatus(false);
            LOGGER.error("调用BMD获取数据异常,查询方案:{}", JSONObject.toJSONString(querySchema), e);
            throw e;
        }
    }

    private ErrorLog buildErrorLog(ActionParam actionParam) {
        ErrorLog errorLog = new ErrorLog();
        errorLog.setErrorCode(ErrorCodeEnum.ACTION_PULLING_DATA_OVERFLOW.getCode());
        errorLog.setErrorLocation("数据拉取组件，actionId:" + actionParam.getActionId());
        errorLog.setErrorDescription("单个查询方案数据量超出限制");
        errorLog.setErrorTimestamp(DateUtils.getCurrentDateTime());
        errorLog.setErrorMessage(ErrorCodeEnum.ACTION_PULLING_DATA_OVERFLOW.getMessage());
        errorLog.setPossibleCausesAndGuidance("数据量超出限制，请尝试添加更多的筛选条件进行查询,节点id:" + actionParam.getActionId());
        return errorLog;
    }

    private static ErrorLog getErrorLog(ActionParam actionParam, CustomException customException) {
        //没有异常，调用边缘报错，构造新异常
        ErrorLog errorLog = new ErrorLog();
        errorLog.setErrorCode(customException.getErrorCode());
        errorLog.setErrorLocation("数据拉取组件，actionId:" + actionParam.getActionId());
        errorLog.setErrorDescription(customException.getErrorMessage());
        errorLog.setErrorTimestamp(DateUtils.getCurrentDateTime());
        errorLog.setErrorMessage(customException.getErrorMessage());
        errorLog.setPossibleCausesAndGuidance("请检查数据拉取节点配置是否正常,节点id:" + actionParam.getActionId());
        return errorLog;
    }

    private Map<String, Object> generatePocSchema(JSONObject querySchema, Object obj) {
        JSONObject dynamicSchema = (JSONObject) obj;
        //维度
        if (!CollectionUtils.isEmpty(dynamicSchema.getJSONArray(FieldNameConstant.DYNAMIC_SCHEMA_DIMENSIONS))) {
            JSONArray dynamicDimensions = dynamicSchema.getJSONArray(FieldNameConstant.DYNAMIC_SCHEMA_DIMENSIONS);
            Map<String, String> map = dynamicDimensions.stream().collect(Collectors.toMap(item -> ((JSONObject) item).getString("field"), item -> ((JSONObject) item).getString("field")));

            querySchema.getJSONArray("dimensions").stream().filter(o -> !map.containsKey(((JSONObject) o).getString("field"))).forEach(o -> ((JSONObject) o).put("defaultValue", "auto"));
        }

        //度量
        if (!CollectionUtils.isEmpty(dynamicSchema.getJSONArray(FieldNameConstant.DYNAMIC_SCHEMA_MEASURES))) {
            querySchema.fluentRemove(FieldNameConstant.DYNAMIC_SCHEMA_MEASURES).put(FieldNameConstant.DYNAMIC_SCHEMA_MEASURES, dynamicSchema.getJSONArray(FieldNameConstant.DYNAMIC_SCHEMA_MEASURES));
        }
        //过滤
        if (!CollectionUtils.isEmpty(dynamicSchema.getJSONArray(FieldNameConstant.DYNAMIC_SCHEMA_FILTER))) {
            JSONObject filter = new JSONObject();
            JSONArray children = new JSONArray();
            filter.put("logical", "and");
            filter.put("children", children);
            children.addAll(dynamicSchema.getJSONArray(FieldNameConstant.DYNAMIC_SCHEMA_FILTER));
            children.add(querySchema.getJSONObject("filter"));
            querySchema.put(FieldNameConstant.DYNAMIC_SCHEMA_FILTER, filter);
        }

        return querySchema;
    }

    private BasePage createBasePage(Map<String, Object> querySchema) {
        BasePage pageObj = Optional.ofNullable(querySchema).map(map -> map.get("page")).map(page -> JsonUtil.getObject(JsonUtil.getJsonString(page), BasePage.class)).orElse(BasePage.createDefaultPage());
        /*if (pageObj.getPageSize() < BasePage.DEFAULT_PAGE_SIZE) {
            pageObj.setPageSize(BasePage.DEFAULT_PAGE_SIZE);
        }*/
        return pageObj;
    }


    private void createDefaultData(List<Object> list, Map<String, Object> detail) {
        if (!detail.containsKey("meta")) {
            return;
        }
        List meta = (List) detail.get("meta");
        Map<String, Object> defaultData = new HashMap<>();
        for (Object metaObj : meta) {
            Map<String, Object> entry = (Map<String, Object>) metaObj;

            switch ((Integer) entry.get("columnDataType")) {
                case 0:
                    defaultData.put((String) entry.get("columnName"), "");
                    break;
                case 1:
                case 7:
                    defaultData.put((String) entry.get("columnName"), new Date());
                    break;
                case 2:
                    defaultData.put((String) entry.get("columnName"), 0);
                    break;
                case 3:
                    defaultData.put((String) entry.get("columnName"), 0.0f);
                    break;
                case 4:
                    defaultData.put((String) entry.get("columnName"), false);
                    break;
                default:
                    break;
            }
        }
        list.add(defaultData);
    }


}
