package com.digiwin.athena.executionengine.component.action;

import com.alibaba.fastjson.JSONObject;
import com.digiwin.app.service.DWServiceContext;
import com.digiwin.athena.executionengine.component.domain.ActionParam;
import com.digiwin.athena.executionengine.constant.FieldNameConstant;
import com.digiwin.athena.executionengine.constant.LogConstant;
import com.digiwin.athena.executionengine.constant.MetaDataConstant;
import com.digiwin.athena.executionengine.constant.ModuleProperty;
import com.digiwin.athena.executionengine.core.container.ExecuteContext;
import com.digiwin.athena.executionengine.core.container.ExecuteErrorMessage;
import com.digiwin.athena.executionengine.dto.LogDataDto;
import com.digiwin.athena.executionengine.dto.LogDto;
import com.digiwin.athena.executionengine.enumtype.ActionTypeEnum;
import com.digiwin.athena.executionengine.util.HttpClientUtils;
import com.digiwin.athena.executionengine.util.MqSendUtils;
import com.digiwin.athena.executionengine.util.JsonResolverUtils;
import com.digiwin.athena.executionengine.util.ServiceResponseUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

/**
 * @description:
 * @author: ZhangJun
 * @date: 2020/7/7 9:30
 */
@Service("ServiceComposerAction")
public class ServiceComposerAction extends ActionBase {
    private static final Logger LOGGER = LoggerFactory.getLogger(ServiceComposerAction.class);

    private static final String START_SERVICE_COMPOSER = "startServiceComposer";

    @Override
    protected Object actionExecute(ExecuteContext context, Map<String, Object> reqMap, ActionParam actionParam) {
        JSONObject metaObj = actionParam.getMetaObj();
        String serviceComposerId = metaObj.getString(MetaDataConstant.SERVICE_COMPOSER_ID);
        boolean asyncComplete = metaObj.getBoolean(MetaDataConstant.ASYNC_COMPLETE);
        /**
         * 20230206临时解决因同步发起服务编排到导致侦测阻塞问题
         * 侦测--》执行--》服务编排
         * 从侦测过来的会在header上带一个monitorProcessType:startServiceComposer
         * 此时执行引擎默认以异步发起服务编排
         */
        if (!asyncComplete) {
            LOGGER.info("header中monitorProcessType:{}", context.getMonitorProcessType());

            asyncComplete = START_SERVICE_COMPOSER.equalsIgnoreCase(context.getMonitorProcessType());
        }
        LOGGER.info("上下文取到的locale:{}", context.getLocale());

        JSONObject actionJson = actionParam.getActionJson();
        String locale = actionJson.getString(MetaDataConstant.ACTION_LOCALE);
        locale = StringUtils.isEmpty(context.getLocale()) ? locale : context.getLocale();
        LOGGER.info("发起服务编排最终使用的locale:{}", locale);
        JSONObject respObj = null;
        try {
            Map<String, String> headerMap = new HashMap<>();
            headerMap.put("token", context.getToken());
            headerMap.put("Content-Type", "application/json;charset=UTF-8");
            headerMap.put(FieldNameConstant.CAMEL_CASE_ROUTER_KEY, context.getRouterKey());
            headerMap.put(MetaDataConstant.ACTION_LOCALE, locale);
            headerMap.put("invokerId", ModuleProperty.CURRENT_MODULE_NAME);

            Map<String, Object> param = new HashMap<>();
            param.put("serviceComposerId", serviceComposerId);
            param.put("asyncComplete", asyncComplete);
            param.put("tenantId", context.getTenantId());
            param.put("eocMap", context.getEocMap());
            param.put("params", reqMap);

            String jsonParam = JsonResolverUtils.toJsonString(param);
            String httpRespContent = HttpClientUtils.doPost(ModuleProperty.SERVICE_COMPOSER_URL, headerMap, jsonParam);
            if (httpRespContent == null) {
                context.setExecuteStatus(false);
                LOGGER.error("执行引擎调用服务编排执行{}:{}返回的结果为：null", actionParam.getActionId(), jsonParam);
                return null;
            }
            LOGGER.info("执行引擎调用服务编排执行{}:{}返回的结果为：{}", actionParam.getActionId(), jsonParam, httpRespContent);
            respObj = ServiceResponseUtils.getServiceComposerResp(httpRespContent);
            if (respObj == null) {
                context.setExecuteStatus(false);
            } else {
                context.setExecuteStatus(true);
            }
            logTrace(respObj, true);
            MqSendUtils.sendSuccessMessage(context, "ServiceComposer", param);
            return respObj;
        } catch (Exception e) {
            logTrace(respObj, false);
            LOGGER.error("执行引擎调用服务编排异常", e);
            context.setExecuteStatus(false);
            ExecuteErrorMessage executeErrorMessage = new ExecuteErrorMessage();
            executeErrorMessage.setErrorMsg(e.getMessage());
            executeErrorMessage.setErrorCode("P.EE.600.0001");
            executeErrorMessage.setActionTypeEnum(ActionTypeEnum.SERVICE_COMPOSER);
            context.setExecuteErrorMessage(executeErrorMessage);
            MqSendUtils.sendExceptionMessage(context, e.getMessage());
            throw e;
        }
    }


    /**
     * 埋点 调用服务编排
     *
     * @param respObj
     */
    private void logTrace(JSONObject respObj, boolean success) {

        String wid = Optional.ofNullable(respObj).map(item -> respObj.getString("wid")).orElse(null);
        String rid = Optional.ofNullable(respObj).map(item -> respObj.getString("rid")).orElse(null);
        StringBuilder value = new StringBuilder();
        value.append("wId=").append(wid).append("&&rId=").append(rid);
        LogDataDto logDataDto = new LogDataDto(value.toString(), "服务编排wid和rid", LogConstant.TYPE_LINK, LogConstant.LABEL_SERVICE_COMPOSER);
        String message = success ? "调用服务编排成功" : "调用服务编排失败";
        LogDto logDto = new LogDto(message, Arrays.asList(logDataDto));
        if (success) {
            LOGGER.info(logDto.toString());
        } else {
            LOGGER.error(logDto.toString());
        }
    }


}
