package com.digiwin.athena.executionengine.service.facade.analyzer.impl;

import com.alibaba.fastjson.JSONObject;
import com.digiwin.athena.executionengine.constant.CommonConstant;
import com.digiwin.athena.executionengine.constant.FieldNameConstant;
import com.digiwin.athena.executionengine.enumtype.ValueTypeEnum;
import com.digiwin.athena.executionengine.service.facade.mapping.data.DataMappingHandler;
import com.digiwin.athena.executionengine.model.ParamElement;
import com.digiwin.athena.executionengine.model.input.InputParamModule;
import com.digiwin.athena.executionengine.service.facade.analyzer.AbstractAnalyzerBase;
import com.digiwin.athena.executionengine.util.AnalysisUtils;
import com.google.common.collect.Maps;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * @description: 执行入参分析
 * @author: renwm
 * @date: 2020/6/12 16:41
 */
@Service("engineInputParamAnalyzer")
public class InputParamAnalyzer extends AbstractAnalyzerBase {

    private static final Logger LOGGER = LoggerFactory.getLogger(InputParamAnalyzer.class);

    /**
     * 入参结构分析
     *
     * @param executionRule 实现抽象方法，抽象方法中该参数指的是执行规则，
     *                      在分析入参结构时是不需要且此时还没有求出执行规则。
     *                      所以不需要传入
     */
    @Override
    protected void analysisHandler(JSONObject executionRule) {

        InputParamModule inputParam = getLocalContext().getInputParam();

        if (MapUtils.isEmpty(inputParam.getInputParam())) {
            LOGGER.warn("执行引擎执行没有需要分析的执行入参");
            return;
        }

        //分析param或者paras入参
        analysisInputParam("", CommonConstant.JSON_PATH_PREFIX, inputParam.getInputParam(), CommonConstant.EXECUTION_PARAM);
        inputParam.getInputParam().clear();

        //分析sysParam入参
        analysisInputParam("sysParam", CommonConstant.JSON_PATH_PREFIX, inputParam.getSysParam(), CommonConstant.EXECUTION_SYSPARAM);
    }

    /**
     * 分析执行入参，解析成对应的paramElement对象
     *
     * @param path
     * @param jsonPath
     * @param inputMap
     */
    private void analysisInputParam(String path, String jsonPath, Map<String, Object> inputMap, String inputParamPart) {
        if (MapUtils.isEmpty(inputMap)) {
            return;
        }
        //如果是首次进入则默认映射到的是根节点
        //如 path : param
        //   path -> $
        if (FieldNameConstant.isValidInputParamPath(path)) {
            path = "$";
        }
        for (Map.Entry<String, Object> entry : inputMap.entrySet()) {

            ParamElement paramElement = new ParamElement();
            paramElement.setInputParamPart(inputParamPart);
            paramElement.setParamName(entry.getKey());
            paramElement.setValueType(ValueTypeEnum.STRING);
            paramElement.setParamPath(entry.getKey());
            paramElement.setArray(false);

            //添加参数的路径，供后面接卸params中的 name 和value的时候  匹配（参数actionId.参数路径作为参数的唯一标识）
            if (StringUtils.isNotBlank(path)) {
                paramElement.setParamPath(AnalysisUtils.concatByDot(path, entry.getKey()));
            }
            //添加jsonPath路径 供后面获取参数值的时候 根据jsonPath解析
            paramElement.setElementJsonPath(AnalysisUtils.concatByDot(jsonPath, entry.getKey()));

            getLocalContext().addParamElement(paramElement.getParamPath(), paramElement);

            // 判断参数类型是否是数组类型，如果是数组类型 需要解析 数组的结构
            if (entry.getValue() instanceof List) {
                //判断数组中的参数类型是否对象 如果是对象 需要解析map对象的结构
                List valList = (List) entry.getValue();

                if (CollectionUtils.isNotEmpty(valList) && valList.get(0) instanceof Map) {

                    for (Object object : valList) {
                        paramElement.setElementJsonPath(AnalysisUtils.concatByDot(jsonPath, entry.getKey()));

                        analysisInputParam(paramElement.getParamPath(),
                                paramElement.getElementJsonPath() + CommonConstant.JSON_PATH_ARRAY_SUFFIX,
                                (Map) object, inputParamPart);
                    }
                } else {
                    if (getLocalContext().isNotExistParamElement(paramElement.getParamPath())) {
                        getLocalContext().addInputParamKey(paramElement.getParamPath());
                    }
                }
                //如果参数 是数组且不是对象（参数没有子元素了），将参数添加到上下文参数集合中
                paramElement.setValueType(ValueTypeEnum.ARRAY);
                paramElement.setArray(true);
            }
            // 判断参数是否是对象 是对象解析结构
            else if (entry.getValue() instanceof Map) {
                paramElement.setValueType(ValueTypeEnum.OBJECT);
                analysisInputParam(paramElement.getParamPath(),
                        paramElement.getElementJsonPath(),
                        (Map) entry.getValue(), inputParamPart);
            } else {

                if (getLocalContext().isNotExistParamElement(paramElement.getParamPath())) {
                    getLocalContext().addInputParamKey(paramElement.getParamPath(), inputParamPart);
                }
            }
        }
    }

    /**
     * 执行引擎多路径入参分析，得到完整的key以及缺失的key
     * todo 需要thememap支持多路径后，才可将该实现对接到现有流程中
     */
    private void inputKeyAnalyze() {
        Set<String> shortageKeySet = new HashSet<>();
        DataMappingHandler dataMappingHandler = getLocalContext().getDataMappingManager().getDatMappingHandler(getLocalContext().getInputParam().getActionId()
                , CommonConstant.EXECUTION_PARAM);
        //获取所有叶子节点的paramPath
        Set<String> allLeafParamPathSet = dataMappingHandler.getLeafParamPath();
        for (String paramPath : allLeafParamPathSet) {
            if (!shortageKeySet.contains(paramPath)) {
                calculateShortageKeys(paramPath, shortageKeySet, dataMappingHandler);
            }
        }
        //完整的key与缺失的key的差集就是不缺失的key
        allLeafParamPathSet.removeAll(shortageKeySet);
    }

    /**
     * 寻找缺失的key
     *
     * @param paramPath          节点的paramPath
     * @param shortageKeySet     缺失key的set集合
     * @param dataMappingHandler
     */
    private void calculateShortageKeys(String paramPath, Set<String> shortageKeySet, DataMappingHandler dataMappingHandler) {
        //父节点paramPath
        String lastCollectionParamPath = dataMappingHandler.getLastCollectionNodeParamPath(paramPath);
        if (lastCollectionParamPath == null) {
            //如果找不到上一个集合节点，比如param入参，顶层就是一个复杂类型的结构。此时无法衡量出key缺失不缺失，直接返回
            return;
        }
        //1 得到分组统计后的cnt groupingZone结构为{"$[0].col1[0]":1,$[0].col1[1]":2}
        Map<String, ? extends Number> groupingZone = getGroupingZone(paramPath, dataMappingHandler);

        //2 如果parentNode是root节点，则拿root节点的totalCount和groupZone的size比。如果不是root节点，则和parentNode的zone比较，判断zone是否完全相同（key，以及 value）
        boolean matchResult;
        boolean isTop = dataMappingHandler.isRootNode(lastCollectionParamPath);
        if (isTop) {
            matchResult = dataMappingHandler.getTotalCountByParamPath(lastCollectionParamPath) == groupingZone.keySet().size();
        } else {
            matchResult = matchWithParentCollectionZone(groupingZone, dataMappingHandler.getZoneByParamPath(lastCollectionParamPath));
        }
        //3 如果node为叶子节点进行了2中的比较，比较结果如果相同，需要找到上层集合节点向上递归判断，如果不相同需要将当前节点的parampath塞入shortageKeySet,如果node为非叶子节点，需要将node下的所有叶子节点放入shortageKeySet
        if (!matchResult) {
            //匹配不上 加入shortageKeySet
            addChildLeafNodeToShortage(paramPath, shortageKeySet, dataMappingHandler);
            return;
        }
        if (!isTop) {
            //匹配ok 且没到root节点 继续往上递归
            calculateShortageKeys(lastCollectionParamPath, shortageKeySet, dataMappingHandler);
        }
    }

    /**
     * path加入shortageKeySet
     *
     * @param paramPath          节点的paramPath
     * @param shortageKeySet     缺失key的set集合
     * @param dataMappingHandler
     */
    private void addChildLeafNodeToShortage(String paramPath, Set<String> shortageKeySet, DataMappingHandler dataMappingHandler) {
        //叶子节点直接放入shorageKeySet
        if (dataMappingHandler.isLeafNode(paramPath)) {
            shortageKeySet.add(paramPath);
            return;
        }
        //如果为集合节点缺失，需要将当前集合节点下的所有叶子节点加入shortageKeySet
        List<String> childrenParamPathList = dataMappingHandler.getChildrenParamPath(paramPath);
        for (String path : childrenParamPathList) {
            //循环递归 child 直到找出所有叶子节点为止
            addChildLeafNodeToShortage(path, shortageKeySet, dataMappingHandler);
        }
    }

    /**
     * 分组后的zone和parentNode zone进行比较
     *
     * @param groupingZone   分组统计后的zone
     * @param parentNodeZone 父节点zone
     * @return
     */
    private boolean matchWithParentCollectionZone(Map<String, ? extends Number> groupingZone, Map<String, Integer> parentNodeZone) {
        for (Map.Entry<String, Integer> entry : parentNodeZone.entrySet()) {
            String key = entry.getKey();
            if (!groupingZone.containsKey(key)) {
                return false;
            }
            int parentNodeZoneValue = entry.getValue();
            int groupZoneValue = groupingZone.get(key).intValue();
            if (parentNodeZoneValue != groupZoneValue) {
                return false;
            }
        }
        return true;
    }

    /**
     * zone的key 进行分组统计
     *
     * @param paramPath
     * @param dataMappingHandler
     * @return
     */
    private Map<String, ? extends Number> getGroupingZone(String paramPath, DataMappingHandler dataMappingHandler) {
        String parentNodeParamPath = dataMappingHandler.getParentParamPath(paramPath);
        //如果parentNode是root节点，那么zone是没有$[0].A[0]...这种结构，挂在root下的dataNode zone为$[0],$[1]，直接返回zone信息和rootNode的totalCount比较
        if (dataMappingHandler.isRootNode(parentNodeParamPath)) {
            return dataMappingHandler.getZoneByParamPath(paramPath);
        }
        //获取处理过的zone信息
        Map<String, Integer> zone = getProcessedZone(paramPath, dataMappingHandler);

        List<String> subKeyList = new ArrayList<>();
        for (String key : zone.keySet()) {
            subKeyList.add(key.substring(0, key.lastIndexOf(".")));
        }
        Map<String, Long> collect = subKeyList.stream().collect(Collectors.groupingBy(item -> item, Collectors.counting()));
        return MapUtils.isEmpty(collect) ? Maps.newHashMap() : collect;
    }

    /**
     * 获取处理过的zone信息
     *
     * @param paramPath
     * @param dataMappingHandler
     * @return
     */
    private Map<String, Integer> getProcessedZone(String paramPath, DataMappingHandler dataMappingHandler) {
        String parentNodeParamPath = dataMappingHandler.getParentParamPath(paramPath);
        Map<String, Integer> zone = dataMappingHandler.getZoneByParamPath(paramPath);
        if (dataMappingHandler.isArray(parentNodeParamPath)) {
            //如果parentNode是集合
            return zone;
        } else {
            Map<String, Integer> newZone = new HashMap<>();
            String lastCollectionParamPath = dataMappingHandler.getLastCollectionNodeParamPath(paramPath);
            int idx = StringUtils.countMatches(lastCollectionParamPath, ".");
            //处理zone信息
            for (Map.Entry<String, Integer> entry : zone.entrySet()) {
                String key = entry.getKey();
                int pos = StringUtils.ordinalIndexOf(key, ".", idx + 1);
                newZone.put(key.substring(0, pos), entry.getValue());
            }
            return newZone;

        }
    }
}
