package com.digiwin.athena.executionengine.trans.components;

import com.digiwin.athena.executionengine.constant.TransConstant;
import com.digiwin.athena.executionengine.enumtype.ErrorCodeEnum;
import com.digiwin.athena.executionengine.model.trans.AddColumnElement;
import com.digiwin.athena.executionengine.model.trans.DealResult;
import com.digiwin.athena.executionengine.model.trans.StepElement;
import com.digiwin.athena.executionengine.trans.Step;
import com.digiwin.athena.executionengine.trans.TransAbstractStep;
import com.digiwin.athena.executionengine.util.ExceptionUtils;
import com.google.common.collect.Lists;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;


/**
 * @description: 添加新列組件
 * @author: ZhangJun
 * @create: 2021/4/7
 */
@Component("addColumn")
public class AddColumnStep extends TransAbstractStep {
    private static final Logger LOGGER = LoggerFactory.getLogger(AddColumnStep.class);

    @Override
    public DealResult doDealData(Step step) {
        DealResult dealResult = new DealResult();
        //输入数据
        Object inputData = getCurrentData(step.getPrevStepNameList());
        if (!(inputData instanceof List)) {
            LOGGER.error("addColumnstep组件只能处理表状数据，非表状数据需要先执行展平节点！Trans执行 {} 失败，数据结构错误！", step.getName());
            dealResult.setDataNullFail();
            return dealResult;
        }
        //检查currentData数据和addColumnStep中的定义的newField是否存在相同
        if (!ruleKeyExistsInCurrentData(step)) {
            LOGGER.error("规则定义中配置的newField，在源数据中存在，请检查stepName:{}", step.getName());
            dealResult.setDataNullFail();
            return dealResult;
        }

        List<Map<String, Object>> dataList = (List<Map<String, Object>>) (inputData);
        // 数据源为空直接返回
        if (CollectionUtils.isEmpty(dataList)) {
            dealResult.setSuccess(Lists.newArrayList());
            return dealResult;
        }
        Map<String, Object> columnMap = getColumnMap(step.getStepElement());
        dataList.forEach(item -> item.putAll(columnMap));
        dealResult.setSuccess(dataList);
        return dealResult;
    }

    /**
     * 根据transrule 获取需要新增列的值
     *
     * @param stepElement
     * @return
     */
    private Map<String, Object> getColumnMap(StepElement stepElement) {
        List<AddColumnElement> rules = stepElement.getRules();
        Map<String, Object> columnMap = new HashMap<>();
        rules.forEach(addColumnElement -> {
            Object value = getColumnValue(addColumnElement, stepElement.getTechnique());
            columnMap.put(addColumnElement.getNewField(), value);
        });
        return columnMap;
    }

    /**
     * 获取column的值
     *
     * @param addColumnElement
     * @param technique
     * @return
     */
    private Object getColumnValue(AddColumnElement addColumnElement, String technique) {
        String type = addColumnElement.getType();
        String valueType = addColumnElement.getValueType();
        Object value = null;
        if (TransConstant.SOURCE_TYPE_CONSTANT.equalsIgnoreCase(type)) {
            value = addColumnElement.getValue();
        } else if (TransConstant.SOURCE_TYPE_MECHANISM.equalsIgnoreCase(type)) {
            value = getTransDataManager().getMechanismData(addColumnElement.getValue());
        }
        if (value == null) {
            LOGGER.error("addColumn组件获取列值失败！");
            throw ExceptionUtils.buildVerificationException(ErrorCodeEnum.TRANS_RULE_DEFINE_ERROR, technique);
        }
        if (TransConstant.VALUE_TYPE_STRING.equalsIgnoreCase(valueType)) {
            return String.valueOf(value);
        }
        try {
            return Integer.parseInt(String.valueOf(value));
        } catch (NumberFormatException e) {
            LOGGER.error("value无法转换成integer！value:{},type:{}", value, type);
            throw e;
        }
    }

    @Override
    public boolean defineCheck(StepElement stepElement) {
        List<AddColumnElement> rules = stepElement.getRules();
        if (CollectionUtils.isEmpty(rules)) {
            LOGGER.error("rules定义为空，请检查stepName:{}", stepElement.getName());
            return false;
        }
        boolean ruleCheck = rules.stream().anyMatch(this::addColumnElementCheck);
        if (!ruleCheck) {
            return ruleCheck;
        }
        return ruleKeyDefinedCheck(stepElement);
    }

    /**
     * 规则定义时候，内部的newField是否 重复定义
     *
     * @return
     */
    private boolean ruleKeyDefinedCheck(StepElement stepElement) {
        List<AddColumnElement> rules = stepElement.getRules();
        Set<String> newFieldSet = new HashSet<>();
        for (AddColumnElement addColumnElement : rules) {
            newFieldSet.add(addColumnElement.getNewField());
        }
        //检查定义中的newField是否存在重复
        if (newFieldSet.size() != rules.size()) {
            LOGGER.error("rules中的newField存在重复的配置，请检查stepName:{}", stepElement.getName());
            return false;
        }
        return true;
    }

    /**
     * 规则定义时候，newField是否和currentData中的key存在相同
     *
     * @param step
     * @return
     */
    private boolean ruleKeyExistsInCurrentData(Step step) {
        List<AddColumnElement> rules = step.getStepElement().getRules();
        Set<String> newFieldSet = new HashSet<>();
        for (AddColumnElement addColumnElement : rules) {
            newFieldSet.add(addColumnElement.getNewField());
        }
        //检查定义的newField和源数据中是否存在相同的
        Set<String> inputDataKeySet = getInputDataKeys(step);
        if (!compare2Set(inputDataKeySet, newFieldSet)) {
            return false;
        }
        return true;
    }

    /**
     * set内容比较
     *
     * @param inputDataKeySet
     * @param newFieldSet
     * @return
     */
    public boolean compare2Set(Set<String> inputDataKeySet, Set<String> newFieldSet) {
        for (String inputKey : inputDataKeySet) {
            if (newFieldSet.contains(inputKey)) {
                return false;
            }
        }
        return true;
    }

    /**
     * 获取表状数据的key
     *
     * @param step
     * @return
     */
    private Set<String> getInputDataKeys(Step step) {
        //此时getCurrentData一定是表状数据，如果不满足在doDealData方法开始执行的时候检查出来
        List<Map<String, Object>> dataList = (List<Map<String, Object>>) (getCurrentData(step.getPrevStepNameList()));
        Set<String> inputDataKeys = new HashSet<>();
        dataList.forEach(item -> inputDataKeys.addAll(item.keySet()));
        return inputDataKeys;
    }

    /**
     * 检查rule
     *
     * @param rule
     * @return
     */
    private boolean addColumnElementCheck(AddColumnElement rule) {
        if (StringUtils.isEmpty(rule.getNewField()) || StringUtils.isEmpty(rule.getValueType())
                || StringUtils.isEmpty(rule.getType()) || rule.getValue() == null) {
            return false;
        }
        if (!TransConstant.SOURCE_TYPE_CONSTANT.equalsIgnoreCase(rule.getType())
                && !TransConstant.SOURCE_TYPE_MECHANISM.equalsIgnoreCase(rule.getType())) {
            return false;
        }

        if (!TransConstant.VALUE_TYPE_NUMBER.equalsIgnoreCase(rule.getValueType())
                && !TransConstant.VALUE_TYPE_STRING.equalsIgnoreCase(rule.getValueType())) {
            return false;
        }
        return true;
    }
}
