package com.digiwin.athena.executionengine.trans.components;

import com.digiwin.athena.executionengine.model.trans.CollectCondition;
import com.digiwin.athena.executionengine.model.trans.DealResult;
import com.digiwin.athena.executionengine.model.trans.StepElement;
import com.digiwin.athena.executionengine.trans.Step;
import com.digiwin.athena.executionengine.trans.TransAbstractStep;
import com.digiwin.athena.executionengine.util.AnalysisUtils;
import com.google.common.collect.Lists;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.LoggerFactory;
import org.slf4j.Logger;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @description: 聚合组件
 * @author: xijj
 * @date: 2020/7/23 18:04
 */
@Component("polymerize")
public class CollectStep extends TransAbstractStep {

    private static final Logger LOGGER = LoggerFactory.getLogger(CollectStep.class);

    @Override
    public boolean defineCheck(StepElement stepElement) {
        return true;
    }

    @Override
    public DealResult doDealData(Step step) {
        DealResult dealResult = new DealResult();
        try {
            //获取数据，并校验
            Object inputData = getCurrentData(step.getPrevStepNameList());
            if (!(inputData instanceof List)) {
                LOGGER.warn("聚合组件只能接受表状数据，非展平数据需要先执行展平节点！Trans执行 {} 失败，数据结构错误！",
                        step.getName());
                dealResult.setDataNullFail();
                return dealResult;
            }
            List<Map<String, Object>> dataList = (List<Map<String, Object>>) (inputData);

            // 数据源为空直接返回
            if (CollectionUtils.isEmpty(dataList)) {
                dealResult.setSuccess(Lists.newArrayList());
                return dealResult;
            }
            // 没有过滤条件返回原数据
            StepElement stepElement = step.getStepElement();
            List<CollectCondition> collectConditionList = stepElement.getCollectConditions();
            if (CollectionUtils.isEmpty(collectConditionList)) {

                if (stepElement.isTruncateKey()) {
                    List<Map<String, Object>> res = new ArrayList<>();
                    for (Map<String, Object> record : dataList) {
                        Map<String, Object> newRecord = new HashMap<>();
                        for (String key : record.keySet()) {
                            newRecord.put(AnalysisUtils.truncateKey(key), record.get(key));
                        }
                        res.add(newRecord);
                    }
                    dealResult.setSuccess(res);
                } else {
                    dealResult.setSuccess(dataList);
                }
                return dealResult;
            }
            // 按照规则中的条件遍历进行归并处理
            for (CollectCondition condition : collectConditionList) {
                // 判断别名是否为空，如果别名为空，不需要分组
                if (StringUtils.isBlank(condition.getNewField())) {
                    continue;
                }
                List<String> filedList = condition.getFileds();
                // 按照给定的field分组
                Map<String, List<Map<String, Object>>> groupMap = dataList.stream().collect(Collectors.groupingBy(map -> {
                    StringBuilder builder = new StringBuilder();
                    filedList.forEach(item -> builder.append(map.get(item)));
                    return builder.toString();
                }));
                // 存放每次归并处理的结果集
                List<Map<String, Object>> rootList = new ArrayList<>();
                // 处理分组后的Map
                for (Map.Entry<String, List<Map<String, Object>>> entry : groupMap.entrySet()) {
                    Map<String, Object> rowMap = new HashMap<>();
                    List<Object> rowElementList = new ArrayList<>();
                    // 遍历分组后的每条记录
                    for (Map<String, Object> data : entry.getValue()) {
                        // key为需要分组的key直接放到新的记录 否则都添加到新的节点中
                        Map<String, Object> elementMap = new HashMap<>();
                        for (Map.Entry<String, Object> entryData : data.entrySet()) {
                            // 用来归并剩下字段值的集合
                            if (filedList.contains(entryData.getKey())) {
                                String key = stepElement.isTruncateKey() ? AnalysisUtils.truncateKey(entryData.getKey()) : entryData.getKey();
                                rowMap.put(key, entryData.getValue());
                            } else {
                                // 值类型
                                if ("value".equals(condition.getType())) {
                                    rowElementList.add(entryData.getValue());
                                } else {
                                    // (对象类型)剩余的添加到数组后 放到rowMap中
                                    String key = stepElement.isTruncateKey() ? AnalysisUtils.truncateKey(entryData.getKey()) : entryData.getKey();
                                    elementMap.put(key, entryData.getValue());
                                }
                            }
                        }
                        rowElementList.add(elementMap);
                    }
                    // 根据给定的别名 作为list节点的key
                    if (CollectionUtils.isNotEmpty(rowElementList)) {
                        rowMap.put(condition.getNewField(), rowElementList);
                    }
                    rootList.add(rowMap);
                }
                // 作为下一轮分组的数据源
                dataList = rootList;
            }

            dealResult.setSuccess(dataList);
            return dealResult;
        } catch (Exception e) {
            LOGGER.error("执行聚合组件出现异常");
            throw e;
        }
    }
}
