package com.digiwin.athena.executionengine.trans.util;

import com.alibaba.fastjson.JSONObject;
import com.digiwin.athena.executionengine.constant.TransConstant;
import com.digiwin.athena.executionengine.model.trans.*;
import com.digiwin.athena.executionengine.model.trans.components.CalculateRuleDto;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.List;

/**
 * @description: trans定义JSON的分析工具类
 * @author: fenglei
 * @date: 2020-12-3
 */
public class TransAnalysisUtils {

    private static final Logger LOGGER = LoggerFactory.getLogger(TransAnalysisUtils.class);

    /**
     * 取trans定义的版本号
     *
     * @param jsonObject
     * @return
     */
    public static String getVersion(JSONObject jsonObject) {
        try {
            String version = jsonObject.getString("version");
            if (version == null) {
                //2之前的版本是没有version字段的，在2之前的版本号默认为1
                return "1";
            }
            return version;
        } catch (Exception e) {
            LOGGER.error("解析rule读取version:{},异常:{}", jsonObject, e);
        }

        return null;
    }

    /**
     * 解析数据处理的规则中step部分
     *
     * @param jsonObject
     */
    public static List<StepElement> parseStep(JSONObject jsonObject) {
        try {
            // 获取rule下所有的节点
            List<JSONObject> ruleList = jsonObject.getJSONArray("step").toJavaList(JSONObject.class);

            if (CollectionUtils.isEmpty(ruleList)) {
                return null;
            }
            List<StepElement> dataElementList = new ArrayList<>();
            for (JSONObject ruleObj : ruleList) {
                // 获取过滤条件
                StepElement stepElement = new StepElement();
                stepElement.setTechnique(ruleObj.getString("technique"));
                //FIXME stepName不能为空，不能为空字符串，也不能重复，需要增加校验
                String stepName = ruleObj.getString("name");
                if (stepName.isEmpty()) {
                    LOGGER.error("解析rule-step规则失败，step name为空", stepName);
                    break;
                }
                stepElement.setName(stepName);

                if ("group".equals(stepElement.getTechnique())) {
                    dataElementList.add(parseGroup(ruleObj, stepElement));
                } else if ("filter".equals(stepElement.getTechnique())) {
                    dataElementList.add(parseFilter(ruleObj, stepElement));
                } else if ("polymerize".equals(stepElement.getTechnique())) {
                    dataElementList.add(parseCollect(ruleObj, stepElement));
                } else if ("mechanism".equals(stepElement.getTechnique())) {
                    dataElementList.add(parseMechanism(ruleObj, stepElement));
                } else if ("distinct".equals(stepElement.getTechnique())) {
                    dataElementList.add(parseDistinct(ruleObj, stepElement));
                } else if ("calculate".equals(stepElement.getTechnique())) {
                    dataElementList.add(parseCalculate(ruleObj, stepElement));
                } else if ("input".equals(stepElement.getTechnique())) {
                    dataElementList.add(parseInput(ruleObj, stepElement));
                } else if ("join".equals(stepElement.getTechnique())) {
                    dataElementList.add(parseJoin(ruleObj, stepElement));
                } else if ("spread".equals(stepElement.getTechnique())) {
                    dataElementList.add(stepElement);
                } else if ("numberRange".equals(stepElement.getTechnique())) {
                    dataElementList.add(parseNumberRange(ruleObj, stepElement));
                } else if ("getRuntimeInfo".equals(stepElement.getTechnique())) {
                    dataElementList.add(parseGetRuntimeInfo(ruleObj, stepElement));
                } else if ("addColumn".equals(stepElement.getTechnique())) {
                    dataElementList.add(parseAddColumn(ruleObj, stepElement));
                } else if ("truncateKey".equals(stepElement.getTechnique())) {
                    dataElementList.add(stepElement);
                } else if ("chooseColumn".equals(stepElement.getTechnique())) {
                    stepElement.setFields(ruleObj.getJSONArray("fields"));
                    stepElement.setType(ruleObj.getString("type"));
                    dataElementList.add(stepElement);
                } else if ("union".equalsIgnoreCase(stepElement.getTechnique())) {
                    stepElement.setType(ruleObj.getString("type"));
                    String dataLeft = ruleObj.getString("data_lt");
                    String dataRight = ruleObj.getString("data_rt");
                    stepElement.setDataLt(dataLeft);
                    stepElement.setDataRt(dataRight);
                    dataElementList.add(stepElement);
                } else {
                    //FIXME 这里需要报错并且退出
                    dataElementList.add(stepElement);
                }
                if (stepElement == null) {
                    LOGGER.error("解析rule-step规则失败，step name: {}", stepName);
                    break;
                }
            }
            return dataElementList;
        } catch (Exception e) {
            LOGGER.error("解析rule-step规则:{},异常:{}", jsonObject, e);
        }
        return null;
    }

    /**
     * 解析添加一列新值的rules
     *
     * @param ruleObject
     * @param stepElement
     * @return
     */
    private static StepElement parseAddColumn(JSONObject ruleObject, StepElement stepElement) {
        List<JSONObject> addColumnRules = ruleObject.getJSONArray("rule").toJavaList(JSONObject.class);

        List<AddColumnElement> rules = new ArrayList<>();
        for (JSONObject ruleObj : addColumnRules) {
            AddColumnElement addColumnElement = new AddColumnElement();
            Object value = ruleObj.get("value");
            Object valueType = ruleObj.get("value_type");
            Object type = ruleObj.get("type");
            Object newField = ruleObj.get("newField");
            addColumnElement.setNewField(newField == null ? "" : String.valueOf(newField));
            addColumnElement.setType(type == null ? "" : String.valueOf(type));
            addColumnElement.setValueType(valueType == null ? "" : String.valueOf(valueType));
            //value字段需要允许为""，所以如果该字段取值为null或者没有该字段，不应该给""作为默认值处理，null值的处理在component子类中校验
            addColumnElement.setValue(value == null ? null : String.valueOf(value));
            rules.add(addColumnElement);
        }
        stepElement.setRules(rules);
        return stepElement;
    }

    /**
     * 解析数据处理的规则order部分
     *
     * @param jsonObject
     */
    public static List<OrderElement> parseOrder(JSONObject jsonObject) {
        try {
            // 获取rule下所有的节点
            List<JSONObject> orderList = jsonObject.getJSONArray("order").toJavaList(JSONObject.class);
            if (CollectionUtils.isEmpty(orderList)) {
                return null;
            }
            List<OrderElement> orderElementList = new ArrayList<>();
            for (JSONObject orderObj : orderList) {
                // 获取过滤条件
                OrderElement orderElement = new OrderElement();
                orderElement.setFrom(orderObj.getString("from"));
                orderElement.setTo(orderObj.getString("to"));
                orderElementList.add(orderElement);
            }
            return orderElementList;
        } catch (Exception e) {
            LOGGER.error("解析rule规则:{},异常:{}", jsonObject, e);
        }
        return null;
    }

    /**
     * 解析合并节点定义结构
     *
     * @param ruleObject
     * @param joinStepElement
     * @return
     */
    private static StepElement parseJoin(JSONObject ruleObject, StepElement joinStepElement) {
        String type = ruleObject.getString("type");
        joinStepElement.setType(type);

        String dataLeft = ruleObject.getString("data_lt");
        String dataRight = ruleObject.getString("data_rt");

        List<String> leftFields = null;
        if (ruleObject.containsKey("left_fields")) {
            //cross join不需要关联字段
            leftFields = ruleObject.getJSONArray("left_fields").toJavaList(String.class);
        }
        List<String> rightFields = null;
        if (ruleObject.containsKey("right_fields")) {
            //cross join不需要关联字段
            rightFields = ruleObject.getJSONArray("right_fields").toJavaList(String.class);
        }
        List<JSONObject> selectLtFields = null;
        List<JSONObject> selectRtFields = null;
        if (ruleObject.containsKey("select_lt_fields")) {
            selectLtFields = ruleObject.getJSONArray("select_lt_fields").toJavaList(JSONObject.class);
        }
        if (ruleObject.containsKey("select_rt_fields")) {
            selectRtFields = ruleObject.getJSONArray("select_rt_fields").toJavaList(JSONObject.class);
        }

        if (StringUtils.isNotEmpty(dataLeft)) {
            joinStepElement.setDataLt(dataLeft);
        }

        if (StringUtils.isNotEmpty(dataRight)) {
            joinStepElement.setDataRt(dataRight);
        }

        List<JoinSelectElement> selectLtFieldList = null;
        if (ruleObject.containsKey("select_lt_fields")) {
            selectLtFieldList = new ArrayList<>();
            for (JSONObject selectLtFieldObj : selectLtFields) {
                JoinSelectElement joinSelectElement = new JoinSelectElement();
                joinSelectElement.setField(selectLtFieldObj.getString("field"));
                joinSelectElement.setNewField(selectLtFieldObj.getString("newField"));
                selectLtFieldList.add(joinSelectElement);
            }
        }

        List<JoinSelectElement> selectRtFieldList = null;
        if (ruleObject.containsKey("select_rt_fields")) {
            selectRtFieldList = new ArrayList<>();
            for (JSONObject selectRtFieldObj : selectRtFields) {
                JoinSelectElement joinSelectElement = new JoinSelectElement();
                joinSelectElement.setField(selectRtFieldObj.getString("field"));
                joinSelectElement.setNewField(selectRtFieldObj.getString("newField"));
                selectRtFieldList.add(joinSelectElement);
            }
        }

        joinStepElement.setSelectLtFields(selectLtFieldList);
        joinStepElement.setSelectRtFields(selectRtFieldList);
        joinStepElement.setLeftFields(leftFields);
        joinStepElement.setRightFields(rightFields);
        return joinStepElement;
    }

    /**
     * 解析数据逻辑判断节点定义结构
     *
     * @param ruleObject
     * @param numberRangeStepElement
     * @return
     */
    private static StepElement parseNumberRange(JSONObject ruleObject, StepElement numberRangeStepElement) {
        String newField = ruleObject.getString("newField");
        String column = ruleObject.getString("column");
        String defaultValue = ruleObject.getString("defaultValue");
        String defaultValueType = ruleObject.getString("defaultValueType");
        String defaultValueDataType = ruleObject.getString("defaultValueDataType");

        List<JSONObject> numberRangeRules = ruleObject.getJSONArray("rule").toJavaList(JSONObject.class);

        List<NumberRangeRuleElement> rules = new ArrayList<>();
        for (JSONObject ruleObj : numberRangeRules) {
            NumberRangeRuleElement numberRangeRuleElement = new NumberRangeRuleElement();
            numberRangeRuleElement.setLower(ruleObj.getString("lower"));
            numberRangeRuleElement.setLowerType(ruleObj.getString("lowerType"));
            numberRangeRuleElement.setLowerDataType(ruleObj.getString("lowerDataType"));
            numberRangeRuleElement.setUpper(ruleObj.getString("upper"));
            numberRangeRuleElement.setUpperType(ruleObj.getString("upperType"));
            numberRangeRuleElement.setUpperDataType(ruleObj.getString("upperDataType"));
            numberRangeRuleElement.setValue(ruleObj.getString("value"));
            numberRangeRuleElement.setDecision(ruleObj.getString("decision"));
            numberRangeRuleElement.setValueType(ruleObj.getString("valueType"));
            numberRangeRuleElement.setValueDataType(ruleObj.getString("valueDataType"));
            rules.add(numberRangeRuleElement);
        }

        numberRangeStepElement.setNewField(newField);
        numberRangeStepElement.setColumn(column);
        numberRangeStepElement.setDefaultValue(defaultValue);
        numberRangeStepElement.setRule(rules);
        numberRangeStepElement.setDefaultValueType(defaultValueType);
        numberRangeStepElement.setDefaultValueDataType(defaultValueDataType);
        return numberRangeStepElement;
    }

    /**
     * 解析获取系统变量节点定义结构
     *
     * @param ruleObject
     * @param getRuntimeInfoStepElement
     * @return
     */
    private static StepElement parseGetRuntimeInfo(JSONObject ruleObject, StepElement getRuntimeInfoStepElement) {
        List<JSONObject> runtimeInfoRules = ruleObject.getJSONArray("rule").toJavaList(JSONObject.class);

        List<RuntimeInfoElement> rules = new ArrayList<>();
        for (JSONObject ruleObj : runtimeInfoRules) {
            RuntimeInfoElement runtimeInfoElement = new RuntimeInfoElement();
            Object format = ruleObj.get("format");
            Object infoType = ruleObj.get("infoType");
            Object newField = ruleObj.get("newField");
            Object method = ruleObj.get("method");
            runtimeInfoElement.setFormat(format == null ? "" : String.valueOf(format));
            runtimeInfoElement.setInfoType(infoType == null ? "" : String.valueOf(infoType));
            runtimeInfoElement.setNewField(newField == null ? "" : String.valueOf(newField));
            runtimeInfoElement.setMethod(method == null ? "" : String.valueOf(method));
            rules.add(runtimeInfoElement);
        }

        getRuntimeInfoStepElement.setRuntimeInfoStepRule(rules);
        return getRuntimeInfoStepElement;
    }

    /**
     * 解析数据读取节点定义结构
     *
     * @param ruleObject
     * @param inputStepElement
     * @return
     */
    private static StepElement parseInput(JSONObject ruleObject, StepElement inputStepElement) {
        String path = ruleObject.getString("path");
        inputStepElement.setPath(path);
        String nullDefaultConstruct = ruleObject.getString("nullDefaultConstruct");
        inputStepElement.setNullDefaultConstruct(nullDefaultConstruct);
        return inputStepElement;
    }

    /**
     * 解析去重节点定义结构
     *
     * @param ruleObject
     * @param distinctStepElement
     * @return
     */
    private static StepElement parseDistinct(JSONObject ruleObject, StepElement distinctStepElement) {
        List<String> fieldJsonList = ruleObject.getJSONArray("fields").toJavaList(String.class);
        if (CollectionUtils.isNotEmpty(fieldJsonList)) {
            distinctStepElement.setFields(fieldJsonList);
        }
        if (ruleObject.containsKey("returnAll")) {
            distinctStepElement.setReturnAll(ruleObject.getBoolean("returnAll"));
        }
        return distinctStepElement;
    }

    /**
     * 解析机制变量节点定义结构
     *
     * @param ruleObject
     * @param mechanismStepElement
     * @return
     */
    private static StepElement parseMechanism(JSONObject ruleObject, StepElement mechanismStepElement) {
        mechanismStepElement.setVariableName(ruleObject.getString("variable_name"));
        if (ruleObject.containsKey("eocLevel")) {
            mechanismStepElement.setEocLevel(ruleObject.getString("eocLevel"));
        }
        return mechanismStepElement;
    }

    private static StepElement parseCalculate(JSONObject ruleObject, StepElement calculateStepElement) {
        List<JSONObject> ruleObjs = ruleObject.getJSONArray("rule").toJavaList(JSONObject.class);
        return parseCalculateRule(ruleObjs, calculateStepElement);
    }

    private static StepElement parseCalculateRule(List<JSONObject> ruleObjs, StepElement calculateStepElement) {
        List<CalculateRuleDto> calculateRuleDtoList = new ArrayList<>();
        for (JSONObject ruleObj : ruleObjs) {
            CalculateRuleDto calculateRuleDto = new CalculateRuleDto();
            calculateRuleDto.setCalcType(ruleObj.getString("calcType"));
            calculateRuleDto.setFieldA(ruleObj.getString("fieldA"));
            calculateRuleDto.setFieldB(ruleObj.getString("fieldB"));
            calculateRuleDto.setFieldC(ruleObj.getString("fieldC"));
            calculateRuleDto.setFormatA(ruleObj.getString("formatA"));
            calculateRuleDto.setFormatB(ruleObj.getString("formatB"));
            calculateRuleDto.setFormatC(ruleObj.getString("formatC"));
            calculateRuleDto.setFieldAType(ruleObj.getString("fieldA_type"));
            calculateRuleDto.setFieldBType(ruleObj.getString("fieldB_type"));
            calculateRuleDto.setFieldCType(ruleObj.getString("fieldC_type"));
            calculateRuleDto.setNewField(ruleObj.getString("newField"));
            calculateRuleDto.setValueType(ruleObj.getString("valueType"));
            calculateRuleDto.setSuffix(ruleObj.getString("suffix"));
            if (!ruleObj.containsKey("valueType")) {
                calculateRuleDto.setValueType("number");
            }

            calculateRuleDto.setValueLength(ruleObj.getString("valueLength"));
            calculateRuleDto.setPrecision(ruleObj.getString("precision"));
            if (calculateRuleDto.getCalcType().equals("Divide") && !ruleObj.containsKey("precision")) {
                calculateRuleDto.setPrecision("2");
            }

            calculateRuleDto.setRetain(ruleObj.getString("retain"));
            calculateRuleDto.setIntervalDays(ruleObj.getIntValue("intervalDays"));
            calculateRuleDto.setPrecisionType(ruleObj.getString("precisionType"));
            if (calculateRuleDto.getCalcType().equals("Divide") && !ruleObj.containsKey("precisionType")) {
                calculateRuleDto.setPrecisionType("rounding");
            }

            calculateRuleDto.setExceptionHandling(ruleObj.getString("exceptionHandling"));
            if (calculateRuleDto.getCalcType().equals("Divide") && !ruleObj.containsKey("exceptionHandling")) {
                calculateRuleDto.setExceptionHandling("divideDefault");
            }
            if (calculateRuleDto.getCalcType().equals("Rooting") && !ruleObj.containsKey("exceptionHandling")) {
                calculateRuleDto.setExceptionHandling("baseLessThanZero");
            }
            if (calculateRuleDto.getCalcType().equals("MOM") && !ruleObj.containsKey("exceptionHandling")) {
                calculateRuleDto.setExceptionHandling("divideDefault");
            }

            calculateRuleDtoList.add(calculateRuleDto);
        }

        calculateStepElement.setCalculateRuleDtoList(calculateRuleDtoList);
        return calculateStepElement;
    }

    /**
     * 解析聚合节点定义结构
     *
     * @param ruleObject
     * @param collectStepElement
     */
    private static StepElement parseCollect(JSONObject ruleObject, StepElement collectStepElement) {
        Boolean isTruncateKey = ruleObject.getBoolean("isTruncateKey");
        if (isTruncateKey != null) {
            collectStepElement.setTruncateKey(isTruncateKey);
        }

        if (ruleObject.containsKey("condition")) {
            List<JSONObject> conditionList = ruleObject.getJSONArray("condition").toJavaList(JSONObject.class);
            if (CollectionUtils.isNotEmpty(conditionList)) {
                List<CollectCondition> collectConditionList = new ArrayList<>();
                for (JSONObject conditionObj : conditionList) {
                    CollectCondition collectCondition = new CollectCondition();
                    collectCondition.setType(conditionObj.getString("type"));
                    collectCondition.setNewField(conditionObj.getString("newField"));
                    List<String> fieldList = conditionObj.getJSONArray("fields").toJavaList(String.class);
                    if (CollectionUtils.isNotEmpty(fieldList)) {
                        collectCondition.setFileds(fieldList);

                    }
                    collectConditionList.add(collectCondition);
                }
                collectStepElement.setCollectConditions(collectConditionList);
            }
        }

        return collectStepElement;
    }

    /**
     * 解析分组节点定义结构
     *
     * @param ruleObject
     * @param groupStepElement
     */
    private static StepElement parseGroup(JSONObject ruleObject, StepElement groupStepElement) {
        List<Statistic> statisticList = new ArrayList<>();
        groupStepElement.setStatistics(statisticList);
        JSONObject conditionObj = ruleObject.getJSONObject("condition");
        // 获取field
        List<String> fieldJsonList = conditionObj.getJSONArray("fields").toJavaList(String.class);
        if (CollectionUtils.isNotEmpty(fieldJsonList)) {
            groupStepElement.setFields(fieldJsonList);
        }
        // 获取里面的condition结构
        List<JSONObject> staticsList = conditionObj.getJSONArray("statistics").toJavaList(JSONObject.class);

        for (JSONObject staticsObj : staticsList) {
            Statistic statistic = new Statistic();
            statistic.setStatisticalMethod(staticsObj.getString("statisticalMethod"));
            statistic.setNewField(staticsObj.getString("newField"));
            statistic.setPrecision(staticsObj.getString("precision"));
            statistic.setPrecisionType(staticsObj.getString("precisionType"));
            statistic.setType(getObjectString(staticsObj.get("type")));
            if (staticsObj.getJSONArray("statisticalFields") != null) {
                List<String> staticList = staticsObj.getJSONArray("statisticalFields").toJavaList(String.class);
                if (CollectionUtils.isNotEmpty(staticList)) {
                    statistic.setStatisticalFields(staticList);
                }
            }
            if (null == staticsObj.get("exceptionHandling")) {
                statistic.setExceptionHandling("divideDefault");
            }
            statistic.setSortName(getObjectString(staticsObj.get("sortName")));
            statistic.setMode(getObjectString(staticsObj.get("mode")));
            statistic.setDirection(getObjectString(staticsObj.get("direction")));
            statistic.setStartPos(getObjectString(staticsObj.get("startPos")));
            statistic.setEndPos(getObjectString(staticsObj.get("endPos")));
            statistic.setRate(getObjectString(staticsObj.get("rate")));
            statistic.setMaximumIterations(getObjectString(staticsObj.get("maximumIterations")));
            statisticList.add(statistic);
        }
        return groupStepElement;
    }


    private static String getObjectString(Object value) {
        if (value == null) {
            return "";
        } else {
            return String.valueOf(value);
        }
    }

    /**
     * 解析过滤节点定义结构
     *
     * @param ruleObject
     * @param filterStepElement
     */
    private static StepElement parseFilter(JSONObject ruleObject, StepElement filterStepElement) {
        filterStepElement.setIsEject(ruleObject.getBoolean("isEject"));
        filterStepElement.setNullJudge(ruleObject.getString("nullJudge"));
        JSONObject conditionsObj = ruleObject.getJSONObject("condition");
        return parseRefactorFilter(conditionsObj, filterStepElement);
    }

    /**
     * 解析重构后的filter
     *
     * @param conditionsObj
     * @param filterStepElement
     * @return
     */
    private static StepElement parseRefactorFilter(JSONObject conditionsObj, StepElement filterStepElement) {
        List<FilterItem> itemList = new ArrayList<>();
        analysisSingleFilterItem(conditionsObj, itemList);
        filterStepElement.setRules(itemList);
        return filterStepElement;
    }

    /**
     * 解析嵌套的item
     *
     * @param conditionsObj
     * @param itemList
     */
    private static void analysisNestingFilterItem(JSONObject conditionsObj, List<FilterItem> itemList) {
        List<JSONObject> items = conditionsObj.getJSONArray("items").toJavaList(JSONObject.class);
        for (JSONObject item : items) {
            analysisSingleFilterItem(item, itemList);
        }
    }

    /**
     * 解析FilterItem
     *
     * @param subConditionObj
     * @param itemList
     * @return
     */
    private static void analysisSingleFilterItem(JSONObject subConditionObj, List<FilterItem> itemList) {
        FilterItem filterItem = new FilterItem();
        String logiType = subConditionObj.getString("logitype");
        filterItem.setLogiType(logiType);
        if (TransConstant.LOGITYPE_SINGLE.equalsIgnoreCase(logiType)) {
            filterItem.setLeft(subConditionObj.getString("left"));
            filterItem.setLeftValueType(subConditionObj.getString("left_value_type"));
            filterItem.setOp(subConditionObj.getString("op"));
            filterItem.setRight(subConditionObj.getString("right"));
            filterItem.setRightValueType(subConditionObj.getString("right_value_type"));
            filterItem.setType(subConditionObj.getString("type"));
            Object leftDateFormat = subConditionObj.get("left_date_format");
            Object rightDateFormat = subConditionObj.get("right_date_format");
            filterItem.setLeftDateFormat((leftDateFormat == null) ? null : String.valueOf(leftDateFormat));
            filterItem.setRightDateFormat((rightDateFormat == null) ? null : String.valueOf(rightDateFormat));
        } else {
            List<FilterItem> innerItemList = new ArrayList<>();
            analysisNestingFilterItem(subConditionObj, innerItemList);
            filterItem.setFilterItems(innerItemList);
        }
        itemList.add(filterItem);
    }
}