package com.digiwin.athena.executionengine.util;

import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.digiwin.app.autoconfigure.DWHttpClientAutoConfiguration;
import com.digiwin.athena.executionengine.constant.CommonConstant;
import com.digiwin.athena.executionengine.constant.ModuleProperty;
import com.digiwin.athena.executionengine.dto.ErrorCodeDto;
import com.digiwin.athena.executionengine.enumtype.ErrorCodeEnum;
import com.digiwin.athena.executionengine.exception.ResponseStatusException;
import com.digiwin.loadbalance.util.HttpRouteUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.HttpStatus;
import org.apache.http.client.HttpClient;
import org.apache.http.client.HttpRequestRetryHandler;
import org.apache.http.client.ServiceUnavailableRetryStrategy;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.conn.routing.HttpRoutePlanner;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Bean;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.ResponseEntity;
import org.springframework.http.client.ClientHttpRequestFactory;
import org.springframework.http.client.HttpComponentsClientHttpRequestFactory;
import org.springframework.http.converter.StringHttpMessageConverter;
import org.springframework.retry.annotation.EnableRetry;
import org.springframework.scheduling.annotation.EnableAsync;
import org.springframework.stereotype.Component;
import org.springframework.web.client.HttpClientErrorException;
import org.springframework.web.client.HttpServerErrorException;
import org.springframework.web.client.RestTemplate;
import org.springframework.web.client.UnknownHttpStatusCodeException;

import javax.print.DocFlavor;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

/**
 * @description: http连接池请求
 * @author: renwm
 * @date: Created in 2020/5/6 17:58
 */
@Component
@EnableRetry
@EnableAsync(proxyTargetClass = true)
public class HttpClientUtils {

    private static final Logger LOGGER = LoggerFactory.getLogger(HttpClientUtils.class);

    private static final String METHOD_POST = "POST";
    private static final String METHOD_GET = "GET";

    @Autowired(required = false)
    @Qualifier(HttpRouteUtils.ATTEMPT_HTTPCLIENT_ROUTE_PLAN)
    HttpRoutePlanner httpRoutePlanner;

//    @Bean
//    public RestTemplate restTemplate() {
//        RestTemplate restTemplate = new RestTemplate(httpRequestFactory());
//        restTemplate.getMessageConverters().set(1, new StringHttpMessageConverter(StandardCharsets.UTF_8));
//        return restTemplate;
//    }

    @Bean("ee-resttemplate")
    public RestTemplate restTemplate(
            @Qualifier("ee-resttemplate-requestfactory")
                    ClientHttpRequestFactory httpRequestFactory) {
        RestTemplate restTemplate = new RestTemplate(httpRequestFactory);
        restTemplate.getMessageConverters().set(1, new StringHttpMessageConverter(StandardCharsets.UTF_8));
        return restTemplate;
    }

//    private ClientHttpRequestFactory httpRequestFactory() {
//        return new HttpComponentsClientHttpRequestFactory(httpClient());
//    }

    @Bean("ee-resttemplate-requestfactory")
    public ClientHttpRequestFactory httpRequestFactory(
            @Qualifier(DWHttpClientAutoConfiguration.BEAN_NAME_DW_HTTPCLIENT_REQUEST_RETRY_HANDLER)
                    HttpRequestRetryHandler requestRetryHandler,
            @Qualifier(DWHttpClientAutoConfiguration.BEAN_NAME_DW_HTTPCLIENT_RESPONSE_RETRY_STRATEGY)
                    ServiceUnavailableRetryStrategy serviceUnavailableRetryStrategy) {

        HttpClientBuilder builder = HttpClientBuilder.create()
                .setRetryHandler(requestRetryHandler)
                .setServiceUnavailableRetryStrategy(serviceUnavailableRetryStrategy);

        PoolingHttpClientConnectionManager connectionManager = getConnectionManager();
        /** 设置整个连接池最大连接数 */
        connectionManager.setMaxTotal(ModuleProperty.HTTP_MAXTOTAL);

        /** 路由是对maxTotal的细分 */
        connectionManager.setDefaultMaxPerRoute(ModuleProperty.HTTP_MAXPERROUTE);
        RequestConfig requestConfig = RequestConfig.custom()
                /** 返回数据的超时时间 */
                .setSocketTimeout(ModuleProperty.HTTP_SOCKE_TTIMEOUT)
                /** 连接上服务器的超时时间 */
                .setConnectTimeout(ModuleProperty.HTTP_CONNECT_TIMEOUT)
                /** 从连接池中获取连接的超时时间 */
                .setConnectionRequestTimeout(ModuleProperty.HTTP_REQUEST_TIMEOUT)
                .build();
        builder.setDefaultRequestConfig(requestConfig).setConnectionManager(connectionManager)
                .setRoutePlanner(httpRoutePlanner).build();

        HttpClient httpClient = builder.build();
        HttpComponentsClientHttpRequestFactory clientHttpRequestFactory = new HttpComponentsClientHttpRequestFactory(httpClient);

        return clientHttpRequestFactory;
    }

//    private HttpClient httpClient() {
//        PoolingHttpClientConnectionManager connectionManager = getConnectionManager();
//        /** 设置整个连接池最大连接数 */
//        connectionManager.setMaxTotal(ModuleProperty.HTTP_MAXTOTAL);
//
//        /** 路由是对maxTotal的细分 */
//        connectionManager.setDefaultMaxPerRoute(ModuleProperty.HTTP_MAXPERROUTE);
//        RequestConfig requestConfig = RequestConfig.custom()
//                /** 返回数据的超时时间 */
//                .setSocketTimeout(ModuleProperty.HTTP_SOCKE_TTIMEOUT)
//                /** 连接上服务器的超时时间 */
//                .setConnectTimeout(ModuleProperty.HTTP_CONNECT_TIMEOUT)
//                /** 从连接池中获取连接的超时时间 */
//                .setConnectionRequestTimeout(ModuleProperty.HTTP_REQUEST_TIMEOUT)
//
//                .build();
//        return HttpClientBuilder.create()
//                .setDefaultRequestConfig(requestConfig)
//                .setConnectionManager(connectionManager)
//                .setRoutePlanner(httpRoutePlanner)
//                .setRetryHandler(getHttpRequestRetryHandler(3))
//                .build();
//    }

    @Bean("httpClientConnectionManager")
    public PoolingHttpClientConnectionManager getConnectionManager() {
        return new PoolingHttpClientConnectionManager();
    }

//    /**
//     * 获取重连处理对象
//     *
//     * @return
//     */
//    private HttpRequestRetryHandler getHttpRequestRetryHandler(final int retryCount) {
//        // 请求重试处理
//        return new HttpRequestRetryHandler() {
//            @Override
//            public boolean retryRequest(IOException exception, int executionCount, HttpContext context) {
//
//                // Unknown host
//                if (exception instanceof UnknownHostException) {
//                    return false;
//                }
//                // SSL handshake exception
//                if (exception instanceof SSLHandshakeException) {
//                    return false;
//                }
//                if (exception instanceof SocketTimeoutException) {
//                    return false;
//                }
//                // 错误次数小于重试次数，那么就重试
//                if (executionCount > retryCount) {
//                    return false;
//                }
//                return true;
//            }
//        };
//    }

    public static String doRequest(String method, String url, Map<String, String> headers, Map<String, Object> params) {
        if (METHOD_POST.equalsIgnoreCase(method)) {
            return doPost(url, headers, JSONObject.toJSONString(params, SerializerFeature.WRITE_MAP_NULL_FEATURES, SerializerFeature.QuoteFieldNames));

        }
        if (METHOD_GET.equalsIgnoreCase(method)) {
            return doGet(url, headers, params);
        }
        return null;
    }

    /**
     * http post请求
     *
     * @param url
     * @param jsonParams
     * @return
     */
    public static String doPost(String url, Map<String, String> headers, String jsonParams) {
        String respContent = null;
        try {
            HttpEntity httpEntity = new HttpEntity(jsonParams, getSpringHttpHeader(headers));
            ResponseEntity<String> responseEntity = ContextUtils.getBean(RestTemplate.class).postForEntity(url, httpEntity, String.class);
            int statusCode = responseEntity.getStatusCodeValue();
            if (statusCode != HttpStatus.SC_OK) {
                throw ExceptionUtils.buildResponseStatusException(ErrorCodeEnum.HTTP_RESPONSE_FAIL_STATUS, METHOD_POST, url, String.valueOf(statusCode));
            }
            respContent = responseEntity.getBody();
        } catch (HttpClientErrorException e) {
            LOGGER.error("doPost请求[" + url + "]出现客户端4XX异常！响应报文为：" + e.getResponseBodyAsString(), e);
            ResponseStatusException responseStatusException = ExceptionUtils.buildResponseStatusException(ErrorCodeEnum.HTTP_CLIENT_EXCEPTION, METHOD_POST, url, e.getResponseBodyAsString());
            responseStatusException.getInstructors().put(CommonConstant.CHAIN_INFO, getChainInfo(e.getResponseBodyAsString()));
            throw responseStatusException;
        } catch (HttpServerErrorException e) {
            LOGGER.error("doPost请求[" + url + "]出现服务端5XX异常！响应报文为：" + e.getResponseBodyAsString(), e);
            ResponseStatusException responseStatusException;
            ErrorCodeDto errorCodeDto = parseErrorInfo(e.getResponseBodyAsString());
            if (errorCodeDto != null) {
                responseStatusException = new ResponseStatusException(errorCodeDto.getErrorCode(), errorCodeDto.getMessage(), e.getResponseBodyAsString());
            } else {
                responseStatusException = ExceptionUtils.buildResponseStatusException(ErrorCodeEnum.HTTP_SERVER_EXCEPTION, METHOD_POST, url, e.getResponseBodyAsString());
            }
            responseStatusException.getInstructors().put(CommonConstant.CHAIN_INFO, getChainInfo(e.getResponseBodyAsString()));
            throw responseStatusException;
        } catch (UnknownHttpStatusCodeException e) {
            LOGGER.error("doPost请求[" + url + "]出现未知状态码异常！响应报文为：" + e.getResponseBodyAsString(), e);
            ResponseStatusException responseStatusException = ExceptionUtils.buildResponseStatusException(ErrorCodeEnum.HTTP_UNKNOWN_STATUS_EXCEPTION, METHOD_POST, url, e.getResponseBodyAsString());
            responseStatusException.getInstructors().put(CommonConstant.CHAIN_INFO, getChainInfo(e.getResponseBodyAsString()));
            throw responseStatusException;
        } catch (Exception e) {
            LOGGER.error("[{" + url + "}]post请求失败", e);
            throw ExceptionUtils.buildResponseStatusException(ErrorCodeEnum.HTTP_UNKNOWN_EXCEPTION, METHOD_POST, url);
        } finally {
            if (respContent == null) {
                LOGGER.error("httpClient post请求错误 请求url：{}", url);
                LOGGER.error("httpClient post请求错误 请求header：{}", headers);
                LOGGER.error("httpClient post请求错误 请求入参：{}", jsonParams);
            }
        }
        return respContent;
    }

    private static HttpHeaders getSpringHttpHeader(Map<String, String> headers) {
        HttpHeaders springHttpHeader = new HttpHeaders();
        for (Map.Entry<String, String> entry : headers.entrySet()) {
            springHttpHeader.set(entry.getKey(), entry.getValue());
        }
        return springHttpHeader;
    }

    /**
     * get请求
     *
     * @param url
     * @return
     */
    private static String doGet(String url, Map<String, String> headers, Map<String, Object> param) {
        String respContent = null;
        try {
            boolean isNeedDecode = true;
            Map<String, String> paramMap = new HashMap();
            if (MapUtils.isNotEmpty(param)) {
                StringBuilder sbParam = new StringBuilder("?");
                for (Map.Entry<String, Object> entry : param.entrySet()) {
                    sbParam.append(entry.getKey()).append("=").append(URLEncoder.encode(entry.getValue().toString(), "UTF-8")).append("&");
                }
                url = url + sbParam;
            } else {
                //处理url里的json数据
                String[] urlParams = url.split("\\?");
                if (urlParams.length > 1) {
                    String[] params = urlParams[1].split("&");

                    String urlParamUnion = "";
                    for (String para : params) {
                        String[] keyValue = para.split("=");
                        String key = keyValue[0];
                        String value = keyValue[1];
                        LOGGER.info("Param: {}, Value: {}", key, value);
                        urlParamUnion = urlParamUnion + key + "={" + key + "}&";
                        paramMap.put(key, java.net.URLDecoder.decode(value, "utf-8"));
                    }
                    url = urlParams[0] + "?" + urlParamUnion;
                    LOGGER.info("doGet调用url: {}", url);
                    isNeedDecode = false;
                }
            }
            //防止get请求中中文乱码
            if (isNeedDecode) {
                url = java.net.URLDecoder.decode(url, "utf-8");
            }
            HttpEntity httpEntity = new HttpEntity<>(getSpringHttpHeader(headers));
            ResponseEntity<String> responseEntity = ContextUtils.getBean(RestTemplate.class).exchange(url, HttpMethod.GET, httpEntity, String.class, paramMap);
            int statusCode = responseEntity.getStatusCodeValue();
            if (statusCode != HttpStatus.SC_OK) {
                throw ExceptionUtils.buildResponseStatusException(ErrorCodeEnum.HTTP_RESPONSE_FAIL_STATUS, METHOD_GET, url, String.valueOf(statusCode));
            }
            respContent = responseEntity.getBody();
        } catch (HttpClientErrorException e) {
            LOGGER.error("doGet请求[" + url + "]出现客户端4XX异常！响应报文为：" + e.getResponseBodyAsString(), e);
            ResponseStatusException responseStatusException = ExceptionUtils.buildResponseStatusException(ErrorCodeEnum.HTTP_CLIENT_EXCEPTION, METHOD_GET, url, e.getResponseBodyAsString());
            responseStatusException.getInstructors().put(CommonConstant.CHAIN_INFO, getChainInfo(e.getResponseBodyAsString()));
            throw responseStatusException;
        } catch (HttpServerErrorException e) {
            LOGGER.error("doGet请求[" + url + "]出现服务端5XX异常！响应报文为：" + e.getResponseBodyAsString(), e);
            ResponseStatusException responseStatusException;
            ErrorCodeDto errorCodeDto = parseErrorInfo(e.getResponseBodyAsString());
            if (errorCodeDto != null) {
                responseStatusException = new ResponseStatusException(errorCodeDto.getErrorCode(), errorCodeDto.getMessage(), e.getResponseBodyAsString());
            } else {
                responseStatusException = ExceptionUtils.buildResponseStatusException(ErrorCodeEnum.HTTP_SERVER_EXCEPTION, METHOD_GET, url, e.getResponseBodyAsString());
            }
            responseStatusException.getInstructors().put(CommonConstant.CHAIN_INFO, getChainInfo(e.getResponseBodyAsString()));
            throw responseStatusException;
        } catch (UnknownHttpStatusCodeException e) {
            LOGGER.error("doGet请求[" + url + "]出现未知状态码异常！响应报文为：" + e.getResponseBodyAsString(), e);
            ResponseStatusException responseStatusException = ExceptionUtils.buildResponseStatusException(ErrorCodeEnum.HTTP_UNKNOWN_STATUS_EXCEPTION, METHOD_GET, url, e.getResponseBodyAsString());
            responseStatusException.getInstructors().put(CommonConstant.CHAIN_INFO, getChainInfo(e.getResponseBodyAsString()));
            throw responseStatusException;
        } catch (Exception e) {
            LOGGER.error("[{" + url + "}]get请求异常", e);
            throw ExceptionUtils.buildResponseStatusException(ErrorCodeEnum.HTTP_UNKNOWN_EXCEPTION, METHOD_GET, url);
        } finally {
            if (respContent == null) {
                LOGGER.error("httpClient get请求错误 请求url：{}", url);
                LOGGER.error("httpClient get请求错误 请求header：{}", headers);
                LOGGER.error("httpClient get请求错误 请求入参：{}", param);
            }
        }
        return respContent;
    }

    /**
     * 解析上层服务抛出的errorCode和message
     *
     * @param message
     */
    private static ErrorCodeDto parseErrorInfo(String message) {
        String errorCode = Optional.ofNullable(JSONObject.parseObject(message))
                .map(item -> item.getString("errorCode")).orElse(null);
        String errorMessage = Optional.ofNullable(JSONObject.parseObject(message))
                .map(item -> item.getString("errorMessage")).orElse(null);

        if (StringUtils.isNotEmpty(errorCode) && StringUtils.isNotEmpty(errorMessage)) {
            return new ErrorCodeDto(errorCode, errorMessage);
        }
        return null;
    }

    /**
     * 获取下层chainInfo
     *
     * @param message
     * @return
     */
    public static String getChainInfo(String message) {
        return Optional.ofNullable(getErrorInstructors(message)).
                map(item -> item.getString(CommonConstant.CHAIN_INFO)).orElse(null);
    }

    private static JSONObject getErrorInstructors(String message) {
        return Optional.ofNullable(JSONObject.parseObject(message))
                .map(item -> item.getJSONObject("errorInstructors")).orElse(null);
    }
}