package com.digiwin.athena.executionengine.trans.components;

import com.digiwin.athena.executionengine.constant.TransConstant;
import com.digiwin.athena.executionengine.model.trans.DealResult;
import com.digiwin.athena.executionengine.model.trans.RuntimeInfoElement;
import com.digiwin.athena.executionengine.model.trans.StepElement;
import com.digiwin.athena.executionengine.trans.Step;
import com.digiwin.athena.executionengine.trans.TransAbstractStep;
import com.digiwin.athena.executionengine.util.DateUtils;
import com.google.common.collect.Lists;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import java.util.List;
import java.util.Map;

/**
 * @description: 获取系统变量的节点（根据开放方法获取变量）
 * @author: zhangww
 * @date: 2021/1/7 9:23
 */
@Component("getRuntimeInfo")
public class GetRuntimeInfoStep extends TransAbstractStep {

    private static final Logger LOGGER = LoggerFactory.getLogger(GetRuntimeInfoStep.class);


    @Override
    public boolean defineCheck(StepElement stepElement) {
        List<RuntimeInfoElement> rules = stepElement.getRuntimeInfoStepRule();
        for (RuntimeInfoElement rule : rules) {
            String infoType = rule.getInfoType();
            if (TransConstant.RUNTIMEINFO_INFOTYPE_CURRENTTIME.equals(infoType)) {
                if (!checkCurrentTime(rule)) {
                    return false;
                }
            } else {
                return false;
            }
        }
        return true;
    }

    /**
     * 对infotype为currentTime的规则类型进行校验
     *
     * @param rule
     * @return
     */
    private boolean checkCurrentTime(RuntimeInfoElement rule) {
        if (StringUtils.isEmpty(rule.getFormat()) ||
                StringUtils.isEmpty(rule.getMethod()) || StringUtils.isEmpty(rule.getNewField())) {
            return false;
        }
        if (!(TransConstant.RUNTIMEINFO_METHOD_DYNAMIC.equals(rule.getMethod())
                || TransConstant.RUNTIMEINFO_METHOD_STATIC.equals(rule.getMethod()))) {
            return false;
        }
        return true;
    }

    @Override
    public DealResult doDealData(Step step) {
        //当前action的开始执行的时间
        DealResult dealResult = new DealResult();
        Object inputData = getCurrentData(step.getPrevStepNameList());
        if (!(inputData instanceof List)) {
            LOGGER.warn("获取系统变量组件只能接受表状数据，非展平数据需要先执行展平节点！Trans执行 {} 失败，数据结构错误！",
                    step.getName());
            dealResult.setDataNullFail();
            return dealResult;
        }

        //输入数据
        List<Map<String, Object>> dataList = (List<Map<String, Object>>) (inputData);
        // 数据源为空直接返回
        if (CollectionUtils.isEmpty(dataList)) {
            dealResult.setSuccess(Lists.newArrayList());
            return dealResult;
        }

        List<RuntimeInfoElement> rules = step.getStepElement().getRuntimeInfoStepRule();
        for (Map<String, Object> data : dataList) {
            if (!getRuntimeInfo(rules, data)) {
                dealResult.setData(null);
                dealResult.setResult(TransConstant.TRANS_FAIL);
                return dealResult;
            }
        }

        dealResult.setSuccess(dataList);
        return dealResult;
    }

    /**
     * 根据规则获取系统运行时的变量
     *
     * @param rules 获取变量的规则
     * @param data  数据
     * @return
     */
    private boolean getRuntimeInfo(List<RuntimeInfoElement> rules, Map<String, Object> data) {
        for (RuntimeInfoElement rule : rules) {
            String infoType = rule.getInfoType();
            String method = rule.getMethod();
            if (TransConstant.RUNTIMEINFO_INFOTYPE_CURRENTTIME.equals(infoType)) {
                setCurrentTime(data, rule, method);
            } else {
                return false;
            }
        }
        return true;
    }

    /**
     * 设置获取系统时间的列
     *
     * @param data
     * @param rule
     * @param method
     */
    private void setCurrentTime(Map<String, Object> data, RuntimeInfoElement rule, String method) {
        String currentTime = DateUtils.getCurrentDateTime();
        if (TransConstant.RUNTIMEINFO_METHOD_STATIC.equals(method)) {
            data.put(rule.getNewField(), DateUtils.transferFormat(currentTime, rule.getFormat()));
        } else {
            data.put(rule.getNewField(), DateUtils.getCurrentDateTime(rule.getFormat()));
        }
    }
}
