package com.digiwin.athena.executionengine.trans.operator;

import com.digiwin.athena.executionengine.util.RegexUtil;
import org.apache.commons.lang3.StringUtils;

import java.math.BigDecimal;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

public class OperatorFactory {

    private static Map<String, OperatorFunction<Object>> FUNCTION_MAP = new HashMap<>();

    static {
        //and
        FUNCTION_MAP.put("and", objects -> {
            boolean isMatch;
            for (Object obj : objects) {
                isMatch = ((Boolean) obj).booleanValue();
                if (!isMatch) {
                    return isMatch;
                }
            }
            return true;
        });

        //eq
        FUNCTION_MAP.put("eq", objects -> {
            String origin = String.valueOf(objects[0]);
            String target = String.valueOf(objects[1]);
            if (StringUtils.isBlank(origin) || StringUtils.isBlank(target)) {
                return false;
            }
            if (RegexUtil.isNumber(origin) && RegexUtil.isNumber(target)) {
                BigDecimal originDecimal = new BigDecimal(String.valueOf(objects[0]));
                BigDecimal targetDecimal = new BigDecimal(String.valueOf(objects[1]));
                return originDecimal.compareTo(targetDecimal) == 0;
            }
            return String.valueOf(objects[0]).equals(String.valueOf(objects[1]));
        });

        //gte
        FUNCTION_MAP.put("gte", objects -> {
            BigDecimal originDecimal = new BigDecimal(String.valueOf(objects[0]));
            BigDecimal targetDecimal = new BigDecimal(String.valueOf(objects[1]));
            return originDecimal.compareTo(targetDecimal) >= 0;
        });

        //gt
        FUNCTION_MAP.put("gt", objects -> {
            BigDecimal originDecimal = new BigDecimal(String.valueOf(objects[0]));
            BigDecimal targetDecimal = new BigDecimal(String.valueOf(objects[1]));
            return originDecimal.compareTo(targetDecimal) > 0;
        });

        //lte
        FUNCTION_MAP.put("lte", objects -> {
            BigDecimal originDecimal = new BigDecimal(String.valueOf(objects[0]));
            BigDecimal targetDecimal = new BigDecimal(String.valueOf(objects[1]));
            return originDecimal.compareTo(targetDecimal) <= 0;
        });

        //lt
        FUNCTION_MAP.put("lt", objects -> {
            BigDecimal originDecimal = new BigDecimal(String.valueOf(objects[0]));
            BigDecimal targetDecimal = new BigDecimal(String.valueOf(objects[1]));
            return originDecimal.compareTo(targetDecimal) < 0;
        });

        //neq
        FUNCTION_MAP.put("neq", objects -> {
            String origin = String.valueOf(objects[0]);
            String target = String.valueOf(objects[1]);
            if (StringUtils.isBlank(origin) || StringUtils.isBlank(target)) {
                return false;
            }
            if (RegexUtil.isNumber(origin) && RegexUtil.isNumber(target)) {
                BigDecimal originDecimal = new BigDecimal(String.valueOf(objects[0]));
                BigDecimal targetDecimal = new BigDecimal(String.valueOf(objects[1]));
                return originDecimal.compareTo(targetDecimal) != 0;
            }
            return !String.valueOf(objects[0]).equals(String.valueOf(objects[1]));
        });

        //or
        FUNCTION_MAP.put("or", objects -> {
            boolean isMatch;
            for (Object obj : objects) {
                isMatch = ((Boolean) obj).booleanValue();
                if (isMatch) {
                    return true;
                }
            }
            return false;
        });

        //Prefix match
        FUNCTION_MAP.put("pm", objects -> {
            String value = String.valueOf(objects[0]);
            String prefix = String.valueOf(objects[1]);
            if (StringUtils.isEmpty(value) || StringUtils.isEmpty(prefix)) {
                return false;
            }
            return value.startsWith(prefix);
        });

        //suffix match
        FUNCTION_MAP.put("sm", objects -> {
            String value = String.valueOf(objects[0]);
            String suffix = String.valueOf(objects[1]);
            if (StringUtils.isEmpty(value) || StringUtils.isEmpty(suffix)) {
                return false;
            }
            return value.endsWith(suffix);
        });

        //full match
        FUNCTION_MAP.put("fm", objects -> {
            String value = String.valueOf(objects[0]);
            String key = String.valueOf(objects[1]);
            if (StringUtils.isEmpty(value) || StringUtils.isEmpty(key)) {
                return false;
            }
            return value.contains(key);
        });

        //in
        FUNCTION_MAP.put("in", objects -> {
            Object value = objects[0];
            List<Object> collections = (List<Object>) objects[1];
            Set<Object> set = new HashSet<>(collections);
            return set.contains(value);
        });

        //not in
        FUNCTION_MAP.put("nin", objects -> {
            Object value = objects[0];
            List<Object> collections = (List<Object>) objects[1];
            Set<Object> set = new HashSet<>(collections);
            return !set.contains(value);
        });
    }

    @FunctionalInterface
    public interface OperatorFunction<T> {
        boolean apply(T... objects);
    }

    public static OperatorFunction getOperatorFunction(String type) {
        return FUNCTION_MAP.get(type);
    }


}
