package com.digiwin.athena.base.application.service.importstatistics.application;

import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.atdm.constant.ErrorCodeEnum;
import com.digiwin.athena.atdm.importstatistics.ServiceException;
import com.digiwin.athena.atdm.importstatistics.application.Application;
import com.digiwin.athena.atdm.importstatistics.application.ApplicationRelation;
import com.digiwin.athena.atdm.importstatistics.application.FieldDescription;
import com.digiwin.athena.atdm.importstatistics.cac.DataEntryCacService;
import com.digiwin.athena.atdm.importstatistics.common.api.DapResponse;
import com.digiwin.athena.atdm.importstatistics.kg.DataEntryKgService;
import com.digiwin.athena.atdm.importstatistics.util.LanguageUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

@Slf4j
@Service
public class ApplicationService {

    @Autowired
    private DataEntryKgService kgService;
    @Autowired
    private DataEntryCacService cacService;

    public List<Application> getApplications(AuthoredUser user) {
        List<Application> applications = this.getByNameAndTags(null, null);
        List<Application> filteredApplications = this.filterApplicationByTenant(applications,
                user.getTenantId());
        List<Application> enhanceApplications = this.processEnhanceApplication(filteredApplications);
        return enhanceApplications;
    }

    public List<Application> getByNameAndTags(String applicationName, List<String> tags){
        List<Application> applications = null;
        DapResponse re = this.kgService.getApplicationByNameAndTags(applicationName, tags);
        if(re.getStatus() != 200){
            throw new ServiceException(-1, re.getErrorMessage());
        }
        if (re.getResponse() != null) {
            applications = JsonUtils.jsonToListObject(JsonUtils.objectToString(re.getResponse()), Application.class);
        }
        if (applications != null && !applications.isEmpty()) {
            for (Application app : applications) {
                try {
                    LanguageUtil.processLocaleLanguage(app, LocaleContextHolder.getLocale().toString());
                } catch (Exception e) {
                    log.error("error parse application language", e);
                }
            }
        }
        return applications;
    }

    public List<Application> filterApplicationByTenant(List<Application> applications, String tenantId){
        if(CollectionUtils.isEmpty(applications)){
            return applications;
        }
        Set<String> matchApplicationCodes = this.findApplicationCodeByTenantId(tenantId);
        List<Application> filteredApplication = applications.stream().filter(e -> matchApplicationCodes.contains(e.getCode())).collect(Collectors.toList());
        return filteredApplication;
    }

    public Set<String> findApplicationCodeByTenantId(String tenantId){
//        List<ApplicationTenant> applicationTenants = this.findApplicationTenantByTenantId(tenantId);
//        Set<String> matchApplication = applicationTenants.stream().map(ApplicationTenant::getApplicationCode).collect(Collectors.toSet());
//        return matchApplication;
        return cacService.getAithorizationsApplication(tenantId);
    }

    public List<Application> processEnhanceApplication(List<Application> applications) {
        List<Application> re = new ArrayList<>();
        if (CollectionUtils.isNotEmpty(applications)) {
            Set<String> appCode = new HashSet<>();
            Set<String> enhanceCode = new HashSet<>();
            for (Application app : applications) {
                if (StringUtils.isEmpty(app.getCategory())) {
                    appCode.add(app.getCode());
                    re.add(app);
                } else if (StringUtils.equals("enhance", app.getCategory())) {
                    enhanceCode.add(app.getCode());
                }
            }

            DapResponse resp = this.kgService.getDigitalStaff(appCode, enhanceCode);
            if (resp == null || resp.getStatus() != 200) {
                throw BusinessException.create(ErrorCodeEnum.KG_V2_GET_DIGITAL_STAFF.getErrCode() ,resp == null ? "NULL" : resp.getErrorMessage());
            }
            Map<String, List> enhanceMap = (Map<String, List>) resp.getResponse();
            if (enhanceMap != null && !enhanceMap.isEmpty()) {
                if (CollectionUtils.isNotEmpty(re)) {
                    for (Application i : re) {
                        i.setEnhance(enhanceMap.get(i.getCode()));
                        setEnhanceParam(i);
                    }
                }
            }
        }
        return re;
    }

    private void setEnhanceParam(Application app) {
        //加入增强应用参数
        DapResponse respEnhance = kgService.getEnhancebyAppCode(app.getCode());
        if (respEnhance.getResponse() != null && !"".equals(respEnhance.getResponse())) {
            List<Application> enhanceApps = JsonUtils.jsonToListObject(JsonUtils.objectToString(respEnhance.getResponse()), Application.class);
            List<Application> filterApps = this.filterApplicationByTenant(enhanceApps, AppAuthContextHolder.getContext().getAuthoredUser().getTenantId());
            if (CollectionUtils.isNotEmpty(filterApps)) {
                if (CollectionUtils.isEmpty(app.getParamDescription())) {
                    List<FieldDescription> paras = new ArrayList<>();
                    app.setParamDescription(paras);
                }
                for (Application i : filterApps) {
                    if (!org.springframework.util.CollectionUtils.isEmpty(i.getParamDescription())) {
                        for (FieldDescription p : i.getParamDescription()) {
                            p.setEnhance(String.format("%s:%s", i.getName(), i.getExtendDisplay()));
                        }
                        app.getParamDescription().addAll(i.getParamDescription());
                    }
                }
            }
        }
    }

    public List<ApplicationRelation> getApplicationRelationsByCodesAndType(List<String> code, String type) {
        DapResponse dapResponse = kgService.postApplicationRelationsByCodesAndType(code, type);
        List<ApplicationRelation> result = JsonUtils.jsonToListObject(JsonUtils.objectToString(dapResponse.getResponse()), ApplicationRelation.class);
        return result;
    }

}