package com.digiwin.athena.base.application.service.commonused;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.appcore.util.SnowflakeIdWorker;
import com.digiwin.athena.base.application.config.BaseAudcDataSourceConfig;
import com.digiwin.athena.base.application.meta.dto.commonused.ItemsDto;
import com.digiwin.athena.base.infrastructure.mapper.audc.commonUsed.ItemLogMapper;
import com.digiwin.athena.base.infrastructure.mapper.audc.commonUsed.ItemMapper;
import com.digiwin.athena.base.infrastructure.meta.po.commonused.ItemData;
import com.digiwin.athena.base.infrastructure.meta.po.commonused.ItemLogData;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.time.LocalDateTime;
import java.util.List;

@Service
public class ItemServiceImpl implements ItemService {

    private static final Logger logger = LoggerFactory.getLogger(ItemServiceImpl.class);
    @Autowired
    private ItemMapper itemMapper;

    @Autowired
    private ItemLogMapper itemLogMapper;


    /**
     * 查询对应的模块常用下子项信息
     *
     * @param user   用户信息
     * @param plat   所属模块0:基础资料录入 1：报表，后续有新增则依次递增
     * @param cateId 分类ID -1：根节点
     */
    public List<ItemData> queryItemList(AuthoredUser user, Integer plat, Long cateId) {

        //获取根节点下的子项信息
        QueryWrapper<ItemData> itemQuery = new QueryWrapper<>();
        if (cateId != null) {
            itemQuery.eq("cate_id", cateId);
        }
        itemQuery.eq("plat", plat);
        itemQuery.eq("tenant_id", user.getTenantId());
        itemQuery.eq("user_id", user.getUserId());
        return itemMapper.selectList(itemQuery);
    }

    /**
     * 新增子项
     *
     * @param user     用户信息
     * @param itemsDto 新增的子项信息
     * @return int
     */
    @Transactional(transactionManager = BaseAudcDataSourceConfig.BASE_AUDC_DATASOURCE_TRANSACTION_MANAGER_BUSINESS)
    public ItemData addItem(AuthoredUser user, ItemsDto itemsDto) {
        ItemData data = new ItemData();
        data.setId(SnowflakeIdWorker.getInstance().newId());
        data.setPlat(itemsDto.getPlat());
        data.setCreateDate(LocalDateTime.now());
        data.setModifyDate(LocalDateTime.now());
        data.setItemCate(itemsDto.getItemCate());
        data.setItemCode(itemsDto.getItemCode());
        data.setCateId(itemsDto.getCateId());
        data.setUserId(user.getUserId());
        data.setUserName(user.getUserName());
        data.setTenantId(user.getTenantId());
        QueryWrapper<ItemData> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("item_code",itemsDto.getItemCode());
        queryWrapper.eq("plat",itemsDto.getPlat());
        queryWrapper.eq("user_id",user.getUserId());
        queryWrapper.eq("tenant_id",user.getTenantId());
        ItemData itemData = itemMapper.selectOne(queryWrapper);
        if(itemData != null){
            return itemData;
        }
        int result = itemMapper.insert(data);
        if (result > 0) {
            itemRecord(user, itemsDto, 1);
            return data;
        }
        return null;
    }

    /**
     * 校验子项是否存在
     *
     * @param user     用户信息
     * @param itemsDto 子项信息
     */
    public boolean isItemExist(AuthoredUser user, ItemsDto itemsDto) {
        QueryWrapper<ItemData> itemQuery = new QueryWrapper<>();
        itemQuery.eq("plat", itemsDto.getPlat());
        itemQuery.eq("item_code", itemsDto.getItemCode());
        itemQuery.eq("user_id", user.getUserId());
        itemQuery.eq("cate_id", itemsDto.getCateId());
        Long result = itemMapper.selectCount(itemQuery);
        return result > 0L;
    }


    /**
     * 移动单个子项
     *
     * @param id         需要移动的子项id
     * @param descCateId 移动后的子项分类 -1：根节点
     * @return int 结果
     */
    @Transactional(transactionManager = BaseAudcDataSourceConfig.BASE_AUDC_DATASOURCE_TRANSACTION_MANAGER_BUSINESS)
    public int moveSingleItem(Long id, Long descCateId) {
        //首先将该分类下的item移动到根路径下
        ItemData itemData = new ItemData();
        itemData.setCateId(descCateId);
        itemData.setModifyDate(LocalDateTime.now());
        UpdateWrapper<ItemData> itemWrapper = new UpdateWrapper<>();
        itemWrapper.eq("id", id);
        return itemMapper.update(itemData, itemWrapper);
    }

    /**
     * 将某个分类下的子项全部移动到指定分类
     *
     * @param srcCateId  源分类ID -1：根节点
     * @param descCateId 目标分类ID -1：根节点
     * @return int 结果
     */
    @Transactional(transactionManager = BaseAudcDataSourceConfig.BASE_AUDC_DATASOURCE_TRANSACTION_MANAGER_BUSINESS)
    public int moveItemsByCate(Long srcCateId, Long descCateId, AuthoredUser user, int plat) {
        //首先将该分类下的item移动到根路径下
        ItemData itemData = new ItemData();
        itemData.setCateId(descCateId);
        itemData.setModifyDate(LocalDateTime.now());
        UpdateWrapper<ItemData> itemWrapper = new UpdateWrapper<>();
        itemWrapper.eq("user_id", user.getUserId());
        itemWrapper.eq("tenant_id", user.getTenantId());
        itemWrapper.eq("plat", plat);
        itemWrapper.eq("cate_id", srcCateId);
        return itemMapper.update(itemData, itemWrapper);
    }


    /**
     * 将某个分类下的子项全部删除
     *
     * @param srcCateId 源分类ID -1：根节点
     * @return int 结果
     */
    @Transactional(transactionManager = BaseAudcDataSourceConfig.BASE_AUDC_DATASOURCE_TRANSACTION_MANAGER_BUSINESS)
    public int deleteItemsByCate(Long srcCateId, AuthoredUser user, int plat) {
        //首先将该分类下的item移动到根路径下
        ItemData itemData = new ItemData();
        itemData.setCateId(srcCateId);
        UpdateWrapper<ItemData> itemWrapper = new UpdateWrapper<>();
        itemWrapper.eq("user_id", user.getUserId());
        itemWrapper.eq("tenant_id", user.getTenantId());
        itemWrapper.eq("plat", plat);
        itemWrapper.eq("cate_id", srcCateId);
        return itemMapper.delete(itemWrapper);
    }


    /**
     * 根据id删除某个子项
     *
     * @param id 子项id
     * @return int 结果
     */
    @Transactional(transactionManager = BaseAudcDataSourceConfig.BASE_AUDC_DATASOURCE_TRANSACTION_MANAGER_BUSINESS)
    public int delete(AuthoredUser user, Long id) {
        ItemData data = itemMapper.selectById(id);
        if (null != data) {
            ItemsDto dto = new ItemsDto();
            BeanUtils.copyProperties(data, dto);
            itemRecord(user, dto, 2);
        }
        return itemMapper.deleteById(id);
    }


    /**
     * 根据id删除某个子项
     *
     * @return int 结果
     */
    @Transactional(transactionManager = BaseAudcDataSourceConfig.BASE_AUDC_DATASOURCE_TRANSACTION_MANAGER_BUSINESS)
    public int delete(AuthoredUser user, Integer plat, String itemCode) {
        QueryWrapper<ItemData> query = new QueryWrapper<>();
        query.eq("user_id", user.getUserId());
        query.eq("tenant_id", user.getTenantId());
        query.eq("item_code", itemCode);
        query.eq("plat", plat);
        ItemData data = itemMapper.selectOne(query);
        if (null != data) {
            ItemsDto dto = new ItemsDto();
            BeanUtils.copyProperties(data, dto);
            itemRecord(user, dto, 2);
            return itemMapper.deleteById(data.getId());
        }
        return -1;
    }


    /**
     * 子项详情查询
     *
     * @param user 用户信息
     * @return int 结果
     */
    public ItemData detail(AuthoredUser user, String itemCode, Integer plat) {

        QueryWrapper<ItemData> query = new QueryWrapper<>();
        query.eq("plat", plat);
        query.eq("user_id", user.getUserId());
        query.eq("tenant_id", user.getTenantId());
        query.eq("item_code", itemCode);
        return itemMapper.selectOne(query);
    }


    /**
     * 判断是否已经设置为常用
     *
     * @param user     用户信息
     * @param plat     所属模块0:基础资料录入 1：报表，后续有新增则依次递增
     * @param itemCode 子项编码
     * @return boolean
     */
    public boolean checkUsed(AuthoredUser user, Integer plat, String itemCode) {

        QueryWrapper<ItemData> query = new QueryWrapper<>();
        query.eq("plat", plat);
        query.eq("user_id", user.getUserId());
        query.eq("tenant_id", user.getTenantId());
        query.eq("item_code", itemCode);
        Long result = itemMapper.selectCount(query);
        return result > 0L;


    }


    /**
     * 打开子项记录打开日志
     *
     * @param user    用户信息
     * @param itemDto 打开的子项洗洗
     * @param type    0：打开 1：取消常用 2：设置常用
     * @return int
     */
    @Transactional(transactionManager = BaseAudcDataSourceConfig.BASE_AUDC_DATASOURCE_TRANSACTION_MANAGER_BUSINESS)
    public int itemRecord(AuthoredUser user, ItemsDto itemDto, int type) {
        try {
            ItemLogData data = new ItemLogData();
            data.setId(SnowflakeIdWorker.getInstance().newId());
            data.setUserId(user.getUserId());
            data.setTenantId(user.getTenantId());
            data.setPlat(itemDto.getPlat());
            data.setCreateDate(LocalDateTime.now());
            data.setModifyDate(LocalDateTime.now());
            data.setItemCode(itemDto.getItemCode());
            data.setItemCate(itemDto.getItemCate());
            data.setType(type);

            //计算访问记录表中的信息，判断是否需要自动设置为常用,只有是访问的时候才会记录
            if (type == 0) {
                int autoResult = autoInUsed(user, itemDto);
                logger.debug("auto calc used result:" + autoResult);
            }
            //将数据插入到记录表中
            int addResult = itemLogMapper.insert(data);
            logger.debug("visit record add result:" + addResult);

            return addResult;
        } catch (Exception e) {
            logger.error("add item:[" + JsonUtils.objectToString(itemDto) + "] record type:[" + type + "] error", e);
            return -1;
        }

    }


    /**
     * 自动添加到常用
     *
     * @param user    用户信息
     * @param itemDto 打开的子项洗洗
     * @return int
     */
    @Transactional(transactionManager = BaseAudcDataSourceConfig.BASE_AUDC_DATASOURCE_TRANSACTION_MANAGER_BUSINESS)
    public int autoInUsed(AuthoredUser user, ItemsDto itemDto) {
        try {
            //判断出该子项是否取消过，取消过则不自动添加到常用
            QueryWrapper<ItemLogData> queryWrapper = new QueryWrapper<>();
            queryWrapper.eq("user_id", user.getUserId());
            queryWrapper.eq("tenant_id", user.getTenantId());
            queryWrapper.eq("item_code", itemDto.getItemCode());
            queryWrapper.eq("plat", itemDto.getPlat());
            queryWrapper.eq("type", 2);
            Long result = itemLogMapper.selectCount(queryWrapper);
            //存在过取消常用操作，则该子项不会在自动加入常用
            if (result > 0L) {
                //该常用子项已经存在取消记录
                return -1;
            }
            //没有取消记录，则查看该子项是否已经存在于常用表中，如果存在，则不添加
            QueryWrapper<ItemData> queryItemWrapper = new QueryWrapper<>();
            queryItemWrapper.eq("user_id", user.getUserId());
            queryItemWrapper.eq("tenant_id", user.getTenantId());
            queryItemWrapper.eq("item_code", itemDto.getItemCode());
            queryItemWrapper.eq("plat", itemDto.getPlat());
            Long itemExist = itemMapper.selectCount(queryItemWrapper);
            if (itemExist > 0L) {
                //该常用子项已经存在
                return -2;
            }
            //判断当前访问时间前三天是否有累计三次访问记录
            //判断出该子项是否取消过，取消过则不自动添加到常用
            QueryWrapper<ItemLogData> visitWapper = new QueryWrapper<>();
            visitWapper.eq("user_id", user.getUserId());
            visitWapper.eq("tenant_id", user.getTenantId());
            visitWapper.eq("item_code", itemDto.getItemCode());
            visitWapper.eq("plat", itemDto.getPlat());
            visitWapper.eq("type", 0);
            LocalDateTime nowTime = LocalDateTime.now();
            visitWapper.and(vw -> vw.between("create_date", nowTime.minusDays(3), nowTime));
            Long vistiResult = itemLogMapper.selectCount(visitWapper);
            //如果3天内没有达到指定的访问次数，则不会新增到常用访问中
            if (vistiResult >= 2L) {
                //将数据插入到item子项常用表中
                ItemData resultData = addItem(user, itemDto);
                return (null != resultData) ? 1 : 0;
            }
            return -3;
        } catch (Exception e) {
            logger.error("auto add itemDto:[" + JsonUtils.objectToString(itemDto) + "] error", e);
            return -4;
        }
    }

    /**
     * 删除一周前的数据，不包含当天
     *
     * @return
     */
    @Transactional(transactionManager = BaseAudcDataSourceConfig.BASE_AUDC_DATASOURCE_TRANSACTION_MANAGER_BUSINESS)
    public boolean deleteItemLogByCreateTime(long weeks) {
        LambdaQueryWrapper<ItemLogData> wrapper = new LambdaQueryWrapper<>();
        wrapper.lt(ItemLogData::getCreateDate, LocalDateTime.now().minusWeeks(weeks));
        int delete = itemLogMapper.delete(wrapper);
        if (delete < 1) {
            logger.error("delete itemlog error");
            return false;
        }
        return true;
    }
}
