package com.digiwin.athena.base.application.service.trial;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.appcore.exception.OperateException;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.appcore.util.SnowflakeIdWorker;
import com.digiwin.athena.base.application.config.BaseAudcDataSourceConfig;
import com.digiwin.athena.base.application.meta.request.trial.apply.AcpReq;
import com.digiwin.athena.base.application.meta.request.trial.apply.ProductInfo;
import com.digiwin.athena.base.application.meta.request.trial.apply.TrialApplyReq;
import com.digiwin.athena.base.infrastructure.manager.emc.BaseSendEmailService;
import com.digiwin.athena.base.infrastructure.mapper.audc.trial.TrialApplyMapper;
import com.digiwin.athena.base.infrastructure.mapper.audc.trial.TrialConfigMapper;
import com.digiwin.athena.base.infrastructure.meta.po.trial.TrialApplyData;
import com.digiwin.athena.base.infrastructure.meta.po.trial.TrialConfigData;
import net.sf.json.JSONArray;
import net.sf.json.JSONObject;
import org.apache.commons.lang.time.DateFormatUtils;
import org.apache.http.HttpStatus;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.DigestUtils;
import org.springframework.util.StringUtils;
import org.springframework.web.client.RestTemplate;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 描述
 *
 * @author gonghongxing
 * @version 1.0
 * @since 2021/08/30 14:00:31
 */
@Service
public class TrialApplyServiceImpl implements TrialApplyService {

    private static final Logger logger = LoggerFactory.getLogger(TrialApplyServiceImpl.class);

    private static final String TRIAL_AUDIT_RECEIVER_DAFAULT = "gonghx@digiwin.com";

    private static final String TRAIL_EMAIL_TEMPLATE_DEFAULT = "ATHENA_TRIAL_INFORMATION_APPLY_ZH_CN";

    private static final String ACP_APP_KEY = "brHxX3zqb24ngOphHJ8t6AHrqksR48X1AhUC9egqFE";

    /**
     * ACP请求的URL
     */
    private static final String ACP_URL = "http://misws.digiwin.com/miswsAPI//ServiceForAthena.asmx/AthenaGetACPdata";

    /**
     * 当前使用人所属的区域，根据系统部署的环境固定微软 TW，华为CN
     */
    private static final String USER_AREA = "USER_AREA";

    private static final String USE_ACP_FLAG ="USE_ACP_FLAG";

    private static final String TAIWAN_SIMPLE = "TW";
    /**
     * 数据库处理对象
     */
    private final TrialApplyMapper trialApplyMapper;

    /**
     * 邮件发送模板
     */
    private final BaseSendEmailService baseSendEmailService;

    private final TrialConfigMapper trialConfigMapper;

    private final RestTemplate restTemplate;


    /**
     * 构造和函数注入mapper对象
     *
     * @param trialApplyMapper trialApplyMapper
     */
    @Autowired
    public TrialApplyServiceImpl(TrialApplyMapper trialApplyMapper, BaseSendEmailService baseSendEmailService, TrialConfigMapper trialConfigMapper, RestTemplate restTemplate) {
        this.trialApplyMapper = trialApplyMapper;
        this.baseSendEmailService = baseSendEmailService;
        this.trialConfigMapper = trialConfigMapper;
        this.restTemplate = restTemplate;
    }

    /**
     * 新增申请数据
     *
     * @param data data
     * @return int
     */
    @Override
    @Transactional(transactionManager = BaseAudcDataSourceConfig.BASE_AUDC_DATASOURCE_TRANSACTION_MANAGER_BUSINESS)
    public int insert(TrialApplyData data) {
        data.setId(SnowflakeIdWorker.getInstance().newId());
        data.setCreateDate(LocalDateTime.now());
        data.setModifyDate(LocalDateTime.now());
        return trialApplyMapper.insert(data);
    }

    /**
     * 判断当前申请使用记录是否存在
     *
     * @param phone       手机号码
     * @param productCode 应用编码
     * @return 结果true:存在 false：不存在
     */
    public boolean isExist(String phone, String productCode) {
        QueryWrapper<TrialApplyData> countWrapper = new QueryWrapper<>();
        countWrapper.eq("phone", phone);
        countWrapper.eq("product_code", productCode);
        Long result = trialApplyMapper.selectCount(countWrapper);
        return result > 0L;
    }

    /**
     * 处理申请数据
     *
     * @param req 申请的数据
     * @return 返回信息
     */
    @Override
    @Transactional(transactionManager = BaseAudcDataSourceConfig.BASE_AUDC_DATASOURCE_TRANSACTION_MANAGER_BUSINESS)
    public int dealApplyInfo(TrialApplyReq req) {
        TrialApplyData data = new TrialApplyData();
        BeanUtils.copyProperties(req, data);
        data.setFaq(JsonUtils.objectToString(req.getFaqList()));

        //获取提交的应用信息
        List<ProductInfo> productList = req.getProductList();
        data.setProductInfo(JsonUtils.objectToString(productList));

        //数据插入数据库s
        int result = insert(data);
        if (!(result > 0)) {
            throw BusinessException.create(-1, "Submit Apply Info Error");
        }
        //创建线程去处理
        new Thread(() -> {
            //获取业务接收人员
            logger.info("Query custom :[" + req.getEmail() + "] with server:[custom server]");
            List<String> serverList = queryServerMail(req);
            logger.info("Service Recevier info:" + JsonUtils.objectToString(serverList));
            //获取试用管理员邮箱
            String trialAuditor = queryAuditEmail();
            serverList.add(trialAuditor);

            //将联系人按照“;”号分隔
            String receivers = org.apache.commons.lang.StringUtils.join(serverList.toArray(), ";");
            //获取邮件模板信息
            String templateId = queryMailTemplateId();
            //具体需要发送的邮件内容
            Map<String, Object> sendMailParams = packageSendParam(req);
            try {
                //模板ID以及发送人需要修改为配置获取
                baseSendEmailService.sendEmailWithTemplate(templateId, receivers, sendMailParams);
            } catch (Exception e) {
                logger.error("send supply info error with:" + req, e);
                throw BusinessException.create(-2, "Send Apply Mail To Worker Error");
            }

            UpdateWrapper<TrialApplyData> updateWrapper = new UpdateWrapper<>();
            updateWrapper.eq("id", data.getId()).set("recInfo", receivers);
            trialApplyMapper.update(data, updateWrapper);
        }, "Thread-Request-ACP").start();
        return 0;

    }

    /**
     * 处理邮件发送的内容
     *
     * @param req req
     * @return Map
     */
    private Map<String, Object> packageSendParam(TrialApplyReq req) {
        Map<String, Object> params = new HashMap<>();
        //获取租户信息
        params.put("company", req.getCompany());
        params.put("phone", req.getPhone());
        params.put("email", req.getEmail());
        params.put("faqList", req.getFaqList());
        params.put("productList", req.getProductList());
        return params;

    }

    /**
     * 获取试用审核的人员信息
     *
     * @return string
     */
    private String queryAuditEmail() {
        //试用的接收者
        String receiver = "TRIAL_AUDIT_RECEIVER";
        QueryWrapper<TrialConfigData> trialQuery1 = new QueryWrapper<>();
        trialQuery1.eq("conf_key", receiver);
        TrialConfigData data1 = trialConfigMapper.selectOne(trialQuery1);
        return StringUtils.isEmpty(data1.getConfVal()) ? TRIAL_AUDIT_RECEIVER_DAFAULT : data1.getConfVal();
    }

    /**
     * 获取申请的邮件模板信息
     *
     * @return String
     */
    private String queryMailTemplateId() {
        String templatePrefix = "TRAIL_EMAIL_TEMPLATE";
        //获取对应的语言环境
        String local = LocaleContextHolder.getLocale().toString();
        if (!StringUtils.isEmpty(local) && local.equalsIgnoreCase("zh_TW")) {
            local = "_ZH_TW";
        }
        if (!StringUtils.isEmpty(local) && local.equalsIgnoreCase("en_US")) {
            local = "_EN_US";
        } else {
            local = "_ZH_CN";
        }
        //需要使用的模板id
        String templateId = templatePrefix + local;
        QueryWrapper<TrialConfigData> trialQuery = new QueryWrapper<>();
        trialQuery.eq("conf_key", templateId);
        TrialConfigData data = trialConfigMapper.selectOne(trialQuery);
        return StringUtils.isEmpty(data.getConfVal()) ? TRAIL_EMAIL_TEMPLATE_DEFAULT : data.getConfVal();
    }


    /**
     * 获取客服对应的处理业务人员邮箱信息
     *
     * @param req 客户申请信息
     * @return list
     */
    private List<String> queryServerMail(TrialApplyReq req) {
        QueryWrapper<TrialConfigData> areaQuery = new QueryWrapper<>();
        areaQuery.eq("conf_key", USE_ACP_FLAG);
        TrialConfigData acpFlagData;
        acpFlagData = trialConfigMapper.selectOne(areaQuery);
        if(acpFlagData == null || StringUtils.isEmpty(acpFlagData.getConfVal())|| !"true".equals(acpFlagData.getConfVal()))
        {
            logger.info("Not Config ACP USER_ACP_FLAG");
            return new ArrayList<>();
        }

        AcpReq acpReq = new AcpReq();
        String area = queryUserArea();
        acpReq.setArea(area);
        acpReq.setEmail(req.getEmail());
        acpReq.setPhone(req.getPhone());
        acpReq.setCustName(req.getCompany());
        acpReq.setAppkey(ACP_APP_KEY);
        acpReq.setPK("AI PK INFO");
        String dateStr = DateFormatUtils.format(new Date(), "yyyyMMdd");
        acpReq.setAppsecret(DigestUtils.md5DigestAsHex((ACP_APP_KEY + dateStr).getBytes()).toLowerCase());
        JSONObject acpRet;
        try {
            acpRet = sendToAcp(acpReq);
        } catch (Exception e) {
            logger.error("query result from acp error", e);
            acpRet = new JSONObject();
        }

        return paresServers(acpRet);


    }

    /**
     * 查询当前用户的使用环境信息
     *
     * @return string
     */
    private String queryUserArea() {
        QueryWrapper<TrialConfigData> areaQuery = new QueryWrapper<>();
        areaQuery.eq("conf_key", USER_AREA);
        TrialConfigData areaData;
        try {
            areaData = trialConfigMapper.selectOne(areaQuery);
        } catch (Exception e) {
            logger.error("Query USER_AREA error,ready to use default", e);
            return TAIWAN_SIMPLE;
        }
        //没有配置数据，则使用默认值
        if (null == areaData) {
            logger.error("Query USER_AREA not Config,ready to use default");
            return TAIWAN_SIMPLE;
        }
        return areaData.getConfVal();

    }

    /**
     * 将数据发送给ACP系统
     *
     * @param acpReq 发送给ACP的请求信息
     * @return 返回的信息
     */
    private JSONObject sendToAcp(AcpReq acpReq) {
        try {
            HttpHeaders headers = new HttpHeaders();
            headers.setContentType(MediaType.TEXT_PLAIN);
            HttpEntity<String> httpEntity = new HttpEntity<>(JsonUtils.objectToString(acpReq), headers);
            ResponseEntity<JSONObject> respEntity = restTemplate.exchange(ACP_URL, HttpMethod.POST, httpEntity, JSONObject.class);
            if (HttpStatus.SC_OK == respEntity.getStatusCodeValue()) {
                if (null == respEntity.getBody()) {
                    throw new OperateException("status code:[ " + HttpStatus.SC_OK + " ],but no data returned");
                }
                return respEntity.getBody();
            } else {
                throw new OperateException(respEntity.getStatusCodeValue(), String.valueOf(respEntity.getBody()));
            }
        } catch (Exception ex) {
            throw new OperateException("Query ACP failed, exception: " + ex.getMessage());
        }
    }

    /**
     * 解析acp返回的业务联系人信息
     *
     * @param jsonObject acp返回的对象信息
     * @return list
     */
    private List<String> paresServers(JSONObject jsonObject) {
        if (jsonObject != null && jsonObject.containsKey("code")) {
            if (jsonObject.getInt("code") == 0) {
                logger.error("can not find server");
                throw BusinessException.create(-1, "find acp server error");
            }
            //业务或者电销咨询
            if (jsonObject.containsKey("sales")) {
                JSONArray sales = jsonObject.getJSONArray("sales");
                return parseServerEmail(sales);

            }
            //业务或者电销主管咨询
            else if (jsonObject.containsKey("manager")) {
                JSONArray manager = jsonObject.getJSONArray("manager");
                return parseServerEmail(manager);

            }
            //业务或者电销总经理咨询
            else if (jsonObject.containsKey("GM")) {
                JSONArray gm = jsonObject.getJSONArray("GM");
                return parseServerEmail(gm);
            }
            //市场处窗口资讯
            else if (jsonObject.containsKey("market")) {
                JSONArray market = jsonObject.getJSONArray("market");
                return parseServerEmail(market);
            }
            //如果都没有获取到对应的咨询信息，则返回异常
            else {
                throw BusinessException.create(-1, "Acp no right servers return");
            }
        }
        return new ArrayList<>();
    }

    /**
     * 遍历返回的不同业务联系人信息
     *
     * @param jsonArray 联系人信息
     * @return string
     */
    private List<String> parseServerEmail(JSONArray jsonArray) {
        List<String> emailList = new ArrayList<>();
        for (int i = 0; i < jsonArray.size(); i++) {
            JSONObject object = jsonArray.getJSONObject(i);
            if (object != null && object.containsKey("mail")) {
                String mail = object.getString("mail");
                if (StringUtils.isEmpty(mail)) {
                    continue;
                }
                emailList.add(mail);
            }
        }
        return emailList;
    }
}
