package com.digiwin.athena.base.application.service.userdefined;

import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.appcore.util.SnowflakeIdWorker;
import com.digiwin.athena.base.application.config.BaseAudcDataSourceConfig;
import com.digiwin.athena.base.application.meta.dto.userdefined.UserOperationMsgEvent;
import com.digiwin.athena.base.application.meta.request.userdefined.UserPageModelBodyDTO;
import com.digiwin.athena.base.application.meta.response.userdefined.TakeUpRecordDTO;
import com.digiwin.athena.base.application.meta.response.userdefined.UserPageModelRespDTO;
import com.digiwin.athena.base.infrastructure.constant.AudcErrorCodeEnum;
import com.digiwin.athena.base.infrastructure.constant.Constants;
import com.digiwin.athena.base.infrastructure.manager.aim.BaseAimService;
import com.digiwin.athena.base.infrastructure.manager.aim.model.MessageDO;
import com.digiwin.athena.base.infrastructure.meta.po.userdefined.UserOperationRecordDTO;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.stream.Collectors;

/**
 * 基础资料编辑态控制
 */
@Slf4j
@Service
@SuppressWarnings({"java:S2259", "java:S1192", "java:S6813"})
public class UserPageModelBaseDataService extends UserPageModelParentService {

    @Autowired
    private MessageUtils messageUtils;

    @Autowired
    private BaseAimService baseAimService;


    /**
     * 获取编辑态用户
     * @param authoredUser
     * @param pageModelBodyDTO 尝试占用人员信息
     * @return 如果当前作业被占用，则返回占用人信息，否则返回当前操作人
     */
    @Transactional(transactionManager = BaseAudcDataSourceConfig.BASE_AUDC_DATASOURCE_TRANSACTION_MANAGER_BUSINESS)
    @Override
    public UserPageModelRespDTO getEdit(AuthoredUser authoredUser, UserPageModelBodyDTO pageModelBodyDTO) {
        log.info("getEdit-start-{} ", genLogStr(authoredUser, pageModelBodyDTO));
        UserPageModelRespDTO respDTO = new UserPageModelRespDTO();
        // 暂校验白名单作业
        if (!super.isCheckItem(pageModelBodyDTO)) {
            return respDTO;
        }

        List<TakeUpRecordDTO> takeUpRecordDTOS;
        try {
            // 1、获取占用人信息
            List<UserOperationRecordDTO> userOperationRecordDTOS = super.getTakeUpUser(buildParams(authoredUser, pageModelBodyDTO));

            // 2、记录操作信息，返回占用人信息
            takeUpRecordDTOS = super.recordNoTakeUpOperation(this.buildNewOperationUser(Constants.USER_OPERATION_TAKE_UP_NO, authoredUser, pageModelBodyDTO), userOperationRecordDTOS);
        } catch (Exception e) {
            log.error("getEdit exception:{}", e);
            takeUpRecordDTOS = new ArrayList<>();
        }
        log.info("getEdit-end-{} ", genLogStr(authoredUser, pageModelBodyDTO));
        respDTO.setList(takeUpRecordDTOS);
        return respDTO;
    }


    /**
     * 更新占用时间
     * @param authoredUser
     * @param pageModelBodyDTO 尝试占用人员信息
     * @return 如果当前作业被占用，则返回占用人信息
     */
    @Transactional(transactionManager = BaseAudcDataSourceConfig.BASE_AUDC_DATASOURCE_TRANSACTION_MANAGER_BUSINESS)
    @Override
    public UserPageModelRespDTO updateEdit(AuthoredUser authoredUser, UserPageModelBodyDTO pageModelBodyDTO) {
        UserPageModelRespDTO respDTO = new UserPageModelRespDTO();
        log.info("updateEdit-start-{} ", genLogStr(authoredUser, pageModelBodyDTO));
        //暂校验白名单作业
        if(!super.isCheckItem(pageModelBodyDTO)){
            respDTO.setResult(true);
            return respDTO;
        }
        //防止并发操作
        if(!super.getLock(authoredUser,pageModelBodyDTO)){
            log.error("获取占用锁失败：{}", JsonUtils.objectToString(pageModelBodyDTO));
            throw AudcErrorCodeEnum.PAGE_MODEL_GET_EDIT.getBusinessException(String.format(messageUtils.getMessage("pageModel.getLock")));
        }

        try {
            // 1、获取占用信息
            List<UserOperationRecordDTO> takeUpUserList = super.getTakeUpUser(buildParams(authoredUser, pageModelBodyDTO));

            // 2、添加占用信息，占用失败则返回
            List<UserOperationRecordDTO> newTakeUpUserList = this.buildNewOperationUserList(Constants.USER_OPERATION_TAKE_UP_YES,authoredUser,pageModelBodyDTO);
            if(!super.recordTakeUpOperation(newTakeUpUserList, takeUpUserList)){
                // 解除并发操作
                super.releaseLock(authoredUser,pageModelBodyDTO);
                respDTO.setResult(false);
                return respDTO;
            }

            // 3、通知非占用者（包含非冲突行的其他占用者）
            List<UserOperationRecordDTO> noTakeUpList = super.queryEffectiveRecords(buildQueryNoTakeUpParams(authoredUser, pageModelBodyDTO));
            if(CollectionUtils.isEmpty(noTakeUpList)){
                respDTO.setResult(true);
                return respDTO;
            }
            List<String> userIdList = noTakeUpList.stream().map(UserOperationRecordDTO::getUserId).distinct().collect(Collectors.toList());
            log.info("用户{}占用成功，并通知其他非占用者{}",authoredUser.getUserId(),userIdList);
            sendMessageToClient(processMsgEvent(null,newTakeUpUserList, Constants.USER_OPERATION_MSG_TYPE_BASE_DATA, Constants.USER_OPERATION_MSG_CATEGORY_TAKE_UP), userIdList);
        } catch (Exception e){
            log.error("updateEdit exception:{}",e);
            respDTO.setResult(false);
            return respDTO;
        } finally {
            // 解除并发操作
            super.releaseLock(authoredUser,pageModelBodyDTO);
        }
        log.info("updateEdit-end-{} ", genLogStr(authoredUser, pageModelBodyDTO));
        respDTO.setResult(true);
        return respDTO;
    }

    /**
     * 组装msgEvent
     * @param token 定时任务需要传入租户虚拟token
     * @param newTakeUpUserList
     * @param msgType
     * @param msgCategory
     * @return
     */
    public UserOperationMsgEvent processMsgEvent(String token, List<UserOperationRecordDTO> newTakeUpUserList, String msgType, String msgCategory) {
        if(CollectionUtils.isEmpty(newTakeUpUserList)){
            log.info("processMsgEvent newTakeUpUserList is empty!");
            return null;
        }
        UserOperationRecordDTO currentUser = newTakeUpUserList.get(0);
        UserOperationMsgEvent userOperationMsgEvent = new UserOperationMsgEvent();
        userOperationMsgEvent.setPageCode(currentUser.getPageCode());
        userOperationMsgEvent.setPageType(currentUser.getPageType());
        userOperationMsgEvent.setTenantId(currentUser.getTenantId());
        userOperationMsgEvent.setTimestamp(new Date());
        userOperationMsgEvent.setId(UUID.randomUUID().toString());
        userOperationMsgEvent.setMsgType(msgType);
        userOperationMsgEvent.setMsgCategory(msgCategory);
        userOperationMsgEvent.setToken(token);
        List<TakeUpRecordDTO> takeUpRecordDTOS = new ArrayList<>();
        // 组装返回值
        newTakeUpUserList.forEach(item ->{
            TakeUpRecordDTO newItem = new TakeUpRecordDTO();
            BeanUtils.copyProperties(item,newItem);
            takeUpRecordDTOS.add(newItem);
        });
        userOperationMsgEvent.setTakeUpRecordList(takeUpRecordDTOS);
        return userOperationMsgEvent;
    }

    /**
     * 发送MQTT消息
     * @param userOperationMsgEvent
     * @param userIdList
     */
    @Override
    public void sendMessageToClient(UserOperationMsgEvent userOperationMsgEvent, List<String> userIdList) {
        if(null == userOperationMsgEvent){
            log.info("sendMessageToClient userOperationMsgEvent is null!");
            return;
        }
        //消息弹框
        try {
            MessageDO messageDO = new MessageDO();
            messageDO.setContent(userOperationMsgEvent);
            messageDO.setType(userOperationMsgEvent.getMsgType());
            messageDO.setCategory(userOperationMsgEvent.getMsgCategory());
            messageDO.setSendDate(LocalDateTime.now());
            messageDO.setGid(UUID.randomUUID().toString());
            messageDO.setTenantId(userOperationMsgEvent.getTenantId());
            baseAimService.sendMessageToClient(userOperationMsgEvent.getToken(),messageDO.getTenantId(), userIdList,messageDO);
        } catch (Exception e) {
            log.error("发送推送消息失败",e);
        }
    }

    private String genLogStr(AuthoredUser authoredUser, UserPageModelBodyDTO pageModelBodyDTO) {
        if (null == authoredUser) {
            return "";
        }

        StringBuilder sb = new StringBuilder()
                .append(authoredUser.getUserId())
                .append("-")
                .append(authoredUser.getTenantId());
        if (null != pageModelBodyDTO) {
            sb.append("-").append(pageModelBodyDTO);
        }
        return sb.toString();
    }


    /**
     * 解除占用
     * @param authoredUser
     * @param pageModelBodyDTO 尝试占用人员信息
     * @return
     */
    @Transactional(transactionManager = BaseAudcDataSourceConfig.BASE_AUDC_DATASOURCE_TRANSACTION_MANAGER_BUSINESS)
    @Override
    public Boolean deleteEdit(AuthoredUser authoredUser, UserPageModelBodyDTO pageModelBodyDTO) {
        log.info("deleteEdit-start-{}", genLogStr(authoredUser, pageModelBodyDTO));
        //暂校验白名单作业
        if(!super.isCheckItem(pageModelBodyDTO)){
            return true;
        }

        try {
            List<UserOperationRecordDTO> deleteUserOperationList = this.buildNewOperationUserList(null, authoredUser, pageModelBodyDTO);
            Map<String, Object> params = buildDeleteParams(authoredUser, pageModelBodyDTO);
            // 1、删除占用记录
            int result = super.batchDeleteTakeUpUser(params);
            if(result <= 0){
                // 删除失败，说明之前已退出（正常退出+定时任务清理退出），无需再重复发MQTT消息通知其他非占用者
                log.error("deleteEdit result <=0!");
                return true;
            }

            // 2、若删除占用记录成功，则通知非占用者（包含非冲突行的其他占用者）
            List<UserOperationRecordDTO> noTakeUpList = queryEffectiveRecords(buildQueryNoTakeUpParams(authoredUser, pageModelBodyDTO));
            if(CollectionUtils.isEmpty(noTakeUpList)){
                return true;
            }
            List<String> userIdList = noTakeUpList.stream().map(UserOperationRecordDTO::getUserId).distinct().collect(Collectors.toList());
            log.info("用户{}取消占用成功，并通知其他非占用者{}",authoredUser.getUserId(),userIdList);
            sendMessageToClient(processMsgEvent(null,deleteUserOperationList, Constants.USER_OPERATION_MSG_TYPE_BASE_DATA, Constants.USER_OPERATION_MSG_CATEGORY_RELEASE), userIdList);
        }catch (Exception e){
            log.error("deleteEdit exception:{}",e);
            return false;
        }
        log.info("deleteEdit-end-{} ", genLogStr(authoredUser, pageModelBodyDTO));
        return true;
    }

    /**
     * 前端定时上报用户状态
     * @param authoredUser
     * @param pageModelBodyDTO 占用人员信息
     * @return
     */
    @Transactional(transactionManager = BaseAudcDataSourceConfig.BASE_AUDC_DATASOURCE_TRANSACTION_MANAGER_BUSINESS)
    @Override
    public Boolean reportStatus(AuthoredUser authoredUser, UserPageModelBodyDTO pageModelBodyDTO) {
        log.info("reportStatus-start-{} ", genLogStr(authoredUser, pageModelBodyDTO));
        //暂校验白名单作业
        if(!super.isCheckItem(pageModelBodyDTO)){
            return true;
        }

        try {
            UserOperationRecordDTO userPageModeDTO = new UserOperationRecordDTO();
            userPageModeDTO.setUserId(authoredUser.getUserId());
            userPageModeDTO.setTenantId(authoredUser.getTenantId());
            userPageModeDTO.setPageType(pageModelBodyDTO.getType());
            userPageModeDTO.setPageCode(pageModelBodyDTO.getActivityCode());
            userPageModeDTO.setClientId(pageModelBodyDTO.getClientId());
            userPageModeDTO.setUpdateTime(LocalDateTime.now());
            // 更新用户操作时间
            if(super.updateUserOperationRecord(userPageModeDTO) <= 0){
                log.error("reportStatus fail,userPageModeDTO：{}", JsonUtils.objectToString(userPageModeDTO));
                return false;
            }
        }catch (Exception e){
            log.error("reportStatus exception:{}",e);
            return false;
        }
        log.info("reportStatus-end-{} ", genLogStr(authoredUser, pageModelBodyDTO));
        return true;
    }

    /**
     * 包装操作用户
     * @param authoredUser
     * @param pageModelBodyDTO
     * @return
     */
    private UserOperationRecordDTO buildNewOperationUser(String takeUp, AuthoredUser authoredUser, UserPageModelBodyDTO pageModelBodyDTO){
        UserOperationRecordDTO newOperationUser = new UserOperationRecordDTO();
        newOperationUser.setUserId(authoredUser.getUserId());
        newOperationUser.setUserName(authoredUser.getUserName());
        newOperationUser.setPageCode(pageModelBodyDTO.getActivityCode());
        newOperationUser.setTenantId(authoredUser.getTenantId());
        newOperationUser.setPageType(pageModelBodyDTO.getType());
        newOperationUser.setTakeUp(takeUp);
        newOperationUser.setClientId(pageModelBodyDTO.getClientId());
        LocalDateTime now = LocalDateTime.now();
        newOperationUser.setCreateTime(now);
        newOperationUser.setUpdateTime(now);
        return newOperationUser;
    }

    /**
     * 包装占用操作用户
     * @param authoredUser
     * @param pageModelBodyDTO
     * @return
     */
    private List<UserOperationRecordDTO> buildNewOperationUserList(String takeUp, AuthoredUser authoredUser, UserPageModelBodyDTO pageModelBodyDTO){
        List<UserOperationRecordDTO> takeUpUserList = new ArrayList<>();
        LocalDateTime now = LocalDateTime.now();
        // 取消占用，businessKeys可能为null,则需要查询表里的占用记录
        if(CollectionUtils.isEmpty(pageModelBodyDTO.getBusinessKeys())){
            return super.queryEffectiveRecords(buildQueryTakeUpForReleaseAllParams(authoredUser, pageModelBodyDTO));
        }else {
            for(String businessKey : pageModelBodyDTO.getBusinessKeys()){
                UserOperationRecordDTO newOperationUser = new UserOperationRecordDTO();
                newOperationUser.setId(SnowflakeIdWorker.getInstance().newId());
                newOperationUser.setUserId(authoredUser.getUserId());
                newOperationUser.setUserName(authoredUser.getUserName());
                newOperationUser.setPageCode(pageModelBodyDTO.getActivityCode());
                newOperationUser.setTenantId(authoredUser.getTenantId());
                newOperationUser.setPageType(pageModelBodyDTO.getType());
                newOperationUser.setTakeUp(takeUp);
                newOperationUser.setBusinessKey(businessKey);
                newOperationUser.setClientId(pageModelBodyDTO.getClientId());
                newOperationUser.setCreateTime(now);
                newOperationUser.setUpdateTime(now);
                takeUpUserList.add(newOperationUser);
            }
        }
        return takeUpUserList;
    }

    /**
     * 获取占用信息
     * @param authoredUser 人员信息
     * @param pageModelBodyDTO 占用信息参数
     * @return
     */
    @Override
    protected Map<String, Object> buildParams(AuthoredUser authoredUser,UserPageModelBodyDTO pageModelBodyDTO) {
        Map<String, Object> params = Maps.newHashMap();
        params.put("tenantId",authoredUser.getTenantId());
        params.put("pageType",pageModelBodyDTO.getType());
        params.put("pageCode",pageModelBodyDTO.getActivityCode());
        params.put("updateTime", LocalDateTime.now().minus(Constants.USER_OPERATION_TTL, ChronoUnit.SECONDS));
        if(!CollectionUtils.isEmpty(pageModelBodyDTO.getBusinessKeys())){
            params.put("businessKeys",pageModelBodyDTO.getBusinessKeys());
        }
        return params;
    }

    /**
     * 组装获取非占用信息入参
     * @param authoredUser 人员信息
     * @param pageModelBodyDTO 占用信息参数
     * @return
     */
    protected Map<String, Object> buildQueryNoTakeUpParams(AuthoredUser authoredUser,UserPageModelBodyDTO pageModelBodyDTO) {
        Map<String, Object> params = Maps.newHashMap();
        params.put("tenantId",authoredUser.getTenantId());
        params.put("pageType",pageModelBodyDTO.getType());
        params.put("pageCode",pageModelBodyDTO.getActivityCode());
        params.put("updateTime", LocalDateTime.now().minus(Constants.USER_OPERATION_TTL, ChronoUnit.SECONDS));
        return params;
    }

    /**
     * 组装删除占用信息入参
     * @param authoredUser 人员信息
     * @param pageModelBodyDTO 占用信息参数
     * @return
     */
    protected Map<String, Object> buildDeleteParams(AuthoredUser authoredUser,UserPageModelBodyDTO pageModelBodyDTO) {
        Map<String, Object> params = Maps.newHashMap();
        params.put("userId",authoredUser.getUserId());
        params.put("tenantId",authoredUser.getTenantId());
        params.put("pageType",pageModelBodyDTO.getType());
        params.put("pageCode",pageModelBodyDTO.getActivityCode());
        params.put("updateTime", LocalDateTime.now().minus(Constants.USER_OPERATION_TTL, ChronoUnit.SECONDS));
        params.put("clientId",pageModelBodyDTO.getClientId());
        if(!CollectionUtils.isEmpty(pageModelBodyDTO.getBusinessKeys())){
            params.put("businessKeys",pageModelBodyDTO.getBusinessKeys());
        }
        return params;
    }

    /**
     * 组装删除占用信息入参
     * @param authoredUser 人员信息
     * @param pageModelBodyDTO 占用信息参数
     * @return
     */
    protected Map<String, Object> buildQueryTakeUpForReleaseAllParams(AuthoredUser authoredUser,UserPageModelBodyDTO pageModelBodyDTO) {
        Map<String, Object> params = Maps.newHashMap();
        params.put("userId",authoredUser.getUserId());
        params.put("tenantId",authoredUser.getTenantId());
        params.put("pageType",pageModelBodyDTO.getType());
        params.put("pageCode",pageModelBodyDTO.getActivityCode());
        params.put("updateTime", LocalDateTime.now().minus(Constants.USER_OPERATION_TTL, ChronoUnit.SECONDS));
        params.put("clientId",pageModelBodyDTO.getClientId());
        params.put("takeUp",Constants.USER_OPERATION_TAKE_UP_YES);
        return params;
    }

    /**
     * 校验非占用人员记录是否存在
     * @param authoredUser 人员信息
     * @return
     */
    @Override
    protected Map<String, Object> buildCheckNoTakeUpParams(UserOperationRecordDTO authoredUser) {
        Map<String, Object> params = Maps.newHashMap();
        params.put("client_id",authoredUser.getClientId());
        params.put("take_up",Constants.USER_OPERATION_TAKE_UP_NO);
        return params;
    }

    @Override
    public String getLockKey(AuthoredUser authoredUser,UserPageModelBodyDTO pageModelBodyDTO){
        StringBuilder lockBuilder = new StringBuilder();
        lockBuilder.append(KEY_PREFIX);
        lockBuilder.append(authoredUser.getTenantId());
        lockBuilder.append(UNDER_PLACE);
        lockBuilder.append(pageModelBodyDTO.getType());
        lockBuilder.append(UNDER_PLACE);
        lockBuilder.append(pageModelBodyDTO.getActivityCode());
        lockBuilder.append(UNDER_PLACE);
        return lockBuilder.toString();
    }


    /**
     * 获取占用key
     *
     * @param authoredUser
     * @param pageModelBodyDTO
     * @return
     */
    @Override
    public String getLockKey(AuthoredUser authoredUser, UserPageModelBodyDTO pageModelBodyDTO, String businessKey) {
        StringBuilder lockBuilder = new StringBuilder();
        lockBuilder.append(KEY_PREFIX);
        lockBuilder.append(authoredUser.getTenantId());
        lockBuilder.append(UNDER_PLACE);
        lockBuilder.append(pageModelBodyDTO.getType());
        lockBuilder.append(UNDER_PLACE);
        lockBuilder.append(pageModelBodyDTO.getActivityCode());
        lockBuilder.append(UNDER_PLACE);
        lockBuilder.append(businessKey);
        return lockBuilder.toString();
    }

}
