package com.digiwin.athena.base.application.service.importstatistics;

import com.digiwin.athena.aim.sdk.meta.dto.response.AccessibleRespDTO;
import com.digiwin.athena.aim.sdk.meta.dto.response.ActivityAccessible;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.atdm.audc.CommonAudcService;
import com.digiwin.athena.atdm.constant.ErrorCodeEnum;
import com.digiwin.athena.atdm.iam.CommonUserService;
import com.digiwin.athena.atdm.importstatistics.application.ApplicationRelation;
import com.digiwin.athena.atdm.importstatistics.common.api.DapResponse;
import com.digiwin.athena.atdm.importstatistics.dto.ActivityStatisticsDTO;
import com.digiwin.athena.atdm.importstatistics.dto.BatchImportListRespDTO;
import com.digiwin.athena.atdm.importstatistics.dto.DownloadBaseDataDTO;
import com.digiwin.athena.atdm.importstatistics.dto.DownloadBaseDataParamDTO;
import com.digiwin.athena.atdm.importstatistics.dto.DownloadExportFileParamDTO;
import com.digiwin.athena.atdm.importstatistics.dto.DownloadHistoryProjectTaskParamDTO;
import com.digiwin.athena.atdm.importstatistics.dto.DownloadTemplateDTO;
import com.digiwin.athena.atdm.importstatistics.dto.DownloadTemplateReqDTO;
import com.digiwin.athena.atdm.importstatistics.dto.ErrorTableDTO;
import com.digiwin.athena.atdm.importstatistics.dto.ExportBasicDataRespDTO;
import com.digiwin.athena.atdm.importstatistics.dto.ExportHeaderRespDTO;
import com.digiwin.athena.atdm.importstatistics.dto.ExportStatisticsDTO;
import com.digiwin.athena.atdm.importstatistics.dto.GetRecordsParamDTO;
import com.digiwin.athena.atdm.importstatistics.dto.ImportStatisticsDTO;
import com.digiwin.athena.atdm.importstatistics.dto.UploadParamDTO;
import com.digiwin.athena.atdm.importstatistics.dto.UserStatisticsDTO;
import com.digiwin.athena.atdm.importstatistics.entity.CellTypeContainer;
import com.digiwin.athena.atdm.importstatistics.entity.ImportStatistics;
import com.digiwin.athena.atdm.importstatistics.entity.valueobject.GetActionLocaleResponseDTO;
import com.digiwin.athena.atdm.importstatistics.esp.api.EspBody;
import com.digiwin.athena.atdm.importstatistics.esp.api.EspResponse;
import com.digiwin.athena.atdm.importstatistics.kg.DataEntryKgService;
import com.digiwin.athena.atdm.importstatistics.util.excel.ExcelTypeEnum;
import com.digiwin.athena.atdm.semc.CommonTddService;
import com.digiwin.athena.atdm.semc.dto.EncryptedConfigDTO;
import com.digiwin.athena.atdm.semc.dto.EncryptedConfigVO;
import com.digiwin.athena.base.application.service.importstatistics.application.ApplicationService;
import com.digiwin.athena.base.application.service.importstatistics.service.DataEntryErrorHandlerService;
import com.digiwin.athena.base.application.service.importstatistics.service.DataEntryEspService;
import com.digiwin.athena.base.application.service.importstatistics.service.helpler.ExcelHelper;
import com.digiwin.athena.base.application.service.importstatistics.service.impl.DataEntryExportStatisticsDomainService;
import com.digiwin.athena.base.application.service.importstatistics.service.impl.DataEntryImportStatisticsDomainService;
import com.digiwin.athena.base.application.service.importstatistics.service.impl.DataEntryMetaDataService;
import com.digiwin.athena.base.application.service.importstatistics.worker.ExportFileMsg;
import com.digiwin.athena.base.infrastructure.manager.abt.AbtService;
import com.digiwin.athena.base.sdk.common.application.util.MessageUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.multipart.MultipartFile;
import com.digiwin.athena.atdm.importstatistics.dto.ExportBatchRecord;

import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * @ClassName BaseDataEntryServiceImpl
 * @Description 基础数据录入应用服务
 * @Author zhuangli
 * @Date 2021/4/2 16:26
 * @Version 1.0
 **/
@Slf4j
@Service
public class BaseDataEntryApplicationServiceImpl {

    @Autowired
    DataEntryImportStatisticsDomainService importStatisticsDomainService;

    @Autowired
    DataEntryExportStatisticsDomainService exportStatisticsDomainService;

    @Autowired
    DataEntryMetaDataService metaDataService;
    @Autowired
    DataEntryEspService espService;
    @Autowired
    ApplicationService applicationService;
    @Autowired
    @Qualifier("rabbitDataEntryErrorHandlerServiceImpl")
    DataEntryErrorHandlerService errorHandlerService;

    @Autowired
    ExcelHelper excelHelper;

    @Autowired
    AbtService abtService;

    @Autowired
    CommonTddService tddService;

    @Autowired
    private CommonUserService userService;

    @Autowired
    private DataEntryKgService kgService;

    @Autowired
    private HistoryProjectTaskServiceImpl historyProjectTaskService;

    /**
     * 处理上传
     */
    public DapResponse dealUpload(UploadParamDTO uploadParam, MultipartFile file){
        String appCode = getAppCode("activity", uploadParam.getActivityId());
        uploadParam.setApplication(appCode);
        abtService.handleUpload(uploadParam, file);
        return DapResponse.ok(MessageUtil.getMessage("delivery.success"));
    }

    public void republish(UploadParamDTO uploadParamDTO) {
        String appCode = getAppCode("activity", uploadParamDTO.getActivityId());
        uploadParamDTO.setApplication(appCode);
        abtService.republish(uploadParamDTO);
    }

    /**
     * 异步上传调用abt服务
     * @param uploadParamDTO 多个上传参数
     */
    public void uploadAsync(List<UploadParamDTO> uploadParamDTO) {
        abtService.uploadAsync(uploadParamDTO);
    }

    @Autowired
    private CommonAudcService commonAudcService;

    public List<ImportStatisticsDTO> getRecordsByUserId(AuthoredUser user,String locale, GetRecordsParamDTO getRecordsParam) {
        List<ImportStatisticsDTO> records = abtService.getRecords(user, locale, getRecordsParam.getGetRecordsNum(), getRecordsParam.getActivityName(), getRecordsParam.getStartTime(), getRecordsParam.getEndTime(), getRecordsParam.getType(), getRecordsParam.getState(), getRecordsParam.getOffset());
        List<String> activityIds = records.stream().map(ImportStatisticsDTO::getActivityId).collect(Collectors.toList());
        List<AccessibleRespDTO> accessibleList = commonAudcService.checkTypeActivitiesAccessible(activityIds);
        Map<String, String> map = new HashMap<>();
        if (accessibleList != null) {
            map = accessibleList.stream().flatMap(entry -> (entry.getActivityAccessibleList()).stream())
                    .collect(Collectors.toMap(
                            ActivityAccessible::getTmActivityId,
                            ActivityAccessible::getAccess,
                            (existing, replacement) -> replacement
                    ));
        }
        Map<String, String> finalMap = map;
        records.forEach(e->e.setAccess(finalMap.get(e.getActivityId())));
        return records;
    }

    public List<ExportStatisticsDTO> getExportRecordsByUserId(AuthoredUser user,String locale, GetRecordsParamDTO getRecordsParamDTO) {
        List<ExportStatisticsDTO> exportRecords = abtService.getExportRecords(user, locale, getRecordsParamDTO);
        List<String> activityIds = exportRecords.stream().flatMap(record -> {
            List<String> ids = new ArrayList<>();
            ids.add(record.getActivityId());
            if (record.getExportBatchRecords() != null) {
                ids.addAll(record.getExportBatchRecords().stream()
                        .map(ExportBatchRecord::getCode)
                        .collect(Collectors.toList()));
            }
            return ids.stream();
        }).collect(Collectors.toList());
        List<AccessibleRespDTO> accessibleList = commonAudcService.checkTypeActivitiesAccessible(activityIds);
        Map<String, String> map = new HashMap<>();
        if (accessibleList != null) {
            map = accessibleList.stream().flatMap(entry -> (entry.getActivityAccessibleList()).stream())
                    .collect(Collectors.toMap(
                            ActivityAccessible::getTmActivityId,
                            ActivityAccessible::getAccess,
                            (existing, replacement) -> replacement
                    ));
        }
        Map<String, String> finalMap = map;
        for (ExportStatisticsDTO exportStatisticsDTO : exportRecords) {
            exportStatisticsDTO.setAccess(finalMap.get(exportStatisticsDTO.getActivityId()));
            List<ExportBatchRecord> exportBatchRecords = exportStatisticsDTO.getExportBatchRecords();
            if (exportBatchRecords != null) {
                for (ExportBatchRecord exportBatchRecord : exportBatchRecords) {
                    exportBatchRecord.setAccess(finalMap.get(exportBatchRecord.getCode()));
                }
            }
        }
        return exportRecords;
    }

    public ErrorTableDTO getErrorTableByMasterId(String masterId, String locale) {
        return abtService.getErrorTable(masterId, locale);
    }

    public void downloadErrorTable(String masterId, String locale, Set<String> requiredFields, HttpServletResponse response) {
        ImportStatistics importStatistics = new ImportStatistics();
        ImportStatisticsDTO importStatisticsDTO = abtService.queryImportStatistics(masterId);
        if (null == importStatisticsDTO) {
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0063.getErrCode(),MessageUtil.getMessage("delivery.dataNotFound"));
        }
        BeanUtils.copyProperties(importStatisticsDTO, importStatistics);
        if (0 != importStatistics.getProcessingNum()) {
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0064.getErrCode(),MessageUtil.getMessage("delivery.existRunningData"));
        }
        if (0 == importStatistics.getFailedNum()) {
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0065.getErrCode(),MessageUtil.getMessage("delivery.notFoundErrorData"));
        }
        if (StringUtils.isEmpty(importStatistics.getFailedUrl())) {
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0066.getErrCode(),MessageUtil.getMessage("delivery.notFoundErrorExcel"));
        }
        List<Map> headers = metaDataService.getHeaders(importStatistics.getActionId(), AppAuthContextHolder.getContext().getAuthoredUser().getToken(), locale);
        errorHandlerService.downloadErrorTable(importStatistics, headers, requiredFields, response);
    }

    public void downloadErrorTable(String masterId, String locale, HttpServletResponse response) {
        downloadErrorTable(masterId, locale, null, response);
    }

    /**
     * 批量下载模版
     * @param params 下载模版参数
     * @param response 响应
     * @param locale 语言
     */
    public void downloadTemplateBatch(List<DownloadTemplateReqDTO> params, HttpServletResponse response,String locale) {
        metaDataService.downloadTemplateBatch(params,response,AppAuthContextHolder.getContext().getAuthoredUser().getToken(),locale);
    }

    public void downloadTemplate(String actionId, String locale, String fileName, DownloadTemplateDTO downloadTemplateDTO, HttpServletResponse response) {
        metaDataService.downloadTemplate(actionId, AppAuthContextHolder.getContext().getAuthoredUser().getToken(), locale, fileName, downloadTemplateDTO, response);
    }

    public void downloadTemplate(String actionId, String locale, String fileName, HttpServletResponse response) {
        metaDataService.downloadTemplate(actionId, AppAuthContextHolder.getContext().getAuthoredUser().getToken(), locale, fileName, response);
    }

    public UserStatisticsDTO getProcessingNum(String userId,String tenantId) {
        return abtService.getProcessingNum(userId, tenantId);
    }

    public UserStatisticsDTO getExportingNum(String userId,String tenantId, String type, List<String> types) {
        return abtService.getExportingNum(userId, tenantId, type, types);
    }

    public ImportStatisticsDTO getImportStatistics(String masterId) {
        return abtService.queryImportStatistics(masterId);
    }

    public ActivityStatisticsDTO getActivityStatistics(String activityCode, String userId) {
        return abtService.getActivityStatistics(activityCode,userId);
    }

    /**
     * @return
     * @Author zhuangli
     * @Description 导出基础数据为excel, 支持数据量为60000条.
     * 1.根据actionid调用espapi获取需要导出的数据
     * 2.元数据api获得column名称并和返回数据匹配组装excel数据结构
     * 3.将数据写入excel并导出
     * @Date 13:31 2021/12/1
     * @Param
     **/
    public void downloadBaseData(String actionId, DownloadBaseDataDTO downloadBaseDataDTO, String locale,
                                 HttpServletResponse response) {
        AuthoredUser athenaUser = AppAuthContextHolder.getContext().getAuthoredUser();
        String productName = metaDataService.getProductName(athenaUser.getToken(),
                actionId.substring(actionId.contains("esp_") ? 4 : 0));
        //获取元数据信息
        GetActionLocaleResponseDTO metadataDTO = metaDataService.getActionMetaData(actionId, athenaUser.getToken(),
                locale);
        //response主key
        String mainKey = metadataDTO.getResponse().getData().getData_name();
        String mainKeyDescription = metadataDTO.getResponse().getData().getDescription();
        //调用espapi获取导出数据
        ExportFileMsg exportFileMsg = new ExportFileMsg();
        exportFileMsg.setLocale(locale);
        exportFileMsg.setTenantId(athenaUser.getTenantId());
        exportFileMsg.setUserToken(athenaUser.getToken());
        exportFileMsg.setActionId(actionId);
        EspResponse espResponse = espService.getBaseData(downloadBaseDataDTO.getActionParas(), exportFileMsg, productName, null);
        if (!espResponse.isOK()) {
            log.error("downloadBaseData actionId:{},name:{},locale:{},userToken:{}", actionId, downloadBaseDataDTO,
                    locale, athenaUser.getToken());
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0067.getErrCode(),MessageUtil.getMessage("delivery.exportDataError"));
        }
        EspBody espBody = (EspBody) espResponse.getData();
        List<Map<String,Object>> tableList = (List<Map<String,Object>>)(espBody.getStd_data().getParameter()).get(mainKey);
        //获取responseKey类型
        Map<String, CellTypeContainer> cellTypeContainerMap = metaDataService.getResponseCellTypeContainers(metadataDTO, athenaUser.getToken(), locale);
        //根据actionId获取excel列名
        List<String> keyList = new ArrayList<>();
        Map<String, String> headers = metaDataService.getResponseHeaders(metadataDTO, keyList);
        //处理下载基础数据
        try {
            importStatisticsDomainService.handleDownloadBaseData(locale, downloadBaseDataDTO.getName(), keyList, headers, cellTypeContainerMap,
                    tableList, response, ExcelHelper.getSheetName(mainKey,mainKeyDescription,true));
        } catch (UnsupportedEncodingException e) {
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0068.getErrCode(),MessageUtil.getMessage("delivery.downloadError"));
        }
    }

    /**
     * @Description 导出基础数据为excel, 支持数据量为60000条.
     * 1.根据actionid调用espapi获取需要导出的数据
     * 2.元数据api获得column名称并和返回数据匹配组装excel数据结构
     * 3.将数据写入excel并导出
     * @Date 13:31 2023/7/26
     * @Param
     **/
    public void downloadBaseData(DownloadBaseDataParamDTO downloadBaseDataParamDTO, HttpServletResponse response) {
        response.setContentType(ExcelTypeEnum.XLSX.value());
        String encoding = "UTF-8";
        String formFileName = downloadBaseDataParamDTO.getActionInfo().getFileName() + ExcelHelper.EXCEL_SUFFIX;
        try (ServletOutputStream out = response.getOutputStream()) {
            String appCode = getAppCode("activity", downloadBaseDataParamDTO.getActionInfo().getCode());
            downloadBaseDataParamDTO.setApplication(appCode);
            formFileName = URLEncoder.encode(formFileName, "UTF-8");
            response.setCharacterEncoding(encoding);
            // 解决URLEncoder.encode方法会把空格变成加号（+）在前台页面显示的时候会多出加号的问题
            formFileName = formFileName.replaceAll("\\+",  "%20");
            response.addHeader("Content-Disposition", "attachment;filename=" + formFileName);
            response.addHeader("Access-Control-Expose-Headers", "Content-Disposition");
            byte[] fileBytes = abtService.downloadBaseData(downloadBaseDataParamDTO);
            //            if(null == fileBytes || fileBytes.length ==0){
            //                log.error("导出的文件流为空！");
            //                return;
            //            }
            out.write(fileBytes, 0, fileBytes.length);
            out.flush();
        } catch (IOException e) {
            log.error("downloadBaseData downloadBaseDataParamDTO:{},e:{}", downloadBaseDataParamDTO, e);
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0067.getErrCode(),MessageUtil.getMessage("delivery.exportDataError"));
        }
    }

    public void downloadExportFile(DownloadExportFileParamDTO downloadExportFileParamDTO, HttpServletResponse response){
        ExportStatisticsDTO exportStatisticsDTO = abtService.queryExportStatistics(downloadExportFileParamDTO);
        if (null == exportStatisticsDTO) {
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0063.getErrCode(),MessageUtil.getMessage("delivery.dataNotFound"));
        }
        if (StringUtils.isEmpty(exportStatisticsDTO.getFileUrl())) {
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0066.getErrCode(),MessageUtil.getMessage("delivery.notFoundExportExcel"));
        }
        Date now = new Date();
        if(Objects.nonNull(exportStatisticsDTO.getExpireTime()) && now.after(exportStatisticsDTO.getExpireTime())){
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0117.getErrCode(),MessageUtil.getMessage("delivery.exportExcelIsExpire"));
        }
        exportStatisticsDomainService.downloadExportFile(exportStatisticsDTO,response);
    }

    public Boolean retryDownload(DownloadExportFileParamDTO downloadExportFileParamDTO) {
        ExportStatisticsDTO exportStatisticsDTO = abtService.queryExportStatistics(downloadExportFileParamDTO);
        if (exportStatisticsDTO.getExportParams()==null || "".equals(exportStatisticsDTO.getExportParams())) {
            return true;
        }
        // 0:业务数据录入; 1:历史项目任务；2:任务；3:项目 4:历史项目任务明细
        if ("4".equals(exportStatisticsDTO.getType())) {
            DownloadHistoryProjectTaskParamDTO downloadHistoryProjectTaskParamDTO = JsonUtils.jsonToObject(exportStatisticsDTO.getExportParams(), DownloadHistoryProjectTaskParamDTO.class);
            historyProjectTaskService.startDownloadHistoryData(downloadHistoryProjectTaskParamDTO);
        } else {
            DownloadBaseDataParamDTO downloadBaseDataParamDTO = JsonUtils.jsonToObject(exportStatisticsDTO.getExportParams(), DownloadBaseDataParamDTO.class);
            startDownload(downloadBaseDataParamDTO);
        }
        abtService.updateRetryDownloadState(downloadExportFileParamDTO.getMasterId());
        return true;
    }


    /**
     * 获取导出表头
     * @param actionId 导出actionId
     * @return 导出表头
     */
    public ExportHeaderRespDTO getTableHeader(String actionId) {
        return abtService.getTableHeader(actionId);
    }



    
    public List<BatchImportListRespDTO> getImportAndExportConfig() {
        return abtService.getImportBasicDataList();
    }

    public List<ExportBasicDataRespDTO> getExportBasicDataList() {
        return abtService.getExportBasicDataList();
    }



    public void startDownloadBatch(List<DownloadBaseDataParamDTO> params){
        params.forEach(this::startDownload);
    }

    /**
     * @Description 发起异步下载（目前不需要添加事物）
     * @Date 13:31 2023/9/20
     * @Param
     **/
    public Boolean startDownload(DownloadBaseDataParamDTO downloadBaseDataParamDTO) {
        //从tdd查询出加密配置
        String code = downloadBaseDataParamDTO.getActionInfo().getCode();
        // sheet暂时不支持加密
        if("split".equals(downloadBaseDataParamDTO.getExportMode())){
            EncryptedConfigDTO encryptedConfigDTO = new EncryptedConfigDTO();
            encryptedConfigDTO.setActivityCode(code);
            EncryptedConfigVO encryptedConfig = tddService.getEncryptedConfig(encryptedConfigDTO);
            AuthoredUser authoredUser = AppAuthContextHolder.getContext().getAuthoredUser();
            AuthoredUser proxyAuthoredUser = AppAuthContextHolder.getContext().getProxyAuthoredUser();
            String token = Objects.nonNull(proxyAuthoredUser) ? proxyAuthoredUser.getToken(): authoredUser.getToken();
            //查询出内外部用户 0：内部用户；1：外部用户
            Integer type = userService.queryUserType(authoredUser.getUserId(), token);
            if (Objects.nonNull(encryptedConfig) && 1 == encryptedConfig.getStatus() && Objects.nonNull(encryptedConfig.getScene()) && Objects.nonNull(encryptedConfig.getScope())){
                EncryptedConfigVO.Scope scope = encryptedConfig.getScope();
                //外部员工
                Boolean externalEmployee = Objects.nonNull(scope.getExternalEmployee()) && scope.getExternalEmployee();
                //内部员工
                Boolean internalEmployee = Objects.nonNull(scope.getInternalEmployee()) && scope.getInternalEmployee();

                EncryptedConfigVO.Scene scene = encryptedConfig.getScene();
                //附件
                Boolean download = Objects.nonNull(scene.getDownload()) && scene.getDownload();

                if (download && ((type == 0 && internalEmployee) || (type == 1 && externalEmployee))) {
                    EncryptedConfigVO.Rule rule = encryptedConfig.getRule();
                    Integer passwordLength = rule.getDefenceLength();
                    Integer passwordComplexity = rule.getDefenceComplexity();
                    //加密 调用tdd生成密码
                    String password = tddService.makePassword(passwordComplexity, passwordLength);
                    downloadBaseDataParamDTO.getActionInfo().setDefence(password);
                }
            }
        }
        String appCode = getAppCode("activity", code);
        downloadBaseDataParamDTO.setApplication(appCode);
       return abtService.startDownload(downloadBaseDataParamDTO);
    }

    private String getAppCode(String type, String code) {
        if (StringUtils.isEmpty(code)) {
            return null;
        }
        String appCode = null;
        try {
            DapResponse dapResponse = kgService.getApplicationByTypeAndCode(type, code);
            if (dapResponse.getResponse() != null && !"".equals(dapResponse.getResponse())) {
                ApplicationRelation applicationRelation = JsonUtils.jsonToObject(JsonUtils.objectToString(dapResponse.getResponse()), ApplicationRelation.class);
                appCode = applicationRelation.getApplication();
            }
        } catch (Exception e) {
            log.error("getAppCode error:{}", e);
        }
        return appCode;
    }
}
