package com.digiwin.athena.base.application.config;

import com.digiwin.athena.appcore.exception.GlobalExceptionHandler;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.google.common.eventbus.AsyncEventBus;
import org.apache.commons.lang.StringUtils;
import org.springframework.context.MessageSource;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.http.converter.HttpMessageConverter;
import org.springframework.http.converter.json.MappingJackson2HttpMessageConverter;
import org.springframework.web.servlet.LocaleResolver;
import org.springframework.web.servlet.config.annotation.CorsRegistry;
import org.springframework.web.servlet.config.annotation.InterceptorRegistry;
import org.springframework.web.servlet.config.annotation.WebMvcConfigurationSupport;
import org.springframework.web.servlet.i18n.AcceptHeaderLocaleResolver;
import org.springframework.web.servlet.i18n.LocaleChangeInterceptor;

import javax.servlet.http.HttpServletRequest;
import java.util.List;
import java.util.Locale;

@Configuration
public class CustomerWebMvcConfigurationSupport extends WebMvcConfigurationSupport {
    @Override
    protected void extendMessageConverters(List<HttpMessageConverter<?>> converters) {
        for (HttpMessageConverter<?> httpMessageConverter : converters) {
            if (MappingJackson2HttpMessageConverter.class.isInstance(httpMessageConverter)) {
                ((MappingJackson2HttpMessageConverter) httpMessageConverter).setObjectMapper(JsonUtils.createObjectMapper());
            }
        }
    }

    @Bean
    MessageUtils messageUtils(MessageSource messageSource, AsyncEventBus eventBus) {
        return new MessageUtils(messageSource);
    }

    /**
     * 全局异常处理器
     *
     * @return
     */
    @Bean
    GlobalExceptionHandler globalExceptionHandler() {
        return new GlobalExceptionHandler();
    }

    /**
     * 允许跨域访问
     *
     * @param registry
     */
    @Override
    public void addCorsMappings(CorsRegistry registry) {
        registry.addMapping("/**")
                .allowedOrigins("*")
                .allowedMethods("POST", "GET", "PUT", "OPTIONS", "DELETE")
                .maxAge(3600)
                .allowCredentials(true);
    }

    /**
     * 根据参数lang切换多语言
     *
     * @param registry
     */
    @Override
    public void addInterceptors(InterceptorRegistry registry) {
        LocaleChangeInterceptor localeInterceptor = new LocaleChangeInterceptor();
        localeInterceptor.setParamName("lang");
        registry.addInterceptor(localeInterceptor);
    }

    /**
     * 默认解析器 其中locale表示默认语言
     */
    @Bean
    public LocaleResolver localeResolver() {
        AcceptHeaderLocaleResolver localeResolver = new AcceptHeaderLocaleResolver() {
            @Override
            public Locale resolveLocale(HttpServletRequest request) {
                if (StringUtils.isNotBlank(request.getHeader("locale"))) {
                    return Locale.forLanguageTag(request.getHeader("locale").replace('_', '-'));
                }
                return this.getDefaultLocale();
            }
        };

        localeResolver.setDefaultLocale(Locale.getDefault());
        return localeResolver;
    }
}
