package com.digiwin.athena.base.application.manager.dmc;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.auth.GlobalConstant;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.appcore.util.RetryableAction;
import com.digiwin.athena.appcore.util.RetryableOperationToolkit;
import com.digiwin.athena.base.application.config.DmcProperties;
import com.digiwin.athena.base.application.meta.request.attachment.DmcAccount;
import com.digiwin.athena.base.application.meta.request.attachment.ShareAttachmentDTO;
import com.digiwin.athena.base.application.meta.request.attachment.UploadParamDTO;
import com.digiwin.athena.base.application.meta.response.attachment.ShareAttachmentRespDTO;
import com.digiwin.athena.base.application.meta.response.attachment.UploadAttachmentRespDTO;
import com.digiwin.athena.base.application.util.CommonInputStreamResource;
import com.digiwin.athena.base.infrastructure.constant.AamErrorCodeEnum;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.tomcat.util.ExceptionUtils;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;
import org.springframework.web.client.RestTemplate;

import javax.annotation.Resource;
import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.TimeUnit;

/**
 * DmcServiceImpl Description
 *
 * @author majianfu
 * @date 2022/3/7
 * @since
 */
@Service
@Slf4j
public class DmcServiceImpl implements DmcService, InitializingBean, DisposableBean {
    @Resource
    private DmcProperties dmcProperties;

    @Resource
    private RestTemplate restTemplate;

    private Cache<String, String> dmcTokenCache;

    @Override
    public void afterPropertiesSet() throws Exception {
        dmcTokenCache = CacheBuilder.newBuilder()
                // 最多缓存1w个键值对
                .maximumSize(10000)
                // 过期时间为1小时
                .expireAfterWrite(1, TimeUnit.HOURS).build();
    }

    @Override
    public void destroy() throws Exception {
        dmcTokenCache.invalidateAll();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String login(String account, String password) {
        String url = dmcProperties.getUri() + "/api/dmc/v1/auth/login";

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        Map<String, String> params = new HashMap<>();
        if (StringUtils.isBlank(account) || StringUtils.isBlank(password)) {
            params.put("username", dmcProperties.getUserName());
            params.put("pwdhash", dmcProperties.getPwdHash());
        } else {
            params.put("username", account);
            params.put("pwdhash", password);
        }

        HttpEntity<Map> httpEntity = new HttpEntity<>(params, headers);
        try {
            ResponseEntity<Map<String, String>> respEntity = this.restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<Map<String, String>>() {
            });
            if (HttpStatus.OK.value() == respEntity.getStatusCodeValue()) {
                return respEntity.getBody().get("userToken");
            } else {
                log.error("dmcLogin: login dmc failed, params: {}, response: {}", params, respEntity.getBody());
                throw BusinessException.create(respEntity.getStatusCodeValue(), "login dmc failed, error: " + JsonUtils.objectToString(respEntity.getBody()));
            }
        } catch (Exception ex) {
            if (ex instanceof BusinessException) {
                throw ex;
            }

            log.error("dmcLogin: login dmc failed, params: {}, error: {}", params, ex);
            throw BusinessException.create("login dmc failed, error: " + ex.getMessage());
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void deleteAttachment(String token, Collection<String> dirIds, Collection<String> fileIds) {
        String url = dmcProperties.getUri() + "/api/dmc/v1/buckets/Athena/files/delete/batch";

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.set(GlobalConstant.IAM_USER_TOKEN, token);

        Map<String, Object> params = new HashMap<>();
        params.put("dirIds", dirIds);
        params.put("fileIds", fileIds);
        HttpEntity<Map> httpEntity = new HttpEntity<>(params, headers);
        try {
            ResponseEntity<String> respEntity = this.restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<String>() {
            });
            if (HttpStatus.OK.value() == respEntity.getStatusCodeValue()) {
                return;
            } else {
                log.error("deleteAttachment: delete dmc file failed, params: {}, response: {}", params, respEntity.getBody());
                throw BusinessException.create(respEntity.getStatusCodeValue(), "delete dmc file failed, error: " + JsonUtils.objectToString(respEntity.getBody()));
            }
        } catch (Exception ex) {
            if (ex instanceof BusinessException) {
                throw ex;
            }
            log.error("deleteAttachment: delete dmc file failed, params: {}, error: {}", params, ex);
            throw BusinessException.create("delete dmc file failed, error: " + ex.getMessage());
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void loginAndDeleteAttachment(Collection<String> dirIds, Collection<String> fileIds, DmcAccount dmcAccount) {
        // 获取当前登录用户信息
        AuthoredUser authoredUser = AppAuthContextHolder.getContext().getAuthoredUser();

        // 最大重试次数为1,，即：可发送2次请求
        int maxRetryTimes = 1;
        RetryableOperationToolkit.operate(maxRetryTimes, new RetryableAction<Boolean>() {
            @Override
            public Boolean active(int retryTime) {
                // retryTime == 0，为第一次请求，无需刷新缓存；retryTime > 0，即因为access_token过期，需刷新缓存重新获取access_token
                boolean flushCache = 0 == retryTime ? false : true;
                String dmcToken = dmcTokenCache.getIfPresent(authoredUser.getToken());
                if (flushCache || StringUtils.isBlank(dmcToken)) {
                    // 先清除缓存，再请求dmc获取token后，更新缓存
                    dmcTokenCache.invalidate(authoredUser.getToken());
                    if (null == dmcAccount) {
                        dmcToken = DmcServiceImpl.this.login(null, null);
                    } else {
                        dmcToken = DmcServiceImpl.this.login(dmcAccount.getAccount(), dmcAccount.getPassword());
                    }
                    dmcTokenCache.put(authoredUser.getToken(), dmcToken);
                }

                try {
                    DmcServiceImpl.this.deleteAttachment(dmcToken, dirIds, fileIds);
                    // 请求成功，不需要重试
                    return false;
                } catch (BusinessException ex) {
                    // 401-token过期 且 没到最大重试次数，则需要重试
                    if (ex.getCode() == HttpStatus.UNAUTHORIZED.value() && retryTime <= maxRetryTimes) {
                        return true;
                    }
                    // 其他异常码 or 达到最大重试次数，抛出异常
                    else {
                        throw ex;
                    }
                }
            }

            @Override
            public boolean needRetry(Boolean result) {
                return result;
            }
        });
    }

    /**
     * 上传附件
     *
     * @return
     */
    @Override
    public UploadAttachmentRespDTO uploadAttachment(UploadParamDTO uploadParamDTO) {
        try {
            //登录token
            AuthoredUser authoredUser = AppAuthContextHolder.getContext().getAuthoredUser();
            String dmcToken = uploadParamDTO.getToken();
            if (StringUtils.isBlank(dmcToken)) {
                dmcToken = getAttachmentToken(dmcTokenCache.getIfPresent(authoredUser.getToken()), authoredUser);
            }

            HttpHeaders headers = new HttpHeaders();
            MultiValueMap<String, Object> param = new LinkedMultiValueMap<>();
            InputStream inputStream = uploadParamDTO.getInputStream();
            //会产生临时文件需要删除，此方法废除
//            MultipartFile fileResult = FileUtil.inputStreamToMultipartFile(StringUtils.isNotBlank(uploadParamDTO.getDownloadName())? uploadParamDTO.getDownloadName() : "下载", inputStream);
//            File file = FileUtil.multipartFileToFile(fileResult);
//            FileSystemResource resource = new FileSystemResource(file);

            String uploadUrl = dmcProperties.getUri() + "/api/dmc/v2/file/Athena/upload";
            Map<String, Object> fileMap = Maps.newHashMap();
            fileMap.put("displayName", uploadParamDTO.getFileName());
            fileMap.put("fileName", uploadParamDTO.getDownloadName());
            //重写inputStreamResource 将file直接转成流
            org.springframework.core.io.Resource inputStreamResource = new CommonInputStreamResource(inputStream, uploadParamDTO.getDownloadName());
            param.add("file", inputStreamResource);
            param.add("tenantId", authoredUser.getTenantId());
            param.add("expireDate", uploadParamDTO.getExpireDate());
            param.add("fileInfo", fileMap);
            headers.setContentType(MediaType.MULTIPART_FORM_DATA);
            headers.set(GlobalConstant.IAM_USER_TOKEN, dmcToken);
            HttpEntity<MultiValueMap<String, Object>> uploadHttpEntity = new HttpEntity<>(param, headers);
            ResponseEntity<String> responseEntity = restTemplate.exchange(uploadUrl, HttpMethod.POST, uploadHttpEntity, String.class);
            if (HttpStatus.OK.value() == responseEntity.getStatusCodeValue()) {
                String uploadResult = responseEntity.getBody();
                JSONObject jsonObject = JSON.parseObject(uploadResult);
                return JSONObject.parseObject(JSON.toJSONString(jsonObject.getJSONObject("data")), UploadAttachmentRespDTO.class);
            }
        } catch (Exception e) {
            log.error("upload attachment failed, error: ", e);
            throw BusinessException.create(AamErrorCodeEnum.UPLOAD_ATTANCHMENT_ERROR.getErrCode(), AamErrorCodeEnum.UPLOAD_ATTANCHMENT_ERROR.getErrMsg(), ExceptionUtils.unwrapInvocationTargetException(e));
        }
        return null;
    }

    /**
     * 分享文件
     *
     * @return
     */
    @Override
    public List<ShareAttachmentRespDTO> shareAttachment(ShareAttachmentDTO shareAttachmentDTO) {
        try {
            //登录token
            String dmcToken = shareAttachmentDTO.getToken();
            if (StringUtils.isBlank(dmcToken)) {
                AuthoredUser authoredUser = AppAuthContextHolder.getContext().getAuthoredUser();
                dmcToken = getAttachmentToken(dmcTokenCache.getIfPresent(authoredUser.getToken()), authoredUser);
            }
            if (StringUtils.isBlank(shareAttachmentDTO.getFileId()) && CollectionUtils.isEmpty(shareAttachmentDTO.getFileIds())) {
                return new ArrayList<>();
            }
            Map<String, Object> params = new HashMap<>();
            if (StringUtils.isNotBlank(shareAttachmentDTO.getFileId())) {
                params.put("fileId", shareAttachmentDTO.getFileId());
            }
            if (CollectionUtils.isNotEmpty(shareAttachmentDTO.getFileIds())) {
                params.put("fileIds", shareAttachmentDTO.getFileIds());
            }
            if (StringUtils.isNotBlank(shareAttachmentDTO.getExpireDate())) {
                params.put("expireDate", shareAttachmentDTO.getExpireDate());
            }
            HttpHeaders headers = new HttpHeaders();
            headers.setContentType(MediaType.APPLICATION_JSON);
            headers.set(GlobalConstant.IAM_USER_TOKEN, dmcToken);
            HttpEntity<Map<String, Object>> httpEntity = new HttpEntity<>(params, headers);

            ResponseEntity<String> respEntity = this.restTemplate.exchange(dmcProperties.getUri() + "/api/dmc/v2/file/Athena/share", HttpMethod.POST, httpEntity, String.class);

            if (HttpStatus.OK.value() == respEntity.getStatusCodeValue()) {
                String body = respEntity.getBody();
                JSONObject jsonObject = JSON.parseObject(body);
                ShareAttachmentRespDTO shareAttachmentRespDTO = JSON.parseObject(JSON.toJSONString(jsonObject.getJSONObject("data")), ShareAttachmentRespDTO.class);
                return Collections.singletonList(shareAttachmentRespDTO);
            }
        } catch (Exception e) {
            throw BusinessException.create(AamErrorCodeEnum.SHARE_ATTANCHMENT_ERROR.getErrCode(), AamErrorCodeEnum.SHARE_ATTANCHMENT_ERROR.getErrMsg(), ExceptionUtils.unwrapInvocationTargetException(e));

        }
        return null;
    }

    /**
     * 上传文件并分享文件(多个文件压缩包 单个文件直接打开)
     *
     * @param uploadParamDTO
     * @return
     */
    @Override
    public Optional<ShareAttachmentRespDTO> uploadAndShareAttachment(UploadParamDTO uploadParamDTO) {
        //登录token
        String dmcToken = uploadParamDTO.getToken();
        if (StringUtils.isBlank(dmcToken)) {
            AuthoredUser authoredUser = AppAuthContextHolder.getContext().getAuthoredUser();
            dmcToken = getAttachmentToken(dmcTokenCache.getIfPresent(authoredUser.getToken()), authoredUser);
            uploadParamDTO.setToken(dmcToken);
        }
        //上传文件到文档中心
        UploadAttachmentRespDTO uploadAttachmentRespDTO = uploadAttachment(uploadParamDTO);
        if (Objects.nonNull(uploadAttachmentRespDTO)) {
            ShareAttachmentDTO shareAttachmentDTO = ShareAttachmentDTO.builder().fileId(uploadAttachmentRespDTO.getId()).expireDate(uploadParamDTO.getExpireDate()).token(dmcToken).build();
            //分享文件生成可访问的url
            List<ShareAttachmentRespDTO> shareAttachmentRespDTOS = shareAttachment(shareAttachmentDTO);
            return shareAttachmentRespDTOS.stream().findAny();
        }
        return Optional.empty();

    }

    @Override
    public InputStream previewFile(String fileId) {
        HttpHeaders headers = new HttpHeaders();
        String url = dmcProperties.getUri() + "/api/dmc/v2/file/Athena/preview/" + fileId;
        ResponseEntity<byte[]> restRes = this.restTemplate.exchange(url, HttpMethod.GET, new HttpEntity<byte[]>(headers), byte[].class);
        return new ByteArrayInputStream(restRes.getBody());
    }

    @Override
    public Map<String, Object> deleteFile(String fileId) {
        String url = dmcProperties.getUri() + "/api/dmc/v2/file/Athena/delete/" + fileId;
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);


        HttpEntity<Map> httpEntity = new HttpEntity<>(null, headers);
        ResponseEntity<Map> responseEntity = restTemplate.exchange(url, HttpMethod.DELETE, httpEntity, new ParameterizedTypeReference<Map>() {
        });

        return responseEntity.getBody();
    }

    /**
     * 批量删除敏捷报表数据
     *
     * @param fileIds
     * @return
     */
    @Override
    public void deleteAgileData(String token, Collection<String> dirIds, Collection<String> fileIds) {
        String url = dmcProperties.getUri() + "/api/dmc/v2/file/Athena/delete";

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.set(GlobalConstant.IAM_USER_TOKEN, token);

        Map<String, Collection<String>> param = new HashMap<>();
        param.put("fileIds", fileIds);
        param.put("dirIds", dirIds);
        HttpEntity<Map> httpEntity = new HttpEntity<>(param, headers);
        try {
            ResponseEntity<String> respEntity = this.restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<String>() {
            });
            if (HttpStatus.OK.value() == respEntity.getStatusCodeValue()) {
                return;
            } else {
                log.error("deleteAgileData: delete dmc AgileData failed, params: {}, response: {}", param, respEntity.getBody());
                throw BusinessException.create(respEntity.getStatusCodeValue(), "delete dmc AgileData failed, error: " + JsonUtils.objectToString(respEntity.getBody()));
            }
        } catch (Exception ex) {
            if (ex instanceof BusinessException) {
                throw ex;
            }
            log.error("deleteAgileData: delete dmc AgileData failed, params: {}, error: {}", param, ex);
            throw BusinessException.create("delete dmc AgileData failed, error: " + ex.getMessage());
        }
    }

    @Override
    public ResponseEntity<byte[]> getAttachment(String fileId, String token) {
        String url = dmcProperties.getUri() + "/api/dmc/v1/buckets/Athena/files/" + fileId;
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.set(GlobalConstant.IAM_USER_TOKEN, token);

        HttpEntity<Map<String, Object>> httpEntity = new HttpEntity<>(headers);
        try {
            ResponseEntity<byte[]> respEntity = restTemplate.exchange(url, HttpMethod.GET, httpEntity, new ParameterizedTypeReference<byte[]>() {
            });
            if (HttpStatus.OK.value() == respEntity.getStatusCodeValue()) {
                return respEntity;
            } else {
                log.error("getAttachment: get dmc Attachment failed, params: {}, response: {}", fileId, respEntity.getBody());
                throw BusinessException.create(respEntity.getStatusCodeValue(), "get dmc Attachment failed, error: " + JsonUtils.objectToString(respEntity.getBody()));
            }
        } catch (Exception ex) {
            if (ex instanceof BusinessException) {
                throw ex;
            }
            log.error("getAttachment: get dmc Attachment failed, params: {}, error: {}", fileId, ex);
            throw BusinessException.create("delete dmc AgileData failed, error: " + ex.getMessage());
        }
    }

    /**
     * 获取token 校验token是否有效
     *
     * @param token
     * @return
     */
    private String getAttachmentToken(String token, AuthoredUser authoredUser) {
        if (StringUtils.isBlank(token)) {
            return getLoginToken(authoredUser);
        }
        try {
            if (StringUtils.isNotEmpty(token)) {
                HttpHeaders headers = new HttpHeaders();
                headers.setContentType(MediaType.APPLICATION_FORM_URLENCODED);
                headers.set(GlobalConstant.IAM_USER_TOKEN, token);
                HttpEntity<Object> httpEntity = new HttpEntity<>(headers);
                ResponseEntity<String> respEntity = this.restTemplate.exchange(dmcProperties.getUri() + "/api/dmc/v1/auth/token/analyze", HttpMethod.POST, httpEntity, String.class);
                if (HttpStatus.OK.value() != respEntity.getStatusCodeValue()) {
                    return getLoginToken(authoredUser);
                }
            }
        } catch (Exception e) {
            return getLoginToken(authoredUser);
        }
        return token;
    }

    private String getLoginToken(AuthoredUser authoredUser) {
        dmcTokenCache.invalidate(authoredUser.getToken());
        String dmcToken = login(null, null);
        dmcTokenCache.put(authoredUser.getToken(), dmcToken);
        return dmcToken;
    }

    @Override
    public byte[] getMultiAttachment(Map<String, Object> param, String token) {
        String url = dmcProperties.getUri() + "/api/dmc/v2/file/Athena/download/multi";

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.set(GlobalConstant.IAM_USER_TOKEN, token);

        HttpEntity<Map> httpEntity = new HttpEntity<>(param, headers);

        try {
            ResponseEntity<byte[]> respEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<byte[]>() {
            });
            if (HttpStatus.OK.value() == respEntity.getStatusCodeValue()) {
                return respEntity.getBody();
            } else {
                log.error("getMultiAttachment failed failed, params: {}, response: {}", JSON.toJSONString(param), JSON.toJSONString(respEntity.getBody()));
                throw BusinessException.create(respEntity.getStatusCodeValue(), "get dmc Attachment failed, error: " + JsonUtils.objectToString(respEntity.getBody()));
            }
        } catch (Exception ex) {
            log.error("getMultiAttachment failed, params: " + JSON.toJSONString(param) + ", error: ", ex);
            throw BusinessException.create("getMultiAttachment failed, error: " + ex.getMessage());
        }
    }

}
