package com.digiwin.athena.base.application.service.permission;

import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.base.application.constant.BaseGlobalConstant;
import com.digiwin.athena.base.application.constant.PermissionConstant;
import com.digiwin.athena.base.application.meta.request.permission.TypeActivities;
import com.digiwin.athena.base.application.meta.request.permission.TypeActivitiesAccessible;
import com.digiwin.athena.base.application.meta.response.permission.ActivityAccessible;
import com.digiwin.athena.base.infrastructure.constant.AudcErrorCodeEnum;
import com.digiwin.athena.base.infrastructure.constant.Constants;
import com.digiwin.athena.base.infrastructure.manager.iam.service.BaseIamService;
import com.digiwin.athena.base.infrastructure.manager.thememap.BaseThemeMapService;
import com.digiwin.athena.base.infrastructure.manager.thememap.dto.AuthorityConfigResp;
import com.digiwin.athena.base.infrastructure.manager.uibot.BaseReportService;
import com.digiwin.athena.cac.sdk.manager.CacManager;
import com.digiwin.athena.cac.sdk.meta.dto.response.AppAuthDTO;
import com.google.common.collect.Lists;
import lombok.extern.log4j.Log4j2;
import net.sf.json.JSONArray;
import net.sf.json.JSONNull;
import net.sf.json.JSONObject;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.BooleanUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * PermissionCheckServiceImpl Description
 *
 * @author majianfu
 * @date 2021/10/10
 * @since
 */
@Service
@Log4j2
public class PermissionCheckServiceImpl implements PermissionCheckService {
    @Resource
    private BaseIamService userPermissionService;

    @Resource
    private BaseThemeMapService baseThemeMapService;

    @Autowired
    private MessageUtils messageUtils;

    @Autowired
    private StringRedisTemplate stringRedisTemplate;

    @Autowired
    private BaseReportService baseReportService;

    private static final String ACCESS_FORBIDDEN = "forbidden";

    private static final String SEPARATOR = ":";

    private static final String EFFECT = "effect";
    private static final String CONDITIONS = "conditions";
    private static final String ID = "id";
    private static final String MODULE_ID = "moduleId";
    private static final String ENABLE = "enable";
    private static final String MODULE_ENABLE = "module-enable";

    /**
     * {@inheritDoc}
     */
    @Override
    @Deprecated
    public List<ActivityAccessible> checkActivityAccessible(AuthoredUser authoredUser, List<String> tmActivityIdList) {
        log.info("check user({}) activity access permission for: {}", authoredUser.getUserId(), tmActivityIdList);
        if (CollectionUtils.isEmpty(tmActivityIdList)) {
            return Collections.emptyList();
        }

        // 请求TM获取权限配置
        List<AuthorityConfigResp> accessAuthConfigList = getActivityAuthorityConfig(tmActivityIdList);

        TypeActivities typeActivities = new TypeActivities();
        typeActivities.setTmActivityIdList(tmActivityIdList);
        return checkActivityAccessible(authoredUser, Collections.singletonList(typeActivities), accessAuthConfigList);
    }

    private List<AuthorityConfigResp> getActivityAuthorityConfig(List<String> tmActivityIdList) {
        if (CollectionUtils.isNotEmpty(tmActivityIdList)) {
            return baseThemeMapService.getActivityAuthorityConfig(tmActivityIdList, LocaleContextHolder.getLocale().toString());
        }
        return Lists.newArrayList();
    }

    private List<AuthorityConfigResp> getTaskAuthorityConfig(List<String> tmTaskIdList) {
        if (CollectionUtils.isNotEmpty(tmTaskIdList)) {
            return baseThemeMapService.getTaskAuthorityConfig(tmTaskIdList, LocaleContextHolder.getLocale().toString());
        }
        return Lists.newArrayList();
    }

    private List<ActivityAccessible> checkActivityAccessible(AuthoredUser authoredUser, List<TypeActivities> typeActivitiesList, List<AuthorityConfigResp> accessAuthConfigList) {
        Map<String, List<AuthorityConfigResp>> needConfirmedMap = new HashMap<>();
        List<String> allowedActivityIdList = new ArrayList<>();

        //按应用编码分组权限
        analyzeActivityAuthorityConfig(typeActivitiesList, accessAuthConfigList, needConfirmedMap, allowedActivityIdList);

        List<ActivityAccessible> permissionCheckResultList = new ArrayList<>();
        if (MapUtils.isNotEmpty(needConfirmedMap)) {
            Set<Map.Entry<String, List<AuthorityConfigResp>>> entrySet = needConfirmedMap.entrySet();
            for (Map.Entry<String, List<AuthorityConfigResp>> entry : entrySet) {
                //获取作业权限
                permissionCheckResultList.addAll(callIamToCheckAccessPermission(authoredUser, entry.getKey(), entry.getValue()));
            }
        }

        for (String allowedActivityId : allowedActivityIdList) {
            permissionCheckResultList.add(new ActivityAccessible(allowedActivityId, PermissionConstant.ACCESS_ALLOW));
        }
        return permissionCheckResultList;
    }

    private void analyzeActivityAuthorityConfig(List<TypeActivities> typeActivitiesList, List<AuthorityConfigResp> activityAuthorityConfigList,
                                                Map<String, List<AuthorityConfigResp>> needConfirmedMap, List<String> allowedActivityIdList) {
        if (CollectionUtils.isNotEmpty(activityAuthorityConfigList)) {
            // 存在TM配置的activityId集合
            List<String> legalActivityIdList = activityAuthorityConfigList.stream()
                    .map(AuthorityConfigResp::getCode)
                    .collect(Collectors.toList());

            // <tmActivityId, 要校验的权限列表>
            Map<String, List<String>> activityPermissionCheckMap = new HashMap<>();
            typeActivitiesList.stream()
                    .forEach(typeActivities -> {
                        for (String activityId : typeActivities.getTmActivityIdList()) {
                            // 作业有TM配置，记录下要校验的权限
                            if (legalActivityIdList.contains(activityId)) {
                                activityPermissionCheckMap.put(activityId, typeActivities.getPermissionCheckList());
                            }
                            // 作业没有TM配置--允许访问
                            else {
                                allowedActivityIdList.add(activityId);
                            }
                        }
                    });

            for (AuthorityConfigResp activityAuthorityConfig : activityAuthorityConfigList) {
                String authorityPrefix = activityAuthorityConfig.getAuthorityPrefix();
                // 作业没有配置权限---则说明不需要权限校验
                if (StringUtils.isBlank(authorityPrefix)) {
                    allowedActivityIdList.add(activityAuthorityConfig.getCode());
                }
                // authorityPrefix配置不正确
                else if (!authorityPrefix.contains(SEPARATOR)) {
                    throw AudcErrorCodeEnum.AUTH_PREFIX_CONFIG_ERROR.getBusinessException(messageUtils.getMessageWithFormat("exception.authority.prefix.config.error", activityAuthorityConfig.getCode(), authorityPrefix));
                }
                // 作业配置了权限---则需要进一步调用IAM进行权限校验
                else {
                    // 应用code
                    String appCode = authorityPrefix.substring(0, authorityPrefix.indexOf(SEPARATOR));
                    List<AuthorityConfigResp> needConfirmedList = needConfirmedMap.computeIfAbsent(appCode, key -> new ArrayList<>());
                    // 塞入要校验的权限列表
                    activityAuthorityConfig.setPermissionCheckList(activityPermissionCheckMap.get(activityAuthorityConfig.getCode()));
                    needConfirmedList.add(activityAuthorityConfig);
                }
            }
        }
        // 所有作业都没有TM配置---允许访问
        else {
            typeActivitiesList.stream()
                    .map(TypeActivities::getTmActivityIdList)
                    .forEach(activityIdList -> allowedActivityIdList.addAll(activityIdList));
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public List<TypeActivitiesAccessible> checkTypeActivitiesAccessible(AuthoredUser authoredUser, List<TypeActivities> typeActivitiesList) {
        List<TypeActivities> typeActivitiesListTmp = new ArrayList<>();
        //获取租户下作业列表(对应权限的行为)
        for (TypeActivities typeActivities : typeActivitiesList) {
            if (CollectionUtils.isNotEmpty(typeActivities.getTmActivityIdList())) {
                typeActivitiesListTmp.add(typeActivities);
            }
        }

        // 汇总所有的tmActivityId
        List<String> taskIdList = new ArrayList<>();
        List<String> activityIdList = new ArrayList<>();
        //获取需要校验的权限作业
        typeActivitiesListTmp.stream().filter(typeActivities -> CollectionUtils.isNotEmpty(typeActivities.getTmActivityIdList()))
                .forEach(typeActivities -> {
                    if (null != typeActivities.getCategory() && TypeActivities.CATEGORY_PROJECT.equals(typeActivities.getCategory())) {
                        taskIdList.addAll(typeActivities.getTmActivityIdList());
                    } else {
                        activityIdList.addAll(typeActivities.getTmActivityIdList());
                    }
                });

        //获取权限作业的详情（所属应用/权限类型（报表/项目））
        List<AuthorityConfigResp> authorityConfigList = new ArrayList<>();
        authorityConfigList.addAll(this.getActivityAuthorityConfig(activityIdList));
        authorityConfigList.addAll(this.getTaskAuthorityConfig(taskIdList));

        // 查询作业的访问权限设置
        List<ActivityAccessible> activityAccessibleList = this.checkActivityAccessible(authoredUser, typeActivitiesListTmp, authorityConfigList);

        // <tmActivityId, ActivityAccessible>
        Map<String, ActivityAccessible> activityAccessMap = activityAccessibleList.stream()
                .collect(Collectors.toMap(ActivityAccessible::getTmActivityId, Function.identity(), (key1, key2) -> key2));

        // 构建返回结果
        List<TypeActivitiesAccessible> typeActivitiesAccessibleList = new ArrayList<>(typeActivitiesList.size());
        for (TypeActivities typeActivities : typeActivitiesListTmp) {
            if (CollectionUtils.isEmpty(typeActivities.getTmActivityIdList())) {
                continue;
            }

            List<String> tmActivityIdList = typeActivities.getTmActivityIdList();
            TypeActivitiesAccessible typeActivitiesAccessible = new TypeActivitiesAccessible(typeActivities.getType(), Lists.newArrayListWithCapacity(tmActivityIdList.size()));
            //将权限数据绑定在作业列表上
            for (String tmActivityId : typeActivities.getTmActivityIdList()) {
                typeActivitiesAccessible.getActivityAccessibleList().add(activityAccessMap.get(tmActivityId));
            }
            typeActivitiesAccessibleList.add(typeActivitiesAccessible);
        }

        return typeActivitiesAccessibleList;
    }

    private List<ActivityAccessible> callIamToCheckAccessPermission(AuthoredUser authoredUser, String appCode, List<AuthorityConfigResp> activityAuthConfigList) {
        //获取个人作业权限（对应条件权限）
        JSONArray permissions = getAppAllPermissions(authoredUser, appCode);

        List<ActivityAccessible> permissionCheckResultList = new ArrayList<>(activityAuthConfigList.size());
        for (AuthorityConfigResp authConfig : activityAuthConfigList) {
            // 默认可以访问
            boolean effect = true;
            int denyReason = 0;//默认不能访问的原因是未授权
            Map<String, String> permissionCheckMap = new HashMap<>();
            // 未查询到应用的权限配置/应用权限配置为空，则默认应用拥有全部权限
            if (null != permissions && !permissions.isEmpty()) {
                // 应用ID:模组ID，如：PCC:basicDataEntry
                String appModule = authConfig.getAuthorityPrefix();

                for (int idx = 0; idx < permissions.size(); idx++) {
                    /**
                     * {
                     * 	"sid": 355228935742016,
                     * 	"id": "projectSetDataEntry",
                     * 	"moduleId": "basicDataEntry",
                     * 	"name": "项目集录入",
                     * 	"target": "drn:iam:app:PCC:basicDataEntry:projectSetDataEntry",
                     * 	"effect": "allow",
                     * 	"conditions": {
                     * 		"drn:iam:app:PCC:basicDataEntry:projectSetDataEntry": {
                     * 			"basic-data-combine-save": "forbidden",
                     * 			"enable": "forbidden",
                     * 			"copy": "forbidden",
                     * 			"combine-save": "forbidden"
                     *      }
                     *   }
                     * }
                     */
                    JSONObject permissionJson = permissions.getJSONObject(idx);
                    if (permissionJson == null || permissionJson.isNullObject()) {
                        log.warn("activityAuthConfigList中的第{}个permissionJson为空", idx);
                        continue;
                    }
                    JSONObject conditions = permissionJson.getJSONObject("conditions");

                    // 固定前缀:应用ID:模组ID:作业ID，如：drn:iam:app:PCC:basicDataEntry:projectSetDataEntry
                    String fullPath = BaseIamService.USER_PERMISSION_TARGET_PREFIX + appModule + SEPARATOR + authConfig.getCode();
                    if (jsonContainsJsonObj(conditions, fullPath)) {
                        // 只根据 effect == allow 来判断作业访问权限是否开启
                        effect = StringUtils.equalsIgnoreCase(PermissionConstant.ACCESS_ALLOW, permissionJson.getString("effect"));
                        if (!effect && permissionJson.containsKey("denyReason")){
                            //当effect不是allow时，获取返回的denyReason字段，后面用来区分没有权限的原因
                            denyReason = permissionJson.getInt("denyReason");
                        }
                        // 其他权限校验
                        if (CollectionUtils.isNotEmpty(authConfig.getPermissionCheckList())) {
                            JSONObject permissionList = conditions.getJSONObject(fullPath);

                            for (String permissionCheck : authConfig.getPermissionCheckList()) {
                                if (permissionList.containsKey(permissionCheck)) {
                                    permissionCheckMap.put(permissionCheck, permissionList.getString(permissionCheck));
                                }
                            }
                        }
                        break;
                    }
                }
            }
            String access;
            if (effect){
                access = PermissionConstant.ACCESS_ALLOW;
            }else {
                //区分没权限的作业是未购买还是购买但未授权
                access = denyReason==0?ACCESS_FORBIDDEN: Constants.ACCESS_NO_BUY;
            }
            permissionCheckResultList.add(new ActivityAccessible(authConfig.getCode(), access, permissionCheckMap));
        }
        return permissionCheckResultList;
    }

    private boolean jsonContainsJsonObj(JSONObject jsonObj, String key) {
        return null != jsonObj && jsonObj.containsKey(key) && null != jsonObj.getJSONObject(key);
    }

    private JSONArray getAppAllPermissions(AuthoredUser authoredUser, String appCode) {
        // 取缓存
        String redisKey = BaseGlobalConstant.AUDC_CACHE_KEY_PREFIX + authoredUser.getTenantId() + ":" + authoredUser.getUserId() + ":" + appCode;
        if (Boolean.TRUE.equals(stringRedisTemplate.hasKey(redisKey))) {
            String value = stringRedisTemplate.opsForValue().get(redisKey);
            if (StringUtils.isNotBlank(value)){
                JSONArray jsonArray = JSONArray.fromObject(value);
                log.info("redis里的permissionList:{}", value);
                return jsonArray;
            }
        }
        JSONObject permissionJsonObj = userPermissionService.getAllUserPermissionV2(authoredUser, appCode);
        if (null != permissionJsonObj && permissionJsonObj.containsKey("data")) {
            JSONArray permissionList = permissionJsonObj.getJSONArray("data");
            log.debug("user({}) app authority config: {}", authoredUser.getUserId(), permissionList);
            // 存缓存
            stringRedisTemplate.opsForValue().set(redisKey, JsonUtils.objectToString(permissionList), 60, TimeUnit.SECONDS);
            return permissionList;
        }
        return null;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Map<String, String> checkActivitiesAccessible(AuthoredUser authoredUser, Integer category, List<String> activityIdList) {
        TypeActivities typeActivities = new TypeActivities();
        typeActivities.setType("activityType");
        typeActivities.setCategory(category);
        typeActivities.setTmActivityIdList(activityIdList);

        List<TypeActivitiesAccessible> typeActivitiesAccessible = this.checkTypeActivitiesAccessible(authoredUser, Collections.singletonList(typeActivities));
        Map<String, String> activityPermissionMap = new HashMap<>();
        if (CollectionUtils.isNotEmpty(typeActivitiesAccessible) && CollectionUtils.isNotEmpty(typeActivitiesAccessible.get(0).getActivityAccessibleList())) {
            typeActivitiesAccessible.get(0).getActivityAccessibleList()
                    .forEach(activityAccessible -> activityPermissionMap.put(activityAccessible.getTmActivityId(), activityAccessible.getAccess()));
        }
        return activityPermissionMap;
    }


    /**
     * 获取拥有权限的模组和菜单
     * @return
     */
    @Override
    public Map<String, List<String>> getModuleOrMenuAccessible(AuthoredUser authoredUser, String appCode) {
        Map<String, List<String>> resultMap = new HashMap<>();
        // 获取模组和菜单权限
        JSONArray accessibleArray =  getAppAllPermissions(authoredUser, appCode);
        if (null == accessibleArray) {
            return resultMap;
        }
        // 获取特殊系统菜单的权限
        Map<String, Boolean> specialMenuMap = getSpecialMenuAccessible(authoredUser);
        // 所有模组集合
        List<String> allModuleList = new ArrayList<>();
        // 有权限的模组集合
        List<String> moduleAccessibleList = new ArrayList<>();
        // 所有菜单集合
        List<String> allMenuList = new ArrayList<>();
        // 有权限的菜单集合
        List<String> menuAccessibleList = new ArrayList<>();

        // 将特殊菜单放入列表里
        specialMenuMap.forEach((menuId, accessible) -> {
            allMenuList.add(menuId);
        });

        for (Object item : accessibleArray) {
            if (Objects.isNull(item) || item instanceof JSONNull) {
                continue;
            }
            JSONObject accessibleObj = (JSONObject) item;
            // 分别存储所有模组和菜单
            if (StringUtils.equalsIgnoreCase(MODULE_ENABLE, accessibleObj.getString(ID))) {
                allModuleList.add(accessibleObj.getString(MODULE_ID));
            } else {
                allMenuList.add(accessibleObj.getString(ID));
            }
            // 1.先判断effect是否为allow
            if (!StringUtils.equalsIgnoreCase(PermissionConstant.ACCESS_ALLOW, accessibleObj.getString(EFFECT))) {
                continue;
            }
            // 2.再判断conditions中的enable是否为allow
            JSONObject conditions = accessibleObj.getJSONObject(CONDITIONS);
            String fullPath = BaseIamService.USER_PERMISSION_TARGET_PREFIX + "Athena:" + accessibleObj.getString(MODULE_ID) + SEPARATOR +  accessibleObj.getString(ID);
            if (!jsonContainsJsonObj(conditions, fullPath) ||  null == conditions.getJSONObject(fullPath).get(ENABLE) ||
                    !StringUtils.equalsIgnoreCase(PermissionConstant.ACCESS_ALLOW, conditions.getJSONObject(fullPath).getString(ENABLE))) {
                continue;
            }
            // 判断特殊菜单的权限
            if (specialMenuMap.containsKey(accessibleObj.getString(ID)) && !specialMenuMap.get(accessibleObj.getString(ID))) {
                continue;
            }
            // 分别存储有权限的模组和菜单
            if (StringUtils.equalsIgnoreCase(MODULE_ENABLE, accessibleObj.getString(ID))) {
                moduleAccessibleList.add(accessibleObj.getString(MODULE_ID));
            } else {
                menuAccessibleList.add(accessibleObj.getString(ID));
            }
        }

        resultMap.put("allModule", allModuleList);
        resultMap.put("allMenu", allMenuList);
        resultMap.put("accessibleModule", moduleAccessibleList);
        resultMap.put("accessibleMenu", menuAccessibleList);
        return resultMap;
    }

    /**
     * ERP单据集成配置器
     */
    private static final String ATHENA_DDSM_DESIGNER = "athena-ddsmdesigner";
    /**
     * 重置测试数据
     */
    private static final String RESET_TEST_DATA = "resetTestData";
    /**
     * 财报管理的参数维护
     */
    private static final String REPORT_PARAMETER_MAINTENANCE = "report-parameter-maintenance";
    /**
     * 函数设计
     */
    private static final String REPORT_FUNCTIONAL_DESIGN = "report-functional-design";


    /**
     * 获取特殊菜单的权限
     * @return
     */
    private Map<String, Boolean> getSpecialMenuAccessible(AuthoredUser authoredUser) {
        Map<String, Boolean> specialMenuMap = new HashMap<>();
        // 判断- ERP单据集成配置器：athena-ddsmdesigner的权限
        AppAuthDTO appAuthDTO = CacManager.queryAppAuth(authoredUser.getUserId(), "EAS");;
        specialMenuMap.put(ATHENA_DDSM_DESIGNER, Objects.nonNull(appAuthDTO) ? appAuthDTO.getIsSuccess() : false);

//        // 判断- 重置测试数据：resetTestData的权限
//        Boolean isTestTenant = atmcService.isTenantTest(authoredUser);
//        specialMenuMap.put(RESET_TEST_DATA,  BooleanUtils.isTrue(isTestTenant));

        // 判断- 财报管理的参数维护：report-parameter-maintenance，函数设计：report-functional-design的权限
        Boolean isSysRole = baseReportService.isSysRole(authoredUser);
        specialMenuMap.put(REPORT_PARAMETER_MAINTENANCE, BooleanUtils.isTrue(isSysRole));
        specialMenuMap.put(REPORT_FUNCTIONAL_DESIGN, BooleanUtils.isTrue(isSysRole));
        return specialMenuMap;
    }
}
