package com.digiwin.athena.base.application.service.trial;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.appcore.util.SnowflakeIdWorker;
import com.digiwin.athena.base.application.config.BaseAudcDataSourceConfig;
import com.digiwin.athena.base.application.meta.request.trial.scene.TrialSceneReq;
import com.digiwin.athena.base.application.meta.request.trial.scene.TrialSceneResp;
import com.digiwin.athena.base.application.meta.request.trial.sceneStep.TrialSceneStepBindReq;
import com.digiwin.athena.base.application.meta.request.trial.step.TrialStepResp;
import com.digiwin.athena.base.infrastructure.mapper.audc.trial.TrialSceneMapper;
import com.digiwin.athena.base.infrastructure.mapper.audc.trial.TrialSceneStepMapper;
import com.digiwin.athena.base.infrastructure.meta.po.trial.TrialSceneData;
import com.digiwin.athena.base.infrastructure.meta.po.trial.TrialSceneStepData;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * 描述
 *
 * @author gonghongxing
 * @version 1.0
 * @since 2021/09/03 16:46:12
 */
@Service
public class TrialSceneServiceImpl implements TrialSceneService {

    private final TrialSceneMapper trialSceneMapper;
    private final TrialSceneStepMapper trialSceneStepMapper;
    private final TrialStepService trialStepService;

    @Autowired
    public TrialSceneServiceImpl(TrialStepService trialStepService, TrialSceneMapper trialSceneMapper, TrialSceneStepMapper trialSceneStepMapper) {
        this.trialSceneMapper = trialSceneMapper;
        this.trialSceneStepMapper = trialSceneStepMapper;
        this.trialStepService = trialStepService;
    }

    /**
     * 新增试用场景
     *
     * @param sceneReq sceneReq
     * @return int
     */
    @Override
    @Transactional(transactionManager = BaseAudcDataSourceConfig.BASE_AUDC_DATASOURCE_TRANSACTION_MANAGER_BUSINESS)
    public Long add(TrialSceneReq sceneReq) {
        TrialSceneData sceneData = new TrialSceneData();
        BeanUtils.copyProperties(sceneReq, sceneData);
        sceneData.setId(SnowflakeIdWorker.getInstance().newId());
        sceneData.setCreateDate(LocalDateTime.now());
        sceneData.setModifyDate(LocalDateTime.now());
        int result = trialSceneMapper.insert(sceneData);
        return sceneData.getId();
    }

    /**
     * 修改试用场景
     *
     * @param sceneReq sceneReq
     * @return int
     */
    @Override
    @Transactional(transactionManager = BaseAudcDataSourceConfig.BASE_AUDC_DATASOURCE_TRANSACTION_MANAGER_BUSINESS)
    public int modify(TrialSceneReq sceneReq) {
        TrialSceneData sceneData = new TrialSceneData();
        sceneData.setId(sceneReq.getId());
        sceneData.setSceneDesc(sceneReq.getSceneDesc());
        sceneData.setModifyDate(LocalDateTime.now());
        return trialSceneMapper.updateById(sceneData);
    }

    /**
     * 查询试用场景列表
     *
     * @return list
     */
    public List<TrialSceneResp> list(TrialSceneReq sceneReq) {
        QueryWrapper<TrialSceneData> wrapper = new QueryWrapper<>();
        //如果有没有输入条件，查询所有
        if (StringUtils.isNotEmpty(sceneReq.getSceneKey())) {
            wrapper.likeRight("scene_key", sceneReq.getSceneKey());
        }
        List<TrialSceneData> sceneDataList = trialSceneMapper.selectList(wrapper);
        return sceneDataToResp(sceneDataList);
    }


    /**
     * 查询试用场景列表
     *
     * @return list
     */
    public TrialSceneResp listWithSteps(TrialSceneReq sceneReq) {
        QueryWrapper<TrialSceneData> wrapper = new QueryWrapper<>();
        //如果有没有输入条件，查询所有
        if (StringUtils.isEmpty(sceneReq.getSceneKey())) {
            throw BusinessException.create(-1, "Scene Key must exist");

        }
        wrapper.eq("scene_key", sceneReq.getSceneKey());
        TrialSceneData sceneData = trialSceneMapper.selectOne(wrapper);

        TrialSceneResp resp = new TrialSceneResp();
        BeanUtils.copyProperties(sceneData, resp);
//        List<TrialSceneResp> sceneRespList = sceneDataToResp(sceneData);
//        //遍历场景关联的步骤列表
//        for (TrialSceneResp resp : sceneRespList) {
        String sceneKey = resp.getSceneKey();
        QueryWrapper<TrialSceneStepData> sceneStepWrapper = new QueryWrapper<>();
        sceneStepWrapper.eq("scene_key", sceneKey);
        List<TrialSceneStepData> sceneStepList = trialSceneStepMapper.selectList(sceneStepWrapper);
        List<TrialStepResp> stepResps = new ArrayList<>();
        //获取步骤的详情以及对应的排序
        for (TrialSceneStepData data : sceneStepList) {
            TrialStepResp stepResp = trialStepService.detail(data.getStepKey());
            stepResp.setRank(data.getRank());
            stepResps.add(stepResp);
        }
        resp.setSteps(stepResps);
//        }
        return resp;
    }


    /**
     * 根据id删除试用场景
     *
     * @param id id
     * @return int
     */
    @Override
    @Transactional(transactionManager = BaseAudcDataSourceConfig.BASE_AUDC_DATASOURCE_TRANSACTION_MANAGER_BUSINESS)
    public int deleteById(Long id) {
        return trialSceneMapper.deleteById(id);
    }

    /**
     * 场景绑定步骤
     *
     * @param req req
     */
    @Override
    @Transactional(transactionManager = BaseAudcDataSourceConfig.BASE_AUDC_DATASOURCE_TRANSACTION_MANAGER_BUSINESS)
    public void bindSteps(TrialSceneStepBindReq req) {
        //场景id
        long sceneId = req.getSceneId();
        String sceneKey = req.getSceneKey();
        //步骤id以及对应的排序
        List<Map<String, Object>> steps = req.getSteps();
        //删除所有的场景与应用的绑定数据
        UpdateWrapper<TrialSceneStepData> delWrapper = new UpdateWrapper<>();
        delWrapper.eq("scene_id", sceneId);
        trialSceneStepMapper.delete(delWrapper);
        steps.forEach(x -> {
            TrialSceneStepData data = new TrialSceneStepData();
            data.setId(SnowflakeIdWorker.getInstance().newId());
            data.setSceneId(sceneId);
            data.setSceneKey(sceneKey);
            data.setStepId(MapUtils.getLongValue(x, "stepId"));
            data.setStepKey(MapUtils.getString(x, "stepKey"));
            data.setRank(MapUtils.getIntValue(x, "rank"));
            data.setCreateDate(LocalDateTime.now());
            data.setModifyDate(LocalDateTime.now());
            trialSceneStepMapper.insert(data);
        });
    }

    private List<TrialSceneResp> sceneDataToResp(List<TrialSceneData> dataList) {
        List<TrialSceneResp> respList = new ArrayList<>();
        dataList.forEach(x -> {
            TrialSceneResp resp = new TrialSceneResp();
            BeanUtils.copyProperties(x, resp);
            respList.add(resp);
        });
        return respList;

    }


    /**
     * 判断场景是否存在
     *
     * @param sceneKey sceneKey
     * @return true or false
     */
    public boolean isSceneExist(String sceneKey) {
        QueryWrapper<TrialSceneData> query = new QueryWrapper<>();
        query.eq("scene_key", sceneKey);
        Long result = trialSceneMapper.selectCount(query);
        return result > 0L;
    }


    /**
     * 判断场景是否存在
     *
     * @param sceneKey sceneKey
     * @return true or false
     */
    @Override
    @Transactional(transactionManager = BaseAudcDataSourceConfig.BASE_AUDC_DATASOURCE_TRANSACTION_MANAGER_BUSINESS)
    public int updateDetail(String sceneKey, String detail) {
        TrialSceneData data = new TrialSceneData();
        UpdateWrapper<TrialSceneData> update = new UpdateWrapper<>();
        update.set("detail", detail).eq("scene_key", sceneKey);
        return trialSceneMapper.update(data, update);
    }


    /**
     * 根据scenekey查询场景详情
     *
     * @param sceneKey sceneKey
     * @return int
     */
    public TrialSceneData sceneDetail(String sceneKey) {
        QueryWrapper<TrialSceneData> query = new QueryWrapper<>();
        query.eq("scene_key", sceneKey);
        return trialSceneMapper.selectOne(query);
    }

}
