package com.digiwin.athena.base.application.service.usertrack;

import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.appcore.util.SnowflakeIdWorker;
import com.digiwin.athena.appcore.util.TimeUtils;
import com.digiwin.athena.base.application.meta.request.usertrack.AccountDTO;
import com.digiwin.athena.base.application.meta.request.usertrack.UserTrackSearchDTO;
import com.digiwin.athena.base.application.service.usertrack.analyzer.ReportDataAnalyzer;
import com.digiwin.athena.base.application.service.usertrack.analyzer.UserTrackAnalyzer;
import com.digiwin.athena.base.infrastructure.constant.AudcErrorCodeEnum;
import com.digiwin.athena.base.infrastructure.meta.po.usertrack.mongo.UserTrackExtendDTO;
import com.digiwin.athena.base.infrastructure.mongo.UserTrackMongoMapper;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.text.SimpleDateFormat;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * UserTrackServiceImpl Description
 *
 * @author majianfu
 * @date 2021/8/27
 * @since
 */
@Service
@Slf4j
public class UserTrackServiceImpl implements UserTrackService, InitializingBean {
    @Autowired
    private List<ReportDataAnalyzer> reportDataAnalyzerList;

    private Map<String, ReportDataAnalyzer> reportDataAnalyzerMap = new HashMap<>();

    @Resource
    private MessageUtils messageUtils;

    @Autowired
    private UserTrackMongoMapper userTrackMongoMapper;

    private static final String CLEAR_OPERATION_PURCHASE_WITH_DRAWING = "Athena01144";

    @Override
    public void afterPropertiesSet() throws Exception {
        reportDataAnalyzerList.forEach(reportDataAnalyzer -> reportDataAnalyzerMap.put(StringUtils.upperCase(reportDataAnalyzer.getSourceType()), reportDataAnalyzer));
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void handleReportData(List<UserTrackExtendDTO> reportDataList) {
        if (CollectionUtils.isEmpty(reportDataList)) {
            return;
        }

        AuthoredUser authoredUser = AppAuthContextHolder.getContext().getAuthoredUser();
        if (null != authoredUser) {
            log.info("currentUser{}:{} start", authoredUser.getTenantId(), authoredUser.getUserId());
            log.info("currentUser{}:{} reportDataSize:{}", authoredUser.getTenantId(), authoredUser.getUserId(), reportDataList.size());
        }

        // 填充记录时间、uniqueId等字段
        regularReportData(reportDataList);
        // 分类上报数据
        Map<String, List<UserTrackExtendDTO>> reportDataMap = classifyReportData(reportDataList);
        // 调用对应sourceType的分析器，分析上报数据
        reportDataMap.forEach((source, sourceReportDataList) -> {
            if (reportDataAnalyzerMap.containsKey(source) && CollectionUtils.isNotEmpty(sourceReportDataList)) {
                printReportLog(sourceReportDataList);

                reportDataAnalyzerMap.get(source).analysis(sourceReportDataList);
            } else {
                log.warn("unsupported source:{}, reportDataSize:{}", source, sourceReportDataList.size());
            }
        });

        if (null != authoredUser) {
            log.info("currentUser{}:{} end", authoredUser.getTenantId(), authoredUser.getUserId());
        }
    }

    private void printReportLog(List<UserTrackExtendDTO> reportDataList) {
        // 防止每次打印的日志过长，每10条打印一次
        int count = 0;
        int countThreshold = 10;
        StringBuilder sb = new StringBuilder();
        for (UserTrackExtendDTO userTrack : reportDataList) {
            if (null == userTrack) {
                continue;
            }
            count++;
            sb.append(userTrack.printInfo());
            if ((count % countThreshold) == 0) {
                log.info("{}", sb.toString());
                sb = new StringBuilder();
            }
        }
        if (sb.length() > 0) {
            log.info("{}", sb.toString());
        }
    }

    private void regularReportData(List<UserTrackExtendDTO> reportDataList) {
        LocalDateTime createTime = LocalDateTime.now();
        long createDate = com.digiwin.athena.base.application.util.TimeUtils.date2Long(createTime);
        String createDateStr = TimeUtils.format(createTime, TimeUtils.DEFAULT_FORMAT);
        reportDataList.stream().forEach(reportData -> {
            // 设置createDate，便于FI按时间范围查询
            reportData.setCreateDate(createDate);
            reportData.setCreateDateStr(createDateStr);
            // 设置uniqueId，便于在uniqueId上创建唯一索引 + 设置TTL
            reportData.setUniqueId(SnowflakeIdWorker.getInstance().newId());
        });
    }

    private Map<String, List<UserTrackExtendDTO>> classifyReportData(List<UserTrackExtendDTO> reportDataList) {
        if (CollectionUtils.isEmpty(reportDataList)) {
            return Collections.emptyMap();
        }

        Map<String, List<UserTrackExtendDTO>> reportDataMap = new HashMap<>();
        for (UserTrackExtendDTO reportData : reportDataList) {
            String sourceType = StringUtils.isNotBlank(reportData.getSource()) ? reportData.getSource() : UserTrackAnalyzer.USER_TRACK_ANALYSIS;
            List<UserTrackExtendDTO> sourceReportDataList = reportDataMap.computeIfAbsent(StringUtils.upperCase(sourceType), source -> new ArrayList<>());
            sourceReportDataList.add(reportData);
        }
        return reportDataMap;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void clearReportData(AccountDTO account) {
        // 校验密码
        if (StringUtils.equals(account.getPassword(), CLEAR_OPERATION_PURCHASE_WITH_DRAWING)) {
            reportDataAnalyzerList.forEach(ReportDataAnalyzer::clear);
        } else {
            throw AudcErrorCodeEnum.INVALID_REQUEST_PARAM.getBusinessException(messageUtils.getMessage("exception.password.error"));
        }
    }

    /**
     * 查询埋点数据
     *
     * @return
     */
    @Override
    public Map<String, Object> findByParam(UserTrackSearchDTO searchDTO, AuthoredUser user) {
        log.info("UserTrackServiceImpl-findByParam-searchDTO:{}", JsonUtils.objectToString(searchDTO));
        log.info("UserTrackServiceImpl-findByParam-user:{}", JsonUtils.objectToString(user));
        String tenantId = searchDTO.getTenantId();
        // 若租户id不作为参数传入，则取token里的租户id
        if (StringUtils.isEmpty(tenantId)) {
            tenantId = user.getTenantId();
        }
        // 查询条件
        Criteria criteria = Criteria.where("tenantId").is(tenantId);
        // 用户id
        if (StringUtils.isNotBlank(searchDTO.getUserId())) {
            criteria.and("userId").is(searchDTO.getUserId());
        }
        // 作业ThemeMap定义的code
        if (StringUtils.isNotBlank(searchDTO.getWorkCode())) {
            criteria.and("workCode").is(searchDTO.getWorkCode());
        }
        // 作业ThemeMap定义的code
        if (StringUtils.isNotBlank(searchDTO.getWorkType())) {
            criteria.and("workType").is(searchDTO.getWorkType());
        }
        // 执行的操作
        if (CollectionUtils.isNotEmpty(searchDTO.getOperations())) {
            criteria.and("operation").in(searchDTO.getOperations());
        }

        /*时间范围 start*/
        // 开始时间
        String startTime = searchDTO.getStartTime();
        // 结束时间
        String endTime = searchDTO.getEndTime();
        // 校验开始时间和结束时间格式
        validStartEndTime(startTime, endTime);
        // 拼接时间范围查询条件
        if (StringUtils.isNotBlank(startTime) && StringUtils.isNotBlank(endTime)) {
            criteria.andOperator(Criteria.where("createDateStr").gte(startTime), Criteria.where("createDateStr").lte(endTime));
        } else if (StringUtils.isNotBlank(searchDTO.getStartTime())) {
            criteria.andOperator(Criteria.where("createDateStr").gte(startTime));
        } else if (StringUtils.isNotBlank(searchDTO.getEndTime())) {
            criteria.andOperator(Criteria.where("createDateStr").lte(endTime));
        }
        /*时间范围 end*/
        Query query = Query.query(criteria);
        // 排除key，与数据库中key一致
        query.fields().exclude("orgs").exclude("empInfo").exclude("workContent").exclude("attachData");

        List<UserTrackExtendDTO> list = userTrackMongoMapper.findByParam(tenantId + "_track", query);

        Map<String, Object> resultMap = new HashMap<>();
        resultMap.put("count", list.size());
        resultMap.put("datas", list);
        return resultMap;
    }

    /**
     * 校验
     *
     * @param startTime
     * @param endTime
     */
    private void validStartEndTime(String startTime, String endTime) {
        // 校验开始时间
        if (StringUtils.isNotBlank(startTime)) {
            if (!validTimeFormat(startTime, "yyyy-MM-dd HH:mm:ss")) {
                throw AudcErrorCodeEnum.INVALID_REQUEST_PARAM.getBusinessException("The format of the startTime is incorrect! The correct is yyyy-MM-dd HH:mm:ss");
            }
        }
        // 校验结束时间
        if (StringUtils.isNotBlank(endTime)) {
            if (!validTimeFormat(endTime, "yyyy-MM-dd HH:mm:ss")) {
                throw AudcErrorCodeEnum.INVALID_REQUEST_PARAM.getBusinessException("The format of the endTime is incorrect! The correct is yyyy-MM-dd HH:mm:ss");
            }
        }
    }

    /**
     * 校验时间格式
     *
     * @param time
     * @return
     */
    private boolean validTimeFormat(String time, String format) {
        try {
            SimpleDateFormat simpleDateFormat = new SimpleDateFormat(format);
            simpleDateFormat.parse(time);
            return true;
        } catch (Exception e) {
            return false;
        }
    }
}
