package com.digiwin.athena.base.application.service.commonused;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.atdm.semc.FineReportService;
import com.digiwin.athena.atdm.semc.dto.FineReportVO;
import com.digiwin.athena.base.application.config.BaseAudcDataSourceConfig;
import com.digiwin.athena.base.application.constant.BaseGlobalConstant;
import com.digiwin.athena.base.application.constant.PermissionConstant;
import com.digiwin.athena.base.application.converter.DataConverter;
import com.digiwin.athena.base.application.meta.dto.commonused.TmDataListDTO;
import com.digiwin.athena.base.application.meta.dto.commonused.TmDataTierListDTO;
import com.digiwin.athena.base.application.meta.request.commonused.BatchSortItemReq;
import com.digiwin.athena.base.application.meta.request.permission.TypeActivities;
import com.digiwin.athena.base.application.meta.request.permission.TypeActivitiesAccessible;
import com.digiwin.athena.base.application.meta.response.commonused.CateAndItemResp;
import com.digiwin.athena.base.application.meta.response.commonused.CategoryResp;
import com.digiwin.athena.base.application.meta.response.commonused.ItemResp;
import com.digiwin.athena.base.application.meta.response.permission.ActivityAccessible;
import com.digiwin.athena.base.application.service.permission.PermissionCheckService;
import com.digiwin.athena.base.infrastructure.config.EnvProperties;
import com.digiwin.athena.base.infrastructure.constant.Constants;
import com.digiwin.athena.base.infrastructure.manager.semc.res.TenantModuleRes;
import com.digiwin.athena.base.infrastructure.manager.semc.service.SemcReqService;
import com.digiwin.athena.base.infrastructure.manager.thememap.BaseThemeMapService;
import com.digiwin.athena.base.infrastructure.manager.thememap.dto.BaseItemDto;
import com.digiwin.athena.base.infrastructure.mapper.audc.commonUsed.BasisReportSortMapper;
import com.digiwin.athena.base.infrastructure.meta.po.commonused.BasisReportSortData;
import com.digiwin.athena.base.infrastructure.meta.po.commonused.CategoryData;
import com.digiwin.athena.base.infrastructure.meta.po.commonused.ItemData;
import com.digiwin.athena.datamap.sdk.manager.DataMapManager;
import com.digiwin.athena.datamap.sdk.meta.dto.response.TmDataEntryDTO;
import com.digiwin.athena.datamap.sdk.meta.dto.response.TmStartProjectDTO;
import com.google.common.collect.Maps;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.BooleanUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.client.RestTemplate;

import javax.annotation.Resource;
import java.text.Collator;
import java.util.*;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import static com.digiwin.athena.base.infrastructure.manager.thememap.BaseThemeMapServiceImpl.BASE_DATA_PATTERN;
import static com.digiwin.athena.base.infrastructure.manager.thememap.BaseThemeMapServiceImpl.REPORT_PATTERN;


/**
 * 常用操作的业务处理类
 *
 * @author gonghongxing at 2021-08-05
 */
@Service
public class CommonUsedServiceImpl implements CommonUsedService {
    private static final Logger logger = LoggerFactory.getLogger(CommonUsedServiceImpl.class);
    private static final long ROOT_CATE = -1;

    @Autowired
    private CategoryService categoryService;

    @Autowired
    private ItemService itemService;

    @Resource
    private MessageUtils messageUtils;

    @Autowired
    EnvProperties envProperties;

    @Autowired
    RestTemplate restTemplate;

    @Autowired
    BaseThemeMapService baseThemeMapService;

    @Autowired
    private PermissionCheckService permissionCheckService;
    @Autowired
    private SemcReqService semcReqService;

    @Autowired
    private FineReportService fineReportService;

    /**
     * 查询对应的模块常用下分类以及分类子项信息
     *
     * @param user   用户信息
     * @param plat   所属模块0:基础资料录入 1：报表，后续有新增则依次递增
     * @param cateId 分类ID -1：根节点
     */
    public CateAndItemResp cateAndItemList(AuthoredUser user, Integer plat, Long cateId) {
        List<BaseItemDto> baseItemDtoList = this.queryTmDataFromKM(user, plat, null);
        Map<String, String> baseItemMaps = Maps.newHashMap();
        Map<String, String> baseItemCateMaps = new HashMap<>();
        if (CollectionUtils.isNotEmpty(baseItemDtoList)) {
            //过滤掉隐藏的报表
            baseItemDtoList = baseItemDtoList.stream().filter(item -> (item.getSubReport() == null || !item.getSubReport())).collect(Collectors.toList());
            baseItemMaps = baseItemDtoList.stream().collect(Collectors.toMap(BaseItemDto::getCode, BaseItemDto::getName, (oldName, newName) -> newName));
            baseItemCateMaps = baseItemDtoList.stream().collect(Collectors.toMap(BaseItemDto::getCode, BaseItemDto::getCategory, (oldValue, newValue) -> newValue));
        }
        return queryCommonList(user, plat, cateId, baseItemMaps, baseItemCateMaps);
    }

    /**
     * 递归的方式查询常用列表
     *
     * @param user   用户信息
     * @param plat   所属模块0:基础资料录入 1：报表，后续有新增则依次递增
     * @param cateId 分类ID -1：根节点
     * @return
     */
    private CateAndItemResp queryCommonList(AuthoredUser user, Integer plat, Long cateId, Map<String, String> baseItemMaps, Map<String, String> baseItemCateMaps) {
        CateAndItemResp resp = new CateAndItemResp();
        //cateid为null或者=-1，表示查询的是根节点,首先获取根节点下节点信息
        if (Objects.isNull(cateId) || ROOT_CATE == cateId) {
            cateId = -1L;

        }
        List<CategoryData> cateList = categoryService.queryCateList(user, plat, cateId);
        List<CategoryResp> cateRespList = cateDataToResp(cateList);

        List<ItemData> itemList = itemService.queryItemList(user, plat, cateId);
        List<ItemResp> itemRespList = itemDataToRespWithFilter(itemList, baseItemMaps);
        if (!itemRespList.isEmpty()) {

            for (ItemResp temp : itemRespList) {
                temp.setItemName(MapUtils.getString(baseItemMaps, temp.getItemCode(), ""));
                temp.setCategory(MapUtils.getString(baseItemCateMaps, temp.getItemCode(), ""));
            }
        } else {
            logger.warn("item list is empty,no need to find invoke queryEntryData");
        }

        resp.setItemsList(itemRespList);
        resp.setCateList(cateRespList);
        for (CategoryResp temp : cateRespList) {
            CateAndItemResp tempResp = queryCommonList(user, plat, temp.getId(), baseItemMaps, baseItemCateMaps);
            temp.setItemsList(tempResp.getItemsList());
        }
        return resp;
    }


    /**
     * 删除指定的分类
     *
     * @param user 用户信息
     * @param id   分类id
     * @return 操作结果
     */
    public int deleteCommUsedCate(AuthoredUser user, Long id, Integer plat) {
        //删除该分组下的所有子项
        int result = itemService.deleteItemsByCate(id, user, plat);
        if (result >= 0) {
            return categoryService.deleteCategory(user, id);
        }
        return -1;
    }

    /**
     * 删除指定的分类，并将对应的子项移动的到默认分组中
     *
     * @param user 用户信息
     * @param id   分类id
     * @return 操作结果
     */
    public int deleteCateAndMoveItems(AuthoredUser user, Long id, Integer plat) {
        int result = itemService.moveItemsByCate(id, -1L, user, plat);
        if (result >= 0) {
            return categoryService.deleteCategory(user, id);
        }
        return -1;
    }


    /**
     * data转resp
     *
     * @param cateList 目录列表
     * @return list
     */
    private List<CategoryResp> cateDataToResp(List<CategoryData> cateList) {
        List<CategoryResp> respList = new ArrayList<>();
        for (CategoryData data : cateList) {
            CategoryResp resp = new CategoryResp();
            BeanUtils.copyProperties(data, resp);
            respList.add(resp);
        }
        return respList;
    }


    /**
     * data转resp
     *
     * @param itemList 子项列表
     * @return list
     */
    private List<ItemResp> itemDataToResp(List<ItemData> itemList) {
        List<ItemResp> respList = new ArrayList<>();
        for (ItemData data : itemList) {
            ItemResp resp = new ItemResp();
            BeanUtils.copyProperties(data, resp);
            respList.add(resp);
        }
        return respList;
    }

    /**
     * data转resp
     *
     * @param itemList 子项列表
     * @return list
     */
    private List<ItemResp> itemDataToRespWithFilter(List<ItemData> itemList, Map<String, String> baseItemMaps) {
        List<ItemResp> respList = new ArrayList<>();
        for (ItemData data : itemList) {
            //如果不存在，则跳过该数据，不展示数据
            if (!baseItemMaps.containsKey(data.getItemCode())) {
                logger.warn("ItemCode:{} not in km result,so ignore", data.getItemCode());
                continue;
            }
            ItemResp resp = new ItemResp();
            BeanUtils.copyProperties(data, resp);
            respList.add(resp);
        }
        return respList;
    }


    /**
     * 根据当前登录的用户语言，去寻找基础资料录入的详情
     *
     * @param user 用户信息
     * @return 返回信息
     */
    public Map<String, String> queryEntryData(AuthoredUser user, Integer plat) {
        List<BaseItemDto> respList = baseThemeMapService.getActivitiesByPattern(user, plat, null, null);
        Map<String, String> itemMaps = new HashMap<>();
        if (null != respList && !respList.isEmpty()) {
            itemMaps = respList.stream().collect(Collectors.toMap(BaseItemDto::getCode, BaseItemDto::getName, (oldName, newName) -> newName));
        }
        return itemMaps;
    }


    /**
     * 获取当前的分类信息
     *
     * @param user 用户信息
     * @return 返回信息
     */
    public Map<String, String> queryEntryDataCate(AuthoredUser user, Integer plat) {
        List<BaseItemDto> respList = baseThemeMapService.getActivitiesByPattern(user, plat, null, null);
        Map<String, String> itemMaps = new HashMap<>();
        if (null != respList && !respList.isEmpty()) {
            itemMaps = respList.stream().collect(Collectors.toMap(BaseItemDto::getCode, BaseItemDto::getCategory, (oldValue, newValue) -> newValue));
        }
        return itemMaps;
    }


    /**
     * 获取基础资料录入列表
     *
     * @Author：SYQ
     * @Date：2021/9/23 17:42
     */
    public List<TmDataListDTO> tmDataList(AuthoredUser user, Integer plat, boolean allowRepeat, Boolean searchData, List<BaseItemDto> baseDataEntryJobList, String pattern, String effect) {
        if (0L == user.getTenantSid()) {
            logger.warn("tmDataList-{}:{}:{}:{}: tenantSid is null", user.getTenantId(), user.getUserId(), plat, LocaleContextHolder.getLocale());
            return new ArrayList<>();
        }

        if (StringUtils.isBlank(effect)) {
            effect = PermissionConstant.ACCESS_ALLOW;
        }
        // 查询作业
        List<BaseItemDto> allTmDataList = getAllTmDataList(user, plat, allowRepeat, pattern, effect);
        // 从KM未查询到任何作业，直接返回空列表
        if (CollectionUtils.isEmpty(allTmDataList)) {
            return new ArrayList<>();
        }

        List<BaseItemDto> candidateDataList = new ArrayList<>();
        if (!searchData || CollectionUtils.isEmpty(baseDataEntryJobList)) {
            candidateDataList.addAll(allTmDataList);
        } else {
            Map<String, BaseItemDto> checkMap = allTmDataList.stream().collect(Collectors.toMap(BaseItemDto::getCode, Function.identity(), (newVal, oldVal) -> newVal));
            for (BaseItemDto baseItemDto : baseDataEntryJobList) {
                BaseItemDto tmData = checkMap.get(baseItemDto.getCode());
                // 过滤掉dataMap未返回的（作业可能不存在或者未购买等）
                if (null != tmData) {
                    // 以dataMap权限结果为准
                    baseItemDto.setAccess(tmData.getAccess());
                    candidateDataList.add(baseItemDto);
                }
            }
        }

        // 获取常用的相关信息 保存为常用的功能
        Map<String, ItemData> commonItemMap = getCommonUseData(user, plat, null);
        Set<String> commonItemCodeSet = commonItemMap.keySet();
        // 公用基础资料code
        Set<String> publicDataCodeSet = new HashSet<>();
        Map<String, List<BaseItemDto>> dataMap = new HashMap<>();
        for (BaseItemDto baseItemDto : candidateDataList) {
            // 若是公共数据，则重置appCode和appName，方便走后续分组逻辑
            this.resetAppDataForPublic(baseItemDto);
            // 根据appCode分组
            String appCode = StringUtils.isNotEmpty(baseItemDto.getAppCode()) ? baseItemDto.getAppCode() : "";
            List<BaseItemDto> list = new ArrayList<>();
            if (dataMap.containsKey(appCode)) {
                list = dataMap.get(appCode);
            }
            // 是否为常用数据
            if (commonItemCodeSet.contains(baseItemDto.getCode())) {
                baseItemDto.setCommonUsed(true);
                baseItemDto.setItemId(commonItemMap.get(baseItemDto.getCode()).getId());
            }
            // 是公共数据且code已经存在
            if (this.isPublicExist(baseItemDto, publicDataCodeSet)) {
                continue;
            }
            list.add(baseItemDto);
            dataMap.put(appCode, list);
        }
        // 一次性获取自定义排序数据
        Map<String, List<BasisReportSortData>> customSort = getCustomSort(plat.toString(), user.getUserId());
        // 租户级别排序
        List<TenantModuleRes> tenantModuleList = semcReqService.findTenantModuleList(getModuleCode(plat, pattern));
        Map<String,List<TenantModuleRes>> appCodeToModuleMap = Maps.newLinkedHashMap();
        if(!tenantModuleList.isEmpty()){
            appCodeToModuleMap = tenantModuleList.stream().sorted(Comparator.comparing(TenantModuleRes::getAppSort)).collect(Collectors.groupingBy(TenantModuleRes::getAppCode, LinkedHashMap::new, Collectors.toList()));
            Map<String, List<BaseItemDto>> sortedDataMap = new LinkedHashMap<>();
            for (Map.Entry<String, List<TenantModuleRes>> entry : appCodeToModuleMap.entrySet()) {
                String key = entry.getKey();
                if (dataMap.containsKey(key)) {
                    sortedDataMap.put(key, dataMap.get(key));
                }
            }
            dataMap.forEach((k,v)->{
                if(!sortedDataMap.containsKey(k)){
                    sortedDataMap.put(k,v);
                }
            });
            dataMap = sortedDataMap;
        }
        List<TmDataListDTO> tmDataListDTOList = new ArrayList<>();
        // 整理数据格式
        for (Map.Entry<String, List<BaseItemDto>> entry : dataMap.entrySet()) {
            String appCode = entry.getKey();
            List<BaseItemDto> baseItemDtoList = entry.getValue();
            TmDataListDTO tmDataListDTO = new TmDataListDTO();
            tmDataListDTO.setAppCode(appCode);
            tmDataListDTO.setAppName(baseItemDtoList.get(0).getAppName());
            // 租户的排序与用户级别的排序进行融合
            Map<String, Integer> tier2 = customSort.get("2").stream().collect(Collectors.toMap(e -> e.getAppCode() + "-" + e.getItemCode(), BasisReportSortData::getSort, (existing, replacement) -> existing));
            Map<String, Integer> tier3 = customSort.get("3").stream().collect(Collectors.toMap(e -> e.getAppCode() + "-" + e.getGroupCode() + "-" + e.getItemCode(), BasisReportSortData::getSort, (existing, replacement) -> existing));
            List<TenantModuleRes> tenantModuleRes = appCodeToModuleMap.get(appCode);
            handleTier(tmDataListDTOList, baseItemDtoList, tmDataListDTO, tier2, tier3,tenantModuleRes);
        }
        if(CollectionUtils.isEmpty(tenantModuleList)){
            this.publicBasicDataMoveToFirst(tmDataListDTOList);
        }
        if (Objects.equals(plat, 1)) {
            getFineReportList(tmDataListDTOList, pattern);
        }
        return tmDataListDTOList;
    }

    /**
     * 查询帆软报表
     *
     * @param tmDataListDTOList 报表
     * @param pattern 报表类型
     */
    private void getFineReportList(List<TmDataListDTO> tmDataListDTOList, String pattern) {
        String locale = LocaleContextHolder.getLocale().toString();
        List<FineReportVO.FineReportVOItem> fineReportList = null;
        try {
            fineReportList = fineReportService.getFineReportList("MOBI-STATEMENT".equals(pattern) ? 2 : 1);
        } catch (Exception e) {
            // 不影响其他报表
            logger.error("查询帆软报表异常：", e);
        }
        if (CollectionUtils.isEmpty(fineReportList)) {
            return;
        }
        TmDataListDTO tmDataListDTO = new TmDataListDTO();
        tmDataListDTO.setAppCode("FineReport");
        String appName = "帆软报表";
        if (locale.equalsIgnoreCase("zh_TW")) {
            appName = "帆軟報表";
        }
        if (locale.equalsIgnoreCase("en_US")) {
            appName = "FineReport";
        }
        tmDataListDTO.setAppName(appName);
        tmDataListDTO.setTier("2");
        String finalAppName = appName;
        List<BaseItemDto> fineReportItemList = fineReportList.stream().map(fineReportVO -> {
            BaseItemDto baseItemDto = new BaseItemDto();
            baseItemDto.setCode(fineReportVO.getId());
            baseItemDto.setName(fineReportVO.getText());
            baseItemDto.setCategory("FINE-REPORT");
            baseItemDto.setAppCode("FineReport");
            baseItemDto.setAppName(finalAppName);
            baseItemDto.setCommonUsed(false);
            baseItemDto.setAccess("allow");
            baseItemDto.setIfCustom(false);
            baseItemDto.setIfCommon(false);
            baseItemDto.setCommonApp(false);
            baseItemDto.setSubReport(false);
            baseItemDto.setPath(fineReportVO.getPath());
            return baseItemDto;
        }).collect(Collectors.toList());
        tmDataListDTO.setBaseItemDtoList(fineReportItemList);
        // po要求帆软报表放在最前面
        tmDataListDTOList.add(0, tmDataListDTO);
    }

    private void handleTenantModule(List<TenantModuleRes> tenantModuleRes,Map<String, Integer> tire,String type){
        if(CollectionUtils.isEmpty(tenantModuleRes) || !tire.isEmpty()){
            return;
        }
        tenantModuleRes.stream().sorted(Comparator.comparing(TenantModuleRes::getItemSort)).forEach(e->{
            tire.put(!"group".equals(type)?(e.getAppCode() + "-" + e.getCode()):(e.getAppCode() + "-" + e.getGroupCode() + "-" + e.getCode()),e.getItemSort());
        });
    }

    private String getModuleCode(Integer plat,String pattern){
        if (Objects.equals(plat, 0)) {
            return BASE_DATA_PATTERN;
        }
        // 报表pattern
        else if (Objects.equals(plat, 1)) {
            return StringUtils.isBlank(pattern)?REPORT_PATTERN:null;
        }
        return null;
    }

    private List<BaseItemDto> getAllTmDataList(AuthoredUser user, Integer plat, boolean allowRepeat, String pattern, String effect) {
        // 整理pattern
        pattern = pattern == null ? "" : pattern;
        // 基础资料pattern
        if (Objects.equals(plat, 0)) {
            pattern = BASE_DATA_PATTERN;
        }
        // 报表pattern
        else if (Objects.equals(plat, 1)) {
            pattern = StringUtils.isNotBlank(pattern) ? pattern : REPORT_PATTERN;
        }

        // 获取web端业务数据录入作业列表，从dataMap获取（dataMap已做过权限校验）
        if (BASE_DATA_PATTERN.equals(pattern)) {
            List<TmDataEntryDTO> allTmDataList = DataMapManager.getDataEntryList(effect).getResponse();
            return null != allTmDataList ? allTmDataList.stream().map(DataConverter.INSTANCE::toBaseItem).collect(Collectors.toList()) : new ArrayList<>();
        }
        // 获取web报表作业列表，从dataMap获取（dataMap已做过权限校验）
        else if (REPORT_PATTERN.equals(pattern)) {
            List<TmDataEntryDTO> allTmDataList = DataMapManager.getStatementList(effect).getResponse();
            return null != allTmDataList ? allTmDataList.stream().map(DataConverter.INSTANCE::toBaseItem).collect(Collectors.toList()) : new ArrayList<>();
        }
        // 获取其他类型作业列表，从KM获取，需要做权限校验
        else {
            List<BaseItemDto> baseItems = queryTmDataFromKM(user, plat, pattern);
            if (CollectionUtils.isEmpty(baseItems)) {
                return new ArrayList<>();
            }

            // 访问权限校验
            List<String> codeList = baseItems.stream().map(BaseItemDto::getCode).distinct().collect(Collectors.toList());
            // 如果设置了过滤，则对数据进行过滤（不明白这里为啥是allowRepeat，而不是!allowRepeat)
            if (allowRepeat) {
                codeList = codeList.stream().distinct().collect(Collectors.toList());
            }
            // 获取作业权限
            Map<String, String> accessMap = getAccessData(user, plat, codeList);
            // 移除掉未购买的
            baseItems.removeIf(baseItem -> Constants.ACCESS_NO_BUY.equals(accessMap.get(baseItem.getCode())));
            // 设置访问权限
            baseItems.forEach(baseItem -> baseItem.setAccess(accessMap.get(baseItem.getCode())));

            return baseItems;
        }
    }

    /**
     * 将公共资料移到首位
     *
     * @param tmDataListDTOList
     */
    private void publicBasicDataMoveToFirst(List<TmDataListDTO> tmDataListDTOList) {
        List<TmDataListDTO> commonAppList = tmDataListDTOList.stream().filter(data -> BaseGlobalConstant.APP_CODE_PUBLIC_BASIC_DATA.equals(data.getAppCode()) || isCommonApp(data)).collect(Collectors.toList());

        if (CollectionUtils.isNotEmpty(commonAppList)) {
            tmDataListDTOList.removeAll(commonAppList);
            // 公共应用置顶
            tmDataListDTOList.addAll(0, commonAppList);
        }
    }

    /**
     * 判断是否有公共应用
     *
     * @param data
     * @return
     */
    private boolean isCommonApp(TmDataListDTO data) {
        if (CollectionUtils.isNotEmpty(data.getBaseItemDtoList())) {
            return data.getBaseItemDtoList().stream().anyMatch(baseItemDto -> BooleanUtils.isTrue(baseItemDto.isCommonApp()));
        } else if (CollectionUtils.isNotEmpty(data.getGroupList())) {
            for (TmDataTierListDTO tmDataTierListDTO : data.getGroupList()) {
                if (CollectionUtils.isNotEmpty(tmDataTierListDTO.getBaseItemDtoList())) {
                    return tmDataTierListDTO.getBaseItemDtoList().stream().anyMatch(baseItemDto -> BooleanUtils.isTrue(baseItemDto.isCommonApp()));
                }
            }
        }
        return true;
    }

    /**
     * 若是公共数据，则重置appCode和appName，方便走后续分组逻辑
     *
     * @param baseItemDto 基础资料
     */
    private void resetAppDataForPublic(BaseItemDto baseItemDto) {
        if (baseItemDto.isIfCommon()) {
            baseItemDto.setAppCode(BaseGlobalConstant.APP_CODE_PUBLIC_BASIC_DATA);
            baseItemDto.setAppName(messageUtils.getMessage(BaseGlobalConstant.APP_NAME_KEY_PUBLIC_BASIC_DATA));
        }
    }

    /**
     * 是公共数据且code已经存在
     *
     * @param baseItemDto    基础资料
     * @param publicDataCode 基础资料code集合，用于去重
     * @return true：是公共数据且已存在
     */
    private boolean isPublicExist(BaseItemDto baseItemDto, Set<String> publicDataCode) {
        return BooleanUtils.isTrue(baseItemDto.isIfCommon()) && !publicDataCode.add(baseItemDto.getCode());
    }

    /**
     * 根据用户信息，从km查询对应的基础资料信息
     *
     * @param user 用户信息
     * @param plat 所属模块 0：基础资料 1：报表
     * @return list
     */
    private List<BaseItemDto> queryTmDataFromKM(AuthoredUser user, Integer plat, String pattern) {
        //获取ThemMap全部数据
        List<BaseItemDto> respList;
        try {
            respList = baseThemeMapService.getActivitiesByPattern(user, plat, null, pattern);
        } catch (Exception e) {
            //增加异常处理，目前km不支持个人租户查询对应的作业信息
            String errMsg = "User id:" + user.getUserId() + " userName:" + user.getUserName() + " query KM With plat" + plat + " Error.";
            logger.error(errMsg, e.getMessage());
            return new ArrayList<>();
        }
        return respList;
    }

    /**
     * 根据用户信息，查询对应模块的常用信息
     *
     * @param user   用户信息
     * @param plat   所属模块
     * @param cateId 分类id
     * @return set
     */
    private Map<String, ItemData> getCommonUseData(AuthoredUser user, Integer plat, Long cateId) {
        //获取常用数据
        List<ItemData> commonItemList = itemService.queryItemList(user, plat, cateId);
        //获取常用数据的code
        Map<String, ItemData> commonItemMap = new HashMap<>();
        if (CollectionUtils.isNotEmpty(commonItemList)) {
            commonItemMap = commonItemList.stream().collect(Collectors.toMap(ItemData::getItemCode, Function.identity(), (oldValue, newValue) -> newValue));
        }
        return commonItemMap;
    }


    private void handleTier(List<TmDataListDTO> tmDataListDTOList, List<BaseItemDto> baseItemDtoList, TmDataListDTO tmDataListDTO, Map<String, Integer> tier2, Map<String, Integer> tier3,List<TenantModuleRes> tenantModuleRes) {
        // 此引作业组code不存在 则说明为2层级 应用下就是作业
        List<String> tier = baseItemDtoList.stream().map(BaseItemDto::getGroupCode).collect(Collectors.toList());
        if (tier.contains(null)) {
            tmDataListDTO.setTier("2");
            tier2.put("tier", 2);
            handleTenantModule(tenantModuleRes,tier2,null);
            sortCommon(baseItemDtoList, tier2);
            tmDataListDTO.setBaseItemDtoList(baseItemDtoList);
        } else {
            tmDataListDTO.setTier("3");
            Map<String, TmDataTierListDTO> duplicateMap = new LinkedHashMap<>();
            for (BaseItemDto baseItemDto : baseItemDtoList) {
                if (duplicateMap.containsKey(baseItemDto.getGroupCode())) {
                    duplicateMap.get(baseItemDto.getGroupCode()).getBaseItemDtoList().add(baseItemDto);
                } else {
                    TmDataTierListDTO tmDataTierListDTO = new TmDataTierListDTO();
                    tmDataTierListDTO.setGroupCode(baseItemDto.getGroupCode());
                    tmDataTierListDTO.setGroupName(baseItemDto.getGroupName());
                    tmDataTierListDTO.setGroupSort(baseItemDto.getGroupSort());
                    List<BaseItemDto> baseList = new ArrayList<>();
                    baseList.add(baseItemDto);
                    tmDataTierListDTO.setBaseItemDtoList(baseList);
                    duplicateMap.put(baseItemDto.getGroupCode(), tmDataTierListDTO);
                }
            }
            List<TmDataTierListDTO> list = new ArrayList<>(duplicateMap.values());
            // 按照租户排序
            boolean hasNoTenantGroup = true;
            if(CollectionUtils.isNotEmpty(tenantModuleRes)){
                // 1. 对tenantModuleRes按groupSort升序排序，并提取groupCode与排序值的映射
                Map<String, Integer> groupSortMap = tenantModuleRes.stream()
                        .filter(e->StringUtils.isNotBlank(e.getGroupCode()))
                        // 按groupSort升序排序
                        .sorted(Comparator.comparingInt(TenantModuleRes::getGroupSort))
                        // 收集为groupCode -> groupSort的映射
                        .collect(Collectors.toMap(
                                TenantModuleRes::getGroupCode,
                                TenantModuleRes::getGroupSort,
                                // 处理groupCode重复的情况，保留排序值小的
                                (existing, replacement) -> existing < replacement ? existing : replacement,
                                // 使用LinkedHashMap保持排序后的插入顺序（便于后续处理不匹配项）
                                LinkedHashMap::new
                        ));
                if(MapUtils.isNotEmpty(groupSortMap)){
                    hasNoTenantGroup = false;
                    list.sort(Comparator.comparing(a-> MapUtils.getInteger(groupSortMap,a.getGroupCode(),Integer.MAX_VALUE)));
                }
            }
            if(hasNoTenantGroup){
                //作业组排序 未设定 按作业组code排序  设定 则按作业组sort排序  作业同理
                List<String> groupSortList = list.stream().map(TmDataTierListDTO::getGroupSort).collect(Collectors.toList());
                if (groupSortList.contains(null)) {
                    list.sort(Comparator.comparing(a -> {
                        try {
                            return Integer.parseInt(a.getGroupCode());
                        } catch (NumberFormatException ex) {
                            return -1;
                        }
                    }));
                } else {
                    list.sort(Comparator.comparing(a -> {
                        try {
                            return Integer.parseInt(a.getGroupSort());
                        } catch (NumberFormatException ex) {
                            return -1;
                        }
                    }));
                }
            }
            for (TmDataTierListDTO tmDataTierListDTO : list) {
                List<BaseItemDto> baseItemDtoList1 = tmDataTierListDTO.getBaseItemDtoList();
                tier3.put("tier", 3);
                handleTenantModule(tenantModuleRes,tier3,"group");
                sortCommon(baseItemDtoList1, tier3);
            }
            tmDataListDTO.setGroupList(list);
        }
        tmDataListDTOList.add(tmDataListDTO);
    }

    private void sortCommon(List<BaseItemDto> baseItemDtoList, Map<String, Integer> customSortMap) {
        List<String> collect = baseItemDtoList.stream().map(BaseItemDto::getSort).collect(Collectors.toList());
        splitList(baseItemDtoList, customSortMap, collect.contains(null));
    }

    //得到自定义排序数据
    private Map<String, List<BasisReportSortData>> getCustomSort(String platform, String userId) {
        //根据platform、userId将数据查询出来
        List<BasisReportSortData> customSortList = customSortService.selectList(new LambdaQueryWrapper<BasisReportSortData>().eq(BasisReportSortData::getPlatform, platform).eq(BasisReportSortData::getUserId, userId));
        //根据tier分为两组
        Map<String, List<BasisReportSortData>> map = customSortList.stream().collect(Collectors.groupingBy(BasisReportSortData::getTier));
        map.computeIfAbsent("2", k -> new ArrayList<>());
        map.computeIfAbsent("3", k -> new ArrayList<>());
        return map;
    }


    private void splitList(List<BaseItemDto> baseItemDtoList, Map<String, Integer> customSortMap, Boolean byCharacterSort) {

        Predicate<BaseItemDto> predicateTiter2Against = e -> customSortMap.get(e.getAppCode() + "-" + e.getCode()) == null;
        Predicate<BaseItemDto> predicateTiter3Against = e -> customSortMap.get(e.getAppCode() + "-" + e.getGroupCode() + "-" + e.getCode()) == null;
        List<BaseItemDto> kmSortList = new ArrayList<>();
        switch (customSortMap.get("tier")) {
            case 2:
                kmSortList = baseItemDtoList.stream().filter(predicateTiter2Against).collect(Collectors.toList());
                doKMSort(kmSortList, byCharacterSort);

                baseItemDtoList.removeIf(predicateTiter2Against);
                baseItemDtoList.sort(Comparator.comparing(e -> customSortMap.get(e.getAppCode() + "-" + e.getCode())));
                break;
            case 3:
                kmSortList = baseItemDtoList.stream().filter(predicateTiter3Against).collect(Collectors.toList());
                doKMSort(kmSortList, byCharacterSort);

                baseItemDtoList.removeIf(predicateTiter3Against);
                baseItemDtoList.sort(Comparator.comparing(e -> customSortMap.get(e.getAppCode() + "-" + e.getGroupCode() + "-" + e.getCode())));
                break;
        }
        //最后将没有权限的都放在末尾
        Comparator<BaseItemDto> accessComparator = (item1, item2) -> {
            if ("forbidden".equals(item1.getAccess()) && !"forbidden".equals(item2.getAccess())) {
                return 1;
            } else if (!"forbidden".equals(item1.getAccess()) && "forbidden".equals(item2.getAccess())) {
                return -1;
            }
            return 0; // 保留原始排序
        };
        baseItemDtoList.addAll(kmSortList);
        baseItemDtoList.sort(accessComparator);
    }

    private void doKMSort(List<BaseItemDto> itemDtos, Boolean byCharacterSort) {
        if (byCharacterSort) {
            itemDtos.sort((a, b) -> Collator.getInstance(Locale.CHINA).compare(a.getCode(), b.getCode()));
        } else {
            itemDtos.sort(Comparator.comparing(a -> Integer.parseInt(a.getSort())));
        }
    }

    /**
     * 常用分组
     *
     * @Author：SYQ
     * @Date：2021/10/13 17:24
     */
    public List<CategoryData> finCategory(AuthoredUser user, Integer plat, Long cateId) {
        List<CategoryData> list = new ArrayList<>();
        List<CategoryData> cateList = categoryService.queryCateList(user, plat, cateId);
        CategoryData defaultData = new CategoryData();

        String name = "{\"zh_TW\":\"默認分組\",\"en_US\":\"Default Grouping\",\"zh_CN\":\"默认分组\"}";
        defaultData.setCateName(MessageUtils.getMessageByCurrentLanguage(name));
        defaultData.setId(-1L);
        defaultData.setParentId(-1L);
        list.add(defaultData);
        list.addAll(cateList);
        return list;
    }

    /**
     * 根据权限获取发起项目
     *
     * @param user
     * @return
     */
    @Override
    public List<TmStartProjectDTO> startProjectList(AuthoredUser user, Boolean needSearchStartProject, List<TmStartProjectDTO> startProjectJobList) {
        // 从dataMap获取该用户有访问权限的手动发起项目列表
        List<TmStartProjectDTO> allAccessibleStartProjectList = DataMapManager.getStartProjectList().getResponse();
        if (CollectionUtils.isEmpty(allAccessibleStartProjectList)) {
            return new ArrayList<>();
        }

        // 规整commonApp值
        for (TmStartProjectDTO accessibleStartProject : allAccessibleStartProjectList) {
            if (null == accessibleStartProject.getCommonApp()) {
                accessibleStartProject.setCommonApp(false);
            }
        }

        // 3. 若不要以入参startProjectJobList为返回结果基础，则返回该用户有权限访问的手动发起项目列表
        if (!needSearchStartProject || CollectionUtils.isEmpty(startProjectJobList)) {
            return allAccessibleStartProjectList;
        }
        // 返回startProjectJobList中有访问权限的作业
        else {
            Map<String, TmStartProjectDTO> checkMap = allAccessibleStartProjectList.stream().collect(Collectors.toMap(TmStartProjectDTO::getCode, Function.identity(), (newVal, oldVal) -> newVal));

            List<TmStartProjectDTO> resultList = new ArrayList<>();
            for (TmStartProjectDTO startProjectJob : startProjectJobList) {
                TmStartProjectDTO accessibleStartProject = checkMap.get(startProjectJob.getCode());
                if (null != accessibleStartProject) {
                    // 以dataMap返回的作业commonApp为准
                    startProjectJob.setCommonApp(accessibleStartProject.getCommonApp());
                    resultList.add(startProjectJob);
                }
            }

            return resultList;
        }
    }

    @Autowired
    BasisReportSortMapper customSortService;

    @Override
    @Transactional(rollbackFor = Exception.class, transactionManager = BaseAudcDataSourceConfig.BASE_AUDC_DATASOURCE_TRANSACTION_MANAGER_BUSINESS)
    public Boolean saveCustomSort(AuthoredUser user, BatchSortItemReq items) {

        Set<String> assistSet = new HashSet<>();
        for (BatchSortItemReq.Item item : items.getItems()) {
            if (!assistSet.add(item.getCode())) {
                logger.error(" Duplicate key :{}", item.getCode());
                throw BusinessException.create(-1, "Duplicate key");
            }
        }

        //首先根据tier判断当前这个是user.app,还是user.app.group的排序
        switch (items.getTier()) {
            case "2":
                customSortService.delete(new LambdaQueryWrapper<BasisReportSortData>().eq(BasisReportSortData::getAppCode, items.getAppCode()).eq(BasisReportSortData::getUserId, user.getUserId()));
                break;
            case "3":
                customSortService.delete(new LambdaQueryWrapper<BasisReportSortData>().eq(BasisReportSortData::getAppCode, items.getAppCode()).eq(BasisReportSortData::getGroupCode, items.getGroupCode()).eq(BasisReportSortData::getUserId, user.getUserId()));
                break;
            default:
                logger.error("illegal tier");
                throw new RuntimeException("illegal tier");
        }
        //先将这个用户的app.group下的数据都删了，然后再插入新的
        List<BasisReportSortData> entityList = sortReq2SortData(items, user.getUserId());
        Integer insertCount = customSortService.insertBatchSomeColumn(entityList);
        return null != insertCount && 0 < insertCount;
    }

    private List<BasisReportSortData> sortReq2SortData(BatchSortItemReq items, String userId) {
        List<BasisReportSortData> list = new ArrayList<>();
        items.getItems().stream().forEach(e -> { //这边不进行filter过滤没有权限的item，都存到数据库，读取的时候再进行权限校验排序
            BasisReportSortData basisReportSortData = new BasisReportSortData();
            basisReportSortData.setAppCode(items.getAppCode());
            basisReportSortData.setAppName(items.getAppName());
            if ("3".equals(items.getTier())) {
                basisReportSortData.setGroupCode(items.getGroupCode());
                basisReportSortData.setGroupName(items.getGroupName());
            }
            basisReportSortData.setUserId(userId);
            basisReportSortData.setPlatform(items.getPlatform());
            basisReportSortData.setItemCode(e.getCode());
            basisReportSortData.setItemName(e.getName());
            basisReportSortData.setSort(e.getSort());
            basisReportSortData.setTier(items.getTier());
            list.add(basisReportSortData);
        });
        return list;
    }

    /**
     * 获取作业权限
     *
     * @Author：SYQ
     * @Date：2021/11/10 10:59
     */
    private Map<String, String> getAccessData(AuthoredUser authoredUser, Integer plat, List<String> codeList) {
        Map<String, String> resultMap = new HashMap<>();
        List<TypeActivities> typeActivitiesList = new ArrayList<>();
        TypeActivities typeActivity = new TypeActivities();
        typeActivity.setType(plat + "");
        typeActivity.setTmActivityIdList(codeList);
        typeActivitiesList.add(typeActivity);
        List<TypeActivitiesAccessible> activityPermission = permissionCheckService.checkTypeActivitiesAccessible(authoredUser, typeActivitiesList);
        if (CollectionUtils.isEmpty(activityPermission)) {
            return resultMap;
        }
        // 每个作业的访问权限
        List<ActivityAccessible> activityAccessibleList = activityPermission.get(0).getActivityAccessibleList();
        if (CollectionUtils.isEmpty(activityAccessibleList)) {
            return resultMap;
        }
        for (ActivityAccessible activityAccessible : activityAccessibleList) {
            if (Objects.isNull(activityAccessible)) {
                continue;
            }
            resultMap.put(activityAccessible.getTmActivityId(), activityAccessible.getAccess());
        }
        return resultMap;
    }

}
