package com.digiwin.athena.base.application.service.importstatistics.service.impl;

import com.alibaba.nacos.common.utils.MapUtils;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.atdm.constant.ErrorCodeEnum;
import com.digiwin.athena.atdm.importstatistics.common.api.DapResponse;
import com.digiwin.athena.atdm.importstatistics.dto.DownloadTemplateDTO;
import com.digiwin.athena.atdm.importstatistics.dto.DownloadTemplateReqDTO;
import com.digiwin.athena.atdm.importstatistics.entity.CellTypeContainer;
import com.digiwin.athena.atdm.importstatistics.entity.valueobject.ApiDataFieldLocaleMetadataDTO;
import com.digiwin.athena.atdm.importstatistics.entity.valueobject.Dictionary;
import com.digiwin.athena.atdm.importstatistics.entity.valueobject.GetActionLocaleResponseDTO;
import com.digiwin.athena.atdm.importstatistics.entity.valueobject.MetaDataType;
import com.digiwin.athena.atdm.importstatistics.entity.valueobject.MultiLanguageDTO;
import com.digiwin.athena.atdm.importstatistics.util.excel.ExcelTypeEnum;
import com.digiwin.athena.atdm.importstatistics.util.excel.ExcelUtil;
import com.digiwin.athena.base.application.service.importstatistics.service.DataEntryEspService;
import com.digiwin.athena.base.application.service.importstatistics.service.helpler.ExcelHelper;
import com.digiwin.athena.base.infrastructure.manager.abt.AbtService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.CellStyle;
import org.apache.poi.ss.usermodel.CellType;
import org.apache.poi.ss.usermodel.DataFormat;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.web.client.RestTemplate;

import javax.annotation.Resource;
import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletResponse;
import java.io.ByteArrayOutputStream;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

/**
 * @ClassName MetaDataService
 * @Description 元数据service
 * @Author zhuangli
 * @Date 2021/4/8 11:16
 * @Version 1.0
 **/
@Slf4j
@Service
public class DataEntryMetaDataService {

    @Resource
    RestTemplate restTemplate;

    @Autowired
    ExcelHelper excelHelper;

    @Value("${themeMap.uri}")
    private String knowledgeGraphUrl;

    @Autowired
    DataEntryEspService espService;

    private final String HEADER_KEY = "key";
    private final String HEADER_NAME = "name";
    private final String METADATA_URL = "/restful/service/knowledgegraph/Action/Metadata?actionId={actionId}";
    private final String DICT_BATCH_URL = "/restful/service/knowledgegraph/Dict/FindDictByKeyList";
    private final String DICT_URL = "/restful/service/knowledgegraph/Dict/DictByKey?key={key}";
    private final String FILE_EXIST_URL = "/restful/service/knowledgegraph/File/FileId?key={key}&version={version}";
    private final String TEMPLATE_DOWNLOAD_URL = "/restful/service/knowledgegraph/File/DownloadTemplate?fileId={fileId}";
    private final String PRODUCT_NAME_URL = "/restful/service/knowledgegraph/Action/ProductName?serviceName={serviceName}";
    private final String ACTIVITY_NAME_URL = "/restful/service/knowledgegraph/task/activityNamesByCode?activityCode={activityCode}";
    private final String TEMPLATE_NAME = "template";
    private final String EXCEL_SUFFIX = ".xlsx";

    @Autowired
    AbtService abtService;

    public String getProductName(String token, String serviceName) {
        HttpHeaders headers = new HttpHeaders();
        headers.setAccept(Arrays.asList(MediaType.APPLICATION_JSON));
        MediaType type = MediaType.parseMediaType("application/json; charset=UTF-8");
        headers.setContentType(type);
        headers.add("token", token);
        Map<String, String> parameters = new HashMap<>();
        parameters.put("serviceName", serviceName);
        HttpEntity<Map<String, String>> entity = new HttpEntity<>(headers);
        ResponseEntity<DapResponse> result = restTemplate.exchange(knowledgeGraphUrl + PRODUCT_NAME_URL, HttpMethod.GET, entity, DapResponse.class, parameters);
        if (result.getStatusCodeValue() == 200 && (result.getBody().getStatus() == 200)) {
            String productName = (String) result.getBody().getResponse();
            return productName;
        } else {
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0094.getErrCode(), "未找到元数据");
        }
    }

    public GetActionLocaleResponseDTO getActionMetaData(String actionId, String token, String locale) {
        HttpHeaders headers = new HttpHeaders();
        headers.setAccept(Arrays.asList(MediaType.APPLICATION_JSON));
        MediaType type = MediaType.parseMediaType("application/json; charset=UTF-8");
        headers.setContentType(type);
        headers.add("token", token);
        if (!StringUtils.isEmpty(locale)) {
            headers.add("locale", locale);
        }
        Map<String, String> parameters = new HashMap<>();
        parameters.put("actionId", actionId);
        HttpEntity<Map<String, String>> entity = new HttpEntity<>(headers);
        ResponseEntity<DapResponse<GetActionLocaleResponseDTO>> result = restTemplate.exchange(knowledgeGraphUrl + METADATA_URL, HttpMethod.GET, entity, new ParameterizedTypeReference<DapResponse<GetActionLocaleResponseDTO>>() {
        }, parameters);
        if (200 == result.getStatusCodeValue() && null != result.getBody() && 200 == result.getBody().getStatus()) {
            return result.getBody().getResponse();
        } else {
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0105.getErrCode(), "调用" + knowledgeGraphUrl + "knowledgegraph/Action/Metadata?actionId=" + actionId + "未找到元数据");
        }
    }

    public Map<String, List<Dictionary>> getDictByKeyList(List<String> keyList, String token, String securityToken, String locale) {
        if (CollectionUtils.isEmpty(keyList)) {
            return Collections.EMPTY_MAP;
        }
        HttpHeaders headers = new HttpHeaders();
        headers.setAccept(Arrays.asList(MediaType.APPLICATION_JSON));
        MediaType type = MediaType.parseMediaType("application/json; charset=UTF-8");
        headers.setContentType(type);
        headers.add("token", token);
        if (!StringUtils.isEmpty(securityToken)) {
            headers.add("security-token", securityToken);
        }
        if (!StringUtils.isEmpty(locale)) {
            headers.add("locale", locale);
        }
        Map<String, Object> parameters = new HashMap<>();
        parameters.put("keyList", keyList);
        HttpEntity<Map<String, Object>> entity = new HttpEntity<>(parameters, headers);
        ResponseEntity<DapResponse<Map<String, List<Dictionary>>>> result = restTemplate.exchange(knowledgeGraphUrl + DICT_BATCH_URL, HttpMethod.POST, entity, new ParameterizedTypeReference<DapResponse<Map<String, List<Dictionary>>>>() {
        });
        if (200 == result.getStatusCodeValue() && null != result.getBody() && 200 == result.getBody().getStatus()) {
            if (MapUtils.isEmpty(result.getBody().getResponse())) {
                throw BusinessException.create(ErrorCodeEnum.NUM_500_0096.getErrCode(), "未找到元数据");
            }
            return result.getBody().getResponse();
        } else {
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0096.getErrCode(), "未找到元数据");
        }
    }

    /**
     * @return
     * @Author zhuangli
     * @Description 根据actionId获取excel列名
     * @Date 13:46 2021/12/1
     * @Param
     **/
    public List<Map> getHeaders(String actionId, String userToken, String locale) {
        GetActionLocaleResponseDTO getActionLocaleResponseDTO = getActionMetaData(actionId, userToken, locale);
        return getHeaders(getActionLocaleResponseDTO);
    }

    public List<Map> getHeaders(GetActionLocaleResponseDTO getActionLocaleResponseDTO) {
        ApiDataFieldLocaleMetadataDTO apiDataFieldLocaleMetadataDTO = getActionLocaleResponseDTO.getRequest().getParameters().get(0);
        List<Map> headers = new LinkedList<>();
        apiDataFieldLocaleMetadataDTO.getField().forEach(item -> {
            Map header = new HashMap(2);
            header.put(HEADER_KEY, item.getData_name());
            header.put(HEADER_NAME, item.getDescription());
            headers.add(header);
        });
        return headers;
    }

    public Map<String, String> getResponseHeaders(GetActionLocaleResponseDTO getActionLocaleResponseDTO, List<String> keyList) {
        ApiDataFieldLocaleMetadataDTO apiDataFieldLocaleMetadataDTO = getActionLocaleResponseDTO.getResponse().getData();
        Map<String, String> headerMap = new HashMap<>();
        apiDataFieldLocaleMetadataDTO.getField().forEach(item -> {
            // 解决无法导出单身问题 bug:63701
            keyList.add(item.getData_name());
            headerMap.put(item.getData_name(), item.getDescription());
        });
        return headerMap;
    }


    public void downloadTemplate(String actionId, String token, String locale, String fileName, DownloadTemplateDTO downloadTemplateDTO, HttpServletResponse response) {
        response.setContentType(ExcelTypeEnum.XLSX.value());
        //response.setCharacterEncoding("utf-8");
        try {
            response.addHeader("Content-Disposition", "attachment;filename=" + (StringUtils.isEmpty(fileName) ? TEMPLATE_NAME : URLEncoder.encode(fileName, "UTF-8")) + EXCEL_SUFFIX);
        } catch (UnsupportedEncodingException e) {
            log.warn("文件名转义失败");
            response.addHeader("Content-Disposition", "attachment;filename=" + TEMPLATE_NAME + EXCEL_SUFFIX);
        }
        response.addHeader("Access-Control-Expose-Headers", "Content-Disposition");

        byte[] fileBytes = abtService.downloadTemplate(actionId, token, locale, downloadTemplateDTO);
        if (null == fileBytes || fileBytes.length == 0) {
            log.error("下载的模板文件流为空！");
            return;
        }
        try (ServletOutputStream out = response.getOutputStream()) {
            out.write(fileBytes, 0, fileBytes.length);
            out.flush();
        } catch (Exception e) {
            log.error("下载模板失败", e);
        }
    }

    /**
     * 批量处理逻辑
     *
     * @param params   下载模版参数
     * @param response 响应
     * @param token    用户token
     * @param locale   语言
     */
    public void downloadTemplateBatch(List<DownloadTemplateReqDTO> params, HttpServletResponse response, String token, String locale) {

        //set response content-type: zip
        response.setContentType("application/zip");

        try {
            response.addHeader("Content-Disposition", "attachment;filename=" + URLEncoder.encode("templates.zip", "UTF-8"));
        } catch (UnsupportedEncodingException e) {
            response.addHeader("Content-Disposition", "attachment;filename=templates.zip");
        }
        response.addHeader("Access-Control-Expose-Headers", "Content-Disposition");


        // loop abtService excel output stream
        try (ByteArrayOutputStream zipOutStream = new ByteArrayOutputStream(); ZipOutputStream zipOutputStream = new ZipOutputStream(zipOutStream); ServletOutputStream out = response.getOutputStream()) {

            for (DownloadTemplateReqDTO e : params) {
                DownloadTemplateDTO downloadTemplateDTO = new DownloadTemplateDTO();
                downloadTemplateDTO.setActionInfo(e.getActionInfo());
                downloadTemplateDTO.setFileName(e.getFileName());
                downloadTemplateDTO.setRequiredFields(e.getRequiredFields());
                byte[] fileBytes = abtService.downloadTemplate(e.getActionId(), token, locale, downloadTemplateDTO);
                if (fileBytes != null && fileBytes.length > 0) {
                    zipOutputStream.putNextEntry(new ZipEntry(e.getFileName() + ".xlsx"));
                    zipOutputStream.write(fileBytes);
                    zipOutputStream.closeEntry();
                }
            }

            zipOutputStream.finish(); // 确保 ZIP 正确关闭
            out.write(zipOutStream.toByteArray());
            out.flush();
        } catch (Exception e) {
            log.error("下载 ZIP 失败", e);
        }
    }


    private void createChildSheet(XSSFWorkbook wb, List<CellTypeContainer> businessKeyContainer, CellTypeContainer cellTypeContainerMain, String locale, Set<String> requiredFields) {
        String mainKey = cellTypeContainerMain.getKeyName();
        String mainKeyDescription = cellTypeContainerMain.getKeyDescription();
        Sheet sheet = wb.createSheet(ExcelHelper.getSheetName(mainKey, mainKeyDescription, false));
        List<CellTypeContainer> cellTypes = cellTypeContainerMain.getChildren();
        // 支持子单身三层机构
        List<CellTypeContainer> childBusinessKeyContainerList = getBusinessKeyContainer(cellTypes);
        // 兼容DF中也配置了BK的场景:先从DF中剔除调BK，再将BK统一添加到最前面
        excelHelper.addBkContainer(cellTypes, businessKeyContainer);
        createSheetHeader(wb, sheet, childBusinessKeyContainerList, cellTypes, locale, requiredFields);
    }

    private void createSheetHeader(XSSFWorkbook wb, Sheet sheet, List<CellTypeContainer> businessKeyContainer, List<CellTypeContainer> cellTypes, String locale, Set<String> requiredFields) {
        DataFormat fmt = wb.createDataFormat();
        int index = 0;
        //设置列属性
        for (int i = 0; i < cellTypes.size(); i++) {
            MetaDataType metaDataType = cellTypes.get(i).getProtoType();
            CellTypeContainer cellTypeContainer = cellTypes.get(i);
            CellStyle cellStyle = wb.createCellStyle();
            switch (metaDataType) {
                case STRING:
                    if (null != cellTypeContainer.getPrecision() && cellTypeContainer.getPrecision().length != 0) {
                        ExcelUtil.excelRuleStringLength(locale, sheet, 0, cellTypeContainer.getPrecision()[0], 2, 65536, index, index);
                    }
                    cellStyle.setDataFormat(fmt.getFormat("@"));
                    sheet.setDefaultColumnStyle(index, cellStyle);
                    break;
                case NUMBER:
                    //cellTypeContainer.setPrecision(new int[]{5, 3});
                    ExcelUtil.excelRuleIsNumber(locale, sheet, 2, 65536, index, index);
                    break;
                case DATE:
                    cellStyle.setDataFormat(fmt.getFormat(cellTypeContainer.getDateFormat()));
                    sheet.setDefaultColumnStyle(index, cellStyle);
                    ExcelUtil.excelRuleDateFormat(locale, sheet, "1720-01-01", "", cellTypeContainer.getDateFormat(), 2, 65536, index, index);
                    /*cellStyle.setDataFormat(fmt.getFormat("@"));
                    sheet.setDefaultColumnStyle(i, cellStyle);*/
                    break;
                case OBJECT:
                    //log.error("不应该出现的类型for actionId:{}", actionId);
                    //判断是否是数组类型,且单头存在businessKey
                    if (cellTypeContainer.getArray() && !CollectionUtils.isEmpty(businessKeyContainer)) {
                        createChildSheet(wb, businessKeyContainer, cellTypeContainer, locale, requiredFields);
                        index--;
                    }
                    if (null != cellTypeContainer.getListEnum() && cellTypeContainer.getListEnum().length != 0) {
                        ExcelUtil.excelRuleSelect(locale, sheet, cellTypeContainer.getListEnum(), 2, 65536, index, index);
                    }
                    break;
                case BOOLEAN:
                    ExcelUtil.excelRuleSelect(locale, sheet, cellTypeContainer.getListEnum(), 2, 65536, index, index);
                    break;
                case ENUM:
                    if (null != cellTypeContainer.getListEnum() && cellTypeContainer.getListEnum().length != 0) {
                        ExcelUtil.excelRuleSelect(locale, sheet, cellTypeContainer.getListEnum(), 2, 65536, index, index);
                    }
                    break;
                default:
                    break;
            }
            index++;
        }
        /*创建行Rows及单元格Cells*/
        //第一二行为标题
        Row headRow1 = sheet.createRow(0);
        Row headRow2 = sheet.createRow(1);
        //重置index
        index = 0;
        for (int i = 0; i < cellTypes.size(); i++) {
            //跳过单身字段
            if (!cellTypes.get(i).getArray()) {
                //创建单元格
                Cell cell1 = headRow1.createCell(index);
                Cell cell2 = headRow2.createCell(index);
                //设置值
                if ((!CollectionUtils.isEmpty(requiredFields)) && requiredFields.contains(cellTypes.get(i).getKeyName())) {
                    cell1.setCellValue(ExcelUtil.addRequiredMark(wb, cellTypes.get(i).getKeyDescription()));
                    cell2.setCellValue(cellTypes.get(i).getKeyName());
                } else {
                    cell1.setCellValue(cellTypes.get(i).getKeyDescription());
                    cell2.setCellValue(cellTypes.get(i).getKeyName());
                }
                index++;
            }
        }
    }

    private List<CellTypeContainer> getBusinessKeyContainer(List<CellTypeContainer> cellTypes) {
        List<CellTypeContainer> cellTypeContainers = new LinkedList<>();
        for (CellTypeContainer cellType : cellTypes) {
            if (cellType.getBusinessKey()) {
                cellTypeContainers.add(cellType);
            }
        }
        return cellTypeContainers;
    }

    /**
     * @return
     * @Author zhuangli
     * @Description 下载模板, 根据actionId&locale先判断在文档中心是否存在自定义模板的模板,如果有自定义模板则优先使用自定义;
     * 否则根据api描述自动生成模板
     * @Date 14:01 2022/2/10
     * @Param
     **/
    public void downloadTemplate(String actionId, String token, String locale, String fileName, HttpServletResponse response) {
        downloadTemplate(actionId, token, locale, fileName, null, response);
    }

    public List<CellTypeContainer> getCellTypeContainers(List<ApiDataFieldLocaleMetadataDTO> field, String token, String securityToken, String locale, int depth) {
        List<CellTypeContainer> cellTypes = new LinkedList<>();
        List<String> keyList = field.stream().map(ApiDataFieldLocaleMetadataDTO::getEnum_key).collect(Collectors.toList());
        keyList.removeAll(Collections.singleton(null));
        Map<String, List<Dictionary>> dictionaryMap = getDictByKeyList(keyList, token, securityToken, locale);
        field.forEach(item -> {
            CellTypeContainer cellTypeContainer;
            int[] precision;
            String[] listEnum;
            List<Dictionary> dictionaries;
            String dateFormat;
            switch (MetaDataType.valueOf(item.getData_type().toUpperCase())) {
                case NUMERIC:
                case NUMBER:
                    //先判断是否是枚举类型,如果是枚举类型则将cellType转为枚举
                    cellTypeContainer = new CellTypeContainer();
                    if (null != item.getEnum_key() && null != dictionaryMap.get(item.getEnum_key())) {
                        //此时为枚举类型
                        cellTypeContainer.setProtoType(MetaDataType.ENUM);
                        cellTypeContainer.setCellType(CellType.STRING);
                        dictionaries = dictionaryMap.get(item.getEnum_key());
                        listEnum = parseListEnum(dictionaries);
                        cellTypeContainer.setListEnum(listEnum);
                        cellTypeContainer.setDictionaries(dictionaries);
                    } else {
                        cellTypeContainer.setProtoType(MetaDataType.NUMBER);
                        cellTypeContainer.setCellType(CellType.NUMERIC);
                        //precision = parsePrecision(productName, item);
                        precision = parsePrecision(item);
                        cellTypeContainer.setPrecision(precision);
                    }
                    break;
                case DATETIME:
                case TIME:
                case DATE:
                    cellTypeContainer = new CellTypeContainer();
                    cellTypeContainer.setProtoType(MetaDataType.DATE);
                    cellTypeContainer.setCellType(CellType.STRING);
                    dateFormat = parseDateFormat(item);
                    cellTypeContainer.setDateFormat(dateFormat);
                    break;
                case STRING:
                    //先判断是否是枚举类型,如果是枚举类型则将cellType转为枚举
                    cellTypeContainer = new CellTypeContainer();
                    if (null != item.getEnum_key() && null != dictionaryMap.get(item.getEnum_key())) {
                        //此时为枚举类型
                        cellTypeContainer.setProtoType(MetaDataType.ENUM);
                        cellTypeContainer.setCellType(CellType.STRING);
                        dictionaries = dictionaryMap.get(item.getEnum_key());
                        listEnum = parseListEnum(dictionaries);
                        cellTypeContainer.setListEnum(listEnum);
                        cellTypeContainer.setDictionaries(dictionaries);
                    } else {
                        cellTypeContainer = new CellTypeContainer();
                        cellTypeContainer.setProtoType(MetaDataType.STRING);
                        cellTypeContainer.setCellType(CellType.STRING);
                        //precision = parsePrecision(productName, item);
                        precision = parsePrecision(item);
                        cellTypeContainer.setPrecision(precision);
                    }
                    break;
                case OBJECT:
                    cellTypeContainer = new CellTypeContainer();
                    //判断是否是单身字段,且深度小于2
                    if (item.getIs_array()) {
                        cellTypeContainer.setChildren(getCellTypeContainers(item.getField(), token, securityToken, locale, depth + 1));
                    }
                    cellTypeContainer.setProtoType(MetaDataType.OBJECT);
                    cellTypeContainer.setCellType(CellType.STRING);
                    listEnum = parseListEnum(item);
                    cellTypeContainer.setListEnum(listEnum);
                    break;
                case BOOLEAN:
                    //先判断是否是枚举类型,如果是枚举类型则将cellType转为枚举
                    cellTypeContainer = new CellTypeContainer();
                    if (null != item.getEnum_key() && null != dictionaryMap.get(item.getEnum_key())) {
                        //此时为枚举类型
                        cellTypeContainer.setProtoType(MetaDataType.ENUM);
                        cellTypeContainer.setCellType(CellType.STRING);
                        dictionaries = dictionaryMap.get(item.getEnum_key());
                        listEnum = parseListEnum(dictionaries);
                        cellTypeContainer.setListEnum(listEnum);
                        cellTypeContainer.setDictionaries(dictionaries);
                    } else {
                        cellTypeContainer.setProtoType(MetaDataType.BOOLEAN);
                        cellTypeContainer.setCellType(CellType.STRING);
                        listEnum = parseBooleanListEnum();
                        cellTypeContainer.setListEnum(listEnum);
                    }
                    break;
                default:
                    cellTypeContainer = new CellTypeContainer();
                    cellTypeContainer.setCellType(CellType.STRING);
                    break;
            }
            cellTypeContainer.setBusinessKey(item.getIs_businesskey());
            cellTypeContainer.setArray(item.getIs_array());
            cellTypeContainer.setKeyName(item.getData_name());
            cellTypeContainer.setKeyDescription(item.getDescription());
            cellTypes.add(cellTypeContainer);
        });
        return cellTypes;
    }

    private String[] parseListEnum(List<Dictionary> dictionaries) {
        List<String> select = dictionaries.stream().map(d -> d.getCode() + "." + d.getValue()).collect(Collectors.toList());
        if (CollectionUtils.isEmpty(select)) {
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0100.getErrCode(), "下拉字典不能为空");
        }
        return select.toArray(new String[0]);
    }


    public Map<String, CellTypeContainer> getResponseCellTypeContainers(GetActionLocaleResponseDTO getActionLocaleResponseDTO, String token, String locale) {
        ApiDataFieldLocaleMetadataDTO apiDataFieldLocaleMetadataDTO = getActionLocaleResponseDTO.getResponse().getData();
        if (CollectionUtils.isEmpty(apiDataFieldLocaleMetadataDTO.getField())) {
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0102.getErrCode(), "元数据出参为空");
        }
        List<String> keyList = apiDataFieldLocaleMetadataDTO.getField().stream().map(ApiDataFieldLocaleMetadataDTO::getEnum_key).collect(Collectors.toList());
        keyList.removeAll(Collections.singleton(null));
        //Map<String, List<Dictionary>> dictionaryMap = getDictByKeyList(keyList, token, null, locale);
        List<CellTypeContainer> cellTypeContainers = getCellTypeContainers(apiDataFieldLocaleMetadataDTO.getField(), token, null, locale, 1);
        Map<String, CellTypeContainer> cellTypeContainerMap = cellTypeContainers.stream().collect(Collectors.toMap(CellTypeContainer::getKeyName, Function.identity()));
        /*apiDataFieldLocaleMetadataDTO.getField().forEach(item -> {
            CellTypeContainer cellTypeContainer;
            List<Dictionary> dictionaries;
            int[] precision;
            String[] listEnum;
            switch (MetaDataType.valueOf(item.getData_type().toUpperCase())) {
                case NUMERIC:
                case NUMBER:
                    cellTypeContainer = new CellTypeContainer();
                    if (null != item.getEnum_key() && null != dictionaryMap.get(item.getEnum_key())) {
                        //此时为枚举类型
                        cellTypeContainer.setProtoType(MetaDataType.ENUM);
                        cellTypeContainer.setCellType(CellType.STRING);
                        listEnum = parseListEnum(JSON.parseArray(JSON.toJSONString(dictionaryMap.get(item.getEnum_key())), Dictionary.class));
                        cellTypeContainer.setListEnum(listEnum);
                        dictionaries = getDictByKey(item.getEnum_key(), token, locale);
                        cellTypeContainer.setDictionaries(dictionaries);
                    } else {
                        cellTypeContainer.setProtoType(MetaDataType.NUMBER);
                        cellTypeContainer.setCellType(CellType.NUMERIC);
                        //precision = parsePrecision(productName, item);
                        precision = parsePrecision(item);
                        cellTypeContainer.setPrecision(precision);
                    }
                    break;
                case DATETIME:
                case TIME:
                case DATE:
                    cellTypeContainer = new CellTypeContainer();
                    cellTypeContainer.setProtoType(MetaDataType.DATE);
                    cellTypeContainer.setCellType(CellType.STRING);
                    String dateFormat = parseDateFormat(item);
                    cellTypeContainer.setDateFormat(dateFormat);
                    break;
                case STRING:
                    cellTypeContainer = new CellTypeContainer();
                    if (null != item.getEnum_key() && null != dictionaryMap.get(item.getEnum_key())) {
                        //此时为枚举类型
                        cellTypeContainer.setProtoType(MetaDataType.ENUM);
                        cellTypeContainer.setCellType(CellType.STRING);
                        listEnum = parseListEnum(JSON.parseArray(JSON.toJSONString(dictionaryMap.get(item.getEnum_key())), Dictionary.class));
                        cellTypeContainer.setListEnum(listEnum);
                        dictionaries = getDictByKey(item.getEnum_key(), token, locale);
                        cellTypeContainer.setDictionaries(dictionaries);
                    } else {
                        cellTypeContainer.setProtoType(MetaDataType.STRING);
                        cellTypeContainer.setCellType(CellType.STRING);
                        //precision = parsePrecision(productName, item);
                        precision = parsePrecision(item);
                        cellTypeContainer.setPrecision(precision);
                    }
                    break;
                case OBJECT:
                    cellTypeContainer = new CellTypeContainer();
                    cellTypeContainer.setProtoType(MetaDataType.OBJECT);
                    cellTypeContainer.setCellType(CellType.STRING);
                    listEnum = parseListEnum(item);
                    cellTypeContainer.setListEnum(listEnum);
                    break;
                case BOOLEAN:
                    cellTypeContainer = new CellTypeContainer();
                    if (null != item.getEnum_key() && null != dictionaryMap.get(item.getEnum_key())) {
                        //此时为枚举类型
                        cellTypeContainer.setProtoType(MetaDataType.ENUM);
                        cellTypeContainer.setCellType(CellType.STRING);
                        listEnum = parseListEnum(JSON.parseArray(JSON.toJSONString(dictionaryMap.get(item.getEnum_key())), Dictionary.class));
                        cellTypeContainer.setListEnum(listEnum);
                        dictionaries = getDictByKey(item.getEnum_key(), token, locale);
                        cellTypeContainer.setDictionaries(dictionaries);
                    } else {
                        cellTypeContainer.setProtoType(MetaDataType.BOOLEAN);
                        cellTypeContainer.setCellType(CellType.STRING);
                        listEnum = parseBooleanListEnum();
                        cellTypeContainer.setListEnum(listEnum);
                    }
                    break;
                default:
                    cellTypeContainer = new CellTypeContainer();
                    cellTypeContainer.setCellType(CellType.STRING);
                    break;
            }
            cellTypeContainerMap.put(item.getData_name(), cellTypeContainer);
        });*/
        parseCellTypeContainerMapDict(cellTypeContainerMap);
        return cellTypeContainerMap;
    }

    /**
     * @return
     * @Author zhuangli
     * @Description 解析字典为map
     * @Date 17:12 2022/2/18
     * @Param
     **/
    private void parseCellTypeContainerMapDict(Map<String, CellTypeContainer> cellTypeContainerMap) {
        cellTypeContainerMap.forEach((k, v) -> {
            if (!CollectionUtils.isEmpty(v.getDictionaries())) {
                Map<String, String> dictMap = v.getDictionaries().stream().collect(Collectors.toMap(Dictionary::getCode, Dictionary::getValue, (entity1, entity2) -> entity1));
                v.setDictMap(dictMap);
            }
            if (!CollectionUtils.isEmpty(v.getChildren())) {
                v.getChildren().forEach(item -> {
                    if (!CollectionUtils.isEmpty(item.getDictionaries())) {
                        Map<String, String> dictMapInner = item.getDictionaries().stream().collect(Collectors.toMap(Dictionary::getCode, Dictionary::getValue, (entity1, entity2) -> entity1));
                        item.setDictMap(dictMapInner);
                    }
                });
            }
        });
    }

    private String[] parseBooleanListEnum() {
        return new String[]{"TRUE", "FALSE"};
    }

    private String[] parseListEnum(ApiDataFieldLocaleMetadataDTO item) {
        return null;
    }

    private String parseDateFormat(ApiDataFieldLocaleMetadataDTO item) {
        //MultiLanguageDTO multiLanguageDTO = item.getRemark();
        MultiLanguageDTO multiLanguageDTO = null;
        if (null == multiLanguageDTO) {
            switch (MetaDataType.valueOf(item.getData_type().toUpperCase())) {
                case DATETIME:
                    return "yyyy-MM-dd HH:mm:ss";
                case TIME:
                    return "HH:mm:ss";
                case DATE:
                    return "yyyy-MM-dd";
            }
            return "yyyy-MM-dd";
        }
        String locale = LocaleContextHolder.getLocale().toString();
        switch (locale) {
            case "zh_TW":
                return multiLanguageDTO.getZh_TW();
            case "zh_CN":
                return multiLanguageDTO.getZh_CN();
            case "en_US":
                return multiLanguageDTO.getEn();
            default:
                return multiLanguageDTO.getZh_CN();
        }
    }

    private int[] parsePrecision(ApiDataFieldLocaleMetadataDTO metadataDTO) {
        if (!StringUtils.isEmpty(metadataDTO.getPrecision())) {
            String precision = metadataDTO.getPrecision();
            if (StringUtils.isEmpty(precision)) {
                return new int[0];
            }
            if (precision.indexOf(",") != -1) {
                String[] values = precision.substring(precision.indexOf("(") + 1, precision.indexOf(")")).split(",");
                // 解决查询API规格 接口返回"(null,null)"导致解析报错问题
                if (StringUtils.isEmpty(values[0]) || StringUtils.isEmpty(values[1]) || "null".equals(values[0]) || "null".equals(values[1])) {
                    return new int[0];
                }
                int[] precisionValue = new int[]{Integer.valueOf(values[0]), Integer.valueOf(values[1])};
                return precisionValue;
            } else {
                int[] precisionValue = new int[]{Integer.valueOf(precision)};
                return precisionValue;
            }
        }
        return new int[0];
    }
}
