package com.digiwin.athena.base.application.service.userdefined;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.auth.service.TokenVerifyService;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.base.application.config.BaseAudcDataSourceConfig;
import com.digiwin.athena.base.application.meta.dto.userdefined.UserOperationMsgEvent;
import com.digiwin.athena.base.application.meta.request.userdefined.UserPageModelBodyDTO;
import com.digiwin.athena.base.application.meta.response.userdefined.TakeUpRecordDTO;
import com.digiwin.athena.base.application.meta.response.userdefined.UserPageModelRespDTO;
import com.digiwin.athena.base.infrastructure.constant.Constants;
import com.digiwin.athena.base.infrastructure.mapper.audc.userdefined.UserOperationRecordMapper;
import com.digiwin.athena.base.infrastructure.meta.po.userdefined.UserOperationRecordDTO;
import com.digiwin.athena.base.infrastructure.util.DistributeLocker;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * 编辑态控制基类
 */
@Slf4j
@Service
@SuppressWarnings({"java:S1172", "java:S3740", "java:S6813"})
public class UserPageModelParentService {

    @Autowired
    private UserOperationRecordMapper userOperationRecordMapper;

    @Resource(name = "redisTemplate")
    private RedisTemplate lockRedisTemplate;

    @Autowired
    private TokenVerifyService tokenVerifyService;


    protected static final String UNDER_PLACE = ":";

    protected static final String KEY_PREFIX = "audc:";
    /**
     * 作业冲突白名单
     */
    protected static final List<String> CHECK_ITEM = Lists.newArrayList("baseDataEntryProjectMemorabilia","DataEntry_testflow_sign_document","DataEntry_sign_document_model");

    /**
     * 支持编辑态作业白名单
     *
     * @param pageModelBodyDTO
     * @return
     */
    protected boolean isCheckItem(UserPageModelBodyDTO pageModelBodyDTO) {
        if (pageModelBodyDTO == null) {
            return false;
        }
        if (StringUtils.isEmpty(pageModelBodyDTO.getActivityCode())) {
            return false;
        }
        return CHECK_ITEM.contains(pageModelBodyDTO.getActivityCode());
    }

    /**
     * 获取占用key
     *
     * @param authoredUser
     * @param pageModelBodyDTO
     * @return
     */
    public String getLockKey(AuthoredUser authoredUser, UserPageModelBodyDTO pageModelBodyDTO) {
        return StringUtils.EMPTY;
    }

    /**
     * 获取占用key
     *
     * @param authoredUser
     * @param pageModelBodyDTO
     * @return
     */
    public String getLockKey(AuthoredUser authoredUser, UserPageModelBodyDTO pageModelBodyDTO, String businessKey) {
        return StringUtils.EMPTY;
    }

    /**
     * 根据租户+作业CODE+类型+BK获取防止并发操作
     *
     * @param authoredUser
     * @param pageModelBodyDTO
     * @return
     */
    public boolean getLock(AuthoredUser authoredUser, UserPageModelBodyDTO pageModelBodyDTO) {
        String lockVal = authoredUser.getUserId();
        try {
            // 取消占用，businessKeys可能为null
            if(CollectionUtils.isEmpty(pageModelBodyDTO.getBusinessKeys())){
                return DistributeLocker.tryLock(lockRedisTemplate, getLockKey(authoredUser, pageModelBodyDTO), lockVal, 3L, TimeUnit.SECONDS);
            }

            for (String businessKey : pageModelBodyDTO.getBusinessKeys()) {
                DistributeLocker.tryLock(lockRedisTemplate, getLockKey(authoredUser, pageModelBodyDTO, businessKey), lockVal, 3L, TimeUnit.SECONDS);
            }
        } catch (InterruptedException e) {
            log.error("页面编辑态获取锁失败:{}", e);
            Thread.currentThread().interrupt();
        }
        return true;
    }

    /**
     * 根据租户+作业CODE+类型+BK获取解除并发操作
     *
     * @param authoredUser
     * @param pageModelBodyDTO
     */
    public void releaseLock(AuthoredUser authoredUser, UserPageModelBodyDTO pageModelBodyDTO) {
        try {
            if(CollectionUtils.isEmpty(pageModelBodyDTO.getBusinessKeys())){
                DistributeLocker.releaseLock(lockRedisTemplate, getLockKey(authoredUser, pageModelBodyDTO));
            }else {
                pageModelBodyDTO.getBusinessKeys().forEach( businessKey ->{
                    DistributeLocker.releaseLock(lockRedisTemplate, getLockKey(authoredUser, pageModelBodyDTO, businessKey));
                });
            }
        } catch (Exception e) {
            log.error("页面编辑态解锁失败:{}", e);
        }
    }

    /**
     * 获取编辑态用户
     *
     * @param authoredUser
     * @param pageModelBodyDTO
     * @return
     */
    public UserPageModelRespDTO getEdit(AuthoredUser authoredUser, UserPageModelBodyDTO pageModelBodyDTO) {
        return new UserPageModelRespDTO();
    }

    /**
     * 更新编辑态用户
     *
     * @param authoredUser
     * @param pageModelBodyDTO
     * @return
     */
    public UserPageModelRespDTO updateEdit(AuthoredUser authoredUser, UserPageModelBodyDTO pageModelBodyDTO) {
        return new UserPageModelRespDTO();
    }

    /**
     * 删除编辑态用户
     *
     * @param authoredUser
     * @param pageModelBodyDTO
     * @return
     */
    public Boolean deleteEdit(AuthoredUser authoredUser, UserPageModelBodyDTO pageModelBodyDTO) {
        return true;
    }

    public Boolean reportStatus(AuthoredUser authoredUser, UserPageModelBodyDTO pageModelBodyDTO) {
        return true;
    }

    /**
     * 删除过期数据
     * @return
     */
    @Transactional(transactionManager = BaseAudcDataSourceConfig.BASE_AUDC_DATASOURCE_TRANSACTION_MANAGER_BUSINESS)
    public int deleteExpire() {
        /**
         * 1、查询过期的占用
         */
        LocalDateTime updateTime = LocalDateTime.now().minus(Constants.USER_OPERATION_TTL, ChronoUnit.SECONDS);
        Map<String, Object> queryExpireRecordsParams = Maps.newHashMap();
        queryExpireRecordsParams.put("takeUp", Constants.USER_OPERATION_TAKE_UP_YES);
        queryExpireRecordsParams.put("updateTime", updateTime);
        List<UserOperationRecordDTO> expiredTakeUpRecords = userOperationRecordMapper.queryExpireRecords(queryExpireRecordsParams);

        /**
         * 2、删除过期的数据（占用+非占用）
         */
        UserOperationRecordDTO userOperationRecordDTO = new UserOperationRecordDTO();
        // 更新时间 = 当前时间的-10分钟
        userOperationRecordDTO.setUpdateTime(updateTime);
        int result = userOperationRecordMapper.deleteExpire(userOperationRecordDTO);

        /**
         * 3、通知非占用者（包含非冲突行的其他占用者）
         */
        if(CollectionUtils.isEmpty(expiredTakeUpRecords) && result >0){
            return result;
        }

        // 按照租户+作业code进行分组 发送MQTT消息
        // key:租户+作业code
        Map<String, List<UserOperationRecordDTO>> expiredTakeUpRecordMap = new HashMap<>();
        expiredTakeUpRecords.forEach(expiredTakeUpRecord ->{
            String key = expiredTakeUpRecord.getTenantId()+UNDER_PLACE+expiredTakeUpRecord.getPageCode();
            if(expiredTakeUpRecordMap.containsKey(key)){
                expiredTakeUpRecordMap.get(key).add(expiredTakeUpRecord);
            }else {
                expiredTakeUpRecordMap.put(key,new ArrayList<>(Arrays.asList(expiredTakeUpRecord)));
            }
        });

        expiredTakeUpRecordMap.values().forEach(expiredTakeUpRecordList -> {
            Map<String, Object> queryNoTakeUpParams = Maps.newHashMap();
            UserOperationRecordDTO expiredTakeUpRecord = expiredTakeUpRecordList.get(0);
            queryNoTakeUpParams.put("tenantId",expiredTakeUpRecord.getTenantId());
            queryNoTakeUpParams.put("pageType",expiredTakeUpRecord.getPageType());
            queryNoTakeUpParams.put("pageCode",expiredTakeUpRecord.getPageCode());
            queryNoTakeUpParams.put("updateTime", updateTime);
            List<UserOperationRecordDTO> noTakeUpList = queryEffectiveRecords(queryNoTakeUpParams);
            if(!CollectionUtils.isEmpty(noTakeUpList)){
                // 定时任务需要传入租户虚拟token
                String token = tokenVerifyService.queryApiVirtualToken(expiredTakeUpRecord.getTenantId());
                List<String> userIdList = noTakeUpList.stream().map(UserOperationRecordDTO::getUserId).distinct().collect(Collectors.toList());
                sendMessageToClient(processMsgEvent(token,expiredTakeUpRecordList, Constants.USER_OPERATION_MSG_TYPE_BASE_DATA, Constants.USER_OPERATION_MSG_CATEGORY_RELEASE), userIdList);
            }
        });
        return result;
    }

    /**
     * 发送MQTT消息
     * @param userOperationMsgEvent
     * @param userIdList
     */
    public void sendMessageToClient(UserOperationMsgEvent userOperationMsgEvent, List<String> userIdList) {

    }

    /**
     * 组装msgEvent
     * @param token 定时任务需要传入租户虚拟token
     * @param newTakeUpUserList
     * @param msgType
     * @param msgCategory
     * @return
     */
    public UserOperationMsgEvent processMsgEvent(String token, List<UserOperationRecordDTO> newTakeUpUserList, String msgType, String msgCategory) {
        return new UserOperationMsgEvent();
    }

    /**
     * 获取占用人员信息
     *
     * @param params
     * @return
     */
    protected List<UserOperationRecordDTO> getTakeUpUser(Map<String, Object> params) {
        if (MapUtils.isEmpty(params)) {
            return Collections.emptyList();
        }
        params.put("takeUp", Constants.USER_OPERATION_TAKE_UP_YES);
        return userOperationRecordMapper.queryEffectiveRecords(params);
    }

    protected QueryWrapper<UserOperationRecordDTO> getWrapper(Map<String, Object> params) {
        QueryWrapper<UserOperationRecordDTO> queryWrapper = new QueryWrapper<>();
        params.forEach(queryWrapper::eq);
        return queryWrapper;
    }

    /**
     * 构建人员信息查询参数
     *
     * @param authoredUser     人员信息
     * @param pageModelBodyDTO 占用信息参数
     * @return
     */
    protected Map<String, Object> buildParams(AuthoredUser authoredUser, UserPageModelBodyDTO pageModelBodyDTO) {
        return Maps.newHashMap();
    }

    /**
     * 构建非占用人员信息查询参数
     *
     * @param authoredUser 人员信息
     * @return
     */
    protected Map<String, Object> buildCheckNoTakeUpParams(UserOperationRecordDTO authoredUser) {
        return Maps.newHashMap();
    }


    /**
     * 记录用户占用操作信息
     *
     * @param newTakeUpUserList    新占用信息
     * @param takeUpUserList 当前占用人信息
     *                   return 是否占用成功
     */
    protected boolean recordTakeUpOperation(List<UserOperationRecordDTO> newTakeUpUserList, List<UserOperationRecordDTO> takeUpUserList) {
        if(CollectionUtils.isEmpty(newTakeUpUserList)){
            return false;
        }

        if (CollectionUtils.isEmpty(takeUpUserList)) {
            // 添加占用记录
            userOperationRecordMapper.batchInsertTakeUpRecords(newTakeUpUserList);
            return true;
        }

        // 新添加的占用记录
        List<UserOperationRecordDTO> addTakeUpRecordList = new ArrayList<>();

        newTakeUpUserList.forEach(newTakeUpUser ->{
            // 该行的已占用记录
            List<UserOperationRecordDTO> occupiedRecords = takeUpUserList.stream().filter
                    (takeUpUser -> StringUtils.equals(newTakeUpUser.getBusinessKey(), takeUpUser.getBusinessKey())).collect(Collectors.toList());

            if(CollectionUtils.isEmpty(occupiedRecords)){
                // 该行未被占用，则新增占用记录
                addTakeUpRecordList.add(newTakeUpUser);
            } else {
                occupiedRecords.forEach( conflictTakeUpRecord -> {
                    // 若是当前用户自己已占用，则更新占用时间
                    if(newTakeUpUser.getClientId().equals(conflictTakeUpRecord.getClientId())){
                        userOperationRecordMapper.updateTime(conflictTakeUpRecord);
                    }else {
                        log.warn("takeUp fail,newTakeUpUser:{},conflictTakeUpRecord:{}",newTakeUpUser,conflictTakeUpRecord);
                    }
                });
            }
        });
        if (CollectionUtils.isNotEmpty(addTakeUpRecordList)) {
            // 添加占用记录
            userOperationRecordMapper.batchInsertTakeUpRecords(addTakeUpRecordList);
            return true;
        }
        return false;
    }

    /**
     * 记录用户非占用操作信息
     *
     * @param currentUser
     * @param takeUpRecords
     */
    protected List<TakeUpRecordDTO> recordNoTakeUpOperation(UserOperationRecordDTO currentUser, List<UserOperationRecordDTO> takeUpRecords) {
        List<TakeUpRecordDTO> takeUpRecordDTOS = new ArrayList<>();
        // 没人占用
        if (CollectionUtils.isEmpty(takeUpRecords)) {
            // 插入自己非占用记录
            this.recordUserOperation(currentUser);
            return takeUpRecordDTOS;
        }

        // 有人占用

        Optional<UserOperationRecordDTO> currentUserTakeUpOpt = takeUpRecords.stream().filter(userOperationRecordDTO -> StringUtils.equals(currentUser.getClientId(), userOperationRecordDTO.getClientId())).findFirst();
        // 若自己也占用
        if (currentUserTakeUpOpt.isPresent()) {
            UserOperationRecordDTO takeUpUser = currentUserTakeUpOpt.get();
            // 更新占用时间
            takeUpUser.setUpdateTime(LocalDateTime.now());
            if (this.updateUserOperationRecord(takeUpUser) <= 0) {
                log.error("recordNoTakeUpOperation fail,userPageModeDTO：{}", JsonUtils.objectToString(takeUpUser));
            }
        }else {
            // 若自己不占用，则插入自己非占用记录
            this.recordUserOperation(currentUser);
        }

        // 组装返回值
        takeUpRecords.forEach(item ->{
            TakeUpRecordDTO newItem = new TakeUpRecordDTO();
            BeanUtils.copyProperties(item,newItem);
            takeUpRecordDTOS.add(newItem);
        });

        return takeUpRecordDTOS;
    }

    /**
     * 记录用户操作
     *
     * @param newUser
     */
    protected void recordUserOperation(UserOperationRecordDTO newUser) {
        List<UserOperationRecordDTO> recordUsers = userOperationRecordMapper.selectList(getWrapper(buildCheckNoTakeUpParams(newUser)));
        if (CollectionUtils.isEmpty(recordUsers)) {
            userOperationRecordMapper.insert(newUser);
        } else {
            recordUsers.forEach(recordUser ->{
                recordUser.setUpdateTime(newUser.getUpdateTime());
                userOperationRecordMapper.updateTime(recordUser);
            });
        }
    }

    /**
     * 查询有效的操作
     *
     * @param params
     * @return
     */
    protected List<UserOperationRecordDTO> queryEffectiveRecords(Map<String, Object> params) {
        return userOperationRecordMapper.queryEffectiveRecords(params);
    }

    /**
     * 批量删除占用操作
     *
     * @param params
     */
    protected int batchDeleteTakeUpUser(Map<String, Object> params) {
        params.put("takeUp", Constants.USER_OPERATION_TAKE_UP_YES);
        return userOperationRecordMapper.batchDeleteTakeUpUser(params);
    }

    /**
     * 批量删除当前用户操作
     *
     * @param idList
     */
    protected void batchDeleteUserUserOperation(List<Long> idList) {
        if(CollectionUtils.isEmpty(idList)){
            return;
        }
        userOperationRecordMapper.deleteBatchIds(idList);
    }

    protected void deleteUserUserOperationById(long id) {
        userOperationRecordMapper.deleteById(id);
    }

    /**
     * 删除占用信息
     *
     * @param userPageModeDTO
     */
    protected int updateUserOperationRecord(UserOperationRecordDTO userPageModeDTO) {
        return userOperationRecordMapper.updateUserOperationRecord(userPageModeDTO);
    }
}
