package com.digiwin.athena.base.application.util;

import net.lingala.zip4j.io.outputstream.ZipOutputStream;
import net.lingala.zip4j.model.ZipParameters;
import net.lingala.zip4j.model.enums.EncryptionMethod;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;

/**
 * @Author mk
 * @Date 2024/10/19 17:12
 * @Description:
 */
public class ZipUtils {

    /**
     * 将输入流（InputStream）中的数据压缩并加密为一个ZIP格式的byte数组。
     *
     * @param sourceFileInputStream 要压缩的原文件的输入流（InputStream）
     * @param password              设置的密码，用于加密ZIP文件中的内容
     * @param zipName               设置在ZIP文件中的文件名，即压缩后的文件名
     * @return                      返回压缩并加密后的byte数组
     * @throws IOException          如果读取输入流或写入输出流时发生IO异常
     */
    public static byte[] extracted(InputStream sourceFileInputStream, String password, String zipName) throws IOException {
        // 使用ByteArrayOutputStream来捕获输出，最终转换为byte数组
        ByteArrayOutputStream baos = new ByteArrayOutputStream();

        // 创建ZipParameters对象，用于设置压缩参数
        ZipParameters zipParameters = new ZipParameters();
        // 设置ZIP文件中的文件名
        zipParameters.setFileNameInZip(zipName);
        // 设置是否加密文件
        zipParameters.setEncryptFiles(true);
        // 设置加密方法为普通加密方式
        zipParameters.setEncryptionMethod(EncryptionMethod.ZIP_STANDARD);

        // 使用try-with-resources语句创建ZipOutputStream，确保资源自动关闭
        // 同时使用给定的密码初始化ZipOutputStream
        try (ZipOutputStream zipOutputStream = new ZipOutputStream(baos, password.toCharArray())) {
            // 使用ZipParameters设置当前要写入的ZIP条目的参数
            zipOutputStream.putNextEntry(zipParameters);

            // 创建一个字节缓冲区，用于临时存储读取的数据
            byte[] buffer = new byte[1024];
            int len;
            // 从源输入流中读取数据，直到结束
            while ((len = sourceFileInputStream.read(buffer)) > 0) {
                // 将读取的数据写入ZipOutputStream
                zipOutputStream.write(buffer, 0, len);
            }
            // 完成当前ZIP条目的写入
            zipOutputStream.closeEntry();
        }
        // 将压缩并加密的数据转换为byte数组并返回
        return baos.toByteArray();
    }
}
