package com.digiwin.athena.base.application.service.commonused;

import cn.hutool.core.io.resource.ResourceUtil;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.appcore.util.SnowflakeIdWorker;
import com.digiwin.athena.base.application.config.BaseAudcDataSourceConfig;
import com.digiwin.athena.base.application.constant.PermissionConstant;
import com.digiwin.athena.base.application.meta.dto.commonused.SystemMenuDTO;
import com.digiwin.athena.base.application.meta.dto.commonused.TmDataListDTO;
import com.digiwin.athena.base.application.meta.dto.commonused.TmDataTierListDTO;
import com.digiwin.athena.base.application.meta.request.commonused.MenuAuthorityReq;
import com.digiwin.athena.base.application.meta.response.commonused.MenuAuthorityResp;
import com.digiwin.athena.base.application.meta.response.commonused.MenuQueryResp;
import com.digiwin.athena.base.application.meta.response.commonused.MenuTopResp;
import com.digiwin.athena.base.application.meta.response.commonused.MenuUnFoldResp;
import com.digiwin.athena.base.application.service.permission.PermissionCheckService;
import com.digiwin.athena.base.application.service.permission.PermissionService;
import com.digiwin.athena.base.application.service.permission.PermissionServiceImpl;
import com.digiwin.athena.base.infrastructure.manager.thememap.dto.BaseItemDto;
import com.digiwin.athena.base.infrastructure.mapper.audc.commonUsed.MenuTopMapper;
import com.digiwin.athena.base.infrastructure.mapper.audc.commonUsed.MenuUnfoldMapper;
import com.digiwin.athena.base.infrastructure.meta.po.commonused.MenuTopData;
import com.digiwin.athena.base.infrastructure.meta.po.commonused.MenuUnfoldData;
import com.digiwin.athena.datamap.sdk.meta.dto.response.TmStartProjectDTO;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.stream.Collectors;

import static com.digiwin.athena.base.infrastructure.manager.thememap.BaseThemeMapServiceImpl.BASE_DATA_PLAT;
import static com.digiwin.athena.base.infrastructure.manager.thememap.BaseThemeMapServiceImpl.REPORT_PLAT;


/**
 * 描述
 *
 * @author gonghongxing
 * @version 1.0
 * @since 2022/02/09 09:28:36
 */
@Service
@Slf4j
public class MenuServiceImpl implements MenuService {
    private static final Logger logger = LoggerFactory.getLogger(MenuServiceImpl.class);

    private static final int TOP_STATUS = 1;

    @Autowired
    private MenuTopMapper menuTopMapper;

    @Autowired
    private MenuUnfoldMapper menuUnfoldMapper;

    @Autowired
    private PermissionCheckService permissionCheckService;

    @Autowired
    private CommonUsedService commonUsedService;

    @Autowired
    private PermissionService permissionService;

    @Autowired
    private PermissionServiceImpl permissionServiceImpl;

    // 发起项目
    private static final String MODULE_START_PROJECT = "start-project";
    // 基础资料
    private static final String MODULE_BASIC_DATA = "athena-basicDataEntry";
    // 报表
    private static final String REPORT = "report";
    // 基础资料（提供给智能入口的type）
    private static final String TYPE_BASE_DATA = "baseDataEntry";
    // 发起项目（提供给智能入口的type）
    private static final String TYPE_START_PROJECT = "startProject";
    // 系统菜单
    private static final String TYPE_SYS_MENU = "sysMenu";


    /**
     * 查询用户的模块信息
     *
     * @param user 用户信息
     * @return 置顶的模块信息
     */
    public MenuQueryResp query(AuthoredUser user) {
        MenuQueryResp resp = new MenuQueryResp();
        resp.setTopInfo(menuTopQuery(user));
        resp.setUnfoldInfo(menuUnfoldQuery(user));
        return resp;
    }


    /**
     * 查询用户置顶的模块信息
     *
     * @param user 用户信息
     * @return 置顶的模块信息
     */
    public List<MenuTopResp> menuTopQuery(AuthoredUser user) {

        QueryWrapper<MenuTopData> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("user_id", user.getUserId());
        queryWrapper.eq("tenant_id", user.getTenantId());

        List<MenuTopData> retData = menuTopMapper.selectList(queryWrapper);
        if (null == retData || retData.isEmpty()) {
            //日志记录
            logger.debug("query user top:[" + user.getUserId() + "] menu is empty.");
            return new ArrayList<>();
        }
        List<MenuTopResp> respList = new ArrayList<>();
        for (MenuTopData menuData : retData) {
            MenuTopResp resp = new MenuTopResp();
            BeanUtils.copyProperties(menuData, resp);
            respList.add(resp);
        }
        return respList;
    }


    /**
     * 查询用户模块展开、收起信息
     *
     * @param user 用户信息
     * @return 置顶的模块信息
     */
    public List<MenuUnFoldResp> menuUnfoldQuery(AuthoredUser user) {

        QueryWrapper<MenuUnfoldData> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("user_id", user.getUserId());
        queryWrapper.eq("tenant_id", user.getTenantId());

        List<MenuUnfoldData> retData = menuUnfoldMapper.selectList(queryWrapper);
        if (null == retData || retData.isEmpty()) {
            //日志记录
            logger.debug("query user unfold:[" + user.getUserId() + "] menu is empty.");
            return new ArrayList<>();
        }
        List<MenuUnFoldResp> respList = new ArrayList<>();
        for (MenuUnfoldData menuData : retData) {
            MenuUnFoldResp resp = new MenuUnFoldResp();
            BeanUtils.copyProperties(menuData, resp);
            respList.add(resp);
        }
        return respList;
    }


    /**
     * 查询用户置顶的模块信息
     *
     * @param user     用户信息
     * @param moduleNo 模块信息baseEntry：基础资料，report：报表 guess:猜你想去
     * @return 置顶的模块信息
     */
    @Transactional(transactionManager = BaseAudcDataSourceConfig.BASE_AUDC_DATASOURCE_TRANSACTION_MANAGER_BUSINESS)
    public boolean setMenuTop(AuthoredUser user, String moduleNo) {

        //删除用户+租户的置顶数据
        UpdateWrapper<MenuTopData> updateWrapper = new UpdateWrapper<>();
        updateWrapper.eq("user_id", user.getUserId());
        updateWrapper.eq("tenant_id", user.getTenantId());
        menuTopMapper.delete(updateWrapper);
        //新增用户的置顶数据
        MenuTopData addData = new MenuTopData();
        addData.setId(SnowflakeIdWorker.getInstance().newId());
        addData.setIsTop(TOP_STATUS);
        addData.setUserId(user.getUserId());
        addData.setTenantId(user.getTenantId());
        addData.setCreateDate(LocalDateTime.now());
        addData.setModifyDate(LocalDateTime.now());
        addData.setModuleNo(moduleNo);
        int result = menuTopMapper.insert(addData);
        return result >= 1;
    }

    /**
     * 查询用户置顶的模块信息
     *
     * @param user     用户信息
     * @param moduleNo 模块信息baseEntry：基础资料，report：报表 guess:猜你想去
     * @return 置顶的模块信息
     */
    @Transactional(transactionManager = BaseAudcDataSourceConfig.BASE_AUDC_DATASOURCE_TRANSACTION_MANAGER_BUSINESS)
    public boolean setMenuUnfold(AuthoredUser user, String moduleNo, Integer flag) {

        //删除用户+租户的展开数据
        UpdateWrapper<MenuUnfoldData> updateWrapper = new UpdateWrapper<>();
        updateWrapper.eq("user_id", user.getUserId());
        updateWrapper.eq("tenant_id", user.getTenantId());
        menuUnfoldMapper.delete(updateWrapper);
        //新增用户的展开数据
        MenuUnfoldData addData = new MenuUnfoldData();
        addData.setId(SnowflakeIdWorker.getInstance().newId());
        addData.setUnfold(flag);
        addData.setUserId(user.getUserId());
        addData.setTenantId(user.getTenantId());
        addData.setCreateDate(LocalDateTime.now());
        addData.setModifyDate(LocalDateTime.now());
        addData.setModuleNo(moduleNo);
        int result = menuUnfoldMapper.insert(addData);
        return result >= 1;
    }

    /**
     * 获取所有系统菜单
     * @return
     */
    @Override
    public List<SystemMenuDTO> getAllSystemMenus() {
        // 读取系统菜单数据
        String menuData = ResourceUtil.readUtf8Str("data/systemMenus.json");
        return JsonUtils.jsonToListObject(menuData, SystemMenuDTO.class);
    }


    /**
     * 用户级权限的菜单
     * @param user
     * @return
     */
    @Override
    public List<MenuAuthorityResp> getAuthorityMenu(AuthoredUser user) {
        /*获取模组和系统菜单权限 start*/
        Map<String, List<String>> permissionMap = permissionCheckService.getModuleOrMenuAccessible(user, "Athena");
        // 所有设置权限的模组
        List<String> allPermissionModules = permissionMap.containsKey("allModule") ? permissionMap.get("allModule") : Lists.newArrayList();
        // 所有设置权限的菜单
        List<String> allPermissionMenus = permissionMap.containsKey("allMenu") ? permissionMap.get("allMenu") : Lists.newArrayList();
        // 拥有权限的模组
        List<String> accessibleModules = permissionMap.containsKey("accessibleModule") ? permissionMap.get("accessibleModule") : Lists.newArrayList();
        // 拥有权限的菜单
        List<String> accessibleMenus = permissionMap.containsKey("accessibleMenu") ?  permissionMap.get("accessibleMenu") : Lists.newArrayList();
        /*获取模组和系统菜单权限 end*/

        // 系统菜单数据
        String sysMenuData = ResourceUtil.readUtf8Str("data/systemMenus.json");
        List<SystemMenuDTO> systemMenuDataList = JsonUtils.jsonToListObject(sysMenuData, SystemMenuDTO.class);

        // 获取有权限的模组
        getAccessibleModules(allPermissionModules, accessibleModules, systemMenuDataList);

        List<MenuAuthorityResp> resultList = new ArrayList<>();
        // 获取有权限的系统菜单
        CompletableFuture<Void> future1 = CompletableFuture.runAsync(() -> {
            getAccessibleSysMenus(resultList, systemMenuDataList, accessibleModules, accessibleMenus, allPermissionMenus);
        });
        // 发起项目作业权限
        CompletableFuture<Void> future2 = CompletableFuture.runAsync(() -> {
            getAllowStartProjects(resultList, user, accessibleModules, Boolean.TRUE, null);
        });
        // 基础资料
        CompletableFuture<Void> future3 = CompletableFuture.runAsync(() -> {
            getAllowBaseDataEntries(resultList, user, accessibleModules, Boolean.TRUE, null);
        });
        // 报表
        CompletableFuture<Void> future4 = CompletableFuture.runAsync(() -> {
            getAllowReports(resultList, user, accessibleModules, Boolean.TRUE, null);
        });
        CompletableFuture<Void> allOf = CompletableFuture.allOf(future1, future2, future3, future4);
        try {
            allOf.get();
        } catch (ExecutionException | InterruptedException e) {
            e.printStackTrace();
            if (e instanceof InterruptedException) {
                Thread.currentThread().interrupt();
            }
        }
        return resultList;
    }

    /**
     * 获取有权限的模组
     * @return
     */
    private void getAccessibleModules(List<String> allPermissionModules, List<String> accessibleModules, List<SystemMenuDTO> systemMenuDTOS) {
        /*所有模组 start*/
        // 系统菜单模组
        List<String> allModuleList = systemMenuDTOS.stream().map(SystemMenuDTO::getModuleId).collect(Collectors.toList());
        // 发起项目、基础资料、报表
        allModuleList.addAll(Arrays.asList(MODULE_START_PROJECT, MODULE_BASIC_DATA, REPORT));
        /*所有模组 end*/

        // 获取iam中没有设置权限的模组
        accessibleModules.addAll(allModuleList.stream().filter(item -> !allPermissionModules.contains(item)).collect(Collectors.toList()));
    }

    /**
     * 获取有权限的系统菜单
     * @return
     */
    private void getAccessibleSysMenus(List<MenuAuthorityResp> resultList, List<SystemMenuDTO> systemMenuDataList, List<String> accessibleModules,
                                            List<String> accessibleMenus, List<String> allPermissionMenus) {
        // 将系统菜单平铺
        List<SystemMenuDTO> allSystemMenus = new ArrayList<>();
        getSystemMenu(systemMenuDataList, allSystemMenus);
        // 获取有权限的菜单
        for (SystemMenuDTO systemMenu : allSystemMenus) {
            if (!accessibleModules.contains(systemMenu.getModuleId())) {
                continue;
            }
            // iam设置了allow或者iam没有对其进行任何的设置
            if (accessibleMenus.contains(systemMenu.getCode()) || !allPermissionMenus.contains(systemMenu.getCode())) {
                MenuAuthorityResp resp = new MenuAuthorityResp();
                resp.setCode(systemMenu.getCode());
                resp.setType("sysMenu");
                resultList.add(resp);
            }
        }
    }

    /**
     * 平铺系统菜单
     * @param systemMenuTrees
     * @param systemMenuList
     */
    private void getSystemMenu(List<SystemMenuDTO> systemMenuTrees, List<SystemMenuDTO> systemMenuList) {
        for (SystemMenuDTO systemMenuDTO : systemMenuTrees) {
            if (CollectionUtils.isNotEmpty(systemMenuDTO.getSubMenus())) {
                getSystemMenu(systemMenuDTO.getSubMenus(), systemMenuList);
            } else if (!systemMenuDTO.getIsDirectory()) {
                systemMenuList.add(systemMenuDTO);
            }
        }
    }

    /**
     * 获取有权限的发起项目作业列表
     * @param resultList
     * @param accessibleModules
     * @param user
     */
    private void getAllowStartProjects(List<MenuAuthorityResp> resultList, AuthoredUser user, List<String> accessibleModules,
                                       Boolean needSearchStartProject, List<Map<String, Object>> startProjectJobList) {
        if (!accessibleModules.contains(MODULE_START_PROJECT)) {
            return;
        }

        List<TmStartProjectDTO> startProjects = JsonUtils.jsonToListObject(JsonUtils.objectToString(startProjectJobList), TmStartProjectDTO.class);
        List<TmStartProjectDTO> startProjectList = commonUsedService.startProjectList(user, needSearchStartProject, startProjects);
        getStartProject(startProjectList, resultList);
    }

    /**
     * 获取有权限的基础资料作业列表
     * @param resultList
     * @param user
     * @param accessibleModules
     */
    private void getAllowBaseDataEntries(List<MenuAuthorityResp> resultList, AuthoredUser user, List<String> accessibleModules,
                                         Boolean searchData, List<BaseItemDto> baseDataEntryJobList) {
        if (!accessibleModules.contains(MODULE_BASIC_DATA)) {
            return;
        }
        List<TmDataListDTO> basicDataList = commonUsedService.tmDataList(user, BASE_DATA_PLAT, false, searchData, baseDataEntryJobList, null, PermissionConstant.ACCESS_ALLOW);
        // 将结果平铺，取access为allow
        getBasicDataOrReport(basicDataList, resultList, BASE_DATA_PLAT);
    }

    /**
     * 获取有权限的报表作业列表
     * @param resultList
     * @param user
     * @param accessibleModules
     */
    private void getAllowReports(List<MenuAuthorityResp> resultList, AuthoredUser user, List<String> accessibleModules,
                                 Boolean searchData, List<BaseItemDto> reportJobList) {
        if (!accessibleModules.contains(REPORT)) {
            return;
        }
        List<TmDataListDTO> reportList = commonUsedService.tmDataList(user, REPORT_PLAT, false, searchData, reportJobList, null, PermissionConstant.ACCESS_ALLOW);
        // 将结果平铺，取access为allow
        getBasicDataOrReport(reportList, resultList, REPORT_PLAT);
    }



    /**
     * 获取有权限的基础资料和报表
     * @param dataListDTOS
     * @param accessibleData
     * @param plat
     */
    private void getBasicDataOrReport(List<TmDataListDTO> dataListDTOS, List<MenuAuthorityResp> accessibleData, Integer plat) {
        for (TmDataListDTO tmDataListDTO : dataListDTOS) {
            if (CollectionUtils.isNotEmpty(tmDataListDTO.getGroupList())) {
                for (TmDataTierListDTO group : tmDataListDTO.getGroupList()) {
                    getChildrenBasicDataOrReport(group.getBaseItemDtoList(), accessibleData, plat);
                }
            }
            if (CollectionUtils.isNotEmpty(tmDataListDTO.getBaseItemDtoList())) {
                getChildrenBasicDataOrReport(tmDataListDTO.getBaseItemDtoList(), accessibleData, plat);
            }
        }
    }

    private void getChildrenBasicDataOrReport(List<BaseItemDto> baseItemDtoList, List<MenuAuthorityResp> accessibleData, Integer plat) {
        for (BaseItemDto baseItemDto : baseItemDtoList) {
            if (!"allow".equalsIgnoreCase(baseItemDto.getAccess())) {
                continue;
            }
            MenuAuthorityResp resp = new MenuAuthorityResp();
            resp.setCode(baseItemDto.getCode());
            resp.setType(0 == plat ? TYPE_BASE_DATA : REPORT);
            accessibleData.add(resp);
        }
    }

    /**
     * 获取有权限的发起项目
     * @param startProjectList
     * @param resultList
     */
    private void getStartProject(List<TmStartProjectDTO> startProjectList, List<MenuAuthorityResp> resultList) {
        if (CollectionUtils.isEmpty(startProjectList)) {
            return;
        }
        for (TmStartProjectDTO startProject : startProjectList) {
            MenuAuthorityResp resp = new MenuAuthorityResp();
            resp.setCode(startProject.getCode());
            resp.setType(TYPE_START_PROJECT);
            resultList.add(resp);
        }
    }



    public List<MenuAuthorityResp> getAuthorityMenu(AuthoredUser user, List<MenuAuthorityReq> allMenuList) {
        log.info("allMenuList:{}", JsonUtils.objectToString(allMenuList));
        if (CollectionUtils.isEmpty(allMenuList)) {
            return Lists.newArrayList();
        }
        // 应用code集合
        List<String> appCodes = new ArrayList<>();
        // 作业集合
        Map<String, List<Map<String, Object>>> allMenuMap = new HashMap<>();
        // 获取应用code、整理作业格式
        this.dealMenus(allMenuList, appCodes, allMenuMap, user);
        // 批量获取应用的权限配置
        appCodes.add("Athena");
        permissionService.batchGetAppAccessible(user, appCodes);

        /*获取模组和系统菜单权限 start*/
        Map<String, List<String>> permissionMap = permissionCheckService.getModuleOrMenuAccessible(user, "Athena");
        // 所有设置权限的模组
        List<String> allPermissionModules = permissionMap.containsKey("allModule") ? permissionMap.get("allModule") : Lists.newArrayList();
        // 所有设置权限的菜单
        List<String> allPermissionMenus = permissionMap.containsKey("allMenu") ? permissionMap.get("allMenu") : Lists.newArrayList();
        // 拥有权限的模组
        List<String> accessibleModules = permissionMap.containsKey("accessibleModule") ? permissionMap.get("accessibleModule") : Lists.newArrayList();
        // 拥有权限的菜单
        List<String> accessibleMenus = permissionMap.containsKey("accessibleMenu") ?  permissionMap.get("accessibleMenu") : Lists.newArrayList();
        /*获取模组和系统菜单权限 end*/

        // 系统菜单数据
        String sysMenuData = ResourceUtil.readUtf8Str("data/systemMenus.json");
        List<SystemMenuDTO> systemMenuDataList = JsonUtils.jsonToListObject(sysMenuData, SystemMenuDTO.class);

        // 获取有权限的模组
        getAccessibleModules(allPermissionModules, accessibleModules, systemMenuDataList);

        List<MenuAuthorityResp> resultList = new ArrayList<>();
        // 获取有权限的系统菜单
        CompletableFuture<Void> future1 = CompletableFuture.runAsync(() -> {
            AppAuthContextHolder.getContext().setAuthoredUser(user);
            getAccessibleSysMenus(resultList, systemMenuDataList, accessibleModules, accessibleMenus, allPermissionMenus);
        });
        // 发起项目作业权限
        CompletableFuture<Void> future2 = CompletableFuture.runAsync(() -> {
            AppAuthContextHolder.getContext().setAuthoredUser(user);
            getAllowStartProjects(resultList, user, accessibleModules, Boolean.FALSE, allMenuMap.get(TYPE_START_PROJECT));
        });
        // 基础资料
        CompletableFuture<Void> future3 = CompletableFuture.runAsync(() -> {
            AppAuthContextHolder.getContext().setAuthoredUser(user);
            getAllowBaseDataEntries(resultList, user, accessibleModules, Boolean.FALSE, JsonUtils.jsonToListObject(JsonUtils.objectToString(allMenuMap.get(TYPE_BASE_DATA)), BaseItemDto.class));
        });
        // 报表
        CompletableFuture<Void> future4 = CompletableFuture.runAsync(() -> {
            AppAuthContextHolder.getContext().setAuthoredUser(user);
            getAllowReports(resultList, user, accessibleModules, Boolean.FALSE, JsonUtils.jsonToListObject(JsonUtils.objectToString(allMenuMap.get(REPORT)), BaseItemDto.class));
        });
        CompletableFuture<Void> allOf = CompletableFuture.allOf(future1, future2, future3, future4);
        try {
            allOf.get();
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new RuntimeException(e);
        } catch (ExecutionException e) {
            throw new RuntimeException(e);
        }
        return resultList;
    }


    private void dealMenus(List<MenuAuthorityReq> allMenuList, List<String> appCodes, Map<String, List<Map<String, Object>>> allMenuMap, AuthoredUser user) {
        // 系统菜单集合
        List<Map<String, Object>> sysMenuList = new ArrayList<>();
        // 发起项目集合
        List<Map<String, Object>> startProjectList = new ArrayList<>();
        // 基础资料集合
        List<Map<String, Object>> baseDataEntryList = new ArrayList<>();
        // 报表集合
        List<Map<String, Object>> reportList = new ArrayList<>();

        // 获取有权限的应用列表
        List<String> authorityAppCodes = permissionServiceImpl.getUserAuthoredAppCode(user.getToken());
        if (CollectionUtils.isEmpty(authorityAppCodes)) {
            return;
        }

        // 整理应用code和作业code
        for (MenuAuthorityReq menuAuthorityReq : allMenuList) {
            // 没有应用权限不做处理
            if (!authorityAppCodes.contains(menuAuthorityReq.getAppCode())) {
                continue;
            }
            appCodes.add(menuAuthorityReq.getAppCode());
            List<MenuAuthorityReq.JobCode> jobCodeList = menuAuthorityReq.getJobCodeList();
            if (CollectionUtils.isEmpty(jobCodeList)) {
                continue;
            }
            for (MenuAuthorityReq.JobCode jobCode : jobCodeList) {
                if (TYPE_SYS_MENU.equals(jobCode.getType())) {
                    Map<String,Object> menuMap = new HashMap<>();
                    menuMap.put("code", jobCode.getCode());
                    menuMap.put("appCode", menuAuthorityReq.getAppCode());
                    sysMenuList.add(menuMap);
                }
                if (TYPE_START_PROJECT.equals(jobCode.getType())) {
                    Map<String,Object> menuMap = new HashMap<>();
                    menuMap.put("code", jobCode.getCode());
                    menuMap.put("appCode", menuAuthorityReq.getAppCode());
                    startProjectList.add(menuMap);
                }
                if (TYPE_BASE_DATA.equals(jobCode.getType())) {
                    Map<String,Object> menuMap = new HashMap<>();
                    menuMap.put("code", jobCode.getCode());
                    menuMap.put("appCode", menuAuthorityReq.getAppCode());
                    baseDataEntryList.add(menuMap);
                }
                if (REPORT.equals(jobCode.getType())) {
                    Map<String,Object> menuMap = new HashMap<>();
                    menuMap.put("code", jobCode.getCode());
                    menuMap.put("appCode", menuAuthorityReq.getAppCode());
                    reportList.add(menuMap);
                }
            }
        }
        allMenuMap.put(TYPE_SYS_MENU, sysMenuList);
        allMenuMap.put(TYPE_START_PROJECT, startProjectList);
        allMenuMap.put(TYPE_BASE_DATA, baseDataEntryList);
        allMenuMap.put(REPORT, reportList);
    }

}
