package com.digiwin.athena.base.application.service.principal;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.base.application.config.BaseAudcDataSourceConfig;
import com.digiwin.athena.base.application.meta.request.principal.TaskTypeQueryReq;
import com.digiwin.athena.base.application.meta.request.proxyinfo.ProxyInfoSaveReq;
import com.digiwin.athena.base.application.meta.response.audc.TaskTypeQueryResp;
import com.digiwin.athena.base.infrastructure.mapper.audc.principal.PrincipalTaskTypeMapper;
import com.digiwin.athena.base.infrastructure.meta.bo.principal.ThePrincipalTaskNum;
import com.digiwin.athena.base.infrastructure.constant.AudcErrorCodeEnum;
import com.digiwin.athena.base.infrastructure.manager.eoc.dto.EocEmpInfoDTO;
import com.digiwin.athena.base.sdk.audc.application.service.principal.QueryPrincipalTaskTypeService;
import com.digiwin.athena.base.sdk.audc.application.service.principal.QueryPrincipalTaskTypeServiceImpl;
import com.digiwin.athena.base.sdk.audc.infrastructure.mapper.QueryPrincipalTaskTypeMapper;
import com.digiwin.athena.base.sdk.audc.infrastructure.meta.po.ThePrincipalTaskType;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * @author chenpeng
 * @date 2024年8月26日
 */
@Service
public class PrincipalTaskTypeServiceImpl implements PrincipalTaskTypeService {
    @Autowired
    private PrincipalTaskTypeMapper principalTaskTypeMapper;

    @Autowired
    private QueryPrincipalTaskTypeService queryPrincipalTaskTypeService;

    @Resource
    private MessageUtils messageUtils;

    /**
     * 保存 被代理人不可被代理的任务类型
     *
     * @param agentId      代理人
     * @param userId       被代理人
     * @param taskDefCodes 不可被代理的任务类型
     * @param tenantId     租户id
     * @param calendarTaskCodes  无权代理的行事历任务列表
     */
    @Override
    @Transactional(transactionManager = BaseAudcDataSourceConfig.BASE_AUDC_DATASOURCE_TRANSACTION_MANAGER_BUSINESS)
    public void saveTaskType(String agentId, String userId, List<String> taskDefCodes, String tenantId, List<String> calendarTaskCodes) {
        // 先删除已保存的数据
        principalTaskTypeMapper.batchDelete(Arrays.asList(userId), LocalDateTime.now(), tenantId);
        // 在新增新的数据
        List<ThePrincipalTaskType> allInsertList = new ArrayList<>();
        if (CollectionUtils.isNotEmpty(taskDefCodes)) {
            List<ThePrincipalTaskType> insertList = taskDefCodes.stream()
                    .map(taskDefCode -> new ThePrincipalTaskType(userId, agentId, tenantId, taskDefCode, 0))
                    .collect(Collectors.toList());
            allInsertList.addAll(insertList);
        }
        if (CollectionUtils.isNotEmpty(calendarTaskCodes)) {
            List<ThePrincipalTaskType> insertList = calendarTaskCodes.stream()
                    .map(taskDefCode -> new ThePrincipalTaskType(userId, agentId, tenantId, taskDefCode, 1))
                    .collect(Collectors.toList());
            allInsertList.addAll(insertList);
        }
        if (CollectionUtils.isNotEmpty(allInsertList)) {
            principalTaskTypeMapper.batchInsert(allInsertList);
        }

    }

    /**
     * 查询被代理人可被代理的任务类型
     *
     * @param taskTypeQueryReq
     * @param tenantId
     * @return
     */
    @Override
    public List<TaskTypeQueryResp> queryTaskType(TaskTypeQueryReq taskTypeQueryReq, String tenantId) {
        if (Objects.isNull(taskTypeQueryReq)) {
            throw AudcErrorCodeEnum.INVALID_REQUEST_PARAM.getBusinessException(messageUtils.getMessage("exception.param.null"));
        }
        if (CollectionUtils.isEmpty(taskTypeQueryReq.getThePrincipalId())) {
            throw AudcErrorCodeEnum.INVALID_REQUEST_PARAM.getBusinessException(messageUtils.getMessage("exception.param.thePrincipalIdList.null"));
        }

        List<ThePrincipalTaskType> queryList = queryPrincipalTaskTypeService.queryPrincipalTaskType(tenantId, taskTypeQueryReq.getThePrincipalId(), taskTypeQueryReq.getAgentId(), taskTypeQueryReq.getTaskDefCode());
        if (CollectionUtils.isEmpty(queryList)) {
            return Collections.emptyList();
        }
        List<TaskTypeQueryResp> response = queryList.stream().map(item -> new TaskTypeQueryResp(item)).collect(Collectors.toList());
        return response;
    }

    /**
     * 查询不可被代理的任务数量
     *
     * @param agentEmpList
     * @param tenantId
     */
    @Override
    public List<ThePrincipalTaskNum> queryTaskTypeNum(List<EocEmpInfoDTO> agentEmpList, String tenantId) {
        if (CollectionUtils.isEmpty(agentEmpList)) {
            return null;
        }
        return principalTaskTypeMapper.queryTaskTypeNum(agentEmpList, tenantId);
    }

    /**
     * 批量保存 被代理人可被代理的任务类型
     *
     * @param needSaveList
     * @param tenantId     租户id
     */
    @Override
    @Transactional(transactionManager = BaseAudcDataSourceConfig.BASE_AUDC_DATASOURCE_TRANSACTION_MANAGER_BUSINESS)
    public void batchSaveTaskType(List<ProxyInfoSaveReq> needSaveList, String tenantId) {
        if (CollectionUtils.isEmpty(needSaveList)) {
            return;
        }
        // 先删除已保存的数据
        List<String> userIds = needSaveList.stream().map(ProxyInfoSaveReq::getUserId).collect(Collectors.toList());
        principalTaskTypeMapper.batchDelete(userIds, LocalDateTime.now(), tenantId);
        // 在新增新的数据
        List<ThePrincipalTaskType> allInsertList = new ArrayList<>();
        for (ProxyInfoSaveReq proxyInfoSaveReq : needSaveList) {
            if (CollectionUtils.isNotEmpty(proxyInfoSaveReq.getTaskDefCodes())) {
                List<ThePrincipalTaskType> insertList = proxyInfoSaveReq.getTaskDefCodes().stream()
                        .map(taskDefCode -> new ThePrincipalTaskType(proxyInfoSaveReq.getUserId(), proxyInfoSaveReq.getAgentId(), tenantId, taskDefCode, 0))
                        .collect(Collectors.toList());
                allInsertList.addAll(insertList);
            }
            if (CollectionUtils.isNotEmpty(proxyInfoSaveReq.getCalendarTaskCodes())) {
                List<ThePrincipalTaskType> insertList = proxyInfoSaveReq.getCalendarTaskCodes().stream()
                        .map(taskDefCode -> new ThePrincipalTaskType(proxyInfoSaveReq.getUserId(), proxyInfoSaveReq.getAgentId(), tenantId, taskDefCode, 1))
                        .collect(Collectors.toList());
                allInsertList.addAll(insertList);
            }
        }
        if (CollectionUtils.isNotEmpty(allInsertList)) {
            principalTaskTypeMapper.batchInsert(allInsertList);
        }
    }

    /**
     * 批量删除被代理人的任务类型
     * @param clearList
     * @param tenantId
     */
    @Override
    @Transactional(transactionManager = BaseAudcDataSourceConfig.BASE_AUDC_DATASOURCE_TRANSACTION_MANAGER_BUSINESS)
    public void batchDeleteTaskType(List<String> clearList, String tenantId) {
        if (CollectionUtils.isNotEmpty(clearList)) {
            principalTaskTypeMapper.batchDelete(clearList, LocalDateTime.now(), tenantId);
        }
    }
}
