package com.digiwin.athena.base.application.util;

import cn.hutool.core.img.ImgUtil;
import cn.hutool.core.util.StrUtil;
import com.digiwin.athena.base.application.meta.request.attachment.AttachmentDownloadReqDTO;
import com.jugg.agile.framework.meta.exception.JaException;
import lombok.extern.slf4j.Slf4j;
import org.apache.pdfbox.pdmodel.PDDocument;
import org.apache.pdfbox.pdmodel.PDPage;
import org.apache.pdfbox.pdmodel.PDPageContentStream;
import org.apache.pdfbox.pdmodel.graphics.image.PDImageXObject;
import org.apache.pdfbox.util.Matrix;
import org.springframework.core.io.ClassPathResource;

import javax.imageio.ImageIO;
import java.awt.*;
import java.awt.geom.AffineTransform;
import java.awt.image.BufferedImage;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;


/**
 * @author mk
 * @date 2024/10/30 13:21
 * @description: 水印工具类
 */
@Slf4j
public class WatermarkUtil {
    /**
     * 图片添加水印
     *
     * @param imageBytes
     * @param watermark
     * @param imageFormat
     * @return
     * @throws IOException
     */
    public static byte[] addWatermarkImage(byte[] imageBytes, AttachmentDownloadReqDTO.Watermark watermark, String imageFormat) throws Exception {
        String topWatermarkText = watermark.getTopWatermarkText();
        String lowerWatermarkText = watermark.getLowerWatermarkText();
        // 读取图片字节数组到BufferedImage
        BufferedImage srcImg = ImgUtil.read(new ByteArrayInputStream(imageBytes));

        // 创建一个新的BufferedImage用于绘制水印
        Graphics2D g2d = srcImg.createGraphics();

        // 设置水印字体和颜色
        Font font = loadFont();
        g2d.setFont(font);
        // 半透明灰色
        g2d.setColor(new Color(192, 197, 203, 102));

        // 计算水印文本的宽度和高度
        FontMetrics fontMetrics = g2d.getFontMetrics();
        int textWidth = fontMetrics.stringWidth(topWatermarkText);
        int textHeight = fontMetrics.getHeight();

        // 设置水印的位置和间距
        // 水平间距
        int waterMarkWidth = textWidth + 200;
        // 垂直间距
        int waterMarkHeight = textHeight + 200;

        // 在整个图片上均匀添加倾斜的水印
        for (int i = 0; i < srcImg.getWidth(); i += waterMarkWidth) {
            for (int j = 0; j < srcImg.getHeight(); j += waterMarkHeight) {
                // 保存原始的Graphics2D状态
                AffineTransform originalTransform = g2d.getTransform();
                // 旋转Graphics2D对象，使用负值实现顺时针旋转
                g2d.rotate(-Math.toRadians(30), i + (double) textWidth / 2, j + (double) textHeight / 2);
                if (StrUtil.isBlank(lowerWatermarkText)) {
                    String utf8TopWatermarkText = new String(topWatermarkText.getBytes(StandardCharsets.UTF_8));
                    // 绘制第一行水印文本
                    g2d.drawString(StrUtil.sub(utf8TopWatermarkText, 0, 20), i, j);
                } else {
                    String utf8TopWatermarkText = new String(topWatermarkText.getBytes(StandardCharsets.UTF_8));
                    String utf8LowerWatermarkText = new String(lowerWatermarkText.getBytes(StandardCharsets.UTF_8));
                    // 绘制第一行水印文本
                    g2d.drawString(StrUtil.sub(utf8TopWatermarkText, 0, 20), i, j);
                    // 绘制第二行水印文本
                    g2d.drawString(StrUtil.sub(utf8LowerWatermarkText, 0, 20), i, j + textHeight);
                }
                // 恢复原始的Graphics2D状态
                g2d.setTransform(originalTransform);
            }
        }

        g2d.dispose(); // 释放资源
        // 将带有水印的图片保存到字节输出流
        ByteArrayOutputStream bao = new ByteArrayOutputStream();
        ImgUtil.write(srcImg, imageFormat, bao);
        // 返回含有水印的图片字节数组
        return bao.toByteArray();


    }

    /**
     * pdf添加水印
     *
     * @param pdfBytes
     * @param watermark
     * @return
     * @throws IOException
     */

    public static byte[] addWatermarkPdf(byte[] pdfBytes, AttachmentDownloadReqDTO.Watermark watermark)  throws Exception {
        try (PDDocument document = PDDocument.load(new ByteArrayInputStream(pdfBytes))) {
            for (PDPage page : document.getPages()) {
                try (PDPageContentStream contentStream = new PDPageContentStream(document, page, PDPageContentStream.AppendMode.APPEND, true, true)) {

                    // 获取页面尺寸
                    float pageWidth = page.getMediaBox().getWidth();
                    float pageHeight = page.getMediaBox().getHeight();

                    // 水印的设置
                    // 水印透明度
                    final float opacity = 0.4F;
                    // 水印倾斜角度
                    final double angle = Math.toRadians(30);
                    // 灰色（Light Gray）
                    Color color = new Color(192, 197, 203);

                    // 设置水印的间隔为100px
                    // 水平间隔
                    final float xInterval = 150F;
                    // 垂直间隔
                    final float yInterval = 150F;

                    // 创建一个透明的图像，用于绘制水印
                    BufferedImage bufferedImage = new BufferedImage(400, 100, BufferedImage.TYPE_INT_ARGB);
                    Graphics2D g2d = bufferedImage.createGraphics();
                    g2d.setComposite(AlphaComposite.getInstance(AlphaComposite.SRC_OVER, opacity));
                    //加载字体的方法
                    Font font = loadFont();
                    g2d.setFont(font);
                    g2d.setColor(color);
                    String topWatermarkText = watermark.getTopWatermarkText();
                    String lowerWatermarkText = watermark.getLowerWatermarkText();

                    if (StrUtil.isNotBlank(topWatermarkText)) {
                        if (StrUtil.isBlank(lowerWatermarkText)) {
                            g2d.drawString(StrUtil.sub(topWatermarkText, 0, 20), 10, 50);
                        } else {
                            // 上行文本
                            g2d.drawString(StrUtil.sub(topWatermarkText, 0, 20), 10, 35);
                            // 下行文本
                            g2d.drawString(StrUtil.sub(lowerWatermarkText, 0, 20), 10, 65);
                        }
                    }
                    g2d.dispose();

                    // 将图像保存到 ByteArrayOutputStream
                    ByteArrayOutputStream baos = new ByteArrayOutputStream();
                    ImageIO.write(bufferedImage, "PNG", baos);
                    baos.flush();

                    // 将图像数据转换为 PDImageXObject
                    PDImageXObject pdImage = PDImageXObject.createFromByteArray(document, baos.toByteArray(), "watermark");

                    // 循环添加水印
                    for (float x = 0; x < pageWidth; x += xInterval) {
                        for (float y = 0; y < pageHeight; y += yInterval) {
                            contentStream.saveGraphicsState();
                            contentStream.transform(new Matrix((float) Math.cos(angle), (float) Math.sin(angle), -(float) Math.sin(angle), (float) Math.cos(angle), x, y));
                            contentStream.drawImage(pdImage, 0, 0);
                            contentStream.restoreGraphicsState();
                        }
                    }
                }
            }

            ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
            document.save(outputStream);
            return outputStream.toByteArray();
        }
    }

    public static Font loadFont() throws Exception {
        //使用 ClassLoader 加载资源文件
        ClassPathResource classPathResource = new ClassPathResource("fonts/msyh.ttf");
        InputStream is =classPathResource.getInputStream();
        if (is == null) {
            throw new JaException("Font file not found in resources: fonts/msyh.ttf");
        }

        // 创建并返回字体对象
        return Font.createFont(Font.TRUETYPE_FONT, is).deriveFont(Font.PLAIN, 12);
    }


}



