package com.digiwin.athena.base.application.service.attachment;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.appcore.util.Base64Converter;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.base.application.config.BaseAttachmentDataSourceConfig;
import com.digiwin.athena.base.application.converter.DataConverter;
import com.digiwin.athena.base.application.manager.dmc.DmcService;
import com.digiwin.athena.base.application.meta.request.attachment.AttachUploadedCountDTO;
import com.digiwin.athena.base.application.meta.request.attachment.AttachmentDownloadReqDTO;
import com.digiwin.athena.base.application.meta.request.attachment.AttachmentReqDTO;
import com.digiwin.athena.base.application.meta.request.attachment.DeleteAttachmentReqDTO;
import com.digiwin.athena.base.application.meta.request.attachment.DmcAccount;
import com.digiwin.athena.base.application.meta.request.attachment.UploadOrDeleteAttachmentReqDTO;
import com.digiwin.athena.base.application.meta.request.attachment.UploadParamDTO;
import com.digiwin.athena.base.application.meta.response.attachment.AttachmentDownloadWatermarkRespDTO;
import com.digiwin.athena.base.application.meta.response.attachment.ShareAttachmentRespDTO;
import com.digiwin.athena.base.application.meta.response.attachment.UploadAttachmentRespDTO;
import com.digiwin.athena.base.application.util.WatermarkUtil;
import com.digiwin.athena.base.application.util.ZipUtils;
import com.digiwin.athena.base.infrastructure.mapper.attachment.AttachmentMapper;
import com.digiwin.athena.base.sdk.aam.application.meta.request.QueryAttachmentReqDTO;
import com.digiwin.athena.base.sdk.aam.application.meta.response.AttachmentRespDTO;
import com.digiwin.athena.base.sdk.aam.application.service.QueryAttachmentService;
import com.digiwin.athena.base.sdk.aam.application.util.CommonUtils;
import com.digiwin.athena.base.sdk.aam.infrastructure.meta.po.AttachmentEntity;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.compress.utils.Lists;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.web.client.RestTemplate;

import javax.annotation.Resource;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.net.HttpURLConnection;
import java.net.URL;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Base64;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import static org.apache.commons.lang3.CharEncoding.UTF_8;

@Slf4j
@Service("attachmentService")
public class AttachmentServiceImpl implements AttachmentService {
    @Autowired
    private QueryAttachmentService queryAttachmentService;

    @Resource
    private AttachmentMapper attachmentMapper;

    @Resource
    private AttachmentServiceImpl attachmentService;

    @Autowired
    private RestTemplate restTemplate;

    @Autowired
    private DmcService dmcService;


    private static final DateTimeFormatter dateFormat = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");

    /**
     * {@inheritDoc}
     */
    @Override
    public List<AttachmentRespDTO> queryByTaskIdAndRowDataKeyList(String tenantId, String taskId, String projectId, List<String> categories, List<String> rowDataKeyList) {
        return queryAttachmentService.queryByTaskIdAndRowDataKeyList(tenantId, taskId, projectId, categories, rowDataKeyList);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public List<AttachmentRespDTO> queryByTaskIdAndRowDataKeyList(QueryAttachmentReqDTO queryAttachmentReqDTO) {
        return queryAttachmentService.queryByTaskIdAndRowDataKeyList(queryAttachmentReqDTO);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    @Transactional(transactionManager = BaseAttachmentDataSourceConfig.BASE_ATTACHMENT_DATASOURCE_TRANSACTION_MANAGER_BUSINESS)
    public void saveAttachment(AttachmentReqDTO attachmentFile) {
        log.info("saveAttachment, body:{}", attachmentFile);
        // 获取当前登录用户信息
        AuthoredUser authoredUser = AppAuthContextHolder.getContext().getAuthoredUser();

        attachmentFile.setUploadUserId(authoredUser.getUserId());
        attachmentFile.setUploadUserName(authoredUser.getUserName());

        log.info("saveAttachment params: {}", attachmentFile);
        attachmentMapper.insert(DataConverter.INSTANCE.toAttachmentEntity(attachmentFile));
    }

    private void saveAttachment(List<AttachmentReqDTO> uploadedList) {
        // 获取当前登录用户信息
        AuthoredUser authoredUser = AppAuthContextHolder.getContext().getAuthoredUser();
        for (AttachmentReqDTO attachmentFile : uploadedList) {
            attachmentFile.setUploadUserId(authoredUser.getUserId());
            attachmentFile.setUploadUserName(authoredUser.getUserName());
        }

        attachmentMapper.insertBatchSomeColumn(uploadedList.stream().map(attachmentReqDTO -> DataConverter.INSTANCE.toAttachmentEntity(attachmentReqDTO)).collect(Collectors.toList()));
    }

    /**
     * {@inheritDoc}
     */
    @Override
    @Transactional(transactionManager = BaseAttachmentDataSourceConfig.BASE_ATTACHMENT_DATASOURCE_TRANSACTION_MANAGER_BUSINESS)
    public void deleteAttachment(AttachmentReqDTO attachmentFile) {
        log.info("deleteAttachment, body:{}", attachmentFile);
        this.deleteAttachment(attachmentFile.getCategory(), attachmentFile.getCategoryId(), attachmentFile.getId());
    }

    private int deleteAttachment(String category, String categoryId, String id) {
        LambdaQueryWrapper<AttachmentEntity> wrapper = Wrappers.lambdaQuery();
        wrapper.eq(StringUtils.isNotBlank(category), AttachmentEntity::getCategory, category).eq(StringUtils.isNotBlank(categoryId), AttachmentEntity::getCategoryId, categoryId).eq(AttachmentEntity::getId, id);

        return attachmentMapper.delete(wrapper);
    }

    private int deleteAttachment(String category, String categoryId, Collection<String> idList) {
        LambdaQueryWrapper<AttachmentEntity> wrapper = Wrappers.lambdaQuery();
        wrapper.eq(StringUtils.isNotBlank(category), AttachmentEntity::getCategory, category).eq(StringUtils.isNotBlank(categoryId), AttachmentEntity::getCategoryId, categoryId).in(AttachmentEntity::getId, idList);

        return attachmentMapper.delete(wrapper);
    }

    @Override
    @Transactional(transactionManager = BaseAttachmentDataSourceConfig.BASE_ATTACHMENT_DATASOURCE_TRANSACTION_MANAGER_BUSINESS)
    public void uploadOrDeleteAttachment(UploadOrDeleteAttachmentReqDTO uploadOrDeleteAttachmentReqDTO) {
        log.info("uploadOrDeleteAttachment, body:{}", uploadOrDeleteAttachmentReqDTO);
        if (CollectionUtils.isNotEmpty(uploadOrDeleteAttachmentReqDTO.getUploadedFileList())) {
            // 获取当前登录用户信息
            AuthoredUser authoredUser = AppAuthContextHolder.getContext().getAuthoredUser();

            // 保存新增的附件, 通过【子事务】隔离保存操作
            for (AttachmentReqDTO attachmentReqDTO : uploadOrDeleteAttachmentReqDTO.getUploadedFileList()) {
                attachmentReqDTO.setUploadUserId(authoredUser.getUserId());
                attachmentReqDTO.setUploadUserName(authoredUser.getUserName());
            }
            attachmentService.saveAttachmentInNewTransaction(uploadOrDeleteAttachmentReqDTO.getUploadedFileList());
        }

        if (CollectionUtils.isNotEmpty(uploadOrDeleteAttachmentReqDTO.getDeletedFileList())) {
            Set<String> fileIdList = uploadOrDeleteAttachmentReqDTO.getDeletedFileList().stream().map(AttachmentReqDTO::getId).filter(id -> StringUtils.isNotBlank(id)).collect(Collectors.toSet());

            // 通过【子事物】隔离删除操作
            attachmentService.deleteAttachmentInNewTransaction(null, null, fileIdList);

            // 调用dmc删除附件信息
            if (BooleanUtils.isTrue(uploadOrDeleteAttachmentReqDTO.getNeedDeleteDmcFile())) {
                dmcService.loginAndDeleteAttachment(Collections.emptyList(), fileIdList, null);
            }
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void deleteAttachmentBatch(DeleteAttachmentReqDTO deleteAttachmentReqDTO) {
        log.info("deleteAttachmentBatch, body:{}", deleteAttachmentReqDTO);

        if (CollectionUtils.isNotEmpty(deleteAttachmentReqDTO.getDeleteFileList())) {
            List<AttachmentReqDTO> deleteFileList = deleteAttachmentReqDTO.getDeleteFileList();
            Map<String, DmcAccount> schemaDmcAccountReqDTOMap = deleteAttachmentReqDTO.getSchemaDmcAccountMap();

            if (MapUtils.isEmpty(schemaDmcAccountReqDTOMap)) {
                Set<String> fileIdList = deleteAttachmentReqDTO.getDeleteFileList().stream().map(AttachmentReqDTO::getId).filter(id -> StringUtils.isNotBlank(id)).collect(Collectors.toSet());
                // 调用dmc删除附件信息
                dmcService.loginAndDeleteAttachment(Collections.emptyList(), fileIdList, null);
                return;
            }

            for (AttachmentReqDTO attachmentReqDTO : deleteFileList) {
                Set<String> fileIdList = new HashSet<>();
                fileIdList.add(attachmentReqDTO.getId());
                DmcAccount DmcAccountReqDTO = schemaDmcAccountReqDTOMap.get(attachmentReqDTO.getTargetSchema());
                if (null == DmcAccountReqDTO) {
                    // 调用dmc删除附件信息
                    dmcService.loginAndDeleteAttachment(Collections.emptyList(), fileIdList, null);
                } else {
                    // 将dmc密码进行 base64解密
                    encodeDmcAccountReqDTOPwd(DmcAccountReqDTO);
                    // hash加密
                    encodeDmcAccountReqDTOPwdByHash(DmcAccountReqDTO);
                    // 调用dmc删除附件信息
                    dmcService.loginAndDeleteAttachment(Collections.emptyList(), fileIdList, DmcAccountReqDTO);
                }
            }
        }
    }

    /**
     * 将dmc密码进行 base64解密
     *
     * @param DmcAccountReqDTO
     */
    private static void encodeDmcAccountReqDTOPwd(DmcAccount DmcAccountReqDTO) {
        if (null == DmcAccountReqDTO || !BooleanUtils.isTrue(DmcAccountReqDTO.isPasswordAlreadyEncoded()) || StringUtils.isBlank(DmcAccountReqDTO.getPassword())) {
            return;
        }

        DmcAccountReqDTO.setPassword(Base64Converter.decode(DmcAccountReqDTO.getPassword()));
        DmcAccountReqDTO.setPasswordAlreadyEncoded(false);
    }

    /**
     * hash加密
     *
     * @param DmcAccountReqDTO
     */
    private static void encodeDmcAccountReqDTOPwdByHash(DmcAccount DmcAccountReqDTO) {
        if (null == DmcAccountReqDTO || DmcAccountReqDTO.isPasswordAlreadyEncoded() || StringUtils.isBlank(DmcAccountReqDTO.getPassword())) {
            return;
        }

        DmcAccountReqDTO.setPassword(CommonUtils.sha256(DmcAccountReqDTO.getPassword(), 2));
        DmcAccountReqDTO.setPasswordAlreadyEncoded(true);
    }

    @Override
    public String uploadAttachmentByUrl(UploadParamDTO uploadParamDTO) {
        HttpURLConnection conn;
        try {
            URL url = new URL(uploadParamDTO.getDownloadUrl());
            conn = (HttpURLConnection) url.openConnection();
            conn.setRequestMethod(HttpMethod.GET.name());
            conn.setConnectTimeout(20 * 1000);
            conn.connect();
            uploadParamDTO.setInputStream(conn.getInputStream());
            if (uploadParamDTO.getInputStream().available() <= 0) {
                throw new RuntimeException();
            }
            if (StringUtils.isBlank(uploadParamDTO.getDownloadName())) {
                //应用未传downloadName 需要从url获取,都获取不到 适配成zip
                String headerField = conn.getHeaderField("Content-Disposition");
                if (headerField == null || !headerField.contains("filename=")) {
                    String fileName = url.getFile();
                    fileName = fileName.substring(fileName.lastIndexOf("/") + 1);
                    uploadParamDTO.setDownloadName(fileName);
                    //自定义文件名称
//                    String filename = UUID.randomUUID().toString();
//                    String fileType = FileTypeUtil.getType(uploadParamDTO.getInputStream());
//                    uploadParamDTO.setDownloadName(filename + "." + (StringUtils.isNotBlank(fileType) ? fileType : "zip"));
                } else {
                    uploadParamDTO.setDownloadName(URLDecoder.decode((headerField.substring(headerField.lastIndexOf("filename=") + 9)), "UTF-8").replaceAll("\"", ""));
                }
            }
        } catch (Exception e) {
            throw BusinessException.create(500, "文件内容为空,文件链接失效", ExceptionUtils.getRootCause(e));
        }
        if (StringUtils.isBlank(uploadParamDTO.getExpireDate())) {
            LocalDateTime localDateTime = LocalDateTime.now().plusYears(1);
            uploadParamDTO.setExpireDate(dateFormat.format(localDateTime));
        }
        //上传和分享文档获取url
        Optional<ShareAttachmentRespDTO> shareAttachmentRespDTO = dmcService.uploadAndShareAttachment(uploadParamDTO);
        return shareAttachmentRespDTO.map(ShareAttachmentRespDTO::getUrl).orElse(null);
    }

    /**
     * 上传文档
     *
     * @param uploadParamDTO
     * @return
     */
    @Override
    public UploadAttachmentRespDTO uploadAgileReport(UploadParamDTO uploadParamDTO) {
        if (StringUtils.isEmpty(uploadParamDTO.getFileJson())) {
            return null;
        }
        try {
            InputStream inputStream = new ByteArrayInputStream(uploadParamDTO.getFileJson().getBytes());
            uploadParamDTO.setInputStream(inputStream);
            if (uploadParamDTO.getInputStream().available() <= 0) {
                throw new RuntimeException();
            }
            uploadParamDTO.setDownloadName(uploadParamDTO.getDownloadName() + ".json");
        } catch (Exception e) {
            throw BusinessException.create(500, "文件内容为空,文件链接失效", ExceptionUtils.getRootCause(e));
        }
        //上传和分享文档获取url
        return dmcService.uploadAttachment(uploadParamDTO);
    }


    @Override
    public Long attachUploadedCount(AttachUploadedCountDTO attachUploadedCountDTO) {

        QueryWrapper<AttachmentEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.lambda()
                .eq(AttachmentEntity::getRowDataKey, attachUploadedCountDTO.getRowDataKey())
                .in(AttachmentEntity::getCategory, attachUploadedCountDTO.getCategoryList())
                .eq(AttachmentEntity::getTenantId, attachUploadedCountDTO.getTenantId())
                .eq(StringUtils.isNotBlank(attachUploadedCountDTO.getTaskId()), AttachmentEntity::getTaskId, attachUploadedCountDTO.getTaskId())
                .eq(StringUtils.isNotBlank(attachUploadedCountDTO.getProjectId()), AttachmentEntity::getProjectId, attachUploadedCountDTO.getProjectId());
        return attachmentMapper.selectCount(queryWrapper);
    }

    @Override
    public boolean checkUploadCount(AttachmentReqDTO attachmentFile) {

        if (null != attachmentFile.getAllowCount() && attachmentFile.getCheckCount()) {
            AttachUploadedCountDTO attachUploadedCountDTO = AttachUploadedCountDTO.builder()
                    .rowDataKey(attachmentFile.getRowDataKey())
                    .tenantId(attachmentFile.getTenantId())
                    .taskId(attachmentFile.getTaskId())
                    .projectId(attachmentFile.getProjectId())
                    .categoryList(Collections.singletonList(attachmentFile.getCategory()))
                    .build();
            Long uploadedCount = attachUploadedCount(attachUploadedCountDTO);
            return uploadedCount >= attachmentFile.getAllowCount();
        }
        return false;
    }

    /**
     * 获取报表文档
     *
     * @param fileId
     * @return
     */
    @Override
    public Map<String, Object> getFile(String fileId) {
        InputStream inputStream = dmcService.previewFile(fileId);
        try {
            String result = IOUtils.toString(inputStream, StandardCharsets.UTF_8);
            if (StringUtils.isNotEmpty(result)) {
                return JsonUtils.jsonToObject(result, Map.class);
            }
        } catch (IOException e) {
            log.error("获取敏捷报表数据异常：" + e.getMessage());
        }
        return null;
    }

    @Override
    public Map<String, Object> deleteFile(String fileId) {
        return dmcService.deleteFile(fileId);
    }

    @Transactional(propagation = Propagation.REQUIRES_NEW, transactionManager = BaseAttachmentDataSourceConfig.BASE_ATTACHMENT_DATASOURCE_TRANSACTION_MANAGER_BUSINESS)
    public void saveAttachmentInNewTransaction(List<AttachmentReqDTO> uploadedList) {
        saveAttachment(uploadedList);
    }

    @Transactional(propagation = Propagation.REQUIRES_NEW, transactionManager = BaseAttachmentDataSourceConfig.BASE_ATTACHMENT_DATASOURCE_TRANSACTION_MANAGER_BUSINESS)
    public int deleteAttachmentInNewTransaction(String category, String categoryId, Collection<String> idList) {
        return deleteAttachment(category, categoryId, idList);
    }

    @Override
    public Object download(AttachmentDownloadReqDTO attachmentDownloadReqDTO) {
        //入参内容校验
        checkDownloadContent(attachmentDownloadReqDTO);
        // 设置响应头
        HttpHeaders responseHeaders = new HttpHeaders();
        responseHeaders.setContentType(MediaType.APPLICATION_OCTET_STREAM);
        responseHeaders.setContentDispositionFormData("attachment", attachmentDownloadReqDTO.getFileName() + "." + attachmentDownloadReqDTO.getSuffixName());
        //fileUrl不为空 设置水印 （图纸控件场景）
        if (StringUtils.isNotBlank(attachmentDownloadReqDTO.getFileUrl())) {
            return ResponseEntity.ok().headers(responseHeaders).body(drawingControl(attachmentDownloadReqDTO));
        }

        //fileId 不为空 先添加水印 在打成zip包 并设置解压密码
        AttachmentDownloadReqDTO.DmcAccount dmcAccount = attachmentDownloadReqDTO.getDmcAccount();
        //获取dmc token
        String token = getDmcToken(dmcAccount);
        //调用dmc获取附件
        ResponseEntity<byte[]> responseEntity = this.dmcService.getAttachment(attachmentDownloadReqDTO.getFileId(), token);
        //如果没有获取到文件内容，返回空文件 不进行压缩
        if (Objects.isNull(responseEntity.getBody())) {
            return ResponseEntity.ok().headers(responseHeaders).body(responseEntity);
        }

        //文件存在添加水印
        byte[] bytes = convertToWatermark(attachmentDownloadReqDTO, responseEntity.getBody());
        //根据密码判断是否需要压缩成zip包设置压缩密码 没有设置密码直接返回文件
        if (Objects.isNull(attachmentDownloadReqDTO.getSecurity()) || StringUtils.isBlank(attachmentDownloadReqDTO.getSecurity().getDefence())) {
            return ResponseEntity.ok().headers(responseHeaders).body(bytes);
        }

        //打成zip包设置密码
        byte[] zipFile = convertToZip(attachmentDownloadReqDTO, bytes);
        responseHeaders.setContentDispositionFormData("attachment", attachmentDownloadReqDTO.getFileName() + ".zip");
        return ResponseEntity.ok().headers(responseHeaders).body(zipFile);
    }

    @Override
    public Object downloadImageWatermarkBase64(List<AttachmentDownloadReqDTO> attachmentDownloadReqDTOList) {
        List<AttachmentDownloadWatermarkRespDTO> result = Lists.newArrayList();
        for (AttachmentDownloadReqDTO attachmentDownloadReqDTO : attachmentDownloadReqDTOList) {
            try {
                // 校验入参内容
                String fileId = attachmentDownloadReqDTO.getFileId();
                String fileUrl = attachmentDownloadReqDTO.getFileUrl();
                String suffixName = attachmentDownloadReqDTO.getSuffixName();
                String name = attachmentDownloadReqDTO.getName();
                Assert.isTrue(StringUtils.isNotBlank(fileId) || StringUtils.isNotBlank(fileUrl), "Parameter fileId fileUrl at least one is not empty");
                Assert.isTrue(StringUtils.isNotBlank(suffixName) || StringUtils.isNotBlank(name), "Parameter suffixName name at least one is not empty");
                if (StringUtils.isBlank(suffixName)) {
                    int lastDotIndex = name.lastIndexOf('.');
                    if (lastDotIndex == -1) {
                        throw new IllegalArgumentException("No file extension found");
                    }
                    suffixName = name.substring(lastDotIndex + 1).toLowerCase();
                    attachmentDownloadReqDTO.setSuffixName(suffixName);
                }

                AttachmentDownloadWatermarkRespDTO respDTO = new AttachmentDownloadWatermarkRespDTO();
                respDTO.setFileId(fileId);
                respDTO.setFileUrl(fileUrl);

                byte[] fileBytes = null;
                // 如果 fileUrl 不为空，直接下载文件
                if (StringUtils.isNotBlank(fileUrl)) {
                    fileBytes = downloadImage(fileUrl);
                } else {
                    // 通过 DMC 获取附件
                    AttachmentDownloadReqDTO.DmcAccount dmcAccount = attachmentDownloadReqDTO.getDmcAccount();
                    String token = getDmcToken(dmcAccount);
                    ResponseEntity<byte[]> responseEntity = dmcService.getAttachment(attachmentDownloadReqDTO.getFileId(), token);
                    if (responseEntity != null && responseEntity.getBody() != null) {
                        fileBytes = responseEntity.getBody();
                    }
                }
                // 添加水印并转换为 Base64
                if (fileBytes != null) {
                    byte[] watermarkedBytes = convertToWatermark(attachmentDownloadReqDTO, fileBytes);
                    String base64 = watermarkedBytes != null ? Base64.getEncoder().encodeToString(watermarkedBytes) : null;
                    // 动态拼接前缀
                    String base64String = "data:image/" + attachmentDownloadReqDTO.getSuffixName() + ";base64," + base64;
                    respDTO.setBase64(base64String);
                }
                result.add(respDTO);
            } catch (Exception e) {
                String file = StringUtils.isNotBlank(attachmentDownloadReqDTO.getFileId()) ? attachmentDownloadReqDTO.getFileId() : attachmentDownloadReqDTO.getFileUrl();
                log.error("downloadWatermarkBase64 error processing file with file: {}", file, e);
            }
        }
        return result;
    }

    /**
     * 将文件打成zip包
     *
     * @param attachmentDownloadReqDTO
     * @return
     */
    private byte[] convertToZip(AttachmentDownloadReqDTO attachmentDownloadReqDTO, byte[] fileBytes) {
        //根据密码判断是否需要压缩成zip包设置压缩密码 没有设置密码直接返回文件
        if (Objects.isNull(attachmentDownloadReqDTO.getSecurity()) || StringUtils.isBlank(attachmentDownloadReqDTO.getSecurity().getDefence())) {
            return fileBytes;
        }
        ByteArrayInputStream byteArrayInputStream = new ByteArrayInputStream(fileBytes);
        try {
            fileBytes = ZipUtils.extracted(byteArrayInputStream, Base64Converter.decode(attachmentDownloadReqDTO.getSecurity().getDefence()), attachmentDownloadReqDTO.getFileName() + "." + attachmentDownloadReqDTO.getSuffixName());
        } catch (IOException e) {
            log.error("ZipUtils.extracted fileId:" + e.getMessage());
        }
        // 返回文件内容和响应头
        return fileBytes;
    }

    /**
     * 加密、水印下载附件内容校验
     *
     * @param attachmentDownloadReqDTO
     */
    private void checkDownloadContent(AttachmentDownloadReqDTO attachmentDownloadReqDTO) {
        String fileId = attachmentDownloadReqDTO.getFileId();
        String fileUrl = attachmentDownloadReqDTO.getFileUrl();
        List<String> fileIds = attachmentDownloadReqDTO.getFileIds();
        Assert.hasText(attachmentDownloadReqDTO.getFileName(), "Parameter fileName must not be null");
        Assert.hasText(attachmentDownloadReqDTO.getSuffixName(), "Parameter suffixName must not be null");
        Assert.isTrue(CollectionUtils.isNotEmpty(fileIds) || StringUtils.isNotBlank(fileId)
                || StringUtils.isNotBlank(fileUrl), "Parameter fileIds fileId fileUrl at least one is not empty");

    }

    /**
     * 设置水印
     *
     * @param attachmentDownloadReqDTO
     * @param file
     * @return
     */
    private byte[] convertToWatermark(AttachmentDownloadReqDTO attachmentDownloadReqDTO, byte[] file) {
        AttachmentDownloadReqDTO.Watermark watermark = attachmentDownloadReqDTO.getWatermark();
        boolean flag = Stream.of("jpg", "jpeg", "gif", "bmp", "png", "pic", "pdf").anyMatch(attachmentDownloadReqDTO.getSuffixName()::equalsIgnoreCase);
        //为空不需要设置水印 不是pdf或者图片 不需要设置水印
        if (Objects.isNull(watermark) || StringUtils.isBlank(watermark.getTopWatermarkText()) || !flag) {
            return file;
        }
        //设置pdf水印
        if ("pdf".equalsIgnoreCase(attachmentDownloadReqDTO.getSuffixName())) {
            try {
                return WatermarkUtil.addWatermarkPdf(file, watermark);
            } catch (Exception e) {
                log.error("WatermarkUtil.addWatermarkPdf failed:", e);
            }
        }
        //设置图片水印
        try {
            return WatermarkUtil.addWatermarkImage(file, watermark, attachmentDownloadReqDTO.getSuffixName());
        } catch (Exception e) {
            log.error("WatermarkUtil.addWatermarkImage failed:", e);
        }
        return file;
    }

    /**
     * 调用ksc获取文件
     *
     * @param url ksc url
     * @return
     */
    private byte[] getKsc(String url) {
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        HttpEntity<Map<String, Object>> httpEntity = new HttpEntity<>(headers);
        try {
            ResponseEntity<byte[]> responseEntity = restTemplate.exchange(url, HttpMethod.GET, httpEntity, new ParameterizedTypeReference<byte[]>() {
            });
            return responseEntity.getBody();
        } catch (Exception ex) {
            log.error("ksc failed, url: {}", url, ex);
            throw BusinessException.create("ksc failed, error: " + ex.getMessage());
        }
    }

    /**
     * 图纸控件处理
     *
     * @param attachmentDownloadReqDTO
     * @return
     */
    private byte[] drawingControl(AttachmentDownloadReqDTO attachmentDownloadReqDTO) {
        String fileUrl = attachmentDownloadReqDTO.getFileUrl();
        //调用ksc获取文件
        byte[] ksc = getKsc(fileUrl);
        return convertToWatermark(attachmentDownloadReqDTO, ksc);
    }

    /**
     * 从dmc获取token
     *
     * @param dmcAccount
     * @return
     */
    private String getDmcToken(AttachmentDownloadReqDTO.DmcAccount dmcAccount) {
        //使用dmc系统账号
        if (Objects.isNull(dmcAccount) || Objects.isNull(dmcAccount.getAccount()) || Objects.isNull(dmcAccount.getPassword())) {
            return this.dmcService.login(null, null);
        } else {
            String account = dmcAccount.getAccount();
            String password = dmcAccount.getPassword();
            Boolean isPasswordAlreadyEncoded = dmcAccount.getIsPasswordAlreadyEncoded();
            if (Objects.isNull(isPasswordAlreadyEncoded) || !isPasswordAlreadyEncoded) {
                //base64解密
                String decodePassword = Base64Converter.decode(dmcAccount.getPassword());
                //hash加密
                password = CommonUtils.sha256(decodePassword, 2);
            }
            return this.dmcService.login(account, password);
        }
    }

    private byte[] downloadImage(String imageUrl) {
        HttpHeaders headers = new HttpHeaders();
        HttpEntity<Map<String, Object>> httpEntity = new HttpEntity<>(headers);

        ResponseEntity<byte[]> responseEntity = restTemplate.exchange(imageUrl, HttpMethod.GET, httpEntity, new ParameterizedTypeReference<byte[]>() {
        });
        return responseEntity.getBody();
    }

    @Override
    public Object batchDownload(AttachmentDownloadReqDTO attachmentDownloadReqDTO) {
        //入参内容校验
        checkDownloadContent(attachmentDownloadReqDTO);
        // 设置响应头
        String fileName = attachmentDownloadReqDTO.getFileName() + "." + attachmentDownloadReqDTO.getSuffixName();
        String encodedFileName;
        try {
            encodedFileName = URLEncoder.encode(fileName, UTF_8).replaceAll("\\+", "%20");
        } catch (UnsupportedEncodingException e) {
            throw new RuntimeException(e);
        }
        HttpHeaders responseHeaders = new HttpHeaders();
        responseHeaders.setContentType(MediaType.APPLICATION_OCTET_STREAM);
        responseHeaders.setContentDispositionFormData("attachment", encodedFileName);

        AttachmentDownloadReqDTO.DmcAccount dmcAccount = attachmentDownloadReqDTO.getDmcAccount();
        //获取dmc token
        String token = getDmcToken(dmcAccount);
        //调用dmc获取附件
        Map<String, Object> param = new HashMap<>();
        param.put("fileId", null);
        param.put("fileIds", attachmentDownloadReqDTO.getFileIds());
        param.put("dirId", null);
        param.put("dirIds", null);
        byte[] zipFile = this.dmcService.getMultiAttachment(param, token);
        return ResponseEntity.ok()
                .headers(responseHeaders)
                .body(zipFile);
    }
}
