package com.digiwin.athena.base.application.service.commonused;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.util.SnowflakeIdWorker;
import com.digiwin.athena.base.application.config.BaseAudcDataSourceConfig;
import com.digiwin.athena.base.application.meta.dto.commonused.CategoriesDto;
import com.digiwin.athena.base.infrastructure.mapper.audc.commonUsed.CategoryMapper;
import com.digiwin.athena.base.infrastructure.meta.po.commonused.CategoryData;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.time.LocalDateTime;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.TimeUnit;

@Service
public class CategoryServiceImpl implements CategoryService {

    private static final Logger logger = LoggerFactory.getLogger(CategoryServiceImpl.class);

    private static final int TOP_STATUS = 1;

    private static final int NOT_TOP_STATUS = 0;

    private static final String CATE_DEFAULT_KEY_PREFIX = "audc:commomused:cate:unfold:";


    @Autowired
    private CategoryMapper categoryMapper;

    @Autowired
    private StringRedisTemplate stringRedisTemplate;

    /**
     * 查询对应的模块常用下分类信息
     *
     * @param user   用户信息
     * @param plat   所属模块0:基础资料录入 1：报表，后续有新增则依次递增
     * @param cateId 分类ID -1：根节点
     */
    public List<CategoryData> queryCateList(AuthoredUser user, Integer plat, Long cateId) {
        //获取节点的分类信息
        QueryWrapper<CategoryData> cateQuery = new QueryWrapper<>();
        if (cateId != null) {
            cateQuery.eq("parent_id", cateId);
        }
        cateQuery.eq("plat", plat);
        cateQuery.eq("tenant_id", user.getTenantId());
        cateQuery.eq("user_id", user.getUserId());
        return categoryMapper.selectList(cateQuery);
    }


    /**
     * 新增分类
     *
     * @param dto dto
     * @return int
     */
    @Transactional(transactionManager = BaseAudcDataSourceConfig.BASE_AUDC_DATASOURCE_TRANSACTION_MANAGER_BUSINESS)
    public CategoryData addCategory(AuthoredUser user, CategoriesDto dto) {
        CategoryData data = new CategoryData();
        data.setId(SnowflakeIdWorker.getInstance().newId());
        data.setCreateDate(LocalDateTime.now());
        data.setModifyDate(LocalDateTime.now());
        data.setCateName(dto.getCateName());
        data.setParentId(Objects.isNull(dto.getParentId()) ? -1 : dto.getParentId());
        data.setPlat(dto.getPlat());
        data.setUserId(user.getUserId());
        data.setTenantId(user.getTenantId());
        data.setUserName(user.getUserName());
        int result = categoryMapper.insert(data);
        if (result > 0) {
            return data;
        }
        return null;
    }

    /**
     * 根据分类id删除分类
     *
     * @param user user
     * @param id   id
     * @return int
     */
    @Transactional(transactionManager = BaseAudcDataSourceConfig.BASE_AUDC_DATASOURCE_TRANSACTION_MANAGER_BUSINESS)
    public int deleteCategory(AuthoredUser user, Long id) {
        return categoryMapper.deleteById(id);
    }

    /**
     * 修改分类名称
     *
     * @param user     user
     * @param id       id
     * @param cateName 分类名称
     * @return 结果
     */
    @Transactional(transactionManager = BaseAudcDataSourceConfig.BASE_AUDC_DATASOURCE_TRANSACTION_MANAGER_BUSINESS)
    public int renameCategory(AuthoredUser user, Long id, String cateName) {
        CategoryData renameData = new CategoryData();
        renameData.setId(id);

        renameData.setModifyDate(LocalDateTime.now());
        renameData.setModifyUserId(user.getUserId());
        renameData.setModifyUserName(user.getUserName());
        renameData.setCateName(cateName);
        return categoryMapper.updateById(renameData);
    }


    /**
     * 判断当前分类名称是否存在
     *
     * @param user 用户信息
     * @param dto  分类信息
     * @return boolean
     */
    public boolean isCategoryExist(AuthoredUser user, CategoriesDto dto) {
        if (StringUtils.isEmpty(dto.getCateName()) || "默認分組".equals(dto.getCateName()) || "Default Grouping".equals(dto.getCateName()) || "默认分组".equals(dto.getCateName())) {
            return true;
        }
        QueryWrapper<CategoryData> cateQuery = new QueryWrapper<>();
        cateQuery.eq("cate_name", dto.getCateName());
        cateQuery.eq("parent_id", Objects.isNull(dto.getParentId()) ? -1 : dto.getParentId());
        cateQuery.eq("plat", dto.getPlat());
        cateQuery.eq("user_id", user.getUserId());
        cateQuery.eq("tenant_id", user.getTenantId());
        Long result = categoryMapper.selectCount(cateQuery);
        return result > 0L;
    }

    /**
     * 根据分类id查询分类的详情
     *
     * @param id id
     * @return CategoryData
     */
    public CategoryData detail(Long id) {
        return categoryMapper.selectById(id);
    }


    /**
     * 设置分类为置顶
     *
     * @param user   用户信息
     * @param plat   所属模块，0:基础资料录入 1：报表，后续有新增则依次递增
     * @param cateId 分类id
     * @return 操作结果
     */
    @Transactional(transactionManager = BaseAudcDataSourceConfig.BASE_AUDC_DATASOURCE_TRANSACTION_MANAGER_BUSINESS)
    public boolean setCateTop(AuthoredUser user, Integer plat, Long cateId) {
        UpdateWrapper<CategoryData> updateWrapper = new UpdateWrapper<>();
        updateWrapper.eq("user_id", user.getUserId());
        updateWrapper.eq("tenant_id", user.getTenantId());
        updateWrapper.eq("plat", plat);
        updateWrapper.set("is_top", NOT_TOP_STATUS);
        categoryMapper.update(null, updateWrapper);

        UpdateWrapper<CategoryData> topWrapper = new UpdateWrapper<>();
        topWrapper.eq("id", cateId);
        topWrapper.set("is_top", TOP_STATUS);
        int result = categoryMapper.update(null, topWrapper);
        return result >= 1;
    }

    /**
     * 设置置顶的分组收起或者展开
     *
     * @param user   用户信息
     * @param cateId 分类id
     * @param flag   0：隐藏 1：展开
     * @return 操作信息
     */
    @Transactional(transactionManager = BaseAudcDataSourceConfig.BASE_AUDC_DATASOURCE_TRANSACTION_MANAGER_BUSINESS)
    public boolean cateUnfold(AuthoredUser user, Long cateId, Integer flag) {
        UpdateWrapper<CategoryData> updateWrapper = new UpdateWrapper<>();
        updateWrapper.eq("id", cateId);
        updateWrapper.set("unfold", flag);
        int result = categoryMapper.update(null, updateWrapper);
        return result >= 1;

    }


    /**
     * 设置置顶的默认分组收起或者展开
     *
     * @param user 用户信息
     * @param plat 所属模块，0:基础资料录入 1：报表，后续有新增则依次递增
     * @param flag 0：隐藏 1：展开
     * @return 操作信息
     */
    public boolean cateDefaultUnfold(AuthoredUser user, Integer plat, Integer flag) {
        //设置常用的默认分组的展开收起状态
        String userId = user.getUserId();
        String tenantId = user.getTenantId();
        String queryKey = CATE_DEFAULT_KEY_PREFIX + plat + ":" + tenantId + ":" + userId;
        try {
            stringRedisTemplate.opsForValue().set(queryKey, String.valueOf(flag), 7, TimeUnit.DAYS);
        } catch (Exception e) {
            logger.error("insert default cate unfold info error for user:" + user.getUserId());
            return false;
        }
        return true;
    }

    /**
     * 从redis中获取默认的展开设置信息
     *
     * @param user user
     * @param plat plat
     * @return String
     */
    public String queryDefaultUnfold(AuthoredUser user, Integer plat) {
        String userId = user.getUserId();
        String tenantId = user.getTenantId();
        String queryKey = CATE_DEFAULT_KEY_PREFIX + plat + ":" + tenantId + ":" + userId;
        String retObj = stringRedisTemplate.opsForValue().get(queryKey);
        if (!StringUtils.isBlank(retObj)) {
            return retObj;
        } else {
            return "";
        }
    }
}
